// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Config/hkProductFeatures.h>

class hkTaskQueue;

/// This is the hi-level interface to the collision geometry optimizer.
/// Note: This is NOT meant as a mesh simplifier.
struct HK_EXPORT_COMMON hkgpCgo
{
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, hkgpCgo );

    /// Tracker.
    struct Tracker
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, Tracker );

        virtual ~Tracker() {}

        /// Called before CGO is applied to the geometry, returns true is changes where applied and another pass is needed, false otherwise.
        virtual bool    prePass( struct hkGeometry& geometry ) { return false; }

        /// Called after CGO has been applied to the geometry, returns true is changes where applied and another pass is needed, false otherwise.
        virtual bool    postPass( struct hkGeometry& geometry ) { return false; }

        /// Called when an edge was flipped.
        virtual void    edgeFlip( int vertexFrom, int vertexTo ) {}

        /// Called when an edge was collapsed.
        virtual void    edgeCollapse( int vertexFrom, int vertexTo, const hkVector4& position, hkReal error ) {}

        /// Called when a vertex is deleted.
        virtual void    vertexDelete( int vertexIndex ) {}

        /// Called to set cluster IDs on triangles.
        virtual void    setClusterId(_In_reads_(numTriangles) const int* triangleIndices, int numTriangles, int cid ) {}
    };

    /// Configuration.
    struct HK_EXPORT_COMMON Config : public hkReferencedObject
    {
        HK_DECLARE_CLASS( Config, New, Reflect );

        /// Defaults
        Config() : hkReferencedObject()
        {
            m_semantic = VS_NONE;
            m_combinator = VC_MIN;
            m_maxDistance = HK_REAL_MAX;
            m_maxShrink = hkReal( 0.0f );
            m_maxAngle = HK_REAL_MAX;
            m_minEdgeRatio = hkReal( 0.0f );
            m_maxAngleDrift = hkReal( HK_REAL_PI );
            m_weldDistance = hkReal( 0.0f );
            m_updateThreshold = hkReal( 0.0f );
            m_degenerateTolerance = hkReal( 1e-7 );
            m_flatAngle = hkReal( 0.0f );
            m_maxVertices = 0;
            m_inverseOrientation = false;
            m_proportionalShrinking = false;
            m_multiPass = false;
            m_protectNakedBoundaries = false;
            m_protectMaterialBoundaries = false;
            m_decimateComponents = false;
            m_taskQueue = HK_NULL;
            m_tracker = HK_NULL;

            m_solverAccuracy = SA_NORMAL;
            m_minDistance = hkReal( -0.0001f );
            m_minConvergence = hkReal( 0.00001f );
            m_project = true;
            m_buildClusters = true;
            m_useAccumulatedError = false;
            m_useLegacySolver = false;
        }

        /// Semantic of the W component of vertices.
        enum VertexSemantic
        {
            VS_NONE,        ///< No semantic.
            VS_WEIGHT,      ///< Max distance fraction as in 'Config::m_minDistance + (Config::m_maxDistance - Config::m_minDistance) * W'.
            VS_DISTANCE,    ///< Max distance.
            VS_FACTOR,      ///< Error * W.
            VS_OFFSET       ///< Error + W.
        };

        /// Combinator for the W components of vertices.
        enum VertexCombinator
        {
            VC_MIN,         ///< Min of W0 and W1.
            VC_MAX,         ///< Max of W0 and W1.
            VC_MEAN         ///< (W0 + W1) / 2.
        };

        /// Solver accuracy setting.
        enum SolverAccuracy
        {
            SA_FAST = 4,
            SA_NORMAL = 8,
            SA_ACCURATE = 16,
            SA_HIGH = 64,
        };

        // Optimization parameters.

        VertexSemantic      m_semantic;                 ///< Vertex semantic.
        VertexCombinator    m_combinator;               ///< Vertex combinator.
        hkReal              m_maxDistance;              ///< Maximum allowed distance to the mesh surface (unit: distance).
        hkReal              m_maxShrink;                ///< Maximum allowed shrinking (unit: fraction between 0 and 1).
        hkReal              m_maxAngle;                 ///< Protect concave edges that form an angle greater than this value (unit: radian between 0 and pi).
        hkReal              m_minEdgeRatio;             ///< Minimum allowed triangle edge length ratio (min edge length divided by max edge length, default 0).
        hkReal              m_maxAngleDrift;            ///< Maximum allowed angle drift (unit: radian between 0 and pi).
        hkReal              m_weldDistance;             ///< Automatically weld vertices below or equal to this distance (unit: distance), set to a negative number to disable.
        hkReal              m_updateThreshold;          ///< Perform multiple optimization up to this threshold (delta between current and previous error).
        hkReal              m_degenerateTolerance;      ///< Tolerance used to classify triangles as degenerated or not, see hkcdTriangleUtil::isDegenerate. (default: 1e-7).
        hkReal              m_flatAngle;                ///< Remove vertices if surrounding triangles form an angle less than this value (default: 0, unit radian between 0 an pi).
        int                 m_maxVertices;              ///< Maximum allowed number of vertices.
        hkBool              m_inverseOrientation;       ///< Invert the mesh orientation, thus shrink instead of expanding.
        hkBool              m_proportionalShrinking;    ///< Shrinking is proportional to error.
        hkBool              m_multiPass;                ///< Allow repeated optimization until no further simplification can be applied.
        hkBool              m_protectNakedBoundaries;   ///< Do not optimize open boundaries if true.
        hkBool              m_protectMaterialBoundaries;///< Do not optimize across material boundaries if true.
        hkBool              m_decimateComponents;       ///< Decimate components smaller the current error.

        /// Create parallel tasks in this queue if set.
        hkTaskQueue*        m_taskQueue;                //+nosave

        /// Tracker.
        Tracker*            m_tracker;                  //+nosave

        // Internals.
        SolverAccuracy  m_solverAccuracy;
        hkReal          m_minDistance;
        hkReal          m_minConvergence;
        hkBool          m_project;
        hkBool          m_buildClusters;
        hkBool          m_useAccumulatedError;
        hkBool          m_useLegacySolver;
    };

    /// Progress interface.
    struct IProgress
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, IProgress );

        virtual         ~IProgress() {}

        /// Called after before optimization step, return if the process should continue (true) or not (false).
        virtual bool    step( const Config& config, hkReal error, int numVertices, int numTriangles ) = 0;
    };

    /// ClusterData, use for analysis only.
    struct ClusterData
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, ClusterData );

        HK_INLINE   bool    operator<( const ClusterData& other ) const { return m_errorRange[ 0 ] < other.m_errorRange[ 0 ]; }

        hkReal          m_errorRange[ 2 ];
        int             m_numVertices;
    };

    /// Optimize geometry.
    /// Note: If a non-null array of clusters if passed, optimize will only perform a analysis and fill clusters->getSize() clusters data but do not changes the geometry.
    static void HK_CALL optimize( const Config& config, struct hkGeometry& geometry, _Inout_opt_ IProgress* progress = HK_NULL, _Inout_opt_ hkArray<ClusterData>* clusters = HK_NULL );

    /// Optimize a convex hull.
    static void HK_CALL optimize( const Config& config, class hkgpConvexHull& hull, _Inout_opt_ IProgress* progress = HK_NULL );

    // Utility class
protected:
    hkgpCgo() {}
    hkgpCgo( const hkgpCgo& ) {}
};
HK_REFLECT_ENUM( HK_EXPORT_COMMON, hkgpCgo::Config::VertexSemantic );
HK_REFLECT_ENUM( HK_EXPORT_COMMON, hkgpCgo::Config::VertexCombinator );
HK_REFLECT_ENUM( HK_EXPORT_COMMON, hkgpCgo::Config::SolverAccuracy );

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
