// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0


#pragma once

// this:  #include <Common/Internal/ConvexHull/hkGeometryUtility.h>


#include <Common/Base/Types/Geometry/hkStridedVertices.h>
#include <Common/Base/Types/Geometry/hkGeometry.h>

class hkSphere;
class hkGeomHull;


/// Used to specify any modifications to the behavior of the convex hull builder.
enum hkGeomConvexHullMode
{
    ///
    HK_GEOM_CONVEXHULL_MODE_INVALID,
    /// Default: fast mode.
    HK_GEOM_CONVEXHULL_MODE_FAST,
    /// Should not be used when the algorithm is being called at runtime,
    /// and really only should be called offline by the hkpConvexPieceMeshBuilder
    /// utility, because it can be very slow and only offers accuracy improvements
    /// in a limited number of cases, such as when the input is nearly completely
    /// a 2d object with a large number of vertices spread over a large distance.
    HK_GEOM_CONVEXHULL_MODE_ACCURATE_BUT_SLOW,
    ///
    HK_GEOM_CONVEXHULL_MODE_MAX_ID
};

/// Used to specify any modifications to the behaviour of the OBB builder.
enum hkGeomObbMode
{
    ///
    HK_GEOM_OBB_MODE_INVALID,
    /// Default: normal mode. The mimics the calculations used prior to Havok 3.3, and
    /// is used in the filter pipeline.
    HK_GEOM_OBB_MODE_DEFAULT,
    /// Produces tighter-fitting bounding boxes, but may be considerably slower.
    /// The algorithm uses the normal result as a starting guess, and then iteratively
    /// improves the box to reduce the total volume.
    HK_GEOM_OBB_MODE_BETTER_FIT_BUT_SLOWER,
    ///
    HK_GEOM_OBB_MODE_MAX_ID
};

///
/// Convex hull generator: Calculated the convex hull of a set of vertices in 3-space.
///
/// The convex hull algorithm is used in several places in our current product:
///  - Generation of plane equations for convex vertices shapes. These are used for ray-casting.
///  - Generation of visualization code for display of convex vertices shapes (in the visual debugger)
///  - Generation of a hull for calculating an inertia tensor for convex vertices shapes.
///
class HK_EXPORT_COMMON hkGeometryUtility
{
public:
        //
        // GEOMETRY CONVEXITY AND CREATION FUNCTIONS
        //

        /// Calculates the convex hull of the points in stridedVertsIn.
    static void HK_CALL createConvexGeometry( const hkStridedVertices& stridedVertsIn, hkGeometry& geometryOut);

        /// Calculates the convex hull of the points in stridedVertsIn and returns the vertices used, the plane equations generated and the geometry of hull.
    static void HK_CALL createConvexGeometry( const hkStridedVertices& stridedVertsIn, hkGeometry& geometryOut, hkArray<hkVector4>& planeEquationsOut);

        /// Calculates the vertices of the convex hull at the inner intersection of the specified plane equations. Only one hull is found, and
        /// more plane equations than the minimum necessary can be specified.
    static void HK_CALL createVerticesFromPlaneEquations(  const hkArray<hkVector4>& planeEquationsIn, hkArray<hkVector4>& vertsOut );

        /// Calculates the geometry of the convex hull whose vertices are the intersections of the planeEquationsIn.
    static void HK_CALL createGeometryFromPlaneEquations(  const hkArray<hkVector4>& planeEquationsIn, hkGeometry& geometryOut );


        // Calculate axis from covariance matrix
    static hkBool HK_CALL getAxesFromCovariance(hkMatrix3 &covArray, hkVector4 &axis0, hkVector4 &axis1, hkVector4 &axis2);


        //
        // OTHER UTILITY FUNCTIONS
        //

        /// A utility class for creating oriented bounding boxes from a point cloud.
        /// This is useful if you want to create primitives from arbitrary geometry.
        /// The OBB utility returns a position for the centre of the box, a set of
        /// 3 axes representing the orientation of the box, and a set of half extents
        /// representing the scale of the box.
    static void HK_CALL calcObb( const hkStridedVertices& stridedVertsIn, hkVector4& halfExtents, hkTransform& transform, hkGeomObbMode mode = HK_GEOM_OBB_MODE_DEFAULT);

    /// Calculates the best-fitting cylinder for a point cloud
    static void HK_CALL calcBestCylinder( const hkArray<hkVector4>& vertsIn, hkVector4Parameter obbHalfExtentsIn, const hkTransform& obbTransformIn, hkVector4& pointAOut, hkVector4& pointBOut, hkReal& radiusOut);

        /// This method duplicates the functionality of the deprecated hkGeometryUtils::minSphere() method.
        /// This computes a minimal bounding sphere for the specified point cloud.  The resulting sphere includes a
        /// small additional margin proportional to the AABB extents of the point cloud.
    static hkSphere HK_CALL createMinBoundingSphere( const hkStridedVertices& stridedVertsIn);

        /// deprecated - use createMinBoundingSphere( const hkStridedVertices& stridedVertsIn) instead
    static hkSphere HK_CALL createMinBoundingSphere( const hkStridedVertices& stridedVertsIn, const hkReal unused);

        /// Simple function to push out given planes by the desired distance. It just alters
        /// the w component (D) of each of the plane equations by the given amount (can be negative to shrink)
    static void HK_CALL expandPlanes(  hkArray<hkVector4>& planeEquations, hkReal byDistance );

};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
