// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY       : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Container/ArrayView/hkArrayView.h>
#include <Common/Base/Types/Color/hkColor.h>
#include <Common/ImageUtilities/Image/hkImage.h>

class hkImageFilter;

/// Image Processing.
class HK_EXPORT_COMMON hkImageProcessing
{
public:
    /// Mip map generation options
    struct MipMapOptions
    {
        MipMapOptions()
        {
            m_filter = HK_NULL;
            m_renormalizeNormals = false;
            m_preserveCoverage = false;
            m_alphaThreshold = 0.5f;

            m_addressModeU = hkImageAddressMode::CLAMP;
            m_addressModeV = hkImageAddressMode::CLAMP;

            m_borderColor = hkColors::Black;
        }

        /// The filter to use for mipmap generation. Defaults to bilinear filtering (Triangle filter) if none is given.
        const hkImageFilter* m_filter;

        /// Rescale RGB components to unit length.
        bool m_renormalizeNormals;

        /// If true, the alpha values are scaled to preserve the average coverage when alpha testing is enabled,
        bool m_preserveCoverage;

        /// The alpha test threshold to use when m_preserveCoverage == true.
        float m_alphaThreshold;

        /// The address mode for samples when filtering outside of the image dimensions in the horizontal direction.
        hkImageAddressMode::Enum m_addressModeU;

        /// The address mode for samples when filtering outside of the image dimensions in the vertical direction.
        hkImageAddressMode::Enum m_addressModeV;

        /// The border color if texture address mode equals BORDER.
        hkColorf m_borderColor;
    };

    /// Scales the image.
    static hkResult scale(const hkImage& source, hkImage& target, hkUint32 width, hkUint32 height, _In_opt_ const hkImageFilter* filter = HK_NULL,
        hkImageAddressMode::Enum addressModeU = hkImageAddressMode::CLAMP, hkImageAddressMode::Enum addressModeV = hkImageAddressMode::CLAMP,
        hkColorfParameter borderColor = hkColors::Black);

    /// Flips the image vertically.
    static void flipVertical(hkImage& image);

    /// Genererates the mip maps for the image. The input texture must be in hkImageFormat::R32_G32_B32_A32_FLOAT
    static void generateMipMaps( const hkImage& source, hkImage& target, const MipMapOptions& options);

    /// Assumes that the Red and Green components of an image contain XY of an unit length normal and reconstructs the Z component into B
    static void reconstructNormalZ( hkImage& source );

    /// Renormalizes a normal map to unit length.
    static void renormalizeNormalMap(hkImage& image);

    /// Adjust the roughness in lower mip levels so it maintains the same look from all distances.
    static void adjustRoughness( hkImage& roughnessMap, const hkImage& normalMap );

    /// Converts the image to cubemap.
    /// The input image must be in hkImageFormat::R32_G32_B32_A32_FLOAT.
    ///
    /// Supported input image layouts:
    ///   horizontal cross
    ///   vertical cross
    ///   horizontal strip
    ///   vertical strip
    ///   spherical map (a.k.a. lattitude-longitude or cylindrical)
    static void convertToCubemap(const hkImage& src, hkImage& dst);

    /// Changes the exposure in an HDR image by 2^bias
    static void changeExposure(hkImage& image, hkFloat32 bias);
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
