// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY       : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

// Represents a function used for filtering an image.
class HK_EXPORT_COMMON hkImageFilter
{
public:
    // Samples the filter function at a single point. Note that the distribution isn't necessarily normalized.
    virtual hkSimdFloat32 samplePoint(hkSimdFloat32Parameter x) const = 0;

    // Returns the width of the filter; outside of the interval [-width, width], the filter function is always zero.
    hkSimdFloat32 getWidth() const;

protected:
    hkImageFilter(float width);

private:
    hkSimdFloat32 m_width;
};

// Box filter
class HK_EXPORT_COMMON hkImageFilterBox : public hkImageFilter
{
public:
    hkImageFilterBox(float width = 0.5f);

    virtual hkSimdFloat32 samplePoint(hkSimdFloat32Parameter x) const HK_OVERRIDE;
};

// Triangle filter
class HK_EXPORT_COMMON hkImageFilterTriangle : public hkImageFilter
{
public:
    hkImageFilterTriangle(float width = 1.0f);

    virtual hkSimdFloat32 samplePoint(hkSimdFloat32Parameter x) const HK_OVERRIDE;
};

// Kaiser-windowed sinc filter
class HK_EXPORT_COMMON hkImageFilterSincWithKaiserWindow : public hkImageFilter
{
public:
    // Construct a sinc filter with a Kaiser window of the given window width and beta parameter.
    // Note that the beta parameter (equaling alpha * pi in the mathematical definition of the Kaiser window) is often incorrectly alpha by other filtering tools.
    hkImageFilterSincWithKaiserWindow(float windowWidth = 3.0f, float beta = 4.0f);

    virtual hkSimdFloat32 samplePoint(hkSimdFloat32Parameter x) const HK_OVERRIDE;

private:
    hkSimdFloat32 m_beta;
    hkSimdFloat32 m_invBesselBeta;
};

class hkImageFilterWeightsRange;

// Precomputes the required filter weights for rescaling a sequence of image samples.
class HK_EXPORT_COMMON hkImageFilterWeights
{
public:
    // Precompute the weights for the given filter for scaling between the given number of samples.
    hkImageFilterWeights(const hkImageFilter& filter, hkUint32 srcSamples, hkUint32 dstSamples);

    // Returns the number of weights.
    hkUint32 getNumWeights() const;

    // Returns the weight used for the source sample getFirstSourceSampleIndex(dstSampleIndex) + weightIndex
    hkSimdFloat32 getWeight(hkUint32 dstSampleIndex, hkUint32 weightIndex) const;

    // Returns the index of the first source sample that needs to be weighted to evaluate the destination sample
    HK_INLINE hkInt32 getFirstSourceSampleIndex(hkUint32 dstSampleIndex) const;

    // Returns a range which will infinitely loop over all weights
    hkImageFilterWeightsRange getWeightsRange() const;

private:
    hkArray<float> m_weights;
    hkSimdFloat32 m_widthInSourceSpace;
    hkSimdFloat32 m_sourceToDestScale;
    hkSimdFloat32 m_destToSourceScale;
    hkUint32 m_numWeights;
    hkUint32 m_dstSamplesReduced;
};

class HK_EXPORT_COMMON hkImageFilterWeightsRange
{
    friend class hkImageFilterWeights;
public:
    HK_INLINE hkImageFilterWeightsRange( const hkImageFilterWeightsRange& other );
    // returns the element currently at the front of the range
    HK_INLINE hkSimdFloat32 front() const;

    // moves range one element forward
    HK_INLINE void popFront();

private:
    HK_INLINE hkImageFilterWeightsRange( hkArrayView<float> m_weights );

    hkArrayView<float> m_weights;
    hkUint32 m_currentWeightIndex;
};

#include <Common/ImageUtilities/Processing/hkImageFilter.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
