// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY       : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/ImageUtilities/hkImageUtilities.h>

#include <Common/ImageUtilities/Processing/hkImageFilter.h>

hkSimdFloat32 hkImageFilter::getWidth() const
{
    return m_width;
}

hkImageFilter::hkImageFilter(float width)
    : m_width(hkSimdFloat32::fromFloat(width))
{

}

hkImageFilterBox::hkImageFilterBox(float width) : hkImageFilter(width)
{

}

hkSimdFloat32 hkImageFilterBox::samplePoint(hkSimdFloat32Parameter x) const
{
    hkSimdFloat32 absX;
    absX.setAbs(x);

    hkSimdFloat32 result;
    result.setSelect(absX.lessEqual(getWidth()), hkSimdFloat32_1, hkSimdFloat32_0);
    return result;
}

hkImageFilterTriangle::hkImageFilterTriangle(float width) : hkImageFilter(width)
{

}

hkSimdFloat32 hkImageFilterTriangle::samplePoint(hkSimdFloat32Parameter x) const
{
    hkSimdFloat32 absX;
    absX.setAbs(x);

    hkSimdFloat32 width = getWidth();

    hkSimdFloat32 result;
    result.setSelect(absX.lessEqual(width), width - absX, hkSimdFloat32_0);
    return result;
}

static hkSimdFloat32 sinc(hkSimdFloat32Parameter x)
{
    hkSimdFloat32 absX;
    absX.setAbs(x);

    // Use Taylor expansion for small values to avoid division
    if (absX < hkSimdFloat32_Eps)
    {
        // sin(x) / x = (x - x^3/6 + x^5/120 - ...) / x = 1 - x^2/6 + x^4/120 - ...
        return hkSimdFloat32_1 - x * x * hkSimdFloat32_Inv6;
    }
    else
    {
        return hkSimdReal::fromFloat(hkMath::sin(x.getReal())) / x;
    }
}

static hkSimdFloat32 modifiedBessel0(hkSimdFloat32Parameter x)
{
    // Implementation as I0(x) = sum((1/4 * x * x) ^ k / (k!)^2, k, 0, inf), see http://mathworld.wolfram.com/ModifiedBesselFunctionoftheFirstKind.html

    hkSimdFloat32 sum = hkSimdFloat32_1;

    hkSimdFloat32 xSquared = x * x * hkSimdFloat32_Inv4;

    hkSimdFloat32 currentTerm = xSquared;

    for (hkUint32 i = 2; currentTerm > hkSimdFloat32_Eps; ++i)
    {
        sum += currentTerm;
        currentTerm *= xSquared / hkSimdFloat32::fromInt32(i * i);
    }

    return sum;
}

hkImageFilterSincWithKaiserWindow::hkImageFilterSincWithKaiserWindow(float width, float beta)
    : hkImageFilter(width)
    , m_beta(hkSimdFloat32::fromFloat(beta))
    , m_invBesselBeta(hkSimdFloat32_1 / modifiedBessel0(m_beta))
{

}

hkSimdFloat32 hkImageFilterSincWithKaiserWindow::samplePoint(hkSimdFloat32Parameter x) const
{
    hkSimdFloat32 scaledX = x / getWidth();

    hkSimdFloat32 xSq = hkSimdFloat32_1 - scaledX * scaledX;

    if (xSq.isLessEqualZero())
    {
        return hkSimdFloat32_0;
    }
    else
    {
        return sinc(x * hkSimdFloat32_Pi) * modifiedBessel0(m_beta * xSq.sqrt()) * m_invBesselBeta;
    }
}

hkImageFilterWeights::hkImageFilterWeights(const hkImageFilter& filter, hkUint32 srcSamples, hkUint32 dstSamples)
{
    // Filter weights repeat after the common phase
    hkUint32 commonPhase = hkMath::greatestCommonDivisor(srcSamples, dstSamples);

    srcSamples /= commonPhase;
    dstSamples /= commonPhase;

    m_dstSamplesReduced = dstSamples;

    m_sourceToDestScale = hkSimdFloat32::fromFloat(float(dstSamples) / float(srcSamples));
    m_destToSourceScale = hkSimdFloat32::fromFloat(float(srcSamples) / float(dstSamples));

    hkSimdFloat32 filterScale, invFilterScale;

    if (dstSamples > srcSamples)
    {
        // When upsampling, reconstruct the source by applying the filter in source space and resampling
        filterScale = hkSimdFloat32_1;
        invFilterScale = hkSimdFloat32_1;
    }
    else
    {
        // When downsampling, widen the filter in order to narrow its frequency spectrum, which effectively combines reconstruction + low-pass filter
        filterScale = m_destToSourceScale;
        invFilterScale = m_sourceToDestScale;
    }

    m_widthInSourceSpace = filter.getWidth() * filterScale;

    m_numWeights = hkUint32(hkMath::ceil(m_widthInSourceSpace.getReal() * 2.0f)) + 1;

    m_weights.setSize(dstSamples * m_numWeights);

    for (hkUint32 dstSample = 0; dstSample < dstSamples; ++dstSample)
    {
        hkSimdFloat32 dstSampleInSourceSpace = (hkSimdFloat32::fromInt32(dstSample) + hkSimdFloat32_Half) * m_destToSourceScale;

        hkInt32 firstSourceIdx = getFirstSourceSampleIndex(dstSample);

        hkSimdFloat32 totalWeight = hkSimdFloat32_0;

        for (hkUint32 weightIdx = 0; weightIdx < m_numWeights; ++weightIdx)
        {
            hkSimdFloat32 sourceSample = hkSimdFloat32::fromInt32(firstSourceIdx + hkInt32(weightIdx)) + hkSimdFloat32_Half;

            hkSimdFloat32 weight = filter.samplePoint((dstSampleInSourceSpace - sourceSample) * invFilterScale);
            totalWeight += weight;
            m_weights[dstSample * m_numWeights + weightIdx] = weight.getReal();
        }

        // Normalize weights
        hkSimdFloat32 invWeight = hkSimdFloat32_1 / totalWeight;

        for (hkUint32 weightIdx = 0; weightIdx < m_numWeights; ++weightIdx)
        {
            m_weights[dstSample * m_numWeights + weightIdx] *= invWeight.getReal();
        }
    }
}

hkUint32 hkImageFilterWeights::getNumWeights() const
{
    return m_numWeights;
}

hkSimdFloat32 hkImageFilterWeights::getWeight(hkUint32 dstSampleIndex, hkUint32 weightIndex) const
{
    HK_ASSERT(0x3dc03799, weightIndex < m_numWeights, "Invalid weight index {} (should be < {})", weightIndex, m_numWeights);

    return hkSimdFloat32::fromFloat(m_weights[(dstSampleIndex % m_dstSamplesReduced) * m_numWeights + weightIndex]);
}

hkImageFilterWeightsRange hkImageFilterWeights::getWeightsRange() const
{
    return hkImageFilterWeightsRange( m_weights );
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
