// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM       : ALL
// PRODUCT        : COMMON
// VISIBILITY     : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/ImageUtilities/hkImageUtilities.h>

#include <Common/ImageUtilities/Image/hkImageFormat.h>

namespace HK_UNITY_ANONYMOUS_NAMESPACE
{

    typedef enum DXGI_FORMAT
    {
        DXGI_FORMAT_UNKNOWN = 0,
        DXGI_FORMAT_R32G32B32A32_TYPELESS = 1,
        DXGI_FORMAT_R32G32B32A32_FLOAT = 2,
        DXGI_FORMAT_R32G32B32A32_UINT = 3,
        DXGI_FORMAT_R32G32B32A32_SINT = 4,
        DXGI_FORMAT_R32G32B32_TYPELESS = 5,
        DXGI_FORMAT_R32G32B32_FLOAT = 6,
        DXGI_FORMAT_R32G32B32_UINT = 7,
        DXGI_FORMAT_R32G32B32_SINT = 8,
        DXGI_FORMAT_R16G16B16A16_TYPELESS = 9,
        DXGI_FORMAT_R16G16B16A16_FLOAT = 10,
        DXGI_FORMAT_R16G16B16A16_UNORM = 11,
        DXGI_FORMAT_R16G16B16A16_UINT = 12,
        DXGI_FORMAT_R16G16B16A16_SNORM = 13,
        DXGI_FORMAT_R16G16B16A16_SINT = 14,
        DXGI_FORMAT_R32G32_TYPELESS = 15,
        DXGI_FORMAT_R32G32_FLOAT = 16,
        DXGI_FORMAT_R32G32_UINT = 17,
        DXGI_FORMAT_R32G32_SINT = 18,
        DXGI_FORMAT_R32G8X24_TYPELESS = 19,
        DXGI_FORMAT_D32_FLOAT_S8X24_UINT = 20,
        DXGI_FORMAT_R32_FLOAT_X8X24_TYPELESS = 21,
        DXGI_FORMAT_X32_TYPELESS_G8X24_UINT = 22,
        DXGI_FORMAT_R10G10B10A2_TYPELESS = 23,
        DXGI_FORMAT_R10G10B10A2_UNORM = 24,
        DXGI_FORMAT_R10G10B10A2_UINT = 25,
        DXGI_FORMAT_R11G11B10_FLOAT = 26,
        DXGI_FORMAT_R8G8B8A8_TYPELESS = 27,
        DXGI_FORMAT_R8G8B8A8_UNORM = 28,
        DXGI_FORMAT_R8G8B8A8_UNORM_SRGB = 29,
        DXGI_FORMAT_R8G8B8A8_UINT = 30,
        DXGI_FORMAT_R8G8B8A8_SNORM = 31,
        DXGI_FORMAT_R8G8B8A8_SINT = 32,
        DXGI_FORMAT_R16G16_TYPELESS = 33,
        DXGI_FORMAT_R16G16_FLOAT = 34,
        DXGI_FORMAT_R16G16_UNORM = 35,
        DXGI_FORMAT_R16G16_UINT = 36,
        DXGI_FORMAT_R16G16_SNORM = 37,
        DXGI_FORMAT_R16G16_SINT = 38,
        DXGI_FORMAT_R32_TYPELESS = 39,
        DXGI_FORMAT_D32_FLOAT = 40,
        DXGI_FORMAT_R32_FLOAT = 41,
        DXGI_FORMAT_R32_UINT = 42,
        DXGI_FORMAT_R32_SINT = 43,
        DXGI_FORMAT_R24G8_TYPELESS = 44,
        DXGI_FORMAT_D24_UNORM_S8_UINT = 45,
        DXGI_FORMAT_R24_UNORM_X8_TYPELESS = 46,
        DXGI_FORMAT_X24_TYPELESS_G8_UINT = 47,
        DXGI_FORMAT_R8G8_TYPELESS = 48,
        DXGI_FORMAT_R8G8_UNORM = 49,
        DXGI_FORMAT_R8G8_UINT = 50,
        DXGI_FORMAT_R8G8_SNORM = 51,
        DXGI_FORMAT_R8G8_SINT = 52,
        DXGI_FORMAT_R16_TYPELESS = 53,
        DXGI_FORMAT_R16_FLOAT = 54,
        DXGI_FORMAT_D16_UNORM = 55,
        DXGI_FORMAT_R16_UNORM = 56,
        DXGI_FORMAT_R16_UINT = 57,
        DXGI_FORMAT_R16_SNORM = 58,
        DXGI_FORMAT_R16_SINT = 59,
        DXGI_FORMAT_R8_TYPELESS = 60,
        DXGI_FORMAT_R8_UNORM = 61,
        DXGI_FORMAT_R8_UINT = 62,
        DXGI_FORMAT_R8_SNORM = 63,
        DXGI_FORMAT_R8_SINT = 64,
        DXGI_FORMAT_A8_UNORM = 65,
        DXGI_FORMAT_R1_UNORM = 66,
        DXGI_FORMAT_R9G9B9E5_SHAREDEXP = 67,
        DXGI_FORMAT_R8G8_B8G8_UNORM = 68,
        DXGI_FORMAT_G8R8_G8B8_UNORM = 69,
        DXGI_FORMAT_BC1_TYPELESS = 70,
        DXGI_FORMAT_BC1_UNORM = 71,
        DXGI_FORMAT_BC1_UNORM_SRGB = 72,
        DXGI_FORMAT_BC2_TYPELESS = 73,
        DXGI_FORMAT_BC2_UNORM = 74,
        DXGI_FORMAT_BC2_UNORM_SRGB = 75,
        DXGI_FORMAT_BC3_TYPELESS = 76,
        DXGI_FORMAT_BC3_UNORM = 77,
        DXGI_FORMAT_BC3_UNORM_SRGB = 78,
        DXGI_FORMAT_BC4_TYPELESS = 79,
        DXGI_FORMAT_BC4_UNORM = 80,
        DXGI_FORMAT_BC4_SNORM = 81,
        DXGI_FORMAT_BC5_TYPELESS = 82,
        DXGI_FORMAT_BC5_UNORM = 83,
        DXGI_FORMAT_BC5_SNORM = 84,
        DXGI_FORMAT_B5G6R5_UNORM = 85,
        DXGI_FORMAT_B5G5R5A1_UNORM = 86,
        DXGI_FORMAT_B8G8R8A8_UNORM = 87,
        DXGI_FORMAT_B8G8R8X8_UNORM = 88,
        DXGI_FORMAT_R10G10B10_XR_BIAS_A2_UNORM = 89,
        DXGI_FORMAT_B8G8R8A8_TYPELESS = 90,
        DXGI_FORMAT_B8G8R8A8_UNORM_SRGB = 91,
        DXGI_FORMAT_B8G8R8X8_TYPELESS = 92,
        DXGI_FORMAT_B8G8R8X8_UNORM_SRGB = 93,
        DXGI_FORMAT_BC6H_TYPELESS = 94,
        DXGI_FORMAT_BC6H_UF16 = 95,
        DXGI_FORMAT_BC6H_SF16 = 96,
        DXGI_FORMAT_BC7_TYPELESS = 97,
        DXGI_FORMAT_BC7_UNORM = 98,
        DXGI_FORMAT_BC7_UNORM_SRGB = 99,
        DXGI_FORMAT_AYUV = 100,
        DXGI_FORMAT_Y410 = 101,
        DXGI_FORMAT_Y416 = 102,
        DXGI_FORMAT_NV12 = 103,
        DXGI_FORMAT_P010 = 104,
        DXGI_FORMAT_P016 = 105,
        DXGI_FORMAT_420_OPAQUE = 106,
        DXGI_FORMAT_YUY2 = 107,
        DXGI_FORMAT_Y210 = 108,
        DXGI_FORMAT_Y216 = 109,
        DXGI_FORMAT_NV11 = 110,
        DXGI_FORMAT_AI44 = 111,
        DXGI_FORMAT_IA44 = 112,
        DXGI_FORMAT_P8 = 113,
        DXGI_FORMAT_A8P8 = 114,
        DXGI_FORMAT_B4G4R4A4_UNORM = 115,
        DXGI_FORMAT_FORCE_UINT = 0xffffffff
    } DXGI_FORMAT;

    static struct hkImageFormatMetaData
    {
        hkImageFormatMetaData()
        {
            m_bitsPerPixel = 0;
            m_blockWidth = 1;
            m_blockHeight = 1;
            m_blockDepth = 1;

            m_numChannels = 0;

            hkString::memSet( m_bitsPerChannel, 0, sizeof( m_bitsPerChannel ) );
            hkString::memSet( m_channelMasks, 0, sizeof( m_channelMasks ) );

            m_isCompressed = false;
            m_isDepth = false;
            m_isStencil = false;

            m_dataType = hkImageFormat::DATA_TYPE_NONE;

            m_dxgiFormat = DXGI_FORMAT_UNKNOWN;
        }

        hkUint8 m_bitsPerPixel;
        hkUint8 m_blockWidth;
        hkUint8 m_blockHeight;
        hkUint8 m_blockDepth;

        hkUint8 m_numChannels;

        hkUint8 m_bitsPerChannel[hkImageFormat::CHANNEL_COUNT];
        hkUint32 m_channelMasks[hkImageFormat::CHANNEL_COUNT];

        bool m_isCompressed;
        bool m_isDepth;
        bool m_isStencil;

        hkImageFormat::DataType m_dataType;

        hkImageFormat::Enum m_asLinear;
        hkImageFormat::Enum m_asSrgb;

        hkUint32 m_dxgiFormat;

    } s_metaData[hkImageFormat::COUNT];     

    struct hkImageFormatMetaData_Init
    {
        hkImageFormatMetaData_Init()
        {
            initFormatLinear( hkImageFormat::R32_G32_B32_A32_FLOAT, hkImageFormat::DATA_TYPE_FLOAT, 128, 32, 32, 32, 32, 4, DXGI_FORMAT_R32G32B32A32_FLOAT );
            initFormatLinear( hkImageFormat::R32_G32_B32_A32_UNSIGNED, hkImageFormat::DATA_TYPE_UNSIGNED, 128, 32, 32, 32, 32, 4, DXGI_FORMAT_R32G32B32A32_UINT );
            initFormatLinear( hkImageFormat::R32_G32_B32_A32_SIGNED, hkImageFormat::DATA_TYPE_SIGNED, 128, 32, 32, 32, 32, 4, DXGI_FORMAT_R32G32B32A32_SINT );

            initFormatLinear( hkImageFormat::R32_G32_B32_FLOAT, hkImageFormat::DATA_TYPE_FLOAT, 96, 32, 32, 32, 0, 3, DXGI_FORMAT_R32G32B32_FLOAT );
            initFormatLinear( hkImageFormat::R32_G32_B32_UNSIGNED, hkImageFormat::DATA_TYPE_UNSIGNED, 96, 32, 32, 32, 0, 3, DXGI_FORMAT_R32G32B32_UINT );
            initFormatLinear( hkImageFormat::R32_G32_B32_SIGNED, hkImageFormat::DATA_TYPE_SIGNED, 96, 32, 32, 32, 0, 3, DXGI_FORMAT_R32G32B32_SINT );

            initFormatLinear( hkImageFormat::R32_G32_FLOAT, hkImageFormat::DATA_TYPE_FLOAT, 64, 32, 32, 0, 0, 2, DXGI_FORMAT_R32G32_FLOAT );
            initFormatLinear( hkImageFormat::R32_G32_UNSIGNED, hkImageFormat::DATA_TYPE_UNSIGNED, 64, 32, 32, 0, 0, 2, DXGI_FORMAT_R32G32_UINT );
            initFormatLinear( hkImageFormat::R32_G32_SIGNED, hkImageFormat::DATA_TYPE_SIGNED, 64, 32, 32, 0, 0, 2, DXGI_FORMAT_R32G32_SINT );

            initFormatLinear( hkImageFormat::R32_FLOAT, hkImageFormat::DATA_TYPE_FLOAT, 32, 32, 0, 0, 0, 1, DXGI_FORMAT_R32_FLOAT );
            initFormatLinear( hkImageFormat::R32_UNSIGNED, hkImageFormat::DATA_TYPE_UNSIGNED, 32, 32, 0, 0, 0, 1, DXGI_FORMAT_R32_UINT );
            initFormatLinear( hkImageFormat::R32_SIGNED, hkImageFormat::DATA_TYPE_SIGNED, 32, 32, 0, 0, 0, 1, DXGI_FORMAT_R32_SINT );

            initFormatLinear( hkImageFormat::R16_G16_B16_A16_FLOAT, hkImageFormat::DATA_TYPE_FLOAT, 64, 16, 16, 16, 16, 4, DXGI_FORMAT_R16G16B16A16_FLOAT );
            initFormatLinear( hkImageFormat::R16_G16_B16_A16_UNSIGNED, hkImageFormat::DATA_TYPE_UNSIGNED, 64, 16, 16, 16, 16, 4, DXGI_FORMAT_R16G16B16A16_UINT );
            initFormatLinear( hkImageFormat::R16_G16_B16_A16_SIGNED, hkImageFormat::DATA_TYPE_SIGNED, 64, 16, 16, 16, 16, 4, DXGI_FORMAT_R16G16B16A16_SINT );
            initFormatLinear( hkImageFormat::R16_G16_B16_A16_UNSIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 64, 16, 16, 16, 16, 4, DXGI_FORMAT_R16G16B16A16_UNORM );
            initFormatLinear( hkImageFormat::R16_G16_B16_A16_SIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_SIGNED_NORMALIZED, 64, 16, 16, 16, 16, 4, DXGI_FORMAT_R16G16B16A16_SNORM );

            initFormatLinear( hkImageFormat::R16_G16_FLOAT, hkImageFormat::DATA_TYPE_FLOAT, 32, 16, 16, 0, 0, 2, DXGI_FORMAT_R16G16_FLOAT );
            initFormatLinear( hkImageFormat::R16_G16_UNSIGNED, hkImageFormat::DATA_TYPE_UNSIGNED, 32, 16, 16, 0, 0, 2, DXGI_FORMAT_R16G16_UINT );
            initFormatLinear( hkImageFormat::R16_G16_SIGNED, hkImageFormat::DATA_TYPE_SIGNED, 32, 16, 16, 0, 0, 2, DXGI_FORMAT_R16G16_SINT );
            initFormatLinear( hkImageFormat::R16_G16_UNSIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 32, 16, 16, 0, 0, 2, DXGI_FORMAT_R16G16_UNORM );
            initFormatLinear( hkImageFormat::R16_G16_SIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_SIGNED_NORMALIZED, 32, 16, 16, 0, 0, 2, DXGI_FORMAT_R16G16_SNORM );

            initFormatLinear( hkImageFormat::R16_FLOAT, hkImageFormat::DATA_TYPE_FLOAT, 16, 16, 0, 0, 0, 1, DXGI_FORMAT_R16_FLOAT );
            initFormatLinear( hkImageFormat::R16_UNSIGNED, hkImageFormat::DATA_TYPE_UNSIGNED, 16, 16, 0, 0, 0, 1, DXGI_FORMAT_R16_UINT );
            initFormatLinear( hkImageFormat::R16_SIGNED, hkImageFormat::DATA_TYPE_SIGNED, 16, 16, 0, 0, 0, 1, DXGI_FORMAT_R16_SINT );
            initFormatLinear( hkImageFormat::R16_UNSIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 16, 16, 0, 0, 0, 1, DXGI_FORMAT_R16_UNORM );
            initFormatLinear( hkImageFormat::R16_SIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_SIGNED_NORMALIZED, 16, 16, 0, 0, 0, 1, DXGI_FORMAT_R16_SNORM );

            initFormatLinear( hkImageFormat::R8_G8_B8_A8_UNSIGNED, hkImageFormat::DATA_TYPE_UNSIGNED, 32, 8, 8, 8, 8, 4, DXGI_FORMAT_R8G8B8A8_UINT );
            initFormatLinear( hkImageFormat::R8_G8_B8_A8_SIGNED, hkImageFormat::DATA_TYPE_SIGNED, 32, 8, 8, 8, 8, 4, DXGI_FORMAT_R8G8B8A8_SINT );
            initFormatLinear( hkImageFormat::R8_G8_B8_A8_SIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_SIGNED_NORMALIZED, 32, 8, 8, 8, 8, 4, DXGI_FORMAT_R8G8B8A8_SNORM );

            initFormatLinear( hkImageFormat::R8_G8_B8_A8_UNSIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 32, 8, 8, 8, 8, 4, DXGI_FORMAT_R8G8B8A8_UNORM );
            initFormatLinear( hkImageFormat::R8_G8_B8_A8_UNSIGNED_NORMALIZED_SRGB, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 32, 8, 8, 8, 8, 4, DXGI_FORMAT_R8G8B8A8_UNORM_SRGB );
            setupSrgbPair( hkImageFormat::R8_G8_B8_A8_UNSIGNED_NORMALIZED, hkImageFormat::R8_G8_B8_A8_UNSIGNED_NORMALIZED_SRGB );

            s_metaData[hkImageFormat::R8_G8_B8_A8_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::R] = 0x000000FF;
            s_metaData[hkImageFormat::R8_G8_B8_A8_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::G] = 0x0000FF00;
            s_metaData[hkImageFormat::R8_G8_B8_A8_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::B] = 0x00FF0000;
            s_metaData[hkImageFormat::R8_G8_B8_A8_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::A] = 0xFF000000;

            initFormatLinear(hkImageFormat::R8_G8_B8_UNSIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 24, 8, 8, 8, 0, 3, DXGI_FORMAT_UNKNOWN);
            initFormatLinear(hkImageFormat::R8_G8_B8_UNSIGNED_NORMALIZED_SRGB, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 24, 8, 8, 8, 0, 3, DXGI_FORMAT_UNKNOWN);
            setupSrgbPair(hkImageFormat::R8_G8_B8_UNSIGNED_NORMALIZED, hkImageFormat::R8_G8_B8_UNSIGNED_NORMALIZED_SRGB);

            initFormatLinear( hkImageFormat::B8_G8_R8_A8_UNSIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 32, 8, 8, 8, 8, 4, DXGI_FORMAT_B8G8R8A8_UNORM );
            initFormatLinear( hkImageFormat::B8_G8_R8_A8_UNSIGNED_NORMALIZED_SRGB, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 32, 8, 8, 8, 8, 4, DXGI_FORMAT_B8G8R8A8_UNORM_SRGB );
            setupSrgbPair( hkImageFormat::B8_G8_R8_A8_UNSIGNED_NORMALIZED, hkImageFormat::B8_G8_R8_A8_UNSIGNED_NORMALIZED_SRGB );

            s_metaData[hkImageFormat::B8_G8_R8_A8_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::R] = 0x00FF0000;
            s_metaData[hkImageFormat::B8_G8_R8_A8_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::G] = 0x0000FF00;
            s_metaData[hkImageFormat::B8_G8_R8_A8_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::B] = 0x000000FF;
            s_metaData[hkImageFormat::B8_G8_R8_A8_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::A] = 0xFF000000;

            initFormatLinear( hkImageFormat::B8_G8_R8_X8_UNSIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 32, 8, 8, 8, 0, 3, DXGI_FORMAT_B8G8R8X8_UNORM );
            initFormatLinear( hkImageFormat::B8_G8_R8_X8_UNSIGNED_NORMALIZED_SRGB, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 32, 8, 8, 8, 0, 3, DXGI_FORMAT_B8G8R8X8_UNORM_SRGB );
            setupSrgbPair( hkImageFormat::B8_G8_R8_X8_UNSIGNED_NORMALIZED, hkImageFormat::B8_G8_R8_X8_UNSIGNED_NORMALIZED_SRGB );

            s_metaData[hkImageFormat::B8_G8_R8_X8_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::R] = 0x00FF0000;
            s_metaData[hkImageFormat::B8_G8_R8_X8_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::G] = 0x0000FF00;
            s_metaData[hkImageFormat::B8_G8_R8_X8_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::B] = 0x000000FF;
            s_metaData[hkImageFormat::B8_G8_R8_X8_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::A] = 0x00000000;

            initFormatLinear( hkImageFormat::B8_G8_R8_UNSIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 24, 8, 8, 8, 0, 3, DXGI_FORMAT_UNKNOWN );
            initFormatLinear( hkImageFormat::B8_G8_R8_UNSIGNED_NORMALIZED_SRGB, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 24, 8, 8, 8, 0, 3, DXGI_FORMAT_UNKNOWN );
            setupSrgbPair( hkImageFormat::B8_G8_R8_UNSIGNED_NORMALIZED, hkImageFormat::B8_G8_R8_UNSIGNED_NORMALIZED_SRGB );

            s_metaData[hkImageFormat::B8_G8_R8_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::R] = 0x00FF0000;
            s_metaData[hkImageFormat::B8_G8_R8_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::G] = 0x0000FF00;
            s_metaData[hkImageFormat::B8_G8_R8_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::B] = 0x000000FF;
            s_metaData[hkImageFormat::B8_G8_R8_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::A] = 0x00000000;

            initFormatLinear( hkImageFormat::R8_G8_UNSIGNED, hkImageFormat::DATA_TYPE_UNSIGNED, 16, 8, 8, 0, 0, 2, DXGI_FORMAT_R8G8_UINT );
            initFormatLinear( hkImageFormat::R8_G8_SIGNED, hkImageFormat::DATA_TYPE_SIGNED, 16, 8, 8, 0, 0, 2, DXGI_FORMAT_R8G8_SINT );
            initFormatLinear( hkImageFormat::R8_G8_UNSIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 16, 8, 8, 0, 0, 2, DXGI_FORMAT_R8G8_UNORM );
            initFormatLinear( hkImageFormat::R8_G8_SIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_SIGNED_NORMALIZED, 16, 8, 8, 0, 0, 2, DXGI_FORMAT_R8G8_SNORM );

            initFormatLinear( hkImageFormat::R8_UNSIGNED, hkImageFormat::DATA_TYPE_UNSIGNED, 8, 8, 0, 0, 0, 1, DXGI_FORMAT_R8_UINT );
            initFormatLinear( hkImageFormat::R8_SIGNED, hkImageFormat::DATA_TYPE_SIGNED, 8, 8, 0, 0, 0, 1, DXGI_FORMAT_R8_SINT );
            initFormatLinear( hkImageFormat::R8_SIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_SIGNED_NORMALIZED, 8, 8, 0, 0, 0, 1, DXGI_FORMAT_R8_SNORM );

            initFormatLinear( hkImageFormat::R8_UNSIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 8, 8, 0, 0, 0, 1, DXGI_FORMAT_R8_UNORM );
            s_metaData[hkImageFormat::R8_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::R] = 0xFF;
            s_metaData[hkImageFormat::R8_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::G] = 0x00;
            s_metaData[hkImageFormat::R8_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::B] = 0x00;
            s_metaData[hkImageFormat::R8_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::A] = 0x00;

            initFormatCompressed( hkImageFormat::BC1_UNSIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 4, 4, 4, 1, 4, DXGI_FORMAT_BC1_UNORM );
            initFormatCompressed( hkImageFormat::BC1_UNSIGNED_NORMALIZED_SRGB, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 4, 4, 4, 1, 4, DXGI_FORMAT_BC1_UNORM_SRGB );
            setupSrgbPair( hkImageFormat::BC1_UNSIGNED_NORMALIZED, hkImageFormat::BC1_UNSIGNED_NORMALIZED_SRGB );

            initFormatCompressed( hkImageFormat::BC2_UNSIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 8, 4, 4, 1, 4, DXGI_FORMAT_BC2_UNORM );
            initFormatCompressed( hkImageFormat::BC2_UNSIGNED_NORMALIZED_SRGB, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 8, 4, 4, 1, 4, DXGI_FORMAT_BC2_UNORM_SRGB );
            setupSrgbPair( hkImageFormat::BC2_UNSIGNED_NORMALIZED, hkImageFormat::BC2_UNSIGNED_NORMALIZED_SRGB );

            initFormatCompressed( hkImageFormat::BC3_UNSIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 8, 4, 4, 1, 4, DXGI_FORMAT_BC3_UNORM );
            initFormatCompressed( hkImageFormat::BC3_UNSIGNED_NORMALIZED_SRGB, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 8, 4, 4, 1, 4, DXGI_FORMAT_BC3_UNORM_SRGB );
            setupSrgbPair( hkImageFormat::BC3_UNSIGNED_NORMALIZED, hkImageFormat::BC3_UNSIGNED_NORMALIZED_SRGB );

            initFormatCompressed( hkImageFormat::BC4_UNSIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 4, 4, 4, 1, 1, DXGI_FORMAT_BC4_UNORM );
            initFormatCompressed( hkImageFormat::BC4_SIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_SIGNED_NORMALIZED, 4, 4, 4, 1, 1, DXGI_FORMAT_BC4_SNORM );

            initFormatCompressed( hkImageFormat::BC5_UNSIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 8, 4, 4, 1, 2, DXGI_FORMAT_BC5_UNORM );
            initFormatCompressed( hkImageFormat::BC5_SIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_SIGNED_NORMALIZED, 8, 4, 4, 1, 2, DXGI_FORMAT_BC5_SNORM );

            initFormatCompressed( hkImageFormat::BC6_F16_UNSIGNED, hkImageFormat::DATA_TYPE_FLOAT, 8, 4, 4, 1, 3, DXGI_FORMAT_BC6H_UF16 );
            initFormatCompressed( hkImageFormat::BC6_F16_SIGNED, hkImageFormat::DATA_TYPE_FLOAT, 8, 4, 4, 1, 3, DXGI_FORMAT_BC6H_SF16 );

            initFormatCompressed( hkImageFormat::BC7_UNSIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 8, 4, 4, 1, 4, DXGI_FORMAT_BC7_UNORM );
            initFormatCompressed( hkImageFormat::BC7_UNSIGNED_NORMALIZED_SRGB, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 8, 4, 4, 1, 4, DXGI_FORMAT_BC7_UNORM_SRGB );
            setupSrgbPair( hkImageFormat::BC7_UNSIGNED_NORMALIZED, hkImageFormat::BC7_UNSIGNED_NORMALIZED_SRGB );

            initFormatLinear( hkImageFormat::B5_G6_R5_UNSIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 16, 5, 6, 5, 0, 3, DXGI_FORMAT_B5G6R5_UNORM );
            s_metaData[hkImageFormat::B5_G6_R5_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::R] = 0xF800;
            s_metaData[hkImageFormat::B5_G6_R5_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::G] = 0x07E0;
            s_metaData[hkImageFormat::B5_G6_R5_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::B] = 0x001F;

            initFormatLinear( hkImageFormat::B5_G5_R5_A1_UNSIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 16, 5, 5, 5, 1, 4, DXGI_FORMAT_B5G5R5A1_UNORM );

            initFormatLinear( hkImageFormat::B4_G4_R4_A4_UNSIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 16, 4, 4, 4, 4, 4, DXGI_FORMAT_B4G4R4A4_UNORM );
            s_metaData[hkImageFormat::B4_G4_R4_A4_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::R] = 0x0F00;
            s_metaData[hkImageFormat::B4_G4_R4_A4_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::G] = 0x00F0;
            s_metaData[hkImageFormat::B4_G4_R4_A4_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::B] = 0x000F;
            s_metaData[hkImageFormat::B4_G4_R4_A4_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::A] = 0xF000;

            initFormatLinear( hkImageFormat::R11_G11_B10_UNSIGNED_FLOAT, hkImageFormat::DATA_TYPE_FLOAT, 32, 11, 11, 10, 0, 3, DXGI_FORMAT_R11G11B10_FLOAT );
            initFormatLinear( hkImageFormat::R10_G10_B10_A2_UNSIGNED, hkImageFormat::DATA_TYPE_UNSIGNED, 32, 10, 10, 10, 2, 4, DXGI_FORMAT_R10G10B10A2_UINT );
            initFormatLinear( hkImageFormat::R10_G10_B10_A2_UNSIGNED_NORMALIZED, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 32, 10, 10, 10, 2, 4, DXGI_FORMAT_R10G10B10A2_UNORM );

            // msdn.microsoft.com/en-us/library/windows/desktop/bb943991(v=vs.85).aspx documents R10G10B10A2 as having an alpha mask of 0
            s_metaData[hkImageFormat::R10_G10_B10_A2_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::R] = 0x000003FF;
            s_metaData[hkImageFormat::R10_G10_B10_A2_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::G] = 0x000FFC00;
            s_metaData[hkImageFormat::R10_G10_B10_A2_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::B] = 0x3FF00000;
            s_metaData[hkImageFormat::R10_G10_B10_A2_UNSIGNED_NORMALIZED].m_channelMasks[hkImageFormat::A] = 0;

            initFormatDepth( hkImageFormat::D32F, hkImageFormat::DATA_TYPE_DEPTH_STENCIL, 32, false, 32, 0, DXGI_FORMAT_D32_FLOAT );
            initFormatDepth( hkImageFormat::D32F_S8_X24, hkImageFormat::DATA_TYPE_DEPTH_STENCIL, 64, true, 32, 8, DXGI_FORMAT_D32_FLOAT_S8X24_UINT );
            initFormatDepth( hkImageFormat::D24_S8, hkImageFormat::DATA_TYPE_DEPTH_STENCIL, 32, true, 24, 8, DXGI_FORMAT_D24_UNORM_S8_UINT );
            initFormatDepth( hkImageFormat::D16, hkImageFormat::DATA_TYPE_DEPTH_STENCIL, 16, false, 16, 0, DXGI_FORMAT_D16_UNORM );

            initFormatCompressed( hkImageFormat::ETC1_RGB_4BPP, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 4, 4, 4, 1, 3, DXGI_FORMAT_UNKNOWN );

            // Technically blocks are 8x4, but due to the 8x8 minimum size restriction and since we can't decompress individual blocks anyway, use 8x8
            // so we can assume block size == minimum mip level size
            initFormatCompressed( hkImageFormat::PVRTC_RGBA_2BPP, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 2, 8, 8, 1, 4, DXGI_FORMAT_UNKNOWN );
            initFormatCompressed( hkImageFormat::PVRTC_RGBA_4BPP, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 4, 8, 8, 1, 4, DXGI_FORMAT_UNKNOWN );
            initFormatCompressed( hkImageFormat::PVRTC_RGB_2BPP, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 2, 8, 8, 1, 3, DXGI_FORMAT_UNKNOWN );
            initFormatCompressed( hkImageFormat::PVRTC_RGB_4BPP, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 4, 8, 8, 1, 3, DXGI_FORMAT_UNKNOWN );

            initFormatLinear( hkImageFormat::NV12, hkImageFormat::DATA_TYPE_UNSIGNED_NORMALIZED, 12, 8, 2, 2, 0, 3, DXGI_FORMAT_NV12 );

            initFormatLinear( hkImageFormat::RGBA, hkImageFormat::DATA_TYPE_GENERIC, 0, 1, 1, 1, 1, 4, DXGI_FORMAT_UNKNOWN );
            initFormatLinear( hkImageFormat::RGB, hkImageFormat::DATA_TYPE_GENERIC, 0, 1, 1, 1, 0, 3, DXGI_FORMAT_UNKNOWN );
            initFormatLinear( hkImageFormat::RG, hkImageFormat::DATA_TYPE_GENERIC, 0, 1, 1, 0, 0, 2, DXGI_FORMAT_UNKNOWN );
            initFormatLinear( hkImageFormat::RED, hkImageFormat::DATA_TYPE_GENERIC, 0, 1, 0, 0, 0, 1, DXGI_FORMAT_UNKNOWN );

            initFormatDepth( hkImageFormat::DEPTH, hkImageFormat::DATA_TYPE_GENERIC, 0, false, 1, 0, DXGI_FORMAT_UNKNOWN );
            initFormatDepth( hkImageFormat::DEPTH_STENCIL, hkImageFormat::DATA_TYPE_GENERIC, 0, true, 1, 1, DXGI_FORMAT_UNKNOWN );
        }

        void initFormatLinear( hkImageFormat::Enum format, hkImageFormat::DataType dataType, hkUint8 bitsPerPixel, hkUint8 bitsR, hkUint8 bitsG, hkUint8 bitsB, hkUint8 bitsA, hkUint8 numChannels, hkUint32 dxgiFormat )
        {
            s_metaData[format].m_bitsPerPixel = bitsPerPixel;
            s_metaData[format].m_dataType = dataType;

            s_metaData[format].m_numChannels = numChannels;

            s_metaData[format].m_bitsPerChannel[hkImageFormat::R] = bitsR;
            s_metaData[format].m_bitsPerChannel[hkImageFormat::G] = bitsG;
            s_metaData[format].m_bitsPerChannel[hkImageFormat::B] = bitsB;
            s_metaData[format].m_bitsPerChannel[hkImageFormat::A] = bitsA;

            s_metaData[format].m_asLinear = format;
            s_metaData[format].m_asSrgb = format;

            s_metaData[format].m_dxgiFormat = dxgiFormat;
        }

        void initFormatCompressed( hkImageFormat::Enum format, hkImageFormat::DataType dataType, hkUint8 bitsPerPixel, hkUint8 blockWidth, hkUint8 blockHeight, hkUint8 blockDepth, hkUint8 numChannels, hkUint32 dxgiFormat )
        {
            s_metaData[format].m_bitsPerPixel = bitsPerPixel;
            s_metaData[format].m_blockWidth = blockWidth;
            s_metaData[format].m_blockHeight = blockHeight;
            s_metaData[format].m_blockDepth = blockDepth;
            s_metaData[format].m_dataType = dataType;

            s_metaData[format].m_numChannels = numChannels;

            s_metaData[format].m_isCompressed = true;

            s_metaData[format].m_asLinear = format;
            s_metaData[format].m_asSrgb = format;

            s_metaData[format].m_dxgiFormat = dxgiFormat;
        }

        void initFormatDepth( hkImageFormat::Enum format, hkImageFormat::DataType dataType, hkUint8 bitsPerPixel, bool isStencil, hkUint8 bitsD, hkUint8 bitsS, hkUint32 dxgiFormat )
        {
            s_metaData[format].m_bitsPerPixel = bitsPerPixel;
            s_metaData[format].m_dataType = dataType;

            s_metaData[format].m_isDepth = true;
            s_metaData[format].m_isStencil = isStencil;

            s_metaData[format].m_numChannels = isStencil ? 2 : 1;

            s_metaData[format].m_bitsPerChannel[hkImageFormat::D] = bitsD;
            s_metaData[format].m_bitsPerChannel[hkImageFormat::S] = bitsS;

            s_metaData[format].m_asLinear = format;
            s_metaData[format].m_asSrgb = format;

            s_metaData[format].m_dxgiFormat = dxgiFormat;
        }

        void setupSrgbPair( hkImageFormat::Enum linearFormat, hkImageFormat::Enum srgbFormat )
        {
            s_metaData[linearFormat].m_asLinear = linearFormat;
            s_metaData[linearFormat].m_asSrgb = srgbFormat;

            s_metaData[srgbFormat].m_asLinear = linearFormat;
            s_metaData[srgbFormat].m_asSrgb = srgbFormat;
        }

    } init;

    static const hkImageFormatMetaData& getMetaData( hkImageFormat::Enum format )
    {
        HK_ASSERT_NO_MSG( 0x657bbd16, format < hkImageFormat::COUNT );

        return s_metaData[format];
    }
}

const char* hkImageFormat::getName(Enum format)
{
    return hkReflect::getPresetsOf<hkImageFormat::Enum>().getNameByPreset(&format);
}

hkUint32 hkImageFormat::getBitsPerPixel(Enum format)
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    return getMetaData(format).m_bitsPerPixel;
}

hkUint32 hkImageFormat::getBitsPerBlock(Enum format)
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    const hkImageFormatMetaData& metaData = getMetaData(format);
    return metaData.m_bitsPerPixel * metaData.m_blockWidth * metaData.m_blockHeight * metaData.m_blockDepth;
}

hkUint32 hkImageFormat::getNumChannels(Enum format)
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    return getMetaData(format).m_numChannels;
}

hkUint32 hkImageFormat::getBitsPerChannel(Enum format, Channel c)
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    return getMetaData(format).m_bitsPerChannel[c];
}

hkUint32 hkImageFormat::getChannelMask(Enum format, Channel c)
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    return getMetaData(format).m_channelMasks[c];
}

hkUint32 hkImageFormat::getBlockWidth(Enum format)
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    return getMetaData(format).m_blockWidth;
}

hkUint32 hkImageFormat::getBlockHeight(Enum format)
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    return getMetaData(format).m_blockHeight;
}

hkUint32 hkImageFormat::getBlockDepth(Enum format)
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    return getMetaData(format).m_blockDepth;
}

hkImageFormat::DataType hkImageFormat::getDataType(Enum format)
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    return getMetaData(format).m_dataType;
}

bool hkImageFormat::isGeneric(Enum format)
{
    return format >= hkImageFormat::RGBA && format <= hkImageFormat::DEPTH_STENCIL;
}

bool hkImageFormat::isSrgbFormat(Enum format)
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    return getMetaData(format).m_asLinear != format;
}

hkImageFormat::Enum hkImageFormat::getSrgbFormat(Enum format)
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    return getMetaData(format).m_asSrgb;
}

hkImageFormat::Enum hkImageFormat::getLinearFormat(Enum format)
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    return getMetaData(format).m_asLinear;
}

bool hkImageFormat::isDepthFormat(Enum format)
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    return getMetaData(format).m_isDepth;
}

bool hkImageFormat::isStencilFormat(Enum format)
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    return getMetaData(format).m_isStencil;
}

bool hkImageFormat::isSameTypeGroup(Enum f1, Enum f2)
{
    if(f1 == f2)
    {
        return true;
    }
    switch(f1)
    {
        case hkImageFormat::R32_G32_B32_A32_FLOAT:
        case hkImageFormat::R32_G32_B32_A32_UNSIGNED:
        case hkImageFormat::R32_G32_B32_A32_SIGNED:
            return (f2 == hkImageFormat::R32_G32_B32_A32_FLOAT || f2 == hkImageFormat::R32_G32_B32_A32_UNSIGNED || f2 == hkImageFormat::R32_G32_B32_A32_SIGNED);
        case hkImageFormat::R32_G32_B32_FLOAT:
        case hkImageFormat::R32_G32_B32_UNSIGNED:
        case hkImageFormat::R32_G32_B32_SIGNED:
            return (f2 == hkImageFormat::R32_G32_B32_FLOAT || f2 == hkImageFormat::R32_G32_B32_UNSIGNED || f2 == hkImageFormat::R32_G32_B32_SIGNED);
        case hkImageFormat::R32_G32_FLOAT:
        case hkImageFormat::R32_G32_UNSIGNED:
        case hkImageFormat::R32_G32_SIGNED:
            return (f2 == hkImageFormat::R32_G32_FLOAT || f2 == hkImageFormat::R32_G32_UNSIGNED || f2 == hkImageFormat::R32_G32_SIGNED);
        case hkImageFormat::R32_FLOAT:
        case hkImageFormat::R32_UNSIGNED:
        case hkImageFormat::R32_SIGNED:
            return (f2 == hkImageFormat::R32_FLOAT || f2 == hkImageFormat::R32_UNSIGNED || f2 == hkImageFormat::R32_SIGNED);
        case hkImageFormat::R16_G16_B16_A16_FLOAT:
        case hkImageFormat::R16_G16_B16_A16_UNSIGNED:
        case hkImageFormat::R16_G16_B16_A16_SIGNED:
        case hkImageFormat::R16_G16_B16_A16_UNSIGNED_NORMALIZED:
        case hkImageFormat::R16_G16_B16_A16_SIGNED_NORMALIZED:
            return (f2 == hkImageFormat::R16_G16_B16_A16_FLOAT || f2 == hkImageFormat::R16_G16_B16_A16_UNSIGNED || f2 == hkImageFormat::R16_G16_B16_A16_SIGNED || f2 == hkImageFormat::R16_G16_B16_A16_UNSIGNED_NORMALIZED || f2 == hkImageFormat::R16_G16_B16_A16_SIGNED_NORMALIZED);
        case hkImageFormat::R16_G16_FLOAT:
        case hkImageFormat::R16_G16_UNSIGNED:
        case hkImageFormat::R16_G16_SIGNED:
        case hkImageFormat::R16_G16_UNSIGNED_NORMALIZED:
        case hkImageFormat::R16_G16_SIGNED_NORMALIZED:
            return (f2 == hkImageFormat::R16_G16_FLOAT || f2 == hkImageFormat::R16_G16_UNSIGNED || f2 == hkImageFormat::R16_G16_SIGNED || f2 == hkImageFormat::R16_G16_UNSIGNED_NORMALIZED || f2 == hkImageFormat::R16_G16_SIGNED_NORMALIZED);
        case hkImageFormat::R8_G8_B8_A8_UNSIGNED:
        case hkImageFormat::R8_G8_B8_A8_SIGNED:
        case hkImageFormat::R8_G8_B8_A8_UNSIGNED_NORMALIZED:
        case hkImageFormat::R8_G8_B8_A8_SIGNED_NORMALIZED:
        case hkImageFormat::R8_G8_B8_A8_UNSIGNED_NORMALIZED_SRGB:
            return (f2 == hkImageFormat::R8_G8_B8_A8_UNSIGNED || f2 == hkImageFormat::R8_G8_B8_A8_SIGNED || f2 == hkImageFormat::R8_G8_B8_A8_UNSIGNED_NORMALIZED || f2 == hkImageFormat::R8_G8_B8_A8_SIGNED_NORMALIZED || f2 == hkImageFormat::R8_G8_B8_A8_UNSIGNED_NORMALIZED_SRGB);
        case hkImageFormat::B8_G8_R8_A8_UNSIGNED_NORMALIZED:
        case hkImageFormat::B8_G8_R8_A8_UNSIGNED_NORMALIZED_SRGB:
            return (f2 == hkImageFormat::B8_G8_R8_A8_UNSIGNED_NORMALIZED || f2 == hkImageFormat::B8_G8_R8_A8_UNSIGNED_NORMALIZED_SRGB);
        case hkImageFormat::B8_G8_R8_X8_UNSIGNED_NORMALIZED:
        case hkImageFormat::B8_G8_R8_X8_UNSIGNED_NORMALIZED_SRGB:
            return (f2 == hkImageFormat::B8_G8_R8_X8_UNSIGNED_NORMALIZED || f2 == hkImageFormat::B8_G8_R8_X8_UNSIGNED_NORMALIZED_SRGB);
        case hkImageFormat::B8_G8_R8_UNSIGNED_NORMALIZED:
        case hkImageFormat::B8_G8_R8_UNSIGNED_NORMALIZED_SRGB:
            return (f2 == hkImageFormat::B8_G8_R8_UNSIGNED_NORMALIZED || f2 == hkImageFormat::B8_G8_R8_UNSIGNED_NORMALIZED_SRGB);
        case hkImageFormat::R8_G8_UNSIGNED:
        case hkImageFormat::R8_G8_SIGNED:
        case hkImageFormat::R8_G8_UNSIGNED_NORMALIZED:
        case hkImageFormat::R8_G8_SIGNED_NORMALIZED:
            return (f2 == hkImageFormat::R8_G8_UNSIGNED || f2 == hkImageFormat::R8_G8_SIGNED || f2 == hkImageFormat::R8_G8_UNSIGNED_NORMALIZED || f2 == hkImageFormat::R8_G8_SIGNED_NORMALIZED);
        case hkImageFormat::R8_UNSIGNED:
        case hkImageFormat::R8_SIGNED:
        case hkImageFormat::R8_UNSIGNED_NORMALIZED:
        case hkImageFormat::R8_SIGNED_NORMALIZED:
            return (f2 == hkImageFormat::R8_UNSIGNED || f2 == hkImageFormat::R8_SIGNED || f2 == hkImageFormat::R8_UNSIGNED_NORMALIZED || f2 == hkImageFormat::R8_SIGNED_NORMALIZED);
        case hkImageFormat::BC1_UNSIGNED_NORMALIZED:
        case hkImageFormat::BC1_UNSIGNED_NORMALIZED_SRGB:
            return (f2 == hkImageFormat::BC1_UNSIGNED_NORMALIZED || f2 == hkImageFormat::BC1_UNSIGNED_NORMALIZED_SRGB);
        case hkImageFormat::BC2_UNSIGNED_NORMALIZED:
        case hkImageFormat::BC2_UNSIGNED_NORMALIZED_SRGB:
            return (f2 == hkImageFormat::BC2_UNSIGNED_NORMALIZED || f2 == hkImageFormat::BC2_UNSIGNED_NORMALIZED_SRGB);
        case hkImageFormat::BC3_UNSIGNED_NORMALIZED:
        case hkImageFormat::BC3_UNSIGNED_NORMALIZED_SRGB:
            return (f2 == hkImageFormat::BC3_UNSIGNED_NORMALIZED || f2 == hkImageFormat::BC3_UNSIGNED_NORMALIZED_SRGB);
        case hkImageFormat::BC4_UNSIGNED_NORMALIZED:
        case hkImageFormat::BC4_SIGNED_NORMALIZED:
            return (f2 == hkImageFormat::BC4_UNSIGNED_NORMALIZED || f2 == hkImageFormat::BC4_SIGNED_NORMALIZED);
        case hkImageFormat::BC5_UNSIGNED_NORMALIZED:
        case hkImageFormat::BC5_SIGNED_NORMALIZED:
            return (f2 == hkImageFormat::BC5_UNSIGNED_NORMALIZED || f2 == hkImageFormat::BC5_SIGNED_NORMALIZED);
        case hkImageFormat::BC6_F16_UNSIGNED:
        case hkImageFormat::BC6_F16_SIGNED:
            return (f2 == hkImageFormat::BC6_F16_UNSIGNED || f2 == hkImageFormat::BC6_F16_SIGNED);
        case hkImageFormat::BC7_UNSIGNED_NORMALIZED:
        case hkImageFormat::BC7_UNSIGNED_NORMALIZED_SRGB:
            return (f2 == hkImageFormat::BC7_UNSIGNED_NORMALIZED || f2 == hkImageFormat::BC7_UNSIGNED_NORMALIZED_SRGB);
        case hkImageFormat::R10_G10_B10_A2_UNSIGNED:
        case hkImageFormat::R10_G10_B10_A2_UNSIGNED_NORMALIZED:
            return (f2 == hkImageFormat::R10_G10_B10_A2_UNSIGNED || f2 == hkImageFormat::R10_G10_B10_A2_UNSIGNED_NORMALIZED);
        default:
            return false;
    }
}

bool hkImageFormat::isCompressed(Enum format)
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    return getMetaData(format).m_isCompressed;
}

hkUint32 hkImageFormat::getRowPitch(Enum format, hkUint32 width)
{
    hkUint32 blockWidth = getBlockWidth(format);

    hkUint32 numBlocksX = (width - 1) / blockWidth + 1;

    return numBlocksX * blockWidth * getBitsPerPixel(format) / 8;
}

hkUint32 hkImageFormat::getDepthPitch(Enum format, hkUint32 width, hkUint32 height)
{
    hkUint32 blockHeight = getBlockHeight(format);

    hkUint32 numBlocksY = (height - 1) / blockHeight + 1;

    return numBlocksY * blockHeight * getRowPitch(format, width);
}

hkImageFormat::Enum hkImageFormat::fromPixelMaskAndCount(hkUint32 redMask, hkUint32 greenMask, hkUint32 blueMask, hkUint32 alphaMask, hkUint32 bitsPerPixel)
{
    // Some DDS files in the wild are encoded as this
    if (bitsPerPixel == 8 && redMask == 0xff && greenMask == 0xff && blueMask == 0xff)
    {
        return R8_UNSIGNED_NORMALIZED;
    }

    if (bitsPerPixel == 8 && redMask == 0 && greenMask == 0 && blueMask == 0 && alphaMask == 0xff)
    {
        // Actually A8, but we currently don't support this format and treat some textures as R8 instead
        return R8_UNSIGNED_NORMALIZED;
    }

    for (hkUint32 index = 0; index < COUNT; index++)
    {
        Enum format = static_cast<Enum>(index);
        if (getChannelMask(format, R) == redMask &&
            getChannelMask(format, G) == greenMask &&
            getChannelMask(format, B) == blueMask &&
            getChannelMask(format, A) == alphaMask &&
            getBitsPerPixel(format) == bitsPerPixel &&
            getDataType(format) == DATA_TYPE_UNSIGNED_NORMALIZED &&
            !isCompressed(format))
        {
            return format;
        }
    }

    return INVALID;
}

hkImageFormat::Enum hkImageFormat::fromDxgiFormat(hkUint32 dxgiFormat)
{
    for (hkUint32 index = 0; index < COUNT; ++index)
    {
        Enum format = static_cast<Enum>(index);
        if (toDxgiFormat(format) == dxgiFormat)
        {
            return format;
        }
    }

    return INVALID;
}

hkUint32 hkImageFormat::toDxgiFormat(Enum format)
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;
    return getMetaData(format).m_dxgiFormat;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
