// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0


void hkImageHeader::setFormat(hkImageFormat::Enum format)
{
    m_format = format;
}

hkImageFormat::Enum hkImageHeader::getFormat() const
{
    return m_format;
}

void hkImageHeader::setWidth(hkUint32 width)
{
    m_width = width;
}

hkUint32 hkImageHeader::getWidth(hkUint32 mipLevel /*= 0*/) const
{
    HK_ASSERT(0x74abe223, mipLevel < m_numMipLevels, "Invalid mip level");
    return hkMath::max2(m_width >> mipLevel, 1);
}

void hkImageHeader::setHeight(hkUint32 height)
{
    m_height = height;
}

hkUint32 hkImageHeader::getHeight(hkUint32 mipLevel /*= 0*/) const
{
    HK_ASSERT(0x26d459b5, mipLevel < m_numMipLevels, "Invalid mip level");
    return hkMath::max2(m_height >> mipLevel, 1);
}

void hkImageHeader::setDepth(hkUint32 depth)
{
    m_depth = depth;
}

hkUint32 hkImageHeader::getDepth(hkUint32 mipLevel /*= 0*/) const
{
    HK_ASSERT(0x1ca39475, mipLevel < m_numMipLevels, "Invalid mip level");
    return hkMath::max2(m_depth >> mipLevel, 1);
}

void hkImageHeader::setNumMipLevels(hkUint32 numMipLevels)
{
    m_numMipLevels = numMipLevels;
}

hkUint32 hkImageHeader::getNumMipLevels() const
{
    return m_numMipLevels;
}

void hkImageHeader::setNumFaces(hkUint32 numFaces)
{
    m_numFaces = numFaces;
}

hkUint32 hkImageHeader::getNumFaces() const
{
    return m_numFaces;
}

void hkImageHeader::setNumArrayElements(hkUint32 numArrayElements)
{
    m_numArrayElements = numArrayElements;
}

hkUint32 hkImageHeader::getNumArrayElements() const
{
    return m_numArrayElements;
}

hkUint32 hkImageHeader::getNumBlocksX(hkUint32 mipLevel /*= 0*/) const
{
    HK_ASSERT(0x1f72a857, mipLevel < m_numMipLevels, "Invalid mip level");
    return (getWidth(mipLevel) - 1) / hkImageFormat::getBlockWidth(m_format) + 1;
}

hkUint32 hkImageHeader::getNumBlocksY(hkUint32 mipLevel /*= 0*/) const
{
    HK_ASSERT(0x588aff3, mipLevel < m_numMipLevels, "Invalid mip level");
    return (getHeight(mipLevel) - 1) / hkImageFormat::getBlockHeight(m_format) + 1;
}

hkUint32 hkImageHeader::getNumBlocksZ(hkUint32 mipLevel /*= 0*/) const
{
    HK_ASSERT(0x3ccffc6d, mipLevel < m_numMipLevels, "Invalid mip level");
    return (getDepth(mipLevel) - 1) / hkImageFormat::getBlockDepth(m_format) + 1;
}

hkUint32 hkImageHeader::getBitsPerPixel() const
{
    return hkImageFormat::getBitsPerPixel(getFormat());
}

hkUint32 hkImageHeader::getRowPitch(hkUint32 mipLevel) const
{
    HK_ASSERT(0xa0e4c1e, mipLevel < m_numMipLevels, "Invalid mip level");
    return hkImageFormat::getRowPitch(getFormat(), getWidth(mipLevel));
}

hkUint32 hkImageHeader::getDepthPitch(hkUint32 mipLevel) const
{
    HK_ASSERT(0x4c2c47af, mipLevel < m_numMipLevels, "Invalid mip level");
    return hkImageFormat::getDepthPitch(getFormat(), getWidth(mipLevel), getHeight(mipLevel));
}

hkUint32 hkImage::getDataSize() const
{
    return m_data.getSize();
}

void hkImage::reset()
{
    *this = hkImage();
}

template<typename T>
_Ret_notnull_ T* hkImage::getDataPointer()
{
    validateDataTypeAccessor<T>();
    return reinterpret_cast<T*>(m_data.begin());
}

template<typename T>
_Ret_notnull_ const T* hkImage::getDataPointer() const
{
    validateDataTypeAccessor<T>();
    return reinterpret_cast<const T*>(m_data.begin());
}

template<typename T>
_Ret_notnull_ T* hkImage::getSubImagePointer(hkUint32 mipLevel /*= 0*/, hkUint32 face /*= 0*/, hkUint32 arrayIndex /*= 0*/)
{
    validateDataTypeAccessor<T>();
    return reinterpret_cast<T*>(&m_data[getSubImage(mipLevel, face, arrayIndex).m_offset]);
}

template<typename T>
_Ret_notnull_ const T* hkImage::getSubImagePointer(hkUint32 mipLevel /*= 0*/, hkUint32 face /*= 0*/, hkUint32 arrayIndex /*= 0*/) const
{
    validateDataTypeAccessor<T>();
    return reinterpret_cast<const T*>(&m_data[getSubImage(mipLevel, face, arrayIndex).m_offset]);
}

template<typename T>
_Ret_notnull_ T* hkImage::getPixelPointer(hkUint32 mipLevel /*= 0*/, hkUint32 face /*= 0*/, hkUint32 arrayIndex /*= 0*/, hkUint32 x /*= 0*/, hkUint32 y /*= 0*/, hkUint32 z /*= 0*/)
{
    HK_ASSERT(0x40673cff, !hkImageFormat::isCompressed(getFormat()), "Pixel pointer can only be retrieved for linear formats.");
    HK_ASSERT(0x5a5b3956, x < getWidth(mipLevel), "x out of bounds");
    HK_ASSERT(0x64e5480f, y < getHeight(mipLevel), "y out of bounds");
    HK_ASSERT(0x68fa92e3, z < getDepth(mipLevel), "z out of bounds");

    T* pointer = getSubImagePointer<T>(mipLevel, face, arrayIndex);

    pointer = hkAddByteOffset(pointer, z * getDepthPitch(mipLevel));
    pointer = hkAddByteOffset(pointer, y * getRowPitch(mipLevel));
    pointer = hkAddByteOffset(pointer, x * getBitsPerPixel() / 8);

    return pointer;
}

template<typename T>
_Ret_notnull_ const T* hkImage::getPixelPointer(hkUint32 mipLevel /*= 0*/, hkUint32 face /*= 0*/, hkUint32 arrayIndex /*= 0*/, hkUint32 x /*= 0*/, hkUint32 y /*= 0*/, hkUint32 z /*= 0*/) const
{
    HK_ASSERT(0x54b8f363, !hkImageFormat::isCompressed(getFormat()), "Pixel pointer can only be retrieved for linear formats.");
    HK_ASSERT(0x23728e32, x < getWidth(mipLevel), "x out of bounds");
    HK_ASSERT(0x1d70546c, y < getHeight(mipLevel), "y out of bounds");
    HK_ASSERT(0x7027a724, z < getDepth(mipLevel), "z out of bounds");

    const T* pointer = getSubImagePointer<T>(mipLevel, face, arrayIndex);

    pointer = hkAddByteOffset(pointer, z * getDepthPitch(mipLevel));
    pointer = hkAddByteOffset(pointer, y * getRowPitch(mipLevel));
    pointer = hkAddByteOffset(pointer, x * getBitsPerPixel() / 8);

    return pointer;
}

template<typename T>
hkArrayView<T> hkImage::getRowView(hkUint32 mipLevel /* = 0 */, hkUint32 face /* = 0 */, hkUint32 arrayIndex /* = 0 */, hkUint32 row /* = 0 */)
{
    return hkArrayView<T>(getPixelPointer<T>(mipLevel, face, arrayIndex, 0, row), getRowPitch(mipLevel) / sizeof(T));
}

template<typename T>
hkArrayView<const T> hkImage::getRowView(hkUint32 mipLevel /* = 0 */, hkUint32 face /* = 0 */, hkUint32 arrayIndex /* = 0 */, hkUint32 row /* = 0 */) const
{
    return hkArrayView<const T>(getPixelPointer<T>(mipLevel, face, arrayIndex, 0, row), getRowPitch(mipLevel) / sizeof(T));
}

template<typename T>
hkArrayView<T> hkImage::getDataView()
{
    return hkArrayView<T>(getDataPointer<T>(), getDataSize() / sizeof(T));
}

template<typename T>
hkArrayView<const T> hkImage::getDataView() const
{
    return hkArrayView<const T>(getDataPointer<T>(), getDataSize() / sizeof(T));
}

template<typename T>
_Ret_notnull_ T* hkImage::getBlockPointer(hkUint32 mipLevel /*= 0*/, hkUint32 face /*= 0*/, hkUint32 arrayIndex /*= 0*/, hkUint32 blockX /*= 0*/, hkUint32 blockY /*= 0*/, hkUint32 blockZ /*= 0*/)
{
    T* pointer = getSubImagePointer<T>(mipLevel, face, arrayIndex);

    pointer = hkAddByteOffset(pointer, blockZ * getDepthPitch(mipLevel));

    const hkUint32 blockSizeX = hkImageFormat::getBlockWidth(getFormat());
    const hkUint32 blockSizeY = hkImageFormat::getBlockHeight(getFormat());

    const hkUint32 bitsPerPixel = hkImageFormat::getBitsPerPixel(getFormat());

    const hkUint32 numBlocksX = getWidth(mipLevel) / blockSizeX;

    const hkUint32 blockIndex = blockX + numBlocksX * blockY;

    pointer = hkAddByteOffset(pointer, blockIndex * blockSizeX * blockSizeY * bitsPerPixel / 8);

    return pointer;
}



template<typename T>
_Ret_notnull_ const T* hkImage::getBlockPointer(hkUint32 mipLevel /*= 0*/, hkUint32 face /*= 0*/, hkUint32 arrayIndex /*= 0*/, hkUint32 blockX /*= 0*/, hkUint32 blockY /*= 0*/, hkUint32 blockZ /*= 0*/) const
{
    const T* pointer = getSubImagePointer<T>(mipLevel, face, arrayIndex);

    pointer = hkAddByteOffset(pointer, blockZ * getDepthPitch(mipLevel));

    const hkUint32 blockSizeX = hkImageFormat::getBlockWidth(getFormat());
    const hkUint32 blockSizeY = hkImageFormat::getBlockHeight(getFormat());

    const hkUint32 bitsPerPixel = hkImageFormat::getBitsPerPixel(getFormat());

    const hkUint32 numBlocksX = getWidth(mipLevel) / blockSizeX;

    const hkUint32 blockIndex = blockX + numBlocksX * blockY;

    pointer = hkAddByteOffset(pointer, blockIndex * blockSizeX * blockSizeY * bitsPerPixel / 8);

    return pointer;
}


hkImage::SubImage& hkImage::getSubImage(hkUint32 mipLevel, hkUint32 face, hkUint32 arrayIndex)
{
    validateIndices(mipLevel, face, arrayIndex);
    return m_subImages[mipLevel + getNumMipLevels() * (face + getNumFaces() * arrayIndex)];
}

const hkImage::SubImage& hkImage::getSubImage(hkUint32 mipLevel, hkUint32 face, hkUint32 arrayIndex) const
{
    validateIndices(mipLevel, face, arrayIndex);
    return m_subImages[mipLevel + getNumMipLevels() * (face + getNumFaces() * arrayIndex)];
}

const hkImageHeader& hkImage::getHeader() const
{
    return *this;
}

void hkImage::validateIndices(hkUint32 mipLevel, hkUint32 face, hkUint32 arrayIndex) const
{
    HK_ASSERT(0x2abce942, mipLevel < getNumMipLevels(), "Invalid mip level");
    HK_ASSERT(0x67622127, face < getNumFaces(), "Invalid face index");
    HK_ASSERT(0xe013e36, arrayIndex < getNumArrayElements(), "Invalid array element index");
}


template<typename T>
void hkImage::validateDataTypeAccessor() const
{
    HK_ON_DEBUG(hkUint32 bytesPerBlock = hkImageFormat::getBitsPerBlock(getFormat()) / 8);
    HK_ASSERT(0x60836650, bytesPerBlock % hkSizeOfTypeOrVoid<T>::val == 0 || hkSizeOfTypeOrVoid<T>::val % bytesPerBlock == 0, "Accessor type is suitable for interpreting contained data");
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
