// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/ImageUtilities/Image/hkImageFormat.h>

template<typename> class hkRect;

class HK_EXPORT_COMMON hkImageHeader
{
public:
    HK_DECLARE_CLASS(hkImageHeader, New, Reflect);

    hkImageHeader();

    /// Set the image format.
    HK_INLINE void setFormat(hkImageFormat::Enum format);

    /// Get the image format.
    HK_INLINE hkImageFormat::Enum getFormat() const;

    /// Set the image width.
    HK_INLINE void setWidth(hkUint32 width);

    /// Get the image width at a given mip level.
    HK_INLINE hkUint32 getWidth(hkUint32 mipLevel = 0) const;

    /// Set the image height.
    HK_INLINE void setHeight(hkUint32 height);

    /// Get the image height at a given mip level.
    HK_INLINE hkUint32 getHeight(hkUint32 mipLevel = 0) const;

    /// Set the image depth.
    HK_INLINE void setDepth(hkUint32 depth);

    /// Get the image depth at a given mip level.
    HK_INLINE hkUint32 getDepth(hkUint32 mipLevel = 0) const;

    /// Set the number of mip levels. The non-scaled top-level image counts as one mip level.
    HK_INLINE void setNumMipLevels(hkUint32 numMipLevels);

    /// Get the number of mip levels.
    HK_INLINE hkUint32 getNumMipLevels() const;

    /// Set the number of cubemap faces; should be 1 for non-cubemaps or 6 for cubemaps.
    HK_INLINE void setNumFaces(hkUint32 numFaces);

    /// Get the number of cubemap faces.
    HK_INLINE hkUint32 getNumFaces() const;

    /// Set the number of elements in an array texture; should be 1 for non-array images.
    HK_INLINE void setNumArrayElements(hkUint32 numArrayElements);

    /// Get the number of elements in an array texture.
    HK_INLINE hkUint32 getNumArrayElements() const;

    /// Get the number of compressed blocks in a dimension for a given mip level; for non-compressed formats, this is the number of pixels.
    HK_INLINE hkUint32 getNumBlocksX(hkUint32 mipLevel = 0) const;
    HK_INLINE hkUint32 getNumBlocksY(hkUint32 mipLevel = 0) const;
    HK_INLINE hkUint32 getNumBlocksZ(hkUint32 mipLevel = 0) const;

    /// Get the number of bits per pixel of the format.
    HK_INLINE hkUint32 getBitsPerPixel() const;

    /// Get the pitch in bytes of a single row of the given mip level.
    HK_INLINE hkUint32 getRowPitch(hkUint32 mipLevel = 0) const;

    /// Get the pitch in bytes of a single 2D slice of the given mip level.
    HK_INLINE hkUint32 getDepthPitch(hkUint32 mipLevel = 0) const;

    /// Computes the data size required for an image with the header's format and dimensions.
    hkUint32 computeDataSize() const;

private:
    hkUint32 m_numMipLevels;
    hkUint32 m_numFaces;
    hkUint32 m_numArrayElements;

    hkUint32 m_width;
    hkUint32 m_height;
    hkUint32 m_depth;

    hkImageFormat::Enum m_format;
};

class HK_EXPORT_COMMON hkImage : private hkImageHeader
{
public:
    HK_DECLARE_CLASS(hkImage, New);

    /// Constructs an empty image.
    hkImage();

    /// Constructs an image with the given header; allocating internal storage for it.
    hkImage(const hkImageHeader& header);

    /// Constructs an image with the given header backed by user-supplied external storage.
    hkImage(const hkImageHeader& header, hkArrayView<void> externalData);

    /// Copy constructor
    hkImage( const hkImage& other );

    ~hkImage();

    /// Assignment operator
    hkImage& operator = ( const hkImage& other );

    /// Swaps the contents of this image with another image
    void swap( hkImage& other );

    using hkImageHeader::getWidth;
    using hkImageHeader::getHeight;
    using hkImageHeader::getDepth;

    using hkImageHeader::getNumMipLevels;
    using hkImageHeader::getNumFaces;
    using hkImageHeader::getNumArrayElements;

    using hkImageHeader::getFormat;

    using hkImageHeader::getBitsPerPixel;
    using hkImageHeader::getRowPitch;
    using hkImageHeader::getDepthPitch;

    using hkImageHeader::getNumBlocksX;
    using hkImageHeader::getNumBlocksY;
    using hkImageHeader::getNumBlocksZ;

    /// Returns the header this image was constructed from.
    HK_INLINE const hkImageHeader& getHeader() const;

    /// Returns the total size of all slices and mip levels of the image in bytes.
    HK_INLINE hkUint32 getDataSize() const;

    /// Resets the image to an empty one.
    HK_INLINE void reset();

    /// Resets the image and constructs it with the given header; allocating internal storage for it.
    void reset(const hkImageHeader& header);

    /// Resets the image and constructs it with the given header backed by user-supplied external storage.
    void reset(const hkImageHeader& header, hkArrayView<void> externalData);

    /// Discards all mip levels except the top-most one.
    void discardMipMaps();

    /// Converts the image contents to the given format.
    hkResult convert(hkImageFormat::Enum format);

    /// Reinterprets the image with a given format; the format must have the same size in bits per pixel as the current one.
    void reinterpretAs(hkImageFormat::Enum format);

    /// Loads the image from a given file name. The file format is determined by the extension.
    hkResult loadFrom(_In_z_ const char* fileName);

    /// Saves the image to a given file name. The file format is determined by the extension.
    hkResult saveTo(_In_z_ const char* fileName) const;

    /// Copies a rectangular region from the sourceImage to this image.
    void copyRectFrom( const hkImage& sourceImage, hkRect<hkInt32> sourceRect, hkRect<hkInt32> targetRect );

    /// Retrieves a pointer to the beginning of the image data.
    template<typename T>
    _Ret_notnull_ T* getDataPointer();
    template<typename T>
    _Ret_notnull_ const T* getDataPointer() const;

    /// Retrieves a pointer to the beginning of a specific subimage.
    template<typename T>
    _Ret_notnull_ T* getSubImagePointer(hkUint32 mipLevel = 0, hkUint32 face = 0, hkUint32 arrayIndex = 0);
    template<typename T>
    _Ret_notnull_ const T* getSubImagePointer(hkUint32 mipLevel = 0, hkUint32 face = 0, hkUint32 arrayIndex = 0) const;

    /// Retrieves a pointer to a specific pixel of a subimage.
    template<typename T>
    _Ret_notnull_ T* getPixelPointer(hkUint32 mipLevel = 0, hkUint32 face = 0, hkUint32 arrayIndex = 0, hkUint32 x = 0, hkUint32 y = 0, hkUint32 z = 0);
    template<typename T>
    _Ret_notnull_ const T* getPixelPointer(hkUint32 mipLevel = 0, hkUint32 face = 0, hkUint32 arrayIndex = 0, hkUint32 x = 0, hkUint32 y = 0, hkUint32 z = 0) const;

    /// Retrieves a view to a specific row of a subimage.
    template<typename T>
    hkArrayView<T> getRowView(hkUint32 mipLevel = 0, hkUint32 face = 0, hkUint32 arrayIndex = 0, hkUint32 row = 0);
    template<typename T>
    hkArrayView<const T> getRowView(hkUint32 mipLevel = 0, hkUint32 face = 0, hkUint32 arrayIndex = 0, hkUint32 row = 0) const;

    // Retrieves a view on all the data stored in the image. E.g. across all mips / faces / indices
    template<typename T>
    hkArrayView<T> getDataView();
    template<typename T>
    hkArrayView<const T> getDataView() const;

    /// Retrieves a pointer to a specific compressed block of a subimage.
    template<typename T>
    _Ret_notnull_ T* getBlockPointer(hkUint32 mipLevel = 0, hkUint32 face = 0, hkUint32 arrayIndex = 0, hkUint32 blockX = 0, hkUint32 blockY = 0, hkUint32 blockZ = 0);
    template<typename T>
    _Ret_notnull_ const T* getBlockPointer(hkUint32 mipLevel = 0, hkUint32 face = 0, hkUint32 arrayIndex = 0, hkUint32 blockX = 0, hkUint32 blockY = 0, hkUint32 blockZ = 0) const;


    /// Computes the number of mip maps needed for the given width and height.
    /// 1 means only the source image, 2 means source image and 1 mip map, ...
    static hkUint32 computeNumberOfMipMaps( hkUint32 width, hkUint32 height, hkUint32 depth = 1 );

private:

    template<typename T> void validateDataTypeAccessor() const;

    struct HK_EXPORT_COMMON SubImage
    {
        HK_DECLARE_CLASS(SubImage, New);
        hkUint32 m_offset;
        hkUint32 m_size;
    };

    hkUint32 computeLayout();

    HK_INLINE void validateIndices(hkUint32 mipLevel, hkUint32 face, hkUint32 arrayIndex) const;

    HK_INLINE SubImage& getSubImage(hkUint32 mipLevel, hkUint32 face, hkUint32 arrayIndex);

    HK_INLINE const SubImage& getSubImage(hkUint32 mipLevel, hkUint32 face, hkUint32 arrayIndex) const;

    hkInplaceArray<SubImage, 16> m_subImages;

    hkArray<hkUint8> m_data;
    bool m_userExternalStorage;
};

/// Reference counted version of hkImage
class hkImageRef : public hkReferencedObject, public hkImage
{
public:
    HK_DECLARE_CLASS(hkImageRef, New);

};

/// Addressing mode for images.
struct hkImageAddressMode
{
    enum Enum
    {
        WRAP = 0,
        MIRROR = 1,
        CLAMP = 2,
        BORDER = 3,
        MIRROR_ONCE = 4,

        COUNT
    };
};

HK_REFLECT_ENUM(HK_EXPORT_COMMON, hkImageAddressMode::Enum);

#include <Common/ImageUtilities/Image/hkImage.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
