// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/ImageUtilities/hkImageUtilities.h>
#include <Common/ImageUtilities/Image/hkImage.h>
#include <Common/ImageUtilities/FileFormats/hkImageFileFormat.h>
#include <Common/Base/Serialize/Detail/hkReadBuffer.h>
#include <Common/Base/Serialize/Detail/hkWriteBuffer.h>
#include <Common/ImageUtilities/Conversion/hkImageConversion.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>

#include <Common/ImageUtilities/FileFormats/hkDdsFileFormat.h>
#include <Common/ImageUtilities/FileFormats/hkBmpFileFormat.h>
#include <Common/ImageUtilities/FileFormats/hkTgaFileFormat.h>
#include <Common/ImageUtilities/FileFormats/hkHdrFileFormat.h>
#include <Common/Base/Types/Geometry/Rect/hkRect.h>


hkImageHeader::hkImageHeader()
    : m_numMipLevels(1),
    m_numFaces(1),
    m_numArrayElements(1),
    m_width(0),
    m_height(0),
    m_depth(1),
    m_format(hkImageFormat::INVALID)
{
}

hkUint32 hkImageHeader::computeDataSize() const
{
    hkUint32 dataSize = 0;

    for (hkUint32 mipLevel = 0; mipLevel < getNumMipLevels(); mipLevel++)
    {
        dataSize += hkImageFormat::getDepthPitch(getFormat(), getWidth(mipLevel), getHeight(mipLevel)) * getDepth(mipLevel);
    }

    return dataSize * getNumArrayElements() * getNumFaces();
}

hkImage::hkImage()
    : m_userExternalStorage(false)
{
}


hkImage::hkImage(const hkImageHeader& header)
    : m_userExternalStorage(false)
{
    reset(header);
}

hkImage::hkImage(const hkImageHeader& header, hkArrayView<void> externalData)
    : m_userExternalStorage(true)
{
    reset(header, externalData);
}

hkImage::hkImage(const hkImage& other)
    : hkImageHeader(other)
    , m_subImages(other.m_subImages)
    , m_data(other.m_data)
    , m_userExternalStorage(false)
{

}

hkImage::~hkImage()
{
}

void hkImage::discardMipMaps()
{
    // Nothing to do
    if (getNumMipLevels() <= 1)
    {
        return;
    }

    // Rebuild list of sub-images (similar to computeLayout) and copy existing top level mip data to new locations.
    // We need to take care with address calculations due to the mismatch between old/new mip level count.
    hkUint8* newMipLocation = m_data.begin();
    const hkUint8* oldMipLocation = m_data.begin();

    hkUint32 numSubImages = getNumFaces() * getNumArrayElements();
    m_subImages.reserveExactly(numSubImages);
    m_subImages.setSize(numSubImages);
    SubImage* subImage = m_subImages.begin();

    hkUint32 dataSize = 0;
    for (hkUint32 elementIndex = 0; elementIndex < getNumArrayElements(); elementIndex++)
    {
        for (hkUint32 face = 0; face < getNumFaces(); face++)
        {
            for (hkUint32 mipLevel = 0; mipLevel < getNumMipLevels(); mipLevel++)
            {
                hkUint32 mipLevelSize = hkImageFormat::getDepthPitch(getFormat(), getWidth(mipLevel), getHeight(mipLevel)) * getDepth(mipLevel);

                // Copy the first mip of each chain, otherwise simply advance the source pointer
                if (mipLevel == 0)
                {
                    subImage->m_offset = dataSize;
                    subImage->m_size = mipLevelSize;
                    dataSize += mipLevelSize;
                    subImage++;

                    // Avoid unnecessary copying of the very first subimage which always remains at the same location
                    if (oldMipLocation != newMipLocation)
                    {
                        hkMemUtil::memMove(newMipLocation, oldMipLocation, mipLevelSize);
                    }

                    newMipLocation = hkAddByteOffset(newMipLocation, mipLevelSize);
                }

                oldMipLocation = hkAddByteOffset(oldMipLocation, mipLevelSize);
            }
        }
    }

    // Replace header
    hkImageHeader newHeader = *this;
    newHeader.setNumMipLevels(1);
    static_cast<hkImageHeader&>(*this) = newHeader;

    // Discard remaining data
    m_data.reserveExactly(dataSize);
    m_data.setSize(dataSize);
}

void hkImage::swap(hkImage& other)
{
    // We have to swap the data array manually because hkArray does not allow swapping user storage arrays
    hkArrayView<hkUint8> thisUserData = m_data;
    hkArrayView <hkUint8> otherUserData = other.m_data;
    if (m_userExternalStorage)
    {
        m_data.setDataAutoFree(HK_NULL, 0, 0);
    }
    if (other.m_userExternalStorage)
    {
        other.m_data.setDataAutoFree(HK_NULL, 0, 0);
    }
    m_data.swap(other.m_data);
    if (m_userExternalStorage)
    {
        other.m_data.setDataUserFree(thisUserData.begin(), thisUserData.getSize(), thisUserData.getSize());
    }
    if (other.m_userExternalStorage)
    {
        m_data.setDataUserFree(otherUserData.begin(), otherUserData.getSize(), otherUserData.getSize());
    }

    // swap the remaining members
    hkAlgorithm::swap(m_subImages, other.m_subImages);
    hkAlgorithm::swap(static_cast<hkImageHeader&>(*this), static_cast<hkImageHeader&>(other));
    hkAlgorithm::swap(m_userExternalStorage, other.m_userExternalStorage);
}

hkImage& hkImage::operator=(const hkImage& other)
{
    hkImageHeader::operator=(other);
    m_data = other.m_data;
    m_subImages = other.m_subImages;
    m_userExternalStorage = false;
    return *this;
}

void hkImage::reset(const hkImageHeader& header)
{
    hkImageHeader::operator=(header);

    hkUint32 requiredSize = computeLayout();

    if (m_userExternalStorage)
    {
        HK_ASSERT(0x495d7e4d, requiredSize == (hkUint32)m_data.getSize(), "attempting a resize on user supplied storage");
    }

    m_data.reserveExactly(requiredSize);
    m_data.setSize(requiredSize);
}

void hkImage::reset(const hkImageHeader& header, hkArrayView<void> externalData)
{
    hkImageHeader::operator=(header);

    hkUint32 size = computeLayout();

    HK_ASSERT(0x7e220c25, size == hkUint32(externalData.getSize()), "Supplied data doesn't match the expected size");
    m_userExternalStorage = true;

    m_data.setDataUserFree(static_cast<hkUint8*>(externalData.begin()), size, size);
}

hkUint32 hkImage::computeLayout()
{
    hkUint32 numSubImages = getNumMipLevels() * getNumFaces() * getNumArrayElements();
    m_subImages.reserveExactly(numSubImages);
    m_subImages.setSize(numSubImages);

    int dataSize = 0;

    for (hkUint32 elementIndex = 0; elementIndex < getNumArrayElements(); elementIndex++)
    {
        for (hkUint32 face = 0; face < getNumFaces(); face++)
        {
            for (hkUint32 mipLevel = 0; mipLevel < getNumMipLevels(); mipLevel++)
            {
                SubImage& subImage = getSubImage(mipLevel, face, elementIndex);

                subImage.m_offset = dataSize;
                subImage.m_size = hkImageFormat::getDepthPitch(getFormat(), getWidth(mipLevel), getHeight(mipLevel)) * getDepth(mipLevel);
                dataSize += subImage.m_size;
            }
        }
    }

    return dataSize;
}

hkResult hkImage::convert(hkImageFormat::Enum format)
{
    return hkImageConversion::convert(*this, *this, format);
}

void hkImage::reinterpretAs(hkImageFormat::Enum format)
{
    HK_ASSERT(0x66b9e90b, hkImageFormat::isCompressed(format) == hkImageFormat::isCompressed(getFormat()), "Cannot reinterpret compressed and non-compressed formats");

    HK_ASSERT(0x72df9ecc, hkImageFormat::getBitsPerPixel(getFormat()) == hkImageFormat::getBitsPerPixel(format), "Cannot reinterpret between formats of different sizes");

    setFormat(format);
}

hkResult hkImage::loadFrom(_In_z_ const char* fileName)
{
    hkStringBuf buf(fileName);
    const char* extension = buf.pathExtension();
    if (extension && extension[0] == '.')
    {
        extension++;
    }

    hkImageFileFormat* format = hkImageFileFormat::getReader(extension);
    if (!format)
    {
        return HK_FAILURE;
    }

    return format->readImage(fileName, *this);
}

hkResult hkImage::saveTo(_In_z_ const char* fileName) const
{
    hkStringBuf buf(fileName);
    const char* extension = buf.pathExtension();
    if (extension && extension[0] == '.')
    {
        extension++;
    }

    hkImageFileFormat* format = hkImageFileFormat::getWriter(extension);
    if (!format)
    {
        return HK_FAILURE;
    }

    return format->writeImage(fileName, *this);
}

void hkImage::copyRectFrom( const hkImage& sourceImage, hkRect<hkInt32> sourceRect, hkRect<hkInt32> targetRect )
{
    hkImage& targetImage = *this;
    HK_ASSERT(0x60ac494f, sourceRect.getWidth() == targetRect.getWidth(), "Width of source and target rect must match" );
    HK_ASSERT(0x67caeb0c, sourceRect.getHeight() == targetRect.getHeight(), "Height of source and target rect must match" );
    HK_ASSERT(0x3558a626, sourceRect.getX() >= 0 && sourceRect.getX2() <= (hkInt32)sourceImage.getWidth(), "source rect is out of bounds" );
    HK_ASSERT(0x375e10ba, sourceRect.getY() >= 0 && sourceRect.getY2() <= (hkInt32)sourceImage.getHeight(), "source rect is out of bounds" );
    HK_ASSERT(0x720b44a5, targetRect.getX() >= 0 && targetRect.getX2() <= (hkInt32)targetImage.getWidth(), "target rect is out of bounds" );
    HK_ASSERT(0x3f9b9c3b, targetRect.getY() >= 0 && targetRect.getY2() <= (hkInt32)targetImage.getHeight(), "target rect is out of bounds" );
    HK_ASSERT(0x169dcb73, sourceImage.getFormat() == targetImage.getFormat(), "Source and target format must match" );
    HK_ASSERT(0x4de35fad, !hkImageFormat::isCompressed( sourceImage.getFormat() ), "Compressed formats are not supported" );

    const int sizeOfLine = sourceRect.getWidth() * hkImageFormat::getBitsPerPixel( sourceImage.getFormat() ) / 8;

    const int numRows = sourceRect.getHeight();
    for (int row = 0; row < numRows; row++)
    {
        const void* sourceData = sourceImage.getPixelPointer<void>( 0, 0, 0, sourceRect.getX(), sourceRect.getY() + row, 0 );
        void* targetData = targetImage.getPixelPointer<void>( 0, 0, 0, targetRect.getX(), targetRect.getY() + row, 0 );
        hkMemUtil::memCpy( targetData, sourceData, sizeOfLine );
    }
}

hkUint32 hkImage::computeNumberOfMipMaps( hkUint32 width, hkUint32 height, hkUint32 depth /* = 1 */ )
{
    hkUint32 numMipMaps = 1;

    while (width > 1 || height > 1 || depth > 1)
    {
        width = hkMath::max2(1, width / 2);
        height = hkMath::max2(1, height / 2);
        depth = hkMath::max2(1, depth / 2);

        numMipMaps++;
    }

    return numMipMaps;
}

hkDdsFileFormat s_ddsFormat;
hkBmpFileFormat s_bmpFormat;
hkTgaFileFormat s_TgaFormat;
hkHdrFileFormat s_HdrFormat;

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
