// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY       : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/ImageUtilities/hkImageUtilities.h>

#include <Common/ImageUtilities/FileFormats/hkTgaFileFormat.h>

#include <Common/Base/Serialize/Detail/hkWriteBuffer.h>
#include <Common/Base/Serialize/Detail/hkReadBuffer.h>

#include <Common/Base/Types/Color/hkColor.h>

#include <Common/ImageUtilities/Conversion/hkImageConversion.h>
#include <Common/ImageUtilities/Image/hkImageFormat.h>
#include <Common/ImageUtilities/Image/hkImage.h>

#define DEBUG_LOG_IDENTIFIER "image.tga"
#include <Common/Base/System/Log/hkLog.hxx>
#include <Common/ImageUtilities/Processing/hkImageProcessing.h>

// see Wikipedia for details:
// http://de.wikipedia.org/wiki/Targa_Image_File
struct TgaHeader
{
    hkInt8 m_imageIDLength;
    hkInt8 m_ignored1;
    hkInt8 m_imageType;
    hkInt8 m_ignored2[9];
    hkInt16 m_imageWidth;
    hkInt16 m_imageHeight;
    hkInt8 m_bitsPerPixel;
    hkInt8 m_imageDescriptor;
};

HK_COMPILE_TIME_ASSERT(sizeof(TgaHeader) == 18);

hkResult hkTgaFileFormat::writeImage(const hkIo::Detail::WriteBufferAdapter& stream, const hkImage& image) const
{
    const hkImageFormat::Enum compatibleFormats[] =
    {
        hkImageFormat::B8_G8_R8_A8_UNSIGNED_NORMALIZED,
        hkImageFormat::B8_G8_R8_A8_UNSIGNED_NORMALIZED_SRGB,
        hkImageFormat::B8_G8_R8_UNSIGNED_NORMALIZED,
        hkImageFormat::B8_G8_R8_UNSIGNED_NORMALIZED_SRGB,
    };

    // Find a compatible format closest to the one the image currently has
    hkImageFormat::Enum format = hkImageConversion::findClosestCompatibleFormat(image.getFormat(), hkArrayViewT::make(compatibleFormats));

    if (format == hkImageFormat::INVALID)
    {
        Log_Error("No conversion from format '{}' to a format suitable for TGA files known.", hkImageFormat::getName(image.getFormat()));
        return HK_FAILURE;
    }

    // Convert if not already in a compatible format
    if (format != image.getFormat())
    {
        hkImage convertedImage;
        if (hkImageConversion::convert(image, convertedImage, format).isFailure())
        {
            // This should never happen
            HK_ASSERT(0x4407b2a5, false, "hkImageConversion::convert failed even though the conversion was to the format returned by findClosestCompatibleFormat.");
            return HK_FAILURE;
        }

        return writeImage(stream, convertedImage);
    }

    hkIo::WriteBuffer buffer;
    buffer.attach(stream);

    const bool compress = true;

    hkUint32 bitsPerPixel = image.getBitsPerPixel();

    // Write the header
    {
        hkUint8 header[18];
        hkString::memSet(header, 0, sizeof(header));

        if (!compress)
        {
            // uncompressed TGA
            header[2] = 2;
        }
        else
        {
            // compressed TGA
            header[2] = 10;
        }

        header[13] = hkUint8(image.getWidth(0) / 256);
        header[15] = hkUint8(image.getHeight(0) / 256);
        header[12] = hkUint8(image.getWidth(0) % 256);
        header[14] = hkUint8(image.getHeight(0) % 256);
        header[16] = hkUint8(bitsPerPixel);

        buffer.writeRaw(header, 18);
    }

    const hkUint32 width = image.getWidth(0);
    const hkUint32 height = image.getHeight(0);

    if (!compress)
    {
        // Write image uncompressed
        for (hkUint32 y = 0; y < height; ++y)
        {
            buffer.writeRaw(image.getPixelPointer<void>(0, 0, 0, 0, height - y - 1, 0), image.getRowPitch(0));
        }
    }
    else
    {
        // write image RLE compressed

        hkInt32 rle = 0;

        hkColorUbLinear comparisonColor;
        hkInplaceArray<hkColorUbLinear, 129> unequal;
        hkInt32 equal = 0;

        for (hkUint32 y = 0; y < height; ++y)
        {
            for (hkUint32 x = 0; x < width; ++x)
            {
                hkColorUbLinear pixelColor;
                pixelColor.a = 0xff;

                hkString::memCpy(&pixelColor, image.getPixelPointer<void>(0, 0, 0, x, height - y - 1, 0), bitsPerPixel / 8);

                if (rle == 0) // no comparison possible yet
                {
                    comparisonColor = pixelColor;
                    rle = 1;
                    unequal.pushBack(pixelColor);
                }
                else if (rle == 1) // has one value gathered for comparison
                {
                    if (pixelColor == comparisonColor)
                    {
                        rle = 2; // two values were equal
                        equal = 2; // go into equal-mode
                    }
                    else
                    {
                        rle = 3; // two values were unequal
                        comparisonColor = pixelColor; // go into unequal-mode
                        unequal.pushBack(pixelColor);
                    }
                }
                else if (rle == 2) // equal values
                {
                    if ((pixelColor == comparisonColor) && (equal < 128))
                        ++equal;
                    else
                    {
                        hkUint8 repeat = hkUint8(equal + 127);

                        buffer.write8(repeat);

                        buffer.writeRaw(&comparisonColor, bitsPerPixel / 8);

                        comparisonColor = pixelColor;
                        rle = 1;
                        unequal.clear();
                        unequal.pushBack(pixelColor);
                    }
                }
                else if (rle == 3)
                {
                    if ((pixelColor != comparisonColor) && (unequal.getSize() < 128))
                    {
                        unequal.pushBack(pixelColor);
                        comparisonColor = pixelColor;
                    }
                    else
                    {
                        hkUint8 repeat = (hkUint8)(unequal.getSize()) - 1;

                        buffer.write8(repeat);

                        for (int i = 0; i < unequal.getSize(); ++i)
                        {
                            buffer.writeRaw(&unequal[i], bitsPerPixel / 8);
                        }

                        comparisonColor = pixelColor;
                        rle = 1;
                        unequal.clear();
                        unequal.pushBack(pixelColor);
                    }
                }
            }
        }

        if (rle == 1) // has one value gathered for comparison
        {
            hkUint8 repeat = 0;

            buffer.write8(repeat);
            buffer.writeRaw(&comparisonColor, bitsPerPixel / 8);
        }
        else if (rle == 2) // equal values
        {
            hkUint8 repeat = hkUint8(equal + 127);

            buffer.write8(repeat);
            buffer.writeRaw(&comparisonColor, bitsPerPixel / 8);
        }
        else if (rle == 3)
        {
            hkUint8 repeat = (hkUint8)(unequal.getSize()) - 1;
            buffer.write8(repeat);

            for (int i = 0; i < unequal.getSize(); ++i)
            {
                buffer.writeRaw(&unequal[i], bitsPerPixel / 8);
            }
        }
    }

    return HK_SUCCESS;
}


hkResult hkTgaFileFormat::readImageHeader(const hkIo::Detail::ReadBufferAdapter& stream, hkImageHeader& out_header) const
{
    hkIo::ReadBuffer buffer;
    buffer.attach(stream);

    TgaHeader tgaHeader;
    tgaHeader.m_imageIDLength = buffer.readPod<hkInt8Le>();
    tgaHeader.m_ignored1 = buffer.readPod<hkInt8Le>();
    tgaHeader.m_imageType = buffer.readPod<hkInt8Le>();
    buffer.read(&tgaHeader.m_ignored2, 9);
    tgaHeader.m_imageWidth = buffer.readPod<hkInt16Le>();
    tgaHeader.m_imageHeight = buffer.readPod<hkInt16Le>();
    tgaHeader.m_bitsPerPixel = buffer.readPod<hkInt8Le>();
    tgaHeader.m_imageDescriptor = buffer.readPod<hkInt8Le>();

    const hkUint32 bytesPerPixel = tgaHeader.m_bitsPerPixel / 8;

    // check whether width, height an BitsPerPixel are valid
    if ((tgaHeader.m_imageWidth <= 0) || (tgaHeader.m_imageHeight <= 0))
    {
        Log_Error("TGA has an invalid header: Width = {}, Height = {}", tgaHeader.m_imageWidth, tgaHeader.m_imageHeight);
        return HK_FAILURE;
    }

    hkImageFormat::Enum format = hkImageFormat::INVALID;
    switch (tgaHeader.m_imageType)
    {
    case 10:
    case 2:
        if (bytesPerPixel == 3)
        {
            format = hkImageFormat::B8_G8_R8_UNSIGNED_NORMALIZED;
        }
        else if (bytesPerPixel == 4)
        {
            format = hkImageFormat::B8_G8_R8_A8_UNSIGNED_NORMALIZED;
        }
        break;

    case 3:
        if (bytesPerPixel == 1)
        {
            format = hkImageFormat::R8_UNSIGNED_NORMALIZED;
        }
        break;
    }

    if (format == hkImageFormat::INVALID)
    {
        Log_Error("TGA has an unsupported format: BPP = {}, ImageType = {}", tgaHeader.m_bitsPerPixel, tgaHeader.m_imageType);
        return HK_FAILURE;
    }

    // Set out_header data
    out_header = hkImageHeader();
    out_header.setFormat(format);

    out_header.setNumMipLevels(1);
    out_header.setNumArrayElements(1);
    out_header.setNumFaces(1);

    out_header.setWidth(tgaHeader.m_imageWidth);
    out_header.setHeight(tgaHeader.m_imageHeight);
    out_header.setDepth(1);

    return HK_SUCCESS;
}

hkResult hkTgaFileFormat::readImage(const hkIo::Detail::ReadBufferAdapter& stream, hkImage& image) const
{
    hkIo::ReadBuffer buffer;
    buffer.attach(stream);

    TgaHeader tgaHeader;
    tgaHeader.m_imageIDLength = buffer.readPod<hkInt8Le>();
    tgaHeader.m_ignored1 = buffer.readPod<hkInt8Le>();
    tgaHeader.m_imageType = buffer.readPod<hkInt8Le>();
    buffer.read(&tgaHeader.m_ignored2, 9);
    tgaHeader.m_imageWidth = buffer.readPod<hkInt16Le>();
    tgaHeader.m_imageHeight = buffer.readPod<hkInt16Le>();
    tgaHeader.m_bitsPerPixel = buffer.readPod<hkInt8Le>();
    tgaHeader.m_imageDescriptor = buffer.readPod<hkInt8Le>();

    const hkUint32 bytesPerPixel = tgaHeader.m_bitsPerPixel / 8;

    // check whether width, height an BitsPerPixel are valid
    if ((tgaHeader.m_imageWidth <= 0) || (tgaHeader.m_imageHeight <= 0))
    {
        Log_Error("TGA has an invalid header: Width = {}, Height = {}", tgaHeader.m_imageWidth, tgaHeader.m_imageHeight);
        return HK_FAILURE;
    }

    hkImageFormat::Enum format = hkImageFormat::INVALID;
    bool rle = false;

    switch (tgaHeader.m_imageType)
    {
    case 10:
        rle = true;
        // Fall-through

    case 2:
        if (bytesPerPixel == 3)
        {
            format = hkImageFormat::B8_G8_R8_UNSIGNED_NORMALIZED;
        }
        else if (bytesPerPixel == 4)
        {
            format = hkImageFormat::B8_G8_R8_A8_UNSIGNED_NORMALIZED;
        }
        break;

    case 3:
        if (bytesPerPixel == 1)
        {
            format = hkImageFormat::R8_UNSIGNED_NORMALIZED;
        }
        break;
    }

    if (format == hkImageFormat::INVALID)
    {
        Log_Error("TGA has an unsupported format: BPP = {}, ImageType = {}", tgaHeader.m_bitsPerPixel, tgaHeader.m_imageType);
        return HK_FAILURE;
    }

    // Set image data
    hkImageHeader header;
    header.setFormat(format);

    header.setNumMipLevels(1);
    header.setNumArrayElements(1);
    header.setNumFaces(1);

    header.setWidth(tgaHeader.m_imageWidth);
    header.setHeight(tgaHeader.m_imageHeight);
    header.setDepth(1);

    image.reset(header);

    if (!rle)
    {
        // uncompressed

        const hkUint32 bytesPerRow = bytesPerPixel * tgaHeader.m_imageWidth;

        // read each row (gets rid of the row pitch
        for (hkInt32 y = 0; y < tgaHeader.m_imageHeight; ++y)
            buffer.read(image.getPixelPointer<void>(0, 0, 0, 0, tgaHeader.m_imageHeight - y - 1, 0), bytesPerRow);
    }
    else
    {
        // compressed

        hkInt32 currentPixel = 0;
        const int pixelCount = tgaHeader.m_imageWidth * tgaHeader.m_imageHeight;

        do
        {
            hkUint8 chunkHeader = buffer.readPod<hkUint8Le>();

            if (chunkHeader < 128)
            {
                // If the header is < 128, it means it is the number of RAW color packets minus 1
                // that follow the header
                // add 1 to get number of following color values

                chunkHeader++;

                // Read RAW color values
                for (hkInt32 i = 0; i < (hkInt32)chunkHeader; ++i)
                {
                    const hkInt32 x = currentPixel % tgaHeader.m_imageWidth;
                    const hkInt32 y = currentPixel / tgaHeader.m_imageWidth;

                    buffer.read(image.getPixelPointer<void>(0, 0, 0, x, tgaHeader.m_imageHeight - y - 1, 0), bytesPerPixel);

                    ++currentPixel;
                }
            }
            else // chunk header > 128 RLE data, next color repeated (chunk header - 127) times
            {
                chunkHeader -= 127; // Subtract 127 to get rid of the ID bit

                hkUint8 pixelBuffer[4] = { 255, 255, 255, 255 };

                // read the current color
                buffer.read(pixelBuffer, bytesPerPixel);

                // if it is a 24-Bit TGA (3 channels), the fourth channel stays at 255 all the time, since the 4th value in ucBuffer is never overwritten

                // copy the color into the image data as many times as dictated
                for (hkInt32 i = 0; i < (hkInt32)chunkHeader; ++i)
                {
                    const hkInt32 x = currentPixel % tgaHeader.m_imageWidth;
                    const hkInt32 y = currentPixel / tgaHeader.m_imageWidth;

                    hkUint8* pPixel = image.getPixelPointer<hkUint8>(0, 0, 0, x, tgaHeader.m_imageHeight - y - 1, 0);

                    hkString::memCpy(pPixel, pixelBuffer, bytesPerPixel); // TOM

                    ++currentPixel;
                }
            }
        } while (currentPixel < pixelCount);
    }

    // Horizontally flipped
    if (tgaHeader.m_imageDescriptor & (1U << 4))
    {
        
    }

    // Vertically flipped
    if (tgaHeader.m_imageDescriptor & (1U << 5))
    {
        hkImageProcessing::flipVertical(image);
    }

    return HK_SUCCESS;
}

hkArrayView<const char* const> hkTgaFileFormat::getReadableExtensions() const
{
    static const char* const extensions[] =
    {
        "tga"
    };
    return hkArrayViewT::make( extensions );
}

hkArrayView<const char* const> hkTgaFileFormat::getWriteableExtensions() const
{
    return getReadableExtensions();
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
