// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY       : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/ImageUtilities/hkImageUtilities.h>
#include <Common/ImageUtilities/FileFormats/hkPvrFileFormat.h>
#include <Common/ImageUtilities/Image/hkImage.h>

#include <Common/Base/Serialize/Detail/hkWriteBuffer.h>
#include <Common/Base/Serialize/Detail/hkReadBuffer.h>

#define DEBUG_LOG_IDENTIFIER "image.pvr"
#include <Common/Base/System/Log/hkLog.hxx>

// There is no official spec for the v1 and v2 PVR file formats. For reference, you will have to look up
// the source code in the ImgTech PowerVR SDK.
struct PVRTexHeader
{
    hkUint32Le headerLength;
    hkUint32Le height;
    hkUint32Le width;
    hkUint32Le numMipmaps;
    hkUint32Le flags;
    hkUint32Le dataLength;
    hkUint32Le bpp;
    hkUint32Le bitmaskRed;
    hkUint32Le bitmaskGreen;
    hkUint32Le bitmaskBlue;
    hkUint32Le bitmaskAlpha;
    hkUint32Le pvrTag;
    hkUint32Le numSurfs;
};

#define PVR_TEXTURE_FLAG_TYPE_MASK   0xFFu
#define PVR_TEXTURE_FLAG_CUBEMAP       (1<<12)
#define PVR_TEXTURE_FLAG_MIPMAPS     (1<<8)
#define PVR_TEXTURE_FLAG_ALPHA       (1 << 15)

enum
{
    kPVRTextureFlagTypePVRTC_2 = 0xC,
    kPVRTextureFlagTypePVRTC_4 = 0xD,
    kPVRTextureFlagTypePVRTC_2GL = 0x18,
    kPVRTextureFlagTypePVRTC_4GL = 0x19,
    kPVRTextureFlagTypeETC = 0x36
};

hkPvrFileFormat s_pvrFormat;

hkResult hkPvrFileFormat::writeImage(const hkIo::Detail::WriteBufferAdapter& stream, const hkImage& image) const
{
    hkIo::WriteBuffer buffer;
    buffer.attach(stream);

    PVRTexHeader pvrHeader;
    pvrHeader.headerLength = sizeof(PVRTexHeader);
    pvrHeader.width = image.getWidth();
    pvrHeader.height = image.getHeight();
    pvrHeader.numMipmaps = image.getNumMipLevels() - 1;

    hkUint32 flags = 0;
    if (image.getNumFaces() == 6)
    {
        flags |= PVR_TEXTURE_FLAG_CUBEMAP;
    }
    if (image.getNumMipLevels() > 1)
    {
        flags |= PVR_TEXTURE_FLAG_MIPMAPS;
    }
    HK_ASSERT(0x10c8cf6d, image.getNumArrayElements() == 1, "Array textures are not supported by the pvr writer");

    switch (image.getFormat())
    {
    case hkImageFormat::PVRTC_RGB_2BPP:
        flags |= kPVRTextureFlagTypePVRTC_2;
        pvrHeader.bitmaskAlpha = 0;
        break;
    case hkImageFormat::PVRTC_RGBA_2BPP:
        flags |= kPVRTextureFlagTypePVRTC_2;
        flags |= PVR_TEXTURE_FLAG_ALPHA;
        pvrHeader.bitmaskAlpha = 1;
        break;
    case hkImageFormat::PVRTC_RGB_4BPP:
        flags |= kPVRTextureFlagTypePVRTC_4;
        pvrHeader.bitmaskAlpha = 0;
        break;
    case hkImageFormat::PVRTC_RGBA_4BPP:
        flags |= kPVRTextureFlagTypePVRTC_4;
        flags |= PVR_TEXTURE_FLAG_ALPHA;
        pvrHeader.bitmaskAlpha = 1;
        break;
    case hkImageFormat::ETC1_RGB_4BPP:
        flags |= kPVRTextureFlagTypeETC;
        pvrHeader.bitmaskAlpha = 0;
        break;
    default:
        Log_Error("The image format '{}' can not be written to pvr/etc files", hkImageFormat::getName(image.getFormat()));
        return HK_FAILURE;
    }

    pvrHeader.flags = flags;

    pvrHeader.dataLength = image.getDataSize();
    pvrHeader.bpp = hkImageFormat::getBitsPerPixel(image.getFormat());
    pvrHeader.bitmaskBlue = 0;
    pvrHeader.bitmaskGreen = 0;
    pvrHeader.bitmaskRed = 0;
    pvrHeader.pvrTag = 0x21525650; // PVR!
    pvrHeader.numSurfs = 1;

    if (buffer.writeRaw(&pvrHeader, sizeof(pvrHeader)) != sizeof(pvrHeader))
        return HK_FAILURE;

    if (buffer.writeRaw(image.getDataPointer<void>(), image.getDataSize()) != hkLong(image.getDataSize()))
        return HK_FAILURE;

    return HK_SUCCESS;
}

hkResult hkPvrFileFormat::readImageHeader(const hkIo::Detail::ReadBufferAdapter& stream, hkImageHeader& out_header /*= hkGlobalLog::GetInstance()*/) const
{
    hkIo::ReadBuffer buffer;
    buffer.attach(stream);

    PVRTexHeader pvrHeader;
    if (buffer.read(&pvrHeader, sizeof(PVRTexHeader)) != sizeof(PVRTexHeader))
    {
        Log_Error("Failed to read PVR header");
        return HK_FAILURE;
    }

    out_header = hkImageHeader();
    out_header.setWidth(pvrHeader.width);
    out_header.setHeight(pvrHeader.height);
    out_header.setNumMipLevels(hkUint32(pvrHeader.numMipmaps) + 1);

    if (hkUint32(pvrHeader.flags) & PVR_TEXTURE_FLAG_CUBEMAP)
    {
        out_header.setNumFaces(6);
    }
    else
    {
        out_header.setNumFaces(1);
    }

    out_header.setNumArrayElements(1);

    hkImageFormat::Enum format;

    bool hasAlpha = hkUint32(pvrHeader.bitmaskAlpha) != 0;

    hkUint32 formatFlags = hkUint32(pvrHeader.flags) & PVR_TEXTURE_FLAG_TYPE_MASK;
    switch (formatFlags)
    {
    case kPVRTextureFlagTypePVRTC_4:
    case kPVRTextureFlagTypePVRTC_4GL:
        format = hasAlpha ? hkImageFormat::PVRTC_RGBA_4BPP : hkImageFormat::PVRTC_RGB_4BPP;
        break;

    case kPVRTextureFlagTypePVRTC_2:
    case kPVRTextureFlagTypePVRTC_2GL:
        format = hasAlpha ? hkImageFormat::PVRTC_RGBA_2BPP : hkImageFormat::PVRTC_RGB_2BPP;
        break;

    case kPVRTextureFlagTypeETC:
        format = hkImageFormat::ETC1_RGB_4BPP;
        break;

    default:
        Log_Error("Unsupported texture format");
        return HK_FAILURE;
    }

    out_header.setFormat(format);

    return HK_SUCCESS;
}

hkResult hkPvrFileFormat::readImage(const hkIo::Detail::ReadBufferAdapter& stream, hkImage& image) const
{
    hkImageHeader header;
    if (readImageHeader(stream, header).isFailure())
    {
        return HK_FAILURE;
    }

    image.reset(header);

    hkIo::ReadBuffer buffer;
    buffer.attach(stream);

    if (buffer.read(image.getDataPointer<void>(), image.getDataSize()) != hkLong(image.getDataSize()))
    {
        Log_Error("Failed to read PVR data");
        return HK_FAILURE;
    }

    return HK_SUCCESS;
}

hkArrayView<const char* const> hkPvrFileFormat::getReadableExtensions() const
{
    static const char* const extensions[] = {
        "pvr",
        "etc"
    };
    return hkArrayViewT::make( extensions );
}

hkArrayView<const char* const> hkPvrFileFormat::getWriteableExtensions() const
{
    return getReadableExtensions();
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
