// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY       : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/ImageUtilities/hkImageUtilities.h>

#include <Common/ImageUtilities/FileFormats/hkHdrFileFormat.h>

#include <Common/ImageUtilities/Image/hkImage.h>
#include <Common/ImageUtilities/Conversion/hkImageConversion.h>

#include <Common/Base/Serialize/Detail/hkWriteBuffer.h>
#include <Common/Base/Serialize/Detail/hkReadBuffer.h>

#include <math.h>
#include <stdio.h>

#define DEBUG_LOG_IDENTIFIER "image.hdr"
#include <Common/Base/System/Log/hkLog.hxx>

// HDR reader adapted from flip-code archive:
// http://www.flipcode.com/archives/HDR_Image_Reader.shtml

namespace
{
    typedef hkUint8 RGBE[ 4 ];

    HK_ALWAYS_INLINE hkUint8 readByte(hkIo::ReadBuffer& buffer ) { hkUint8 byte; buffer.read( &byte, 1 ); return byte; }

    HK_INLINE float convertComponent( int expo, int val )
    {
        float v = val / 256.0f;
        float d = (float)hkMath::pow( 2.0, expo );
        return v * d;
    }

    void workOnRGBE( RGBE *scan, int len, float *cols )
    {
        while ( len-- > 0 )
        {
            int expo = scan[ 0 ][ 3 ] - 128;
            cols[ 0 ] = convertComponent( expo, scan[ 0 ][ 0 ] );
            cols[ 1 ] = convertComponent( expo, scan[ 0 ][ 1 ] );
            cols[ 2 ] = convertComponent( expo, scan[ 0 ][ 2 ] );
            cols += 3;
            scan++;
        }
    }

    bool oldDecrunch( RGBE *scanline, int len, hkIo::ReadBuffer& buffer )
    {
        int i;
        int rshift = 0;

        while ( len > 0 )
        {
            scanline[ 0 ][ 0 ] = readByte( buffer );
            scanline[ 0 ][ 1 ] = readByte( buffer );
            scanline[ 0 ][ 2 ] = readByte( buffer );
            scanline[ 0 ][ 3 ] = readByte( buffer );

            if ( scanline[ 0 ][ 0 ] == 1 &&
                scanline[ 0 ][ 1 ] == 1 &&
                scanline[ 0 ][ 2 ] == 1 )
            {
                for ( i = scanline[ 0 ][ 3 ] << rshift; i > 0; i-- )
                {
                    hkString::memCpy( &scanline[ 0 ][ 0 ], &scanline[ -1 ][ 0 ], 4 );
                    scanline++;
                    len--;
                }
                rshift += 8;
            }
            else
            {
                scanline++;
                len--;
                rshift = 0;
            }
        }
        return true;
    }

    bool decrunch( RGBE *scanline, int len, hkIo::ReadBuffer& buffer )
    {
        int  i, j;

        if ( len < 8 || len > 0x7fff ) return oldDecrunch( scanline, len, buffer );

        if ( *buffer.peekAt<hkUint8>() != 2 )
        {
            return oldDecrunch( scanline, len, buffer );
        }
        else
        {
            i = readByte( buffer );
        }

        scanline[ 0 ][ 1 ] = readByte( buffer );
        scanline[ 0 ][ 2 ] = readByte( buffer );
        i = readByte( buffer );

        if ( scanline[ 0 ][ 1 ] != 2 || scanline[ 0 ][ 2 ] & 128 )
        {
            scanline[ 0 ][ 0 ] = (hkUint8)2;
            scanline[ 0 ][ 3 ] = (hkUint8)i;
            return oldDecrunch( scanline + 1, len - 1, buffer );
        }

        // read each component
        for ( i = 0; i < 4; i++ )
        {
            for ( j = 0; j < len; )
            {
                unsigned char code = readByte( buffer );
                if ( code > 128 )
                { // run
                    code &= 127;
                    unsigned char val = readByte( buffer );
                    while ( code-- ) scanline[ j++ ][ i ] = val;
                }
                else
                {   // non-run
                    while ( code-- ) scanline[ j++ ][ i ] = readByte( buffer );
                }
            }
        }

        return true;
    }

    bool readHeader( hkIo::ReadBuffer& buffer, int& width, int& height )
    {
        width = height = 0;

        int i;
        char str[ 200 ];
        buffer.read( str, 10 );
        if ( hkString::memCmp( str, "#?RADIANCE", 10 ) ) return false;

        readByte( buffer );

        char cmd[ 200 ];
        i = 0;
        int c = 0, oldc;
        while ( true )
        {
            oldc = c;
            c = readByte( buffer );
            if ( c == 0xa && oldc == 0xa ) break;
            cmd[ i++ ] = (char)c;
        }

        char reso[ 200 ];
        i = 0;
        while ( true )
        {
            c = readByte( buffer );
            reso[ i++ ] = (char)c;
            if ( c == 0xa ) break;
        }

        long w, h;
        if ( !sscanf( reso, "-Y %ld +X %ld", &h, &w ) )
        {
            return false;
        }

        width = (int)w;
        height = (int)h;

        return true;
    }

    bool readHeader( hkIo::ReadBuffer& buffer, hkImageHeader& out_header )
    {
        int width, height;
        if ( readHeader( buffer, width, height ) )
        {
            out_header = hkImageHeader();
            out_header.setFormat( hkImageFormat::R32_G32_B32_FLOAT );
            out_header.setNumMipLevels( 1 );
            out_header.setNumArrayElements( 1 );
            out_header.setNumFaces( 1 );

            out_header.setWidth( width );
            out_header.setHeight( height );
            out_header.setDepth( 1 );

            return true;
        }
        return false;
    }

    bool readImageContent( hkIo::ReadBuffer& buffer, int width, int height, float* output )
    {
        buffer.prefetchAll();

        hkArray<RGBE>::Temp scanline( width );
        for ( int y = height - 1; y >= 0; y-- )
        {
            if ( decrunch( scanline.begin(), width, buffer ) == false ) break;
            workOnRGBE( scanline.begin(), width, output );
            output += width * 3;
        }

        return true;
    }
}

hkResult hkHdrFileFormat::writeImage( const hkIo::Detail::WriteBufferAdapter& stream, const hkImage& image ) const
{
    return HK_FAILURE;
}

hkResult hkHdrFileFormat::readImageHeader( const hkIo::Detail::ReadBufferAdapter& stream, hkImageHeader& out_header ) const
{
    hkIo::ReadBuffer buffer( stream );
    if ( readHeader( buffer, out_header ) )
        return HK_SUCCESS;
    else
        return HK_FAILURE;
}

hkResult hkHdrFileFormat::readImage( const hkIo::Detail::ReadBufferAdapter& stream, hkImage& image ) const
{
    hkIo::ReadBuffer buffer( stream ); buffer.prefetchAll();

    hkImageHeader header;
    if ( readHeader( buffer, header ) )
    {
        image.reset( header );
        if ( readImageContent( buffer, header.getWidth(), header.getHeight(), image.getDataPointer<float>() ) )
        {
            return HK_SUCCESS;
        }
    }

    return HK_FAILURE;
}

hkArrayView<const char* const> hkHdrFileFormat::getReadableExtensions() const
{
    static const char* const extensions[] = { "hdr" };
    return hkArrayViewT::make( extensions );
}

hkArrayView<const char* const> hkHdrFileFormat::getWriteableExtensions() const
{
    return hkArrayViewT::make<const char* const>( HK_NULL, (hkUlong)0 );
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
