// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY       : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/ImageUtilities/hkImageUtilities.h>

#include <Common/ImageUtilities/FileFormats/hkDdsFileFormat.h>

#include <Common/Base/Serialize/Detail/hkWriteBuffer.h>
#include <Common/Base/Serialize/Detail/hkReadBuffer.h>
#include <Common/ImageUtilities/Image/hkImageFormat.h>
#include <Common/ImageUtilities/Image/hkImage.h>

#define DEBUG_LOG_IDENTIFIER "image.dds"
#include <Common/Base/System/Log/hkLog.hxx>

struct hkDdsPixelFormat {
    hkUint32Le m_size;
    hkUint32Le m_flags;
    hkUint32Le m_fourCC;
    hkUint32Le m_rgbBitCount;
    hkUint32Le m_rBitMask;
    hkUint32Le m_gBitMask;
    hkUint32Le m_bBitMask;
    hkUint32Le m_aBitMask;
};

struct hkDdsHeader {
    hkUint32Le         m_magic;
    hkUint32Le         m_size;
    hkUint32Le         m_flags;
    hkUint32Le         m_height;
    hkUint32Le         m_width;
    hkUint32Le         m_pitchOrLinearSize;
    hkUint32Le         m_depth;
    hkUint32Le         m_mipMapCount;
    hkUint32Le         m_reserved1[11];
    hkDdsPixelFormat m_ddspf;
    hkUint32Le         m_caps;
    hkUint32Le         m_caps2;
    hkUint32Le         m_caps3;
    hkUint32Le         m_caps4;
    hkUint32Le         m_reserved2;
};

struct hkDdsResourceDimension {
    enum Enum
    {
        TEXTURE1D = 2,
        TEXTURE2D = 3,
        TEXTURE3D = 4,
    };
};

struct hkDdsResourceMiscFlags {
    enum Enum
    {
        TEXTURECUBE = 0x4,
    };
};

struct hkDdsHeaderDxt10 {
    hkUint32Le m_dxgiFormat;
    hkUint32Le m_resourceDimension;
    hkUint32Le m_miscFlag;
    hkUint32Le m_arraySize;
    hkUint32Le m_miscFlags2;
};

struct hkDdsdFlags
{
    enum Enum {
        CAPS = 0x000001,
        HEIGHT = 0x000002,
        WIDTH = 0x000004,
        PITCH = 0x000008,
        PIXELFORMAT = 0x001000,
        MIPMAPCOUNT = 0x020000,
        LINEARSIZE = 0x080000,
        DEPTH = 0x800000,
    };
};

struct hkDdpfFlags
{
    enum Enum {
        ALPHAPIXELS = 0x00001,
        ALPHA = 0x00002,
        FOURCC = 0x00004,
        RGB = 0x00040,
        YUV = 0x00200,
        LUMINANCE = 0x20000,
    };
};

struct hkDdsCaps
{
    enum Enum {
        COMPLEX = 0x000008,
        MIPMAP = 0x400000,
        TEXTURE = 0x001000,
    };
};

struct hkDdsCaps2
{
    enum Enum {
        CUBEMAP = 0x000200,
        CUBEMAP_POSITIVEX = 0x000400,
        CUBEMAP_NEGATIVEX = 0x000800,
        CUBEMAP_POSITIVEY = 0x001000,
        CUBEMAP_NEGATIVEY = 0x002000,
        CUBEMAP_POSITIVEZ = 0x004000,
        CUBEMAP_NEGATIVEZ = 0x008000,
        VOLUME = 0x200000,
    };
};

static const hkUint32 hkDdsMagic = 0x20534444;
static const hkUint32 hkDdsDxt10FourCc = 0x30315844;

#define MAKE_FOURCC(a, b, c, d) (a) | ((b) << 8) | ((c) << 16) | ((d) << 24)

namespace
{
    struct hkDdsFileInfo : public hkDdsHeader
    {
        bool m_pitch;
        hkImageFormat::Enum m_pixelFormat;
        bool m_complex;
        bool m_cubeMap;
        bool m_volume;
    };


    hkUint32 toFourCc(hkImageFormat::Enum format)
    {
        switch (format)
        {
        case hkImageFormat::BC1_UNSIGNED_NORMALIZED:
            return MAKE_FOURCC('D', 'X', 'T', '1');

        case hkImageFormat::BC2_UNSIGNED_NORMALIZED:
            return MAKE_FOURCC('D', 'X', 'T', '3');

        case hkImageFormat::BC3_UNSIGNED_NORMALIZED:
            return MAKE_FOURCC('D', 'X', 'T', '5');

        case hkImageFormat::BC4_UNSIGNED_NORMALIZED:
            return MAKE_FOURCC('A', 'T', 'I', '1');

        case hkImageFormat::BC5_UNSIGNED_NORMALIZED:
            return MAKE_FOURCC('A', 'T', 'I', '2');

        case hkImageFormat::R16_G16_B16_A16_UNSIGNED_NORMALIZED:
            return 36;

        case hkImageFormat::R16_FLOAT:
            return 111;

        case hkImageFormat::R16_G16_FLOAT:
            return 112;

        case hkImageFormat::R16_G16_B16_A16_FLOAT:
            return 113;

        case hkImageFormat::R32_FLOAT:
            return 114;

        case hkImageFormat::R32_G32_FLOAT:
            return 115;

        case hkImageFormat::R32_G32_B32_A32_FLOAT:
            return 116;

        default:
            return 0;
        }
    }

    hkImageFormat::Enum fromFourCc(hkUint32 fourCc)
    {
        switch (fourCc)
        {
        case MAKE_FOURCC('D', 'X', 'T', '1'):
            return hkImageFormat::BC1_UNSIGNED_NORMALIZED;

        case MAKE_FOURCC('D', 'X', 'T', '2'):
        case MAKE_FOURCC('D', 'X', 'T', '3'):
            return hkImageFormat::BC2_UNSIGNED_NORMALIZED;

        case MAKE_FOURCC('D', 'X', 'T', '4'):
        case MAKE_FOURCC('D', 'X', 'T', '5'):
            return hkImageFormat::BC3_UNSIGNED_NORMALIZED;

        case MAKE_FOURCC('A', 'T', 'I', '1'):
        case MAKE_FOURCC('B', 'C', '4', 'U'):
            return hkImageFormat::BC4_UNSIGNED_NORMALIZED;

        case MAKE_FOURCC('B', 'C', '4', 'S'):
            return hkImageFormat::BC4_SIGNED_NORMALIZED;

        case MAKE_FOURCC('A', 'T', 'I', '2'):
        case MAKE_FOURCC('B', 'C', '5', 'U'):
            return hkImageFormat::BC5_UNSIGNED_NORMALIZED;

        case MAKE_FOURCC('B', 'C', '5', 'S'):
            return hkImageFormat::BC5_SIGNED_NORMALIZED;

        case 36:
            return hkImageFormat::R16_G16_B16_A16_UNSIGNED_NORMALIZED;

        case 111:
            return hkImageFormat::R16_FLOAT;

        case 112:
            return hkImageFormat::R16_G16_FLOAT;

        case 113:
            return hkImageFormat::R16_G16_B16_A16_FLOAT;

        case 114:
            return hkImageFormat::R32_FLOAT;

        case 115:
            return hkImageFormat::R32_G32_FLOAT;

        case 116:
            return hkImageFormat::R32_G32_B32_A32_FLOAT;

        default:
            return hkImageFormat::INVALID;
        }
    }
}

static hkResult readDdsInfo(hkIo::ReadBuffer& buffer, hkDdsFileInfo& out_fileInfo)
{
    hkDdsHeader& fileHeader = out_fileInfo;
    if (buffer.read(&fileHeader, sizeof(hkDdsHeader)) != sizeof(hkDdsHeader))
    {
        Log_Error("Failed to read file header.");
        return HK_FAILURE;
    }

    if (hkUint32(fileHeader.m_magic) != hkDdsMagic)
    {
        Log_Error("The file is not a recognized DDS file.");
        return HK_FAILURE;
    }

    if (hkUint32(fileHeader.m_size) != 124)
    {
        Log_Error("The file header size {} doesn't match the expected size of 124.", fileHeader.m_size);
        return HK_FAILURE;
    }

    out_fileInfo.m_pitch = (hkUint32(fileHeader.m_flags) & hkDdsdFlags::PITCH) != 0;

    if (hkUint32(fileHeader.m_ddspf.m_size) != 32)
    {
        Log_Error("The pixel format size {} doesn't match the expected value of 32.", fileHeader.m_ddspf.m_size);
        return HK_FAILURE;
    }

    hkDdsHeaderDxt10 headerDxt10;

    hkImageFormat::Enum format = hkImageFormat::INVALID;

    if ((hkUint32(fileHeader.m_ddspf.m_flags) & hkDdpfFlags::FOURCC) != 0)
    {
        if (hkUint32(fileHeader.m_ddspf.m_fourCC) == hkDdsDxt10FourCc)
        {
            if (buffer.read(&headerDxt10, sizeof(hkDdsHeaderDxt10)) != sizeof(hkDdsHeaderDxt10))
            {
                Log_Error("Failed to read file header.");
                return HK_FAILURE;
            }

            format = hkImageFormat::fromDxgiFormat(headerDxt10.m_dxgiFormat);

            if (format == hkImageFormat::INVALID)
            {
                Log_Error("The DXGI format {} has no equivalent image format.", headerDxt10.m_dxgiFormat);
                return HK_FAILURE;
            }
        }
        else
        {
            format = fromFourCc(fileHeader.m_ddspf.m_fourCC);

            if (format == hkImageFormat::INVALID)
            {
                char fourCcString[5] = {
                    char((hkUint32(fileHeader.m_ddspf.m_fourCC) >> 0) & 0xFF),
                    char((hkUint32(fileHeader.m_ddspf.m_fourCC) >> 8) & 0xFF),
                    char((hkUint32(fileHeader.m_ddspf.m_fourCC) >> 16) & 0xFF),
                    char((hkUint32(fileHeader.m_ddspf.m_fourCC) >> 24) & 0xFF),
                    0
                };
                Log_Error("The FourCC code '{}' was not recognized.", fourCcString);
                return HK_FAILURE;
            }
        }
    }
    else if ((hkUint32(fileHeader.m_ddspf.m_flags) & (hkDdpfFlags::ALPHAPIXELS | hkDdpfFlags::ALPHA | hkDdpfFlags::RGB | hkDdpfFlags::LUMINANCE)) != 0)
    {
        format = hkImageFormat::fromPixelMaskAndCount(
            fileHeader.m_ddspf.m_rBitMask, fileHeader.m_ddspf.m_gBitMask,
            fileHeader.m_ddspf.m_bBitMask, fileHeader.m_ddspf.m_aBitMask,
            fileHeader.m_ddspf.m_rgbBitCount);

        if (format == hkImageFormat::INVALID)
        {
            Log_Error("The pixel mask specified was not recognized (R: 0x{%08x}, G: 0x{%08x}, B: 0x{%08x}, A: 0x{%08x}, BPP: {}).",
                fileHeader.m_ddspf.m_rBitMask, fileHeader.m_ddspf.m_gBitMask, fileHeader.m_ddspf.m_bBitMask, fileHeader.m_ddspf.m_aBitMask,
                fileHeader.m_ddspf.m_rgbBitCount);
            return HK_FAILURE;
        }
    }
    else
    {
        Log_Error("The image format is neither specified as a pixel mask nor as a FourCC code.");
        return HK_FAILURE;
    }

    out_fileInfo.m_pixelFormat = format;

    out_fileInfo.m_complex = (hkUint32(fileHeader.m_caps) & hkDdsCaps::COMPLEX) != 0;
    out_fileInfo.m_cubeMap = (hkUint32(fileHeader.m_caps2) & hkDdsCaps2::CUBEMAP) != 0;
    out_fileInfo.m_volume = (hkUint32(fileHeader.m_caps2) & hkDdsCaps2::VOLUME) != 0;

    // Cubemap and volume texture are mutually exclusive
    if (out_fileInfo.m_volume && out_fileInfo.m_cubeMap)
    {
        Log_Error("The header specifies both the VOLUME and CUBEMAP flags.");
        return HK_FAILURE;
    }

    return HK_SUCCESS;
}

hkResult hkDdsFileFormat::readImageHeader(const hkIo::Detail::ReadBufferAdapter& stream, hkImageHeader& out_header) const
{
    hkIo::ReadBuffer buffer;
    buffer.attach(stream);

    hkDdsFileInfo info;
    if (readDdsInfo(buffer, info).isFailure())
    {
        return HK_FAILURE;
    }

    out_header = hkImageHeader();
    out_header.setWidth(info.m_width);
    out_header.setHeight(info.m_height);

    out_header.setFormat(info.m_pixelFormat);

    out_header.setNumMipLevels(hkMath::max2(1U, info.m_mipMapCount));

    if (info.m_cubeMap)
    {
        out_header.setNumFaces(6);
    }
    else if (info.m_volume)
    {
        out_header.setDepth(info.m_depth);
    }

    return HK_SUCCESS;
}

hkResult hkDdsFileFormat::readImage(const hkIo::Detail::ReadBufferAdapter& stream, hkImage& image) const
{
    hkIo::ReadBuffer buffer;
    buffer.attach(stream);

    hkDdsFileInfo info;
    if (readDdsInfo(buffer, info).isFailure())
    {
        return HK_FAILURE;
    }

    hkImageHeader header;
    header.setWidth(info.m_width);
    header.setHeight(info.m_height);

    header.setFormat(info.m_pixelFormat);

    header.setNumMipLevels(hkMath::max2(1U, hkUint32(info.m_mipMapCount)));

    if (info.m_cubeMap)
    {
        header.setNumFaces(6);
    }
    else if (info.m_volume)
    {
        header.setDepth(info.m_depth);
    }

    image.reset(header);

    // If pitch is specified, it must match the computed value
    if (info.m_pitch && image.getRowPitch(0) != hkUint32(info.m_pitchOrLinearSize))
    {
        Log_Error("The row pitch specified in the header doesn't match the expected pitch.");
        return HK_FAILURE;
    }

    hkUint32 dataSize = image.getDataSize();

    if (buffer.read(image.getDataPointer<void>(), dataSize) != hkLong(dataSize))
    {
        Log_Error("Failed to read image data.");
        return HK_FAILURE;
    }

    return HK_SUCCESS;
}

hkResult hkDdsFileFormat::writeImage(const hkIo::Detail::WriteBufferAdapter& stream, const hkImage& image) const
{
    return writeImage(stream, image, false);
}

hkResult hkDdsFileFormat::writeImage(const hkIo::Detail::WriteBufferAdapter& stream, const hkImage& image, bool forceDxt10) const
{
    hkIo::WriteBuffer buffer;
    buffer.attach(stream);

    const hkImageFormat::Enum format = image.getFormat();

    const hkUint32 numFaces = image.getNumFaces();
    const hkUint32 numMipLevels = image.getNumMipLevels();
    const hkUint32 numArrayIndices = image.getNumArrayElements();

    const hkUint32 width = image.getWidth(0);
    const hkUint32 height = image.getHeight(0);
    const hkUint32 depth = image.getDepth(0);

    bool hasMipMaps = numMipLevels > 1;
    bool volume = depth > 1;
    bool cubeMap = numFaces > 1;
    bool array = numArrayIndices > 1;

    hkDdsHeader fileHeader;
    hkDdsHeaderDxt10 headerDxt10;

    hkString::memSet(&fileHeader, 0, sizeof(fileHeader));
    hkString::memSet(&headerDxt10, 0, sizeof(headerDxt10));

    fileHeader.m_magic = hkDdsMagic;
    fileHeader.m_size = 124;
    fileHeader.m_width = width;
    fileHeader.m_height = height;

    // Required in every .dds file.
    hkUint32 flags = hkDdsdFlags::WIDTH | hkDdsdFlags::HEIGHT | hkDdsdFlags::CAPS | hkDdsdFlags::PIXELFORMAT;

    if (hasMipMaps)
    {
        flags |= hkDdsdFlags::MIPMAPCOUNT;
        fileHeader.m_mipMapCount = numMipLevels;
    }

    if (volume)
    {
        // Volume and array are incompatible
        if (array)
        {
            Log_Error("The image is both an array and volume texture. This is not supported.");
            return HK_FAILURE;
        }

        flags |= hkDdsdFlags::DEPTH;
        fileHeader.m_depth = depth;
    }

    if (hkImageFormat::isCompressed(image.getFormat()))
    {
        flags |= hkDdsdFlags::LINEARSIZE;
        fileHeader.m_pitchOrLinearSize = 0;
    }
    else
    {
        flags |= hkDdsdFlags::PITCH;
        fileHeader.m_pitchOrLinearSize = image.getRowPitch(0);
    }

    hkUint32 caps = hkDdsCaps::TEXTURE;
    hkUint32 caps2 = 0;

    if (cubeMap)
    {
        if (numFaces != 6)
        {
            Log_Error("The image is a cubemap, but has {} faces instead of the expected 6.", numFaces);
            return HK_FAILURE;
        }

        if (volume)
        {
            Log_Error("The image is both a cubemap and volume texture. This is not supported.");
            return HK_FAILURE;
        }

        caps |= hkDdsCaps::COMPLEX;
        caps2 |= hkDdsCaps2::CUBEMAP |
            hkDdsCaps2::CUBEMAP_POSITIVEX | hkDdsCaps2::CUBEMAP_NEGATIVEX |
            hkDdsCaps2::CUBEMAP_POSITIVEY | hkDdsCaps2::CUBEMAP_NEGATIVEY |
            hkDdsCaps2::CUBEMAP_POSITIVEZ | hkDdsCaps2::CUBEMAP_NEGATIVEZ;
    }

    if (array)
    {
        caps |= hkDdsCaps::COMPLEX;

        // Must be written as DXT10
        forceDxt10 = true;
    }

    if (volume)
    {
        caps |= hkDdsCaps::COMPLEX;
        caps2 |= hkDdsCaps2::VOLUME;
    }

    if (hasMipMaps)
    {
        caps |= hkDdsCaps::MIPMAP | hkDdsCaps::COMPLEX;
    }

    fileHeader.m_flags = flags;
    fileHeader.m_caps = caps;
    fileHeader.m_caps2 = caps2;

    fileHeader.m_ddspf.m_size = 32;

    hkUint32 redMask = hkImageFormat::getChannelMask(format, hkImageFormat::R);
    hkUint32 greenMask = hkImageFormat::getChannelMask(format, hkImageFormat::G);
    hkUint32 blueMask = hkImageFormat::getChannelMask(format, hkImageFormat::B);
    hkUint32 alphaMask = hkImageFormat::getChannelMask(format, hkImageFormat::A);
    hkUint32 bpp = hkImageFormat::getBitsPerPixel(format);

    hkUint32 fourCc = toFourCc(format);
    hkUint32 dxgiFormat = hkImageFormat::toDxgiFormat(format);

    // When not required to use a DXT10 texture, try to write a legacy DDS by specifying FourCC or pixel masks
    if (!forceDxt10)
    {
        // The format has a known mask and we would also recognize it as the same when reading back in, since multiple formats may have the same pixel masks
        if ((redMask | greenMask | blueMask | alphaMask) && format == hkImageFormat::fromPixelMaskAndCount(redMask, greenMask, blueMask, alphaMask, bpp))
        {
            fileHeader.m_ddspf.m_flags = hkDdpfFlags::ALPHAPIXELS | hkDdpfFlags::RGB;
            fileHeader.m_ddspf.m_rBitMask = redMask;
            fileHeader.m_ddspf.m_gBitMask = greenMask;
            fileHeader.m_ddspf.m_bBitMask = blueMask;
            fileHeader.m_ddspf.m_aBitMask = alphaMask;
            fileHeader.m_ddspf.m_rgbBitCount = bpp;
        }
        // The format has a known FourCC
        else if (fourCc != 0)
        {
            fileHeader.m_ddspf.m_flags = hkDdpfFlags::FOURCC;
            fileHeader.m_ddspf.m_fourCC = fourCc;
        }
        else
        {
            // Fallback to DXT10 path
            forceDxt10 = true;
        }
    }

    if (forceDxt10)
    {
        // We must write a DXT10 file, but there is no matching DXGI_FORMAT - we could also try converting, but that is rarely intended when writing .dds
        if (dxgiFormat == 0)
        {
            Log_Error("The image needs to be written as a DXT10 file, but no matching DXGI format was found for '{}'.",
                hkImageFormat::getName(format));
            return HK_FAILURE;
        }

        fileHeader.m_ddspf.m_flags = hkDdpfFlags::FOURCC;
        fileHeader.m_ddspf.m_fourCC = hkDdsDxt10FourCc;

        headerDxt10.m_dxgiFormat = dxgiFormat;

        if (volume)
        {
            headerDxt10.m_resourceDimension = hkDdsResourceDimension::TEXTURE3D;
        }
        else if (height > 1)
        {
            headerDxt10.m_resourceDimension = hkDdsResourceDimension::TEXTURE2D;
        }
        else
        {
            headerDxt10.m_resourceDimension = hkDdsResourceDimension::TEXTURE1D;
        }

        if (cubeMap)
        {
            headerDxt10.m_miscFlag = hkDdsResourceMiscFlags::TEXTURECUBE;
        }

        // NOT multiplied by number of cubemap faces
        headerDxt10.m_arraySize = numArrayIndices;

        // Can be used to describe the alpha channel usage, but automatically makes it incompatible with the D3DX libraries if not 0.
        headerDxt10.m_miscFlags2 = 0;
    }

    if (buffer.writeRaw(&fileHeader, sizeof(fileHeader)) != sizeof(fileHeader))
    {
        Log_Error("Failed to write image header.");
        return HK_FAILURE;
    }

    if (forceDxt10)
    {
        if (buffer.writeRaw(&headerDxt10, sizeof(headerDxt10)) != sizeof(headerDxt10))
        {
            Log_Error("Failed to write image DX10 header.");
            return HK_FAILURE;
        }
    }

    if (buffer.writeRaw(image.getDataPointer<void>(), image.getDataSize()) != hkLong(image.getDataSize()))
    {
        Log_Error("Failed to write image data.");
        return HK_FAILURE;
    }

    return HK_SUCCESS;
}

hkArrayView<const char* const> hkDdsFileFormat::getReadableExtensions() const
{
    static const char* const extensions[] = { "dds" };
    return hkArrayViewT::make( extensions );
}

hkArrayView<const char* const> hkDdsFileFormat::getWriteableExtensions() const
{
    return getReadableExtensions();
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
