// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY       : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/ImageUtilities/hkImageUtilities.h>

#include <Common/ImageUtilities/FileFormats/hkBmpFileFormat.h>

#include <Common/ImageUtilities/Image/hkImage.h>
#include <Common/ImageUtilities/Conversion/hkImageConversion.h>

#include <Common/Base/Serialize/Detail/hkWriteBuffer.h>
#include <Common/Base/Serialize/Detail/hkReadBuffer.h>

#define DEBUG_LOG_IDENTIFIER "image.bmp"
#include <Common/Base/System/Log/hkLog.hxx>

enum hkBmpCompression
{
    RGB = 0L,
    RLE8 = 1L,
    RLE4 = 2L,
    BITFIELDS = 3L,
    JPEG = 4L,
    PNG = 5L,
};


#pragma pack(push,1)
struct hkBmpFileHeader
{
    hkUint16Le m_type;
    hkUint32Le m_size;
    hkUint16Le m_reserved1;
    hkUint16Le m_reserved2;
    hkUint32Le m_offBits;
};
#pragma pack(pop)

struct hkBmpFileInfoHeader {
    hkUint32Le          m_size;
    hkUint32Le          m_width;
    hkUint32Le          m_height;
    hkUint16Le          m_planes;
    hkUint16Le          m_bitCount;
    hkUint32Le          m_compression;
    hkUint32Le          m_sizeImage;
    hkUint32Le          m_xPelsPerMeter;
    hkUint32Le          m_yPelsPerMeter;
    hkUint32Le          m_clrUsed;
    hkUint32Le          m_clrImportant;
};

struct hkCIEXYZ
{
    hkInt32Le ciexyzX;
    hkInt32Le ciexyzY;
    hkInt32Le ciexyzZ;
};

struct hkCIEXYZTRIPLE
{
    hkCIEXYZ ciexyzRed;
    hkCIEXYZ ciexyzGreen;
    hkCIEXYZ ciexyzBlue;
};

struct hkBmpFileInfoHeaderV4 {
    hkUint32Le        m_redMask;
    hkUint32Le        m_greenMask;
    hkUint32Le        m_blueMask;
    hkUint32Le        m_alphaMask;
    hkUint32Le        m_csType;
    hkCIEXYZTRIPLE  m_endpoints;
    hkUint32Le        m_gammaRed;
    hkUint32Le        m_gammaGreen;
    hkUint32Le        m_gammaBlue;
};

HK_COMPILE_TIME_ASSERT(sizeof(hkCIEXYZTRIPLE) == 3 * 3 * 4);

struct hkBmpFileInfoHeaderV5 {
    hkUint32Le        m_intent;
    hkUint32Le        m_profileData;
    hkUint32Le        m_profileSize;
    hkUint32Le        m_reserved;
};

static const hkUint16 hkBmpFileMagic = 0x4D42u;

struct hkBmpBgrxQuad {
    hkBmpBgrxQuad()
    {
    }

    hkBmpBgrxQuad(hkUint8 red, hkUint8 green, hkUint8 blue) : m_blue(blue), m_green(green), m_red(red), m_reserved(0)
    {
    }

    hkUint8 m_blue;
    hkUint8 m_green;
    hkUint8 m_red;
    hkUint8 m_reserved;
};

hkResult hkBmpFileFormat::writeImage(const hkIo::Detail::WriteBufferAdapter& stream, const hkImage& image) const
{
    // Technically almost arbitrary formats are supported, but we only use the common ones.
    hkImageFormat::Enum compatibleFormats[] =
    {
        hkImageFormat::B8_G8_R8_X8_UNSIGNED_NORMALIZED,
        hkImageFormat::B8_G8_R8_X8_UNSIGNED_NORMALIZED_SRGB,
        hkImageFormat::B8_G8_R8_A8_UNSIGNED_NORMALIZED,
        hkImageFormat::B8_G8_R8_A8_UNSIGNED_NORMALIZED_SRGB,
        hkImageFormat::B8_G8_R8_UNSIGNED_NORMALIZED,
        hkImageFormat::B8_G8_R8_UNSIGNED_NORMALIZED_SRGB,
        //  hkImageFormat::X1R5G5B5_UNORM,
        //  hkImageFormat::X1R5G5B5_UNORM_SRGB,
        hkImageFormat::B5_G6_R5_UNSIGNED_NORMALIZED
    };

    // Find a compatible format closest to the one the image currently has
    hkImageFormat::Enum format = hkImageConversion::findClosestCompatibleFormat(image.getFormat(), hkArrayViewT::make(compatibleFormats));

    if (format == hkImageFormat::INVALID)
    {
        Log_Error("No conversion from format '{}' to a format suitable for BMP files known.", hkImageFormat::getName(image.getFormat()));
        return HK_FAILURE;
    }

    // Convert if not already in a compatible format
    if (format != image.getFormat())
    {
        hkImage convertedImage;
        if (hkImageConversion::convert(image, convertedImage, format).isFailure())
        {
            // This should never happen
            HK_REPORT_FAILURE(0x67a8058d, "hkImageConversion::Convert failed even though the conversion was to the format returned by findClosestCompatibleFormat.");
            return HK_FAILURE;
        }

        return writeImage(stream, convertedImage);
    }

    hkIo::WriteBuffer buffer;
    buffer.attach(stream);

    hkUint32 rowPitch = image.getRowPitch(0);

    hkUint32 height = image.getHeight(0);

    int dataSize = rowPitch * height;

    hkBmpFileInfoHeader fileInfoHeader;
    fileInfoHeader.m_width = image.getWidth(0);
    fileInfoHeader.m_height = height;
    fileInfoHeader.m_planes = 1;
    fileInfoHeader.m_bitCount = hkImageFormat::getBitsPerPixel(format);

    fileInfoHeader.m_sizeImage = 0; // Can be zero unless we store the data compressed

    fileInfoHeader.m_xPelsPerMeter = 0;
    fileInfoHeader.m_yPelsPerMeter = 0;
    fileInfoHeader.m_clrUsed = 0;
    fileInfoHeader.m_clrImportant = 0;

    bool writeColorMask = false;

    // Prefer to write a V3 header
    hkUint32 headerVersion = 3;

    switch (format)
    {
    case hkImageFormat::B8_G8_R8_X8_UNSIGNED_NORMALIZED:
    case hkImageFormat::B8_G8_R8_X8_UNSIGNED_NORMALIZED_SRGB:
        //case hkImageFormat::X1R5G5B5_UNORM:
    case hkImageFormat::B8_G8_R8_UNSIGNED_NORMALIZED:
    case hkImageFormat::B8_G8_R8_UNSIGNED_NORMALIZED_SRGB:
        fileInfoHeader.m_compression = RGB;
        break;

    case hkImageFormat::B8_G8_R8_A8_UNSIGNED_NORMALIZED:
    case hkImageFormat::B8_G8_R8_A8_UNSIGNED_NORMALIZED_SRGB:
        fileInfoHeader.m_compression = BITFIELDS;
        headerVersion = 4;
        break;

    case hkImageFormat::B5_G6_R5_UNSIGNED_NORMALIZED:
        fileInfoHeader.m_compression = BITFIELDS;
        writeColorMask = true;
        break;

    default:
        return HK_FAILURE;
    }

    hkUint32 fileInfoHeaderSize = sizeof(hkBmpFileInfoHeader);
    hkUint32 headerSize = sizeof(hkBmpFileHeader);

    if (headerVersion >= 4)
    {
        fileInfoHeaderSize += sizeof(hkBmpFileInfoHeaderV4);
    }
    else if (writeColorMask)
    {
        headerSize += 3 * sizeof(hkUint32);
    }

    headerSize += fileInfoHeaderSize;

    fileInfoHeader.m_size = fileInfoHeaderSize;

    hkBmpFileHeader header;
    header.m_type = hkBmpFileMagic;
    header.m_size = headerSize + dataSize;
    header.m_reserved1 = 0;
    header.m_reserved2 = 0;
    header.m_offBits = headerSize;

    // Write all data
    if (buffer.writeRaw(&header, sizeof(header)) != sizeof(header))
    {
        Log_Error("Failed to write header.");
        return HK_FAILURE;
    }

    if (buffer.writeRaw(&fileInfoHeader, sizeof(fileInfoHeader)) != sizeof(fileInfoHeader))
    {
        Log_Error("Failed to write fileInfoHeader.");
        return HK_FAILURE;
    }

    if (headerVersion >= 4)
    {
        hkBmpFileInfoHeaderV4 fileInfoHeaderV4;
        hkString::memSet(&fileInfoHeaderV4, 0, sizeof(fileInfoHeaderV4));

        fileInfoHeaderV4.m_redMask = hkImageFormat::getChannelMask(format, hkImageFormat::R);
        fileInfoHeaderV4.m_greenMask = hkImageFormat::getChannelMask(format, hkImageFormat::G);
        fileInfoHeaderV4.m_blueMask = hkImageFormat::getChannelMask(format, hkImageFormat::B);
        fileInfoHeaderV4.m_alphaMask = hkImageFormat::getChannelMask(format, hkImageFormat::A);

        if (buffer.writeRaw(&fileInfoHeaderV4, sizeof(fileInfoHeaderV4)) != sizeof(fileInfoHeaderV4))
        {
            Log_Error("Failed to write fileInfoHeaderV4.");
            return HK_FAILURE;
        }
    }
    else if (writeColorMask)
    {
        struct
        {
            hkUint32Le m_red;
            hkUint32Le m_green;
            hkUint32Le m_blue;
        } colorMask;


        colorMask.m_red = hkImageFormat::getChannelMask(format, hkImageFormat::R);
        colorMask.m_green = hkImageFormat::getChannelMask(format, hkImageFormat::G);
        colorMask.m_blue = hkImageFormat::getChannelMask(format, hkImageFormat::B);

        if (buffer.writeRaw(&colorMask, sizeof(colorMask)) != sizeof(colorMask))
        {
            Log_Error("Failed to write colorMask.");
            return HK_FAILURE;
        }
    }

    const hkUint32 paddedRowPitch = ((rowPitch - 1) / 4 + 1) * 4;
    // Write rows in reverse order
    for (hkInt32 iRow = height - 1; iRow >= 0; iRow--)
    {
        if (buffer.writeRaw(image.getPixelPointer<void>(0, 0, 0, 0, iRow, 0), paddedRowPitch) != hkLong(paddedRowPitch))
        {
            Log_Error("Failed to write data.");
            return HK_FAILURE;
        }
    }

    return HK_SUCCESS;
}

namespace
{
    hkUint32 extractBits(const void* pData, hkUint32 bitAddress, hkUint32 numBits)
    {
        hkUint32 mask = (1U << numBits) - 1;
        hkUint32 byteAddress = bitAddress / 8;
        hkUint32 shiftAmount = 7 - (bitAddress % 8 + numBits - 1);

        return (reinterpret_cast<const hkUint8*>(pData)[byteAddress] >> shiftAmount) & mask;
    }

    struct hkBmpInfo : public hkBmpFileInfoHeader
    {
        bool indexed;
        bool compressed;
        hkUint32 bpp;
        hkImageFormat::Enum format;
        hkUint32 dataSize;
        int rowPitchIn;
    };

    hkResult readBmpInfo(hkIo::ReadBuffer& buffer, hkBmpInfo& info)
    {
        hkBmpFileHeader fileHeader;
        if (buffer.read(&fileHeader, sizeof(hkBmpFileHeader)) != sizeof(hkBmpFileHeader))
        {
            Log_Error("Failed to read header data.");
            return HK_FAILURE;
        }

        // Some very old BMP variants may have different magic numbers, but we don't support them.
        if (hkUint32(fileHeader.m_type) != hkBmpFileMagic)
        {
            Log_Error("The file is not a BMP file. Expected magic number %04x, got %04x.", hkBmpFileMagic, fileHeader.m_type);
            return HK_FAILURE;
        }

        // We expect at least header version 3
        hkUint32 headerVersion = 3;
        if (buffer.read(&info, sizeof(hkBmpFileInfoHeader)) != sizeof(hkBmpFileInfoHeader))
        {
            Log_Error("Failed to read header data (V3).");
            return HK_FAILURE;
        }

        int remainingHeaderBytes = hkUint32(info.m_size) - sizeof(hkBmpFileInfoHeader);

        // File header shorter than expected - happens with corrupt files or e.g. with OS/2 BMP files which may have shorter headers
        if (remainingHeaderBytes < 0)
        {
            Log_Error("The file header was shorter than expected.");
            return HK_FAILURE;
        }

        // Newer files may have a header version 4 (required for transparency)
        hkBmpFileInfoHeaderV4 fileInfoHeaderV4;
        if (remainingHeaderBytes >= sizeof(hkBmpFileInfoHeaderV4))
        {
            headerVersion = 4;
            if (buffer.read(&fileInfoHeaderV4, sizeof(hkBmpFileInfoHeaderV4)) != sizeof(hkBmpFileInfoHeaderV4))
            {
                Log_Error("Failed to read header data (V4).");
                return HK_FAILURE;
            }
            remainingHeaderBytes -= sizeof(hkBmpFileInfoHeaderV4);
        }

        // Skip rest of header
        if (buffer.skip(remainingHeaderBytes) != remainingHeaderBytes)
        {
            Log_Error("Failed to skip remaining header data.");
            return HK_FAILURE;
        }

        info.bpp = info.m_bitCount;

        // Find target format to load the image
        info.format = hkImageFormat::INVALID;
        info.indexed = false;
        info.compressed = false;

        switch (hkUint32(info.m_compression))
        {
            // RGB or indexed data
        case  RGB:
            switch (info.bpp)
            {
            case 1:
            case 4:
            case 8:
                info.indexed = true;

                // We always decompress indexed to BGRX, since the palette is specified in this format
                info.format = hkImageFormat::B8_G8_R8_X8_UNSIGNED_NORMALIZED;
                break;

            case 16:
                //info.format = hkImageFormat::X1R5G5B5_UNORM;
                break;

            case 24:
                info.format = hkImageFormat::B8_G8_R8_UNSIGNED_NORMALIZED;
                break;

            case 32:
                info.format = hkImageFormat::B8_G8_R8_X8_UNSIGNED_NORMALIZED;
                break;
            }
            break;

            // RGB data, but with the color masks specified in place of the palette
        case BITFIELDS:
            switch (info.bpp)
            {
            case 16:
            case 32:
                // In case of old headers, the color masks appear after the header (and aren't counted as part of it)
                if (headerVersion < 4)
                {
                    // Color masks (w/o alpha channel)
                    struct
                    {
                        hkUint32Le m_red;
                        hkUint32Le m_green;
                        hkUint32Le m_blue;
                    } colorMask;

                    if (buffer.read(&colorMask, sizeof(colorMask)) != sizeof(colorMask))
                    {
                        return HK_FAILURE;
                    }

                    info.format = hkImageFormat::fromPixelMaskAndCount(colorMask.m_red, colorMask.m_green, colorMask.m_blue, 0, info.bpp);
                }
                else
                {
                    // For header version four and higher, the color masks are part of the header
                    info.format = hkImageFormat::fromPixelMaskAndCount(
                        fileInfoHeaderV4.m_redMask, fileInfoHeaderV4.m_greenMask,
                        fileInfoHeaderV4.m_blueMask, fileInfoHeaderV4.m_alphaMask, info.bpp);
                }

                break;
            }
            break;

        case RLE4:
            if (info.bpp == 4)
            {
                info.indexed = true;
                info.compressed = true;
                info.format = hkImageFormat::B8_G8_R8_X8_UNSIGNED_NORMALIZED;
            }
            break;

        case RLE8:
            if (info.bpp == 8)
            {
                info.compressed = true;
                info.indexed = true;
                info.format = hkImageFormat::B8_G8_R8_X8_UNSIGNED_NORMALIZED;
            }
            break;
        }

        if (info.format == hkImageFormat::INVALID)
        {
            Log_Error("Unknown or unsupported BMP encoding.");
            return HK_FAILURE;
        }

        const hkUint32 width = info.m_width;

        if (width > 65536)
        {
            Log_Error("Image specifies width > 65536. Header corrupted?");
            return HK_FAILURE;
        }

        const hkUint32 height = info.m_height;

        if (height > 65536)
        {
            Log_Error("Image specifies height > 65536. Header corrupted?");
            return HK_FAILURE;
        }

        info.dataSize = info.m_sizeImage;

        if (info.dataSize > 1024 * 1024 * 1024)
        {
            Log_Error("Image specifies data size > 1GiB. Header corrupted?");
            return HK_FAILURE;
        }

        info.rowPitchIn = (width * info.bpp + 31) / 32 * 4;

        if (info.dataSize == 0)
        {
            if (hkUint32(info.m_compression) != RGB)
            {
                Log_Error("The data size wasn't specified in the header.");
                return HK_FAILURE;
            }
            info.dataSize = info.rowPitchIn * height;
        }

        return HK_SUCCESS;
    }
}

hkResult hkBmpFileFormat::readImageHeader(const hkIo::Detail::ReadBufferAdapter& stream, hkImageHeader& out_header) const
{
    hkIo::ReadBuffer buffer;
    buffer.attach(stream);

    hkBmpInfo info;
    if (readBmpInfo(buffer, info).isFailure())
    {
        return HK_FAILURE;
    }

    out_header = hkImageHeader();
    out_header.setFormat(info.format);
    out_header.setNumMipLevels(1);
    out_header.setNumArrayElements(1);
    out_header.setNumFaces(1);

    out_header.setWidth(info.m_width);
    out_header.setHeight(info.m_height);
    out_header.setDepth(1);

    return HK_SUCCESS;
}

hkResult hkBmpFileFormat::readImage(const hkIo::Detail::ReadBufferAdapter& stream, hkImage& image) const
{
    hkIo::ReadBuffer buffer;
    buffer.attach(stream);

    hkBmpInfo info;
    if (readBmpInfo(buffer, info).isFailure())
    {
        return HK_FAILURE;
    }

    const hkUint32 width = info.m_width;
    const hkUint32 height = info.m_height;

    // Set image data
    hkImageHeader header;

    header.setFormat(info.format);
    header.setNumMipLevels(1);
    header.setNumArrayElements(1);
    header.setNumFaces(1);

    header.setWidth(width);
    header.setHeight(height);
    header.setDepth(1);

    image.reset(header);

    hkUint32 rowPitch = image.getRowPitch(0);

    if (info.indexed)
    {
        // If no palette size was specified, the full available palette size will be used
        hkUint32 paletteSize = info.m_clrUsed;
        if (paletteSize == 0)
        {
            paletteSize = 1U << info.bpp;
        }
        else if (paletteSize > 65536)
        {
            Log_Error("Palette size > 65536.");
            return HK_FAILURE;
        }

        hkArray<hkBmpBgrxQuad> palette;
        palette.setSize(paletteSize);
        if (buffer.read(&palette[0], paletteSize * sizeof(hkBmpBgrxQuad)) != hkLong(paletteSize * sizeof(hkBmpBgrxQuad)))
        {
            Log_Error("Failed to read palette data.");
            return HK_FAILURE;
        }

        if (info.compressed)
        {
            // Compressed data is always in pairs of bytes
            if (info.dataSize % 2 != 0)
            {
                Log_Error("The data size is not a multiple of 2 bytes in an RLE-compressed file.");
                return HK_FAILURE;
            }

            hkArray<hkUint8> compressedData;
            compressedData.setSize(info.dataSize);

            if (buffer.read(&compressedData[0], info.dataSize) != hkLong(info.dataSize))
            {
                Log_Error("Failed to read data.");
                return HK_FAILURE;
            }

            const hkUint8* pIn = &compressedData[0];
            const hkUint8* pInEnd = pIn + info.dataSize;

            // Current output position
            hkUint32 row = height - 1;
            hkUint32 col = 0;

            hkBmpBgrxQuad* pLine = image.getPixelPointer<hkBmpBgrxQuad>(0, 0, 0, 0, row, 0);

            // Decode RLE data directly to RGBX
            while (pIn < pInEnd)
            {
                hkUint32 byte1 = *pIn++;
                hkUint32 byte2 = *pIn++;

                // Relative mode - the first byte specified a number of indices to be repeated, the second one the indices
                if (byte1 > 0)
                {
                    // Clamp number of repetitions to row width.
                    // The spec isn't clear on this point, but some files pad the number of encoded indices for some reason.
                    byte1 = hkMath::min2(byte1, width - col);

                    if (info.bpp == 4)
                    {
                        // Alternate between two indices.
                        for (hkUint32 rep = 0; rep < byte1 / 2; rep++)
                        {
                            pLine[col++] = palette[byte2 >> 4];
                            pLine[col++] = palette[byte2 & 0x0F];
                        }

                        // Repeat the first index for odd numbers of repetitions.
                        if (byte1 & 1)
                        {
                            pLine[col++] = palette[byte2 >> 4];
                        }
                    }
                    else // if (bpp == 8)
                    {
                        // Repeat a single index.
                        for (hkUint32 rep = 0; rep < byte1; rep++)
                        {
                            pLine[col++] = palette[byte2];
                        }
                    }
                }
                else
                {
                    // Absolute mode - the first byte specifies a number of indices encoded separately, or is a special marker
                    switch (byte2)
                    {
                        // End of line marker
                    case 0:
                    {

                        // Fill up with palette entry 0
                        while (col < width)
                        {
                            pLine[col++] = palette[0];
                        }

                        // Begin next line
                        col = 0;
                        row--;
                        pLine -= width;
                    }

                    break;

                    // End of image marker
                    case 1:
                        // Check that we really reached the end of the image.
                        if (row != 0 && col != height - 1)
                        {
                            Log_Error("Unexpected end of image marker found.");
                            return HK_FAILURE;
                        }
                        break;

                    case 2:
                        Log_Error("Found a RLE compression position delta - this is not supported.");
                        return HK_FAILURE;

                    default:
                        // Read byte2 number of indices

                        // More data than fits into the image or can be read?
                        if (col + byte2 > width || pIn + (byte2 + 1) / 2 > pInEnd)
                        {
                            return HK_FAILURE;
                        }

                        if (info.bpp == 4)
                        {
                            for (hkUint32 rep = 0; rep < byte2 / 2; rep++)
                            {
                                hkUint32 indices = *pIn++;
                                pLine[col++] = palette[indices >> 4];
                                pLine[col++] = palette[indices & 0x0f];
                            }

                            if (byte2 & 1)
                            {
                                pLine[col++] = palette[*pIn++ >> 4];
                            }

                            // Pad to word boundary
                            pIn += (byte2 / 2 + byte2) & 1;
                        }
                        else // if (info.bpp == 8)
                        {
                            for (hkUint32 rep = 0; rep < byte2; rep++)
                            {
                                pLine[col++] = palette[*pIn++];
                            }

                            // Pad to word boundary
                            pIn += byte2 & 1;
                        }
                    }
                }
            }
        }
        else
        {
            hkArray<hkUint8> indexedData;
            indexedData.setSize(info.dataSize);
            if (buffer.read(&indexedData[0], info.dataSize) != hkLong(info.dataSize))
            {
                Log_Error("Failed to read data.");
                return HK_FAILURE;
            }

            // Convert to non-indexed
            for (hkUint32 row = 0; row < height; row++)
            {
                hkUint8* pIn = &indexedData[info.rowPitchIn * row];

                // Convert flipped vertically
                hkBmpBgrxQuad* pOut = image.getPixelPointer<hkBmpBgrxQuad>(0, 0, 0, 0, height - row - 1, 0);
                for (hkUint32 col = 0; col < image.getWidth(0); col++)
                {
                    hkUint32 index = extractBits(pIn, col * info.bpp, info.bpp);
                    if (index >= (hkUint32)palette.getSize())
                    {
                        Log_Error("Image contains invalid palette indices.");
                        return HK_FAILURE;
                    }
                    pOut[col] = palette[index];
                }
            }
        }
    }
    else
    {
        // Format must match the number of bits in the file
        if (hkImageFormat::getBitsPerPixel(info.format) != info.bpp)
        {
            Log_Error("The number of bits per pixel specified in the file ({}) does not match the expected value of {} for the format '{}'.",
                info.bpp, hkImageFormat::getBitsPerPixel(info.format), hkImageFormat::getName(info.format));
            return HK_FAILURE;
        }

        // Skip palette data. Having a palette here doesn't make sense, but is not explicitly disallowed by the standard.
        hkLong paletteSize = hkUint32(info.m_clrUsed) * sizeof(hkBmpBgrxQuad);
        if (buffer.skip(paletteSize) != paletteSize)
        {
            Log_Error("Failed to skip palette data.");
            return HK_FAILURE;
        }

        // Read rows in reverse order
        for (hkInt32 row = height - 1; row >= 0; row--)
        {
            if (buffer.read(image.getPixelPointer<void>(0, 0, 0, 0, row, 0), rowPitch) != hkLong(rowPitch))
            {
                Log_Error("Failed to read row data.");
                return HK_FAILURE;
            }
            if (buffer.skip(info.rowPitchIn - rowPitch) != hkLong(info.rowPitchIn - rowPitch))
            {
                Log_Error("Failed to skip row data.");
                return HK_FAILURE;
            }
        }
    }

    return HK_SUCCESS;
}

hkArrayView<const char* const> hkBmpFileFormat::getReadableExtensions() const
{
    static const char* const extensions[] =
    {
        "bmp",
        "dip",
        "rle"
    };
    return hkArrayViewT::make( extensions );
}

hkArrayView<const char* const> hkBmpFileFormat::getWriteableExtensions() const
{
    return getReadableExtensions();
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
