// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY       : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

// this: #include <Common/ImageUtilities/Conversion/hkImageConversion.h>

#include <Common/Base/Container/Enumerable/hkEnumerableClass.h>
#include <Common/ImageUtilities/Image/hkImageFormat.h>

class hkImage;
class hkThreadPool;

/// Describes the properties of an image conversion routine.
struct hkImageConversionFlags
{
    /// Describes the properties of an image conversion routine.
    enum Enum
    {
        /// No flags.
        None = 0,

        /// The conversion can be performed in-place, ie. source and target image are allowed to be the same. If this flag is not set,
        /// a temporary copy is provided automatically.
        InPlace = 1,
    };
};

/// A structure describing the pairs of source/target format that may be converted using the conversion routine.
struct hkImageConversionEntry
{
    hkImageConversionEntry(hkImageFormat::Enum source, hkImageFormat::Enum target, hkImageConversionFlags::Enum flags) : m_sourceFormat(source), m_targetFormat(target), m_flags(flags)
    {
    }

    const hkImageFormat::Enum m_sourceFormat;
    const hkImageFormat::Enum m_targetFormat;
    const hkImageConversionFlags::Enum m_flags;
};

/// Interface for a single image conversion step.
///
/// The actual functionality is implemented as either hkImageConversionStepLinear or hkImageConversionStepDecompressBlocks.
/// Depending on the types on conversion advertised by GetSupportedConversions(), users of this class need to cast it to a derived type first to access
/// the desired functionality.
class HK_EXPORT_COMMON hkImageConversionStep : public hkEnumerable<hkImageConversionStep>
{
protected:
    hkImageConversionStep();
    virtual ~hkImageConversionStep();

public:
    /// Returns an array pointer of supported conversions.
    ///
    /// \note The returned array must have the same entries each time this method is called.
    virtual hkArrayView<const hkImageConversionEntry> getSupportedConversions() const = 0;

    /// Returns true if the conversion can be called in parallel (default). Conversions that are already multi-threaded or launch external processes should return false.
    virtual bool allowExternalThreading() const;
};

/// Interface for a single image conversion step where both the source and target format are uncompressed.
class HK_EXPORT_COMMON hkImageConversionStepLinear : public hkImageConversionStep
{
public:
    /// converts a batch of pixels.
    virtual hkResult convertPixels(hkArrayView<const void> source, hkArrayView<void> target, hkUint32 numElements, hkImageFormat::Enum sourceFormat, hkImageFormat::Enum targetFormat) const = 0;
};

/// Interface for a single image conversion step where the source format is compressed and the target format is uncompressed.
class HK_EXPORT_COMMON hkImageConversionStepDecompressBlocks : public hkImageConversionStep
{
public:
    /// Decompresses the given number of blocks.
    virtual hkResult decompressBlocks(hkArrayView<const void> source, hkArrayView<void> target, hkUint32 numBlocks, hkImageFormat::Enum sourceFormat, hkImageFormat::Enum targetFormat) const = 0;
};

/// Interface for a single image conversion step where the source format is uncompressed and the target format is compressed.
class HK_EXPORT_COMMON hkImageConversionStepCompressBlocks : public hkImageConversionStep
{
public:
    /// Compresses the given number of blocks.
    virtual hkResult compressBlocks(hkArrayView<const void> source, hkArrayView<void> target, hkUint32 numBlocksX, hkUint32 numBlocksY, hkImageFormat::Enum sourceFormat, hkImageFormat::Enum targetFormat) const = 0;
};


/// Helper class containing utilities to convert between different image formats and layouts.
class HK_EXPORT_COMMON hkImageConversion
{
public:
    /// Checks if there is a known conversion path between the two formats
    static bool isConvertible(hkImageFormat::Enum sourceFormat, hkImageFormat::Enum targetFormat);

    /// Finds the image format from a given list of formats which is the cheapest to convert to.
    static hkImageFormat::Enum
        findClosestCompatibleFormat(hkImageFormat::Enum format, hkArrayView<const hkImageFormat::Enum> compatibleFormats);

    /// A single node along a computed conversion path.
    struct ConversionPathNode
    {
        const hkImageConversionStep* m_step;
        hkImageFormat::Enum m_sourceFormat;
        hkImageFormat::Enum m_targetFormat;
        hkUint32 m_sourceBufferIndex;
        hkUint32 m_targetBufferIndex;
        bool m_inPlace;
    };

    /// Precomputes an optimal conversion path between two formats and the minimal number of required scratch buffers.
    ///
    /// The generated path can be cached by the user if the same conversion is performed multiple times. The path must not be reused if the set of supported conversions changes, e.g.
    /// when plugins are loaded or unloaded.
    ///
    /// \param sourceFormat           The source format.
    /// \param targetFormat           The target format.
    /// \param sourceEqualsTarget     If true, the generated path is applicable if source and target memory regions are equal, and may contain additional copy-steps if the conversion can't be performed in-place.
    ///                               A path generated with sourceEqualsTarget == true will work correctly even if source and target are not the same, but may not be optimal.
    ///                               A path generated with sourceEqualsTarget == false will not work correctly when source and target are the same.
    /// \param path_out               The generated path.
    /// \param numScratchBuffers_out The number of scratch buffers required for the conversion path.
    /// \returns                      HK_SUCCESS if a path was found, HK_FAILURE otherwise.
    static hkResult buildPath(hkImageFormat::Enum sourceFormat, hkImageFormat::Enum targetFormat, bool sourceEqualsTarget, hkInplaceArray<ConversionPathNode, 16> &path_out, hkUint32& numScratchBuffers_out);

    /// converts the source image into a target image with the given format. Source and target may be the same.
    static hkResult convert(const hkImage& source, hkImage& target, hkImageFormat::Enum targetFormat, _In_opt_ hkThreadPool* threadPool = HK_NULL);

    /// converts the source image into a target image using a precomputed conversion path.
    static hkResult convert(const hkImage& source, hkImage& target, hkArrayView<ConversionPathNode> path, hkUint32 numScratchBuffers, _In_opt_ hkThreadPool* threadPool = HK_NULL);

    /// converts the raw source data into a target data buffer with the given format. Source and target may be the same.
    static hkResult convertRaw(hkArrayView<const void> source, hkArrayView<void> target, hkUint32 numElements, hkImageFormat::Enum sourceFormat, hkImageFormat::Enum targetFormat);

    /// converts the raw source data into a target data buffer using a precomputed conversion path.
    static hkResult convertRaw(hkArrayView<const void> source, hkArrayView<void> target, hkUint32 numElements, hkArrayView<ConversionPathNode> path, hkUint32 numScratchBuffers);

private:

    hkImageConversion();
    hkImageConversion( const hkImageConversion& );

    static hkResult convertSingleStep(_In_opt_ const hkImageConversionStep* pStep, const hkImage& source, hkImage& target, hkImageFormat::Enum targetFormat, _In_opt_ hkThreadPool* threadPool);

    static hkResult convertSingleStepDecompress(const hkImage &source, hkImage &target, hkImageFormat::Enum sourceFormat, hkImageFormat::Enum targetFormat, _In_ const hkImageConversionStep* pStep);

    static hkResult convertSingleStepCompress(const hkImage &source, hkImage &target, hkImageFormat::Enum sourceFormat, hkImageFormat::Enum targetFormat, _In_ const hkImageConversionStep* pStep, _In_opt_ hkThreadPool* threadPool);

    static void rebuildConversionTable();
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
