// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/GeometryUtilities/hkGeometryUtilities.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>
#include <Common/GeometryUtilities/Inertia/hkInertiaTensorComputer.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>


static void compareMassProperties(hkMassProperties& mpA, hkMassProperties& mpB, hkReal maxRelError)
{
    HK_TEST(hkMath::equal(mpA.m_mass, mpB.m_mass, maxRelError * mpA.m_mass));
    HK_TEST(hkMath::equal(mpA.m_volume, mpB.m_volume, maxRelError * mpA.m_volume));

    hkSimdReal maxRelErrorSimd; maxRelErrorSimd.setFromFloat(maxRelError);
    {
        hkVector4 maxErrorVector; maxErrorVector.setMul(mpA.m_centerOfMass, maxRelErrorSimd);
        maxErrorVector.setAbs(maxErrorVector);

        // We need to increase the maximum error for the COM as it seems specially sensitive to the vertex values
        maxErrorVector.mul(hkSimdReal::fromFloat(10.0f));

        hkVector4 diffAbs; diffAbs.setSub(mpA.m_centerOfMass, mpB.m_centerOfMass);
        diffAbs.setAbs(diffAbs);
        HK_TEST(diffAbs.lessEqual(maxErrorVector).allAreSet<hkVector4ComparisonMask::MASK_XYZ>());
    }

    for (int i = 0; i < 3; ++i)
    {
        const hkVector4& vA = mpA.m_inertiaTensor.getColumn(i);
        hkVector4 maxErrorVector; maxErrorVector.setMul(vA, maxRelErrorSimd);
        maxErrorVector.setAbs(maxErrorVector);
        hkVector4 diffAbs; diffAbs.setSub(vA, mpB.m_inertiaTensor.getColumn(i));
        diffAbs.setAbs(diffAbs);
        HK_TEST(diffAbs.lessEqual(maxErrorVector).allAreSet<hkVector4ComparisonMask::MASK_XYZ>());
    }
}


static void testHkMassPropertiesApplyScale()
{
    hkPseudoRandomGenerator rng(1);
    hkSimdReal maxExtent; maxExtent.setFromFloat(10.0f);
    hkSimdReal maxScale; maxScale.setFromFloat(4.0f);

    for (int i = 0; i < 80; ++i)
    {
        // Randomize scale and mass
        hkVector4 scale; rng.getRandomVector11(scale);
        scale.mul(maxScale);
        if (i < 16)
        {
            // Make first 16 scales uniform
            scale.setAll(scale.getComponent<0>());
        }
        hkReal mass = rng.getRandRange(0.1, 100);

        // Randomize vertices
        hkLocalArray<hkVector4> vertices(32);
        vertices.setSize(vertices.getCapacity());
        for (int j = 0; j < vertices.getSize(); ++j)
        {
            rng.getRandomVector11(vertices[j]);
            vertices[j].mul(maxExtent);
        }

        // Compute mass properties and scale them
        hkMassProperties scaledMp;
        hkInertiaTensorComputer::computeVertexHullVolumeMassProperties((hkReal*)vertices.begin(), sizeof(hkVector4), vertices.getSize(), mass, scaledMp);
        scaledMp.applyScale(scale);

        // Scale vertices and compute mass properties
        hkMassProperties preScaledMp;
        for (int j = 0; j < vertices.getSize(); ++j)
        {
            vertices[j].mul(scale);
        }
        hkVector4 scaleAbs; scaleAbs.setAbs(scale);
        hkSimdReal scaleFactor = scaleAbs.horizontalMul<3>();
        hkInertiaTensorComputer::computeVertexHullVolumeMassProperties((hkReal*)vertices.begin(), sizeof(hkVector4), vertices.getSize(), mass * scaleFactor.getReal(), preScaledMp);

        // Compare results
        compareMassProperties(preScaledMp, scaledMp, 0.001f);
    }
}


int inertiaTensorComputer_main()
{
    testHkMassPropertiesApplyScale();
    return 0;
}

HK_TEST_REGISTER(inertiaTensorComputer_main, "Fast", "Common/Test/UnitTest/GeometryUtilities/", __FILE__);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
