// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/GeometryUtilities/Skeleton/hkaSkeleton.h>

class hkAabb;

/// This utility class contains many functions related to the manipulation of poses and skeletons
namespace hkaSkeletonUtils
{
        /// A user definable function useful for filtering strings before a name-matching operations.
        /// This is easily implemented to remove/replace prefixes, replace characters, etc.
        /// An implementation should return 0 if the strings should match
    typedef int (HK_CALL* hkStringCompareFunc)(const char*, const char*);

        /// Converts a local space pose to a pose in model space.
        /// This routine is alias safe if the indices are parent first order i.e. (parentIndices[i] < i)
    HK_EXPORT_COMMON void HK_CALL transformLocalPoseToModelPose(
        int numTransforms,
        const hkInt16* parentIndices,
        const hkQsTransform* HK_RESTRICT poseLocal,
        hkQsTransform* HK_RESTRICT poseModelOut );

        /// Converts a bone transform from a local space to model space.
    HK_EXPORT_COMMON void HK_CALL transformLocalBoneToModelBone(
        int boneIndex,
        const hkInt16* parentIndices,
        const hkQsTransform* poseLocal,
        hkQsTransform& boneModelOut );

        /// Converts a model space pose to a pose in local space.
        /// This routine is not alias safe.
    HK_EXPORT_COMMON void HK_CALL transformModelPoseToLocalPose(
        int numTransforms,
        const hkInt16* parentIndices,
        const hkQsTransform* HK_RESTRICT poseModel,
        hkQsTransform* HK_RESTRICT poseLocalOut );

        /// Converts a model space pose to a pose in world space.
        /// This simply multiplies through the model pose by the reference frame.
        /// The worldFromModel transform gives the reference frame in world space.
    HK_EXPORT_COMMON void HK_CALL transformModelPoseToWorldPose(
        int numTransforms,
        const hkQsTransform& worldFromModel,
        const hkQsTransform* HK_RESTRICT poseModel,
        hkQsTransform* HK_RESTRICT poseWorldOut );

        /// Converts a world space pose to a pose in model space.
        /// This simply multiplies through the world pose by the inverse of the reference frame.
        /// The worldFromModel transform gives the reference frame in world space.
    HK_EXPORT_COMMON void HK_CALL transformWorldPoseToModelPose(
        int numTransforms,
        const hkQsTransform& worldFromModel,
        const hkQsTransform* HK_RESTRICT poseWorld,
        hkQsTransform* HK_RESTRICT poseModelOut );

        /// Converts a local space pose to a pose in world space.
        /// This first converts to model space and then to world space using the reference frame.
        /// The worldFromModel transform gives the reference frame in world space.
        /// This routine is alias safe if the indices are parent first order i.e. (parentIndices[i] < i)
    HK_EXPORT_COMMON void HK_CALL transformLocalPoseToWorldPose(
        int numTransforms,
        const hkInt16* parentIndices,
        const hkQsTransform& worldFromModel,
        const hkQsTransform* HK_RESTRICT poseLocal,
        hkQsTransform* HK_RESTRICT poseWorldOut );

        /// Converts a local space pose to a pose in world space.
        /// This first converts to model space and then to local space using the reference frame.
        /// The worldFromModel transform gives the reference frame in world space.
        /// This routine is not alias safe.
    HK_EXPORT_COMMON void HK_CALL transformWorldPoseToLocalPose(
        int numTransforms,
        const hkInt16* parentIndices,
        const hkQsTransform& worldFromModel,
        const hkQsTransform* HK_RESTRICT poseWorld,
        hkQsTransform* HK_RESTRICT poseLocalOut );

        /// Blends two (usually local) poses using per-bone weight array (0.0 = A, 1.0 = B)
    HK_EXPORT_COMMON void HK_CALL blendPoses(
        hkUint32 numBones,
        const hkQsTransform* HK_RESTRICT poseA,
        const hkQsTransform* poseB,
        const hkReal* HK_RESTRICT weights,
        hkQsTransform* poseOut );

        /// Blends two (usually local) poses using a single weight, where 0.0 = A and 1.0 = B
    HK_EXPORT_COMMON void HK_CALL blendPoses(
        hkUint32 numBones,
        const hkQsTransform* HK_RESTRICT poseA,
        const hkQsTransform* poseB,
        const hkReal weight,
        hkQsTransform* poseOut );

        /// Blends two (usually local) poses using per-bone weight array (0.0 = A, 1.0 = B)
    HK_EXPORT_COMMON void HK_CALL blendPosesNoAlias(
        hkUint32 numBones,
        const hkQsTransform* HK_RESTRICT poseA,
        const hkQsTransform* HK_RESTRICT poseB,
        const hkReal* HK_RESTRICT weights,
        hkQsTransform* HK_RESTRICT poseOut );

        /// Blends two (usually local) poses using a single weight, where 0.0 = A and 1.0 = B
    HK_EXPORT_COMMON void HK_CALL blendPosesNoAlias(
        hkUint32 numBones,
        const hkQsTransform* HK_RESTRICT poseA,
        const hkQsTransform* HK_RESTRICT poseB,
        const hkReal weight,
        hkQsTransform* HK_RESTRICT poseOut );

        /// Blends two bone chain (usually local) poses using a single weight, where 0.0 = A and 1.0 = B
    HK_EXPORT_COMMON void HK_CALL blendPartialPoses(
        hkUint32 numBones,
        hkInt16* bones,
        const hkQsTransform* HK_RESTRICT poseA,
        const hkQsTransform* HK_RESTRICT poseB,
        const hkReal weight,
        hkQsTransform* HK_RESTRICT poseOut );

        /// Given a local space pose, it will ensure that any bone constraint in the skeleton are enforced.
        /// Currently the only bone constraint is "lockTranslation".
        /// This ensures the pose translations match the translations in the skeleton reference pose.
    HK_EXPORT_COMMON void HK_CALL enforceSkeletonConstraintsLocalSpace(
        const hkaSkeleton& skeleton,
        hkQsTransform* poseLocalInOut );

        /// Given a model space pose, it will ensure that any bone constraint in the skeleton are enforced.
        /// Currently the only bone constraint is "lockTranslation".
        /// This ensures the pose translations match the translations in the skeleton reference pose.
    HK_EXPORT_COMMON void HK_CALL enforceSkeletonConstraintsModelSpace(
        const hkaSkeleton& skeleton,
        hkQsTransform* poseModelInOut );

        /// Given a model space pose, it will ensure that any bone constraint in the skeleton are enforced.
        /// Currently the only bone constraint is "lockTranslation".
        /// This ensures the pose translations match the translations in the constraint pose (given in local space).
    HK_EXPORT_COMMON void HK_CALL enforcePoseConstraintsLocalSpace(
        const hkaSkeleton& skeleton,
        const hkQsTransform* constraintsLocal,
        hkQsTransform* poseInOut );

        /// Given a model space pose, it will ensure that any bone constraint in the skeleton are enforced.
        /// Currently the only bone constraint is "lockTranslation".
        /// This ensures the pose translations match the translations in the constraint pose (given in local space).
    HK_EXPORT_COMMON void HK_CALL enforcePoseConstraintsModelSpace(
        const hkaSkeleton& skeleton,
        const hkQsTransform* constraintsLocal,
        hkQsTransform* poseModelInOut );

        /// Looks for a bone with the specified name (possibly filtering it). Returns -1 if not found, or the bone ID if found
    HK_EXPORT_COMMON hkInt16 HK_CALL findBoneWithName(
        const hkaSkeleton& skeleton,
        const char* name,
        hkStringCompareFunc compare = HK_NULL );

        /// Locks the translation of all bones in the given skeleton (except roots)
    HK_EXPORT_COMMON void HK_CALL lockTranslations( hkaSkeleton& skeleton, hkBool exceptRoots = true );

        /// Unlocks all the translations for the given skeleton
    HK_EXPORT_COMMON void HK_CALL unlockTranslations( hkaSkeleton& skeleton );

        /// Fills bonesOut with the chain between start bone and end bone (where end bone is deeper in the hierarchy).
        /// Returns false if there is no possible chain.
        /// This method possibly allocates space in the array (since its length is unknown by the caller).
    HK_EXPORT_COMMON hkBool HK_CALL getBoneChain(
        const hkaSkeleton& skeleton,
        hkInt16 startBone,
        hkInt16 endBone,
        hkArray<hkInt16>& bonesOut );

        /// Fills bonesOut with the chain between start bone and end bone (where end bone is deeper in the hierarchy).
        /// Returns false if there is no possible chain or the chain crosses a partition boundary.
        /// This method possibly allocates space in the array (since its length is unknown by the caller).
    HK_EXPORT_COMMON hkBool HK_CALL getBoneChainWithinPartition(
        const hkaSkeleton& skeleton,
        hkInt16 startBone,
        hkInt16 endBone,
        hkArray<hkInt16>& bonesOut );

        /// Fills bonesOut with the id of all bones descending from the given bone.
        /// This methods possibly allocates space in the array (since its length is unknown by the caller)
    HK_EXPORT_COMMON void HK_CALL getDescendants(
        const hkaSkeleton& skeleton,
        hkInt16 startBone,
        hkArray<hkInt16>& bonesOut,
        bool includeStart = false );

        /// Fills bonesOut with the id of all bones that are ancestors of the given bone.  This method possibly
        /// allocates space in the array.
    HK_EXPORT_COMMON void HK_CALL getAncestors(
        const hkaSkeleton& skeleton,
        hkInt16 startPoint,
        hkArray<hkInt16>& bonesOut );

        /// Get the distance in the skeleton from bone1 to bone2
    HK_EXPORT_COMMON int HK_CALL getDistance( const hkaSkeleton& skeleton, hkInt16 bone1, hkInt16 bone2 );

        /// Given an array of boolean skeleton->m_numBones big, the array will be marked true for all bones which are
        /// descendants of the start bone and false otherwise.
    HK_EXPORT_COMMON void HK_CALL markDescendants(
        const hkaSkeleton* skeleton,
        int startBone,
        bool* out,
        bool includeStart = false );

        /// Fills bonesOut with the id of all bones in the given skeleton with no children
    HK_EXPORT_COMMON void HK_CALL getLeaves( const hkaSkeleton& skeleton, hkArray<hkInt16>& leavesOut );

        /// Given a local pose, gets the scale in model space of the associated bone
    HK_EXPORT_COMMON void HK_CALL getModelSpaceScale(
        const hkaSkeleton& skeleton,
        const hkQsTransform* poseLocal,
        int boneId,
        hkVector4& scaleOut );

        /// This method iterates through the array and normalizes the quaternions in the array of hkQsTransforms
    HK_EXPORT_COMMON void HK_CALL normalizeRotations( hkQsTransform* transformsInOut, int numTransforms );

        /// Calculate the AABB containing the given pose in worldspace
    HK_EXPORT_COMMON void HK_CALL calcAabb(
        hkUint32 numBones,
        const hkQsTransform* poseLocal,
        const hkInt16* parentIndices,
        const hkQsTransform& worldFromModel,
        hkAabb& aabbOut );

        /// Determines if the partitions are valid.
        /// If no partitions are specified, this is a valid setup.
        /// Otherwise, the partitions must be non-overlapping and fully cover the skeleton.
    HK_EXPORT_COMMON hkBool HK_CALL hasValidPartitions( const hkaSkeleton& skeleton );
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
