// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/GeometryUtilities/hkGeometryUtilities.h>
#include <Common/GeometryUtilities/Skeleton/hkaSkeletonUtils.h>

#include <Common/GeometryUtilities/Skeleton/hkaBone.h>
#include <Common/Base/Math/Vector/hkVector4Util.h>
#include <Common/Base/Container/BitField/hkBitField.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>
#include <Common/Base/Types/Geometry/Aabb/hkAabb.h>

void hkaSkeletonUtils::transformLocalPoseToWorldPose ( int numTransforms, const hkInt16* parentIndices, const hkQsTransform& worldFromModel, const hkQsTransform* HK_RESTRICT poseLocal, hkQsTransform* HK_RESTRICT poseWorldOut)
{
    HK_ASSERT(0x7b94f4a9, ( (hkUlong)poseLocal >= (hkUlong)(poseWorldOut+numTransforms) ) || ( (hkUlong)poseWorldOut >= (hkUlong)(poseLocal+numTransforms) ), "transform arrays overlap");
    for( int i = 0; i < numTransforms; i++ )
    {
        const int parentId = parentIndices[i];
        HK_ASSERT_NO_MSG(0x675efb2b,  (parentIndices[i] == -1) || (parentIndices[i] < i) );
        const hkQsTransform& worldFromParent = ( -1 != parentId ) ? poseWorldOut[parentId] : worldFromModel;
        poseWorldOut[i].setMul(worldFromParent, poseLocal[i] );
    }
}

void hkaSkeletonUtils::transformLocalBoneToModelBone( int boneIndex, const hkInt16* parentIndices, const hkQsTransform* poseLocal, hkQsTransform& boneModelOut )
{
    boneModelOut = poseLocal[boneIndex];

    int parentIndex = parentIndices[boneIndex];

    while( parentIndex >= 0 )
    {
        HK_ASSERT( 0x5f7a612b, parentIndex < boneIndex, "bones out of order" );

        boneModelOut.setMul( poseLocal[parentIndex], boneModelOut );

        parentIndex = parentIndices[parentIndex];
    }
}

void hkaSkeletonUtils::transformLocalPoseToModelPose ( int numTransforms, const hkInt16* parentIndices, const hkQsTransform* HK_RESTRICT poseLocal, hkQsTransform* HK_RESTRICT poseModelOut)
{
    transformLocalPoseToWorldPose( numTransforms, parentIndices, hkQsTransform::getIdentity(), poseLocal, poseModelOut);
}

void hkaSkeletonUtils::transformWorldPoseToLocalPose ( int numTransforms, const hkInt16* parentIndices, const hkQsTransform& worldFromModel, const hkQsTransform* HK_RESTRICT poseWorld, hkQsTransform* HK_RESTRICT poseLocalOut)
{
    HK_ASSERT(0xe52dabb, ( (hkUlong)poseWorld >= (hkUlong)(poseLocalOut+numTransforms) ) || ( (hkUlong)poseLocalOut >= (hkUlong)(poseWorld+numTransforms) ), "transform arrays overlap");
    for( int i = 0; i < numTransforms; i++ )
    {
        const int parentId = parentIndices[i];
        const hkQsTransform& parentFromWorld = ( -1 != parentId ) ? poseWorld[parentId] : worldFromModel;
        HK_ASSERT_NO_MSG(0x675efb2b,  (parentIndices[i] == -1) || (parentIndices[i] < i) );
        poseLocalOut[i].setMulInverseMul( parentFromWorld, poseWorld[i] );
    }
}

void hkaSkeletonUtils::transformModelPoseToLocalPose ( int numTransforms, const hkInt16* parentIndices, const hkQsTransform* HK_RESTRICT poseModel, hkQsTransform* HK_RESTRICT poseLocalOut)
{
    transformWorldPoseToLocalPose( numTransforms, parentIndices, hkQsTransform::getIdentity(), poseModel, poseLocalOut);
}

void hkaSkeletonUtils::transformModelPoseToWorldPose ( int numTransforms, const hkQsTransform& worldFromModel, const hkQsTransform* HK_RESTRICT poseModel, hkQsTransform* HK_RESTRICT poseWorldOut)
{
    HK_ASSERT(0x1081643b, ( (hkUlong)poseModel >= (hkUlong)(poseWorldOut+numTransforms) ) || ( (hkUlong)poseWorldOut >= (hkUlong)(poseModel+numTransforms) ), "transform arrays overlap");
    for( int i = 0; i < numTransforms; i++ )
    {
        poseWorldOut[i].setMul( worldFromModel, poseModel[i] );
    }
}

void hkaSkeletonUtils::transformWorldPoseToModelPose ( int numTransforms, const hkQsTransform& worldFromModel, const hkQsTransform* HK_RESTRICT poseWorld, hkQsTransform* HK_RESTRICT poseModelOut)
{
    HK_ASSERT(0x36bc7900, ( (hkUlong)poseWorld >= (hkUlong)(poseModelOut+numTransforms) ) || ( (hkUlong)poseModelOut >= (hkUlong)(poseWorld+numTransforms) ), "transform arrays overlap");
    for( int i = 0; i < numTransforms; i++ )
    {
        poseModelOut[i].setMulInverseMul( worldFromModel, poseWorld[i] );
    }
}

void hkaSkeletonUtils::enforceSkeletonConstraintsLocalSpace (const hkaSkeleton& skeleton, hkQsTransform* poseInOut)
{
    const int numBones = skeleton.m_bones.getSize();

    for (int i=0; i< numBones; i++)
    {
        const hkaBone& bone = skeleton.m_bones[i];

        if (bone.m_lockTranslation)
        {
            poseInOut[i].m_translation = skeleton.m_referencePose[i].m_translation;
        }
    }
}

void hkaSkeletonUtils::enforcePoseConstraintsLocalSpace (const hkaSkeleton& skeleton, const hkQsTransform* constraintsLocal, hkQsTransform* poseInOut)
{
    const int numBones = skeleton.m_bones.getSize();

    for (int i=0; i< numBones; i++)
    {
        const hkaBone& bone = skeleton.m_bones[i];

        if (bone.m_lockTranslation)
        {
            poseInOut[i].m_translation = constraintsLocal[i].m_translation;
        }
    }
}

void hkaSkeletonUtils::enforceSkeletonConstraintsModelSpace (const hkaSkeleton& skeleton, hkQsTransform* poseModelInOut)
{
    const int numBones = skeleton.m_bones.getSize();

    for (int boneIdx=0; boneIdx< numBones; ++boneIdx)
    {
        const hkaBone& bone = skeleton.m_bones[boneIdx];
        const hkInt16 parentIdx = skeleton.m_parentIndices[boneIdx];

        if (bone.m_lockTranslation)
        {
            const hkVector4& localTranslation = skeleton.m_referencePose[boneIdx].m_translation;

            if (parentIdx != -1)
            {
                const hkQsTransform& parentModelSpace = poseModelInOut[parentIdx];
                poseModelInOut[boneIdx].m_translation._setRotatedDir(parentModelSpace.getRotation(), localTranslation);
                poseModelInOut[boneIdx].m_translation.add(parentModelSpace.getTranslation());
            }
            else
            {
                poseModelInOut[boneIdx].m_translation = localTranslation;
            }
        }
    }
}

void hkaSkeletonUtils::enforcePoseConstraintsModelSpace (const hkaSkeleton& skeleton, const hkQsTransform* constraintsLocal, hkQsTransform* poseModelInOut)
{
    const int numBones = skeleton.m_bones.getSize();

    for (int boneIdx=0; boneIdx< numBones; ++boneIdx)
    {
        const hkaBone& bone = skeleton.m_bones[boneIdx];
        const hkInt16 parentIdx = skeleton.m_parentIndices[boneIdx];

        if (bone.m_lockTranslation)
        {
            const hkVector4& localTranslation = constraintsLocal[boneIdx].m_translation;

            if (parentIdx != -1)
            {
                const hkQsTransform& parentModelSpace = poseModelInOut[parentIdx];
                hkVector4 temp;
                temp._setRotatedDir(parentModelSpace.getRotation(), localTranslation);
                poseModelInOut[boneIdx].m_translation.setAdd( temp, parentModelSpace.getTranslation() );
            }
            else
            {
                poseModelInOut[boneIdx].m_translation = localTranslation;
            }
        }
    }
}

void hkaSkeletonUtils::lockTranslations (hkaSkeleton& skeleton, hkBool exceptRoots)
{
    const int numBones = skeleton.m_bones.getSize();

    for (int i=0; i< numBones; i++)
    {
        if ((skeleton.m_parentIndices[i]>0) || (!exceptRoots))
        {
            hkaBone& bone = skeleton.m_bones[i];
            bone.m_lockTranslation = true;
        }
    }
}

// This one does allocate space in the array (since length is unknown)
hkBool hkaSkeletonUtils::getBoneChain(const hkaSkeleton& skeleton, hkInt16 startBone, hkInt16 endBone, hkArray<hkInt16>& bonesOut)
{
    HK_ASSERT( 0x982651b4, bonesOut.getSize() == 0, "the output array should be empty" );
    if ( HK_VERY_UNLIKELY( startBone > endBone ) )
    {
        HK_ASSERT(0x22440048, false, "Incorrect bone ordering.");
        bonesOut.clear();
        return false;
    }

    // We can only collect the chain in the opposite order that we need to output it, so we collect it into a local array and then
    // reverse it at the end as we put it into the output array.
    hkLocalArray<hkInt16> chain( skeleton.m_bones.getSize() );

    // Go backwards from the end of the chain to the start
    hkInt16 boneId;
    for (boneId = endBone; (boneId > -1) && (boneId != startBone); boneId = skeleton.m_parentIndices[boneId])
    {
        chain.pushBack( boneId );
    }

    const hkBool ok = (boneId > -1);

    // If we found a valid chain then reverse it and return it.
    if ( ok )
    {
        chain.pushBack( startBone );

        const int count = chain.getSize();
        bonesOut.setSize( count );

        for( int i = 0; i < count; i++ )
        {
            bonesOut[i] = chain[count-i-1];
        }
    }

    return ok;
}

// This one does allocate space in the array (since length is unknown)

hkBool hkaSkeletonUtils::getBoneChainWithinPartition(const hkaSkeleton& skeleton, hkInt16 startBone, hkInt16 endBone, hkArray<hkInt16>& bonesOut)
{
    HK_ASSERT( 0x982651b4, bonesOut.getSize() == 0, "the output array should be empty" );
    if ( HK_VERY_UNLIKELY( startBone > endBone ) )
    {
        HK_ASSERT(0x22440048, false, "Incorrect bone ordering.");
        bonesOut.clear();
        return false;
    }

    // We can only collect the chain in the opposite order that we need to output it, so we collect it into a local array and then
    // reverse it at the end as we put it into the output array.
    hkLocalArray<hkInt16> chain( skeleton.m_bones.getSize() );

    int bonePartitionIndex = -1;

    const hkArray<hkaSkeleton::Partition>& partitions = skeleton.m_partitions;
    const int numPartitions = partitions.getSize();

    for ( int i = 0; i < numPartitions; ++i )
    {
        if (partitions[i].isBonePresentWithinPartition( endBone ))
        {
            bonePartitionIndex = i;
        }
    }

    // Go backwards from the end of the chain to the start
    hkInt16 boneId;
    for (boneId = endBone; (boneId > -1) && (boneId != startBone); boneId = skeleton.m_parentIndices[boneId])
    {
        if ((bonePartitionIndex != -1) && !(partitions[bonePartitionIndex].isBonePresentWithinPartition(boneId)))
        {
            boneId = -1;
            break;
        }
        chain.pushBack( boneId );
    }

    const hkBool ok = (boneId > -1);

    // If we found a valid chain then reverse it and return it.
    if ( ok )
    {
        chain.pushBack( startBone );

        const int count = chain.getSize();
        bonesOut.setSize( count );

        for( int i = 0; i < count; i++ )
        {
            bonesOut[i] = chain[count-i-1];
        }
    }

    return ok;
}

hkInt16 hkaSkeletonUtils::findBoneWithName (const hkaSkeleton& skeleton, const char* name, hkStringCompareFunc compare)
{
    const int numBones = skeleton.m_bones.getSize();

    if (compare == HK_NULL)
        compare = hkString::strCasecmp;

    for (hkInt16 boneId=0; boneId < numBones; boneId++)
    {
        if ( compare(name, skeleton.m_bones[boneId].m_name) == 0 )
        {
            return boneId;
        }
    }

    return -1;
}




// Unlocks all the translations for the given skeleton
void hkaSkeletonUtils::unlockTranslations (hkaSkeleton& skeleton)
{
    const int numBones = skeleton.m_bones.getSize();

    for (int i=0; i< numBones; i++)
    {
        hkaBone& bone = skeleton.m_bones[i];

        bone.m_lockTranslation = false;
    }

}


void hkaSkeletonUtils::getDescendants(const hkaSkeleton& skeleton, hkInt16 startBone, hkArray<hkInt16>& bonesOut, bool includeStart )
{
    hkLocalArray<hkBool> included(skeleton.m_bones.getSize());
    included.setSize(skeleton.m_bones.getSize(), false);

    if ( includeStart )
    {
        bonesOut.pushBack( startBone );
    }

    included[startBone] = true;
    for (hkInt16 b=startBone+1; b<skeleton.m_bones.getSize(); b++)
    {
        const hkInt16 parent = skeleton.m_parentIndices[b];
        if (parent != -1 && included[parent])
        {
            included[b] = true;
            bonesOut.pushBack(b);
        }
    }
}

void hkaSkeletonUtils::getAncestors(const hkaSkeleton& skeleton, hkInt16 startBone, hkArray<hkInt16>& bonesOut )
{
    hkInt16 parent = skeleton.m_parentIndices[startBone];

    while( parent != -1 )
    {
        bonesOut.pushBack(parent);
        startBone = parent;
        parent = skeleton.m_parentIndices[startBone];
    }
}

int hkaSkeletonUtils::getDistance( const hkaSkeleton& skeleton, hkInt16 bone1, hkInt16 bone2 )
{
    if ( bone1 == bone2 )
    {
        return 0;
    }

    hkInt16 leafward = ( bone1 > bone2 ) ? bone1 : bone2;
    hkInt16 rootward = ( leafward == bone1 ) ? bone2 : bone1;

    hkInt16 count = 1;
    hkInt16 parent = skeleton.m_parentIndices[leafward];
    while( parent != -1 )
    {
        if ( rootward == parent )
        {
            return count;
        }

        count++;
        parent = skeleton.m_parentIndices[parent];
    }

    return -1;
}

void hkaSkeletonUtils::markDescendants( const hkaSkeleton* skeleton, int startBone, bool* out, bool includeStart )
{
    HK_ASSERT(0x4e0d723a, startBone >= 0 && startBone < skeleton->m_bones.getSize(), "StartBone is not in range" );

    // Bones of a lower index cannot be part of the branch
    for ( int i = 0; i < startBone; i++ )
    {
        out[ i ] = false;
    }

    // Temporarily mark the start bone as being included
    out[ startBone ] = true;

    // Mark all bones who's parents are included
    for ( int i = startBone + 1; i < skeleton->m_bones.getSize(); i++ )
    {
        const hkInt16& parent = skeleton->m_parentIndices[ i ];

        out[ i ] = ( parent >= 0 ) && out[ parent ];
    }

    // Test if the start bone is to be included or not
    out[ startBone ] = includeStart;
}

void hkaSkeletonUtils::getLeaves (const hkaSkeleton& skeleton, hkArray<hkInt16>& leavesOut)
{
    hkLocalArray<hkBool> isParent(skeleton.m_bones.getSize());
    isParent.setSize(skeleton.m_bones.getSize(), false);

    for (int b=0; b<skeleton.m_parentIndices.getSize(); ++b)
    {
        hkInt16 parentIdx = skeleton.m_parentIndices[b];
        if (parentIdx == -1)
        {
            // root bone has no parent
            continue;
        }
        isParent[parentIdx] = true;
    }

    for (hkInt16 b=0; b<skeleton.m_bones.getSize(); ++b)
    {
        if ( !isParent[b] )
        {
            leavesOut.pushBack(b);
        }
    }
}


HK_INLINE void blendTwoTransforms (const hkQsTransform& transformA, const hkQsTransform& transformB, hkSimdRealParameter weight, hkQsTransform& transformOut)
{
    transformOut.m_translation.setInterpolate(
        transformA.m_translation,
        transformB.m_translation, weight);

    // Make sure we blend the closest representation of the quaternion; flip if necessary
    hkVector4 bRot;
    bRot.setFlipSign(transformB.m_rotation.m_vec, transformA.m_rotation.m_vec.dot<4>(transformB.m_rotation.m_vec));

    hkQuaternion rotationQuat;
    rotationQuat.m_vec.setInterpolate(
        transformA.m_rotation.m_vec,
        bRot, weight);

    transformOut.m_scale.setInterpolate(
        transformA.m_scale,
        transformB.m_scale, weight);

    // Check Rotation
    const hkSimdReal quatNorm = rotationQuat.m_vec.lengthSquared<4>();
#if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED)
    rotationQuat.normalize();
    transformOut.m_rotation.m_vec.setSelect(quatNorm.less(hkSimdReal_Eps), hkVector4::getConstant<HK_QUADREAL_0001>(), rotationQuat.m_vec);
#else
    if (quatNorm.isLess(hkSimdReal_Eps))
    {
        // no rotations blended (or canceled each other) -> set to identity
        rotationQuat.setIdentity();
    }
    else
    {
        // normalize
        rotationQuat.normalize();
    }
    transformOut.m_rotation = rotationQuat;
#endif
}

void hkaSkeletonUtils::blendPoses( hkUint32 numBones, const hkQsTransform* HK_RESTRICT poseA, const hkQsTransform* poseB, const hkReal* HK_RESTRICT weights, hkQsTransform* poseOut )
{
    
    for (hkUint32 i=0; i < numBones; i++)
    {
        blendTwoTransforms( poseA[i], poseB[i], hkSimdReal::fromFloat(weights[i]), poseOut[i] );
    }
}

void hkaSkeletonUtils::blendPoses( hkUint32 numBones, const hkQsTransform* HK_RESTRICT poseA, const hkQsTransform* poseB, const hkReal weight, hkQsTransform* poseOut )
{
    
    hkSimdReal sWeight; sWeight.setFromFloat(weight);

    for (hkUint32 i=0; i < numBones; i++)
    {
        blendTwoTransforms( poseA[i], poseB[i], sWeight, poseOut[i] );
    }
}

void hkaSkeletonUtils::blendPosesNoAlias( hkUint32 numBones, const hkQsTransform* HK_RESTRICT poseA, const hkQsTransform* HK_RESTRICT poseB, const hkReal* HK_RESTRICT weights, hkQsTransform* HK_RESTRICT poseOut )
{
    HK_ASSERT(0x33fe89e8, ( (hkUlong)poseA >= (hkUlong)(poseOut+numBones) ) || ( (hkUlong)poseOut >= (hkUlong)(poseA+numBones) ), "transform arrays overlap");
    HK_ASSERT(0x15febee8, ( (hkUlong)poseB >= (hkUlong)(poseOut+numBones) ) || ( (hkUlong)poseOut >= (hkUlong)(poseB+numBones) ), "transform arrays overlap");
    HK_ASSERT(0x1318d556, ( (hkUlong)poseA >= (hkUlong)(poseB  +numBones) ) || ( (hkUlong)poseB   >= (hkUlong)(poseA+numBones) ), "transform arrays overlap");

    HK_ASSERT(0x43bc98b6, ( (hkUlong)poseA   >= (hkUlong)(weights+numBones) ) || ( (hkUlong)weights >= (hkUlong)(poseA+numBones) ),   "transform and weight arrays overlap");
    HK_ASSERT(0x37fe5fc3, ( (hkUlong)poseB   >= (hkUlong)(weights+numBones) ) || ( (hkUlong)weights >= (hkUlong)(poseB+numBones) ),   "transform and weight arrays overlap");
    HK_ASSERT(0x453fc82, ( (hkUlong)poseOut >= (hkUlong)(weights+numBones) ) || ( (hkUlong)weights >= (hkUlong)(poseOut+numBones) ), "transform and weight arrays overlap");

    for (hkUint32 i=0; i < numBones; i++)
    {
        blendTwoTransforms (poseA[i], poseB[i], hkSimdReal::fromFloat(weights[i]), poseOut[i]);
    }
}

void hkaSkeletonUtils::blendPosesNoAlias( hkUint32 numBones, const hkQsTransform* HK_RESTRICT poseA, const hkQsTransform* HK_RESTRICT poseB, const hkReal weight, hkQsTransform* HK_RESTRICT poseOut )
{
    HK_ASSERT(0x6ec0ab54, ( (hkUlong)poseA >= (hkUlong)(poseOut+numBones) ) || ( (hkUlong)poseOut >= (hkUlong)(poseA+numBones) ), "transform arrays overlap");
    HK_ASSERT(0x7fc2897a, ( (hkUlong)poseB >= (hkUlong)(poseOut+numBones) ) || ( (hkUlong)poseOut >= (hkUlong)(poseB+numBones) ), "transform arrays overlap");
    HK_ASSERT(0x1a82821b, ( (hkUlong)poseA >= (hkUlong)(poseB  +numBones) ) || ( (hkUlong)poseB   >= (hkUlong)(poseA+numBones) ), "transform arrays overlap");

    hkSimdReal sWeight; sWeight.setFromFloat(weight);

    for (hkUint32 i=0; i < numBones; i++)
    {
        blendTwoTransforms( poseA[i], poseB[i], sWeight, poseOut[i] );
    }
}

void hkaSkeletonUtils::blendPartialPoses( hkUint32 numBones, hkInt16* bones, const hkQsTransform* HK_RESTRICT poseA, const hkQsTransform* HK_RESTRICT poseB, const hkReal weight, hkQsTransform* HK_RESTRICT poseOut )
{
    HK_ASSERT(0x4f1d3f35, ( (hkUlong)poseA >= (hkUlong)(poseOut+numBones) ) || ( (hkUlong)poseOut >= (hkUlong)(poseA+numBones) ), "transform arrays overlap");
    HK_ASSERT(0x150f9244, ( (hkUlong)poseB >= (hkUlong)(poseOut+numBones) ) || ( (hkUlong)poseOut >= (hkUlong)(poseB+numBones) ), "transform arrays overlap");
    HK_ASSERT(0x3e116f79, ( (hkUlong)poseA >= (hkUlong)(poseB  +numBones) ) || ( (hkUlong)poseB   >= (hkUlong)(poseA+numBones) ), "transform arrays overlap");

    hkSimdReal sWeight = hkSimdReal::fromFloat(weight);

    for( hkUint32 i = 0; i < numBones; i++ )
    {
        blendTwoTransforms( poseA[ bones[i] ], poseB[ bones[i] ], sWeight, poseOut[ bones[i] ] );
    }
}

void hkaSkeletonUtils::getModelSpaceScale (const hkaSkeleton& skeleton, const hkQsTransform* poseLocal, int boneId, hkVector4& scaleOut)
{
    // Use temp here to avoid LHS. See HKA-1019
    hkVector4 tmpScale = hkVector4::getConstant<HK_QUADREAL_1>();
    while(boneId!=-1)
    {
        tmpScale.mul(poseLocal[boneId].getScale());
        boneId = skeleton.m_parentIndices[boneId];
    }
    scaleOut = tmpScale;
}

void hkaSkeletonUtils::normalizeRotations(hkQsTransform* transformsInOut, int numTransforms )
{
    // now normalize 4 quaternions at once
    hkQsTransform* blockStart = transformsInOut;
    unsigned numTransformsOver4 = numTransforms/4;
    for (unsigned i=0; i< numTransformsOver4; i++)
    {
        hkVector4 dots;
        hkVector4Util::dot4_4vs4(blockStart[0].m_rotation.m_vec, blockStart[0].m_rotation.m_vec,
            blockStart[1].m_rotation.m_vec, blockStart[1].m_rotation.m_vec,
            blockStart[2].m_rotation.m_vec, blockStart[2].m_rotation.m_vec,
            blockStart[3].m_rotation.m_vec, blockStart[3].m_rotation.m_vec,
            dots);
        hkVector4 inverseSqrtDots;
        inverseSqrtDots.setSqrtInverse<HK_ACC_23_BIT,HK_SQRT_IGNORE>(dots);

        blockStart[0].m_rotation.m_vec.mul(inverseSqrtDots.getComponent<0>());
        blockStart[1].m_rotation.m_vec.mul(inverseSqrtDots.getComponent<1>());
        blockStart[2].m_rotation.m_vec.mul(inverseSqrtDots.getComponent<2>());
        blockStart[3].m_rotation.m_vec.mul(inverseSqrtDots.getComponent<3>());

        blockStart += 4;
    }

    unsigned leftovers = numTransforms%4;
    for (unsigned j=0; j<leftovers; j++)
    {
        blockStart[j].m_rotation.normalize();
    }
}

void HK_CALL hkaSkeletonUtils::calcAabb(hkUint32 numBones, const hkQsTransform* poseLocal, const hkInt16* parentIndices, const hkQsTransform& worldFromModel, hkAabb& aabbOut )
{
    // Clear aabb
    aabbOut.setEmpty();

    // Transform the local pose into world space
    hkArray<hkQsTransform> poseWorld;
    poseWorld.setSize(numBones);
    transformLocalPoseToWorldPose( numBones, parentIndices, worldFromModel, poseLocal, poseWorld.begin() );

    // Iterate over the worldspace skeleton, expanding the AABB to contain the bone
    for ( hkUint32 i = 0; i < numBones; ++i )
    {
        aabbOut.includePoint(poseWorld[i].getTranslation());
    }
}

hkBool HK_CALL hkaSkeletonUtils::hasValidPartitions( const hkaSkeleton& skeleton )
{
    if ( skeleton.m_partitions.getSize() == 0 )
    {
        return true;
    }

    HK_ASSERT_NO_MSG(0x22440029, skeleton.m_bones.getSize() <= 256);
    hkInplaceBitField<256> partitionsCoverage;
    hkInplaceBitField<256> currentPartitionCoverage;
    hkInplaceBitField<256> buffer;
    for ( int partitionsIt = 0; partitionsIt < skeleton.m_partitions.getSize(); partitionsIt++ )
    {
        const hkaSkeleton::Partition& partition = skeleton.m_partitions[ partitionsIt ];
        currentPartitionCoverage.assignRange( partition.m_startBoneIndex, partition.m_numBones, 1 );

        buffer = partitionsCoverage;
        buffer.andWith( currentPartitionCoverage );
        if ( buffer.anyIsSet() )
        {
            return false;
        }

        partitionsCoverage.orWith( currentPartitionCoverage );
        currentPartitionCoverage.assignAll( 0 );
    }

    return partitionsCoverage.bitCount() == skeleton.m_bones.getSize();
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
