// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once
// this: #include <Common/GeometryUtilities/Misc/hkGeometryUtils.h>

#include <Common/Base/Types/Geometry/hkGeometry.h>
#include <Common/Base/Types/Geometry/Aabb/hkAabb.h>
#include <Common/Base/Algorithm/Sort/hkRadixSort.h>

namespace hk1AxisSweep { struct AabbInt; }
struct hkStridedVertices;
class hkSphere;

/// Utility class, contains methods that operate with hkGeometry objects.
class HK_EXPORT_COMMON hkGeometryUtils
{
public:

    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, hkGeometryUtils );

    /// input structure into createGrid
    struct HK_EXPORT_COMMON GridInput
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_GEOMETRY, hkGeometryUtils::GridInput );
        GridInput( int numVertsX = 16, hkVector4Parameter up = hkVector4::getConstant( HK_QUADREAL_0010 ) );
        void centerGrid(); // sets the origin so that the grid is centered around 0

        int m_numVertsX;    ///< Number of vertices in X-direction
        int m_numVertsY;    ///< Number of vertices in Y-direction

        hkVector4 m_origin;
        hkVector4 m_stepX;
        hkVector4 m_stepY;
    };

    /// Adds a grid to an existing geometry.
    static void HK_CALL appendGrid( const GridInput& input, hkGeometry& geom, int defaultMaterial = 0 );

    /// Adds a sphere to an existing geometry.
    static void HK_CALL appendSphere( hkVector4Parameter center, hkReal radius, int numSteps, hkGeometry& geom, int material = -1 );

    /// Adds a random convex geometry where the vertices are inside maxRadius and outside minRadius to geom.
    static void HK_CALL appendRandomConvex( int numPoints, hkReal minRadius, hkReal maxRadius, int randomSeed, hkGeometry& geom, int material = -1 );

    /// Adds a hkGeometry that has the shape of a box to geom.
    static void HK_CALL appendBox( hkVector4Parameter center, hkVector4Parameter halfExtents, hkGeometry& geom, int material = -1 );

    /// Adds a hkGeometry that has the shape of a capsule to geom.
    static void HK_CALL appendCapsule( hkVector4Parameter top, hkVector4Parameter bottom, hkReal radius, int numHeightSegments, int numSides, const hkTransform& transform, hkGeometry& geom );

    /// Adds a hkGeometry that has the shape of a tapered capsule to geom.
    static void HK_CALL appendTaperedCapsule( hkVector4Parameter start, hkVector4Parameter end, hkReal startRadius, hkReal endRadius, int heightSamples, int thetaSamples, const hkTransform& transform, hkGeometry& geom );

    /// Adds a hkGeometry that has the shape of a cylinder to geom.
    static void HK_CALL appendCylinder( hkVector4Parameter top, hkVector4Parameter bottom, hkReal radius, int numSides, int numHeightSegments, hkGeometry& geom );

    /// Adds a hkGeometry that has the shape of a plane to geom.
    static void HK_CALL appendPlane( hkVector4Parameter normal, hkVector4Parameter perpToNormal, hkVector4Parameter center, hkVector4Parameter extents, hkGeometry& geom );

    /// Adds a hkGeometry that has the shape of a bone to geom.
    static void HK_CALL appendBone( hkVector4Parameter parentPos, hkVector4Parameter childPos, hkGeometry& geom );

    /// triangulates a tetrahedron
    static void HK_CALL triangulateTetrahedron( const hkVector4* positionsAndDistance, hkGeometry& geom );

    /// Virtual interface to access vertices.
    struct HK_EXPORT_COMMON IVertices
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, IVertices );

        virtual         ~IVertices() {}
        virtual int     getNumVertices() const = 0;
        virtual void    getVertex( int index, hkVector4& vertexOut ) const = 0;
        virtual bool    isWeldingAllowed( int vertexA, int vertexB ) const { return true; }
    };

    /// Finds identical vertices and generate the map from original indices to welded ones.
    /// Note this method virtualize access to vertices.
    /// Returns the number of unique vertices found.
    static int HK_CALL weldVerticesVirtual(_In_ const IVertices* vertices, hkArray<int>& remap, hkReal threshold = hkReal( 1e-3f ) );

    /// Given an hkGeometry, it finds identical vertices and collapses them, updating triangles as well.
    /// Degenerate triangles are automatically removed
    static void HK_CALL weldVertices( hkGeometry& meshGeometry, hkReal threshold = hkReal( 1e-3f ) );

    /// Given an hkGeometry, it finds identical vertices and collapses them, updating triangles as well.
    /// This version returns a map from the original vertex indices to the new vertex indices.
    /// Degenerate triangles are automatically removed and the triangleRemapOut array contains -1 for each degenerate.
    /// If "keep vertex order" is set to true, the original vertex ordering will be maintained as much as possible
    static void HK_CALL weldVertices( hkGeometry& meshGeometry, hkReal threshold, hkBool keepVertexOrder, hkArray<int>& vertexRemapOut, hkArray<int>& triangleRemapOut );

    /// Given an hkGeometry, it finds identical vertices and collapses them, updating triangles as well.
    /// All arrays used by the internal sorting are explicitly provided.
    static void HK_CALL _weldVertices( hkGeometry& meshGeometry, hkReal threshold, hkBool keepVertexOrder, hkArray<int>& vertexRemapOut, hkArray<int>& triangleRemapOut, hkArray<hkVector4>& uniqueVerts, hkArrayBase<hk1AxisSweep::AabbInt>& sweepAabbs, hkArrayBase<hkRadixSort::SortData32>& sortData, hkArrayBase<hk1AxisSweep::AabbInt>& sortedAabbs );

    /// Given an hkGeometry, it finds triangles that use the same three vertices. It removes these extra triangles,
    /// and returns a map from the original triangles to the new triangles.
    /// If ignoreWinding is true (the default), triangles with the same indices but different winding are considered duplicates (and will be removed).
    /// If ignoreWinding is false, triangles with the same indices but different winding are NOT considered duplicates (i.e., only duplicates with identical triangle indices will be removed).
    static void HK_CALL removeDuplicateTriangles( hkGeometry& meshGeometry, hkArray<int>& triangleMapOut, bool ignoreWinding = true );

    /// Given an hkGeometry, it finds and removes triangles that use the same three vertices with the same winding order.
    /// This is much faster than removeDuplicateTriangles for large geometries, but doesn't compute the map.
    /// Returns HK_FAILURE if any memory allocations failed
    static hkResult HK_CALL removeDuplicateTrianglesFast( hkGeometry& meshGeometry, hkArray<hkGeometry::Triangle>& newTriangles );

    /// Optimize the memory layout of a given geometry by reordering vertices and triangles.
    /// - If not null, vertexMap will map new vertex indices to old ones.
    /// - If not null, triangleMap will map new triangle indices to old ones.
    static void HK_CALL optimizeMemoryLayout( hkGeometry& geometry, _Inout_opt_ hkArray<int>* vertexMap = HK_NULL, _Inout_opt_ hkArray<int>* triangleMap = HK_NULL );

    ///
    static void HK_CALL quantize( hkGeometry& geomInOut, int resolution = ( 1 << 16 ) );


    /// Transform a geometry
    static void HK_CALL transformGeometry( const hkMatrix4& transform, hkGeometry& geometryInOut );

    static void HK_CALL transformGeometry(const hkTransform& transform, hkGeometry& geometryInOut);

    /// Append one geometry to another.
    static void HK_CALL appendGeometry( const hkGeometry& input, hkGeometry& geomInOut );

    /// Create the Minkowski sum A-B
    static void HK_CALL appendMinkowskiSum(
        const hkTransform& tA, const hkStridedVertices& vertsA,
        const hkTransform& tB, const hkStridedVertices& vertsB,
        hkGeometry& geometryOut );

    /// Controls the behavior of getGeometryInsideAabb.
    enum GetGeometryInsideAabbMode
    {
        /// No data is copied to the output hkGeometry, but its arrays will be set to the minimal size
        MODE_PRESIZE_ARRAYS,

        /// Copy the triangles and vertices in the specified AABB (default behavior)
        MODE_COPY_DATA,
    };

    /// Extract the triangles from the input geometry that are within the specified AABB.
    /// The AABB of each triangle is tested against the specified AABB, so some
    /// triangles that are outside of the AABB may be included.
    static hkResult HK_CALL getGeometryInsideAabb( const hkGeometry& geomIn, hkGeometry& geomOut, const hkAabb& aabb, GetGeometryInsideAabbMode mode = MODE_COPY_DATA );

    /// As above, but with the vertexRemap array specified
    static hkResult HK_CALL getGeometryInsideAabb( const hkGeometry& geomIn, hkGeometry& geomOut, const hkAabb& aabb, hkArray<int>& vertexRemap, GetGeometryInsideAabbMode mode = MODE_COPY_DATA );

    /// Computes the volume of the given geometry
    static void HK_CALL computeVolume( const hkGeometry& geometry, hkSimdReal& volume );

    /// Computes the AABB of the given geometry
    static void HK_CALL computeAabb( const hkGeometry& geomIn, hkAabb& aabbOut );

    /// Computes the AABB of the given geometry, using the triangle list.
    /// This gives a more accurate AABB calculation if there are unreferenced vertices.
    static void HK_CALL computeAabbFromTriangles( const hkGeometry& geomIn, hkAabb& aabbOut );

    /// Data structure to allow linear time operations on triangle meshes
    struct HK_EXPORT_COMMON Node
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, hkGeometryUtils::Node );
        Node( hkUint32 vertexIndex );
        Node( const Node& other );
        Node& operator= ( const Node& other );

        struct HK_EXPORT_COMMON Triangle
        {
            Triangle( hkUint32 a, hkUint32 b, hkUint32 c );

            hkUint32 m_indices[ 3 ];
            hkUint32 m_sortedIndices[ 3 ];

            template<typename T>
            inline void _sort3( T& a, T& b, T& c );
        };

        struct HK_EXPORT_COMMON Edge
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, hkGeometryUtils::Node::Edge );
            Edge( hkUint32 endpointIndex, const Node::Triangle& triangle, hkUint32 triangleIndex );
            Edge& operator= ( const Edge& other );
            Edge( const Edge& other );

            // Return true if this edge has a parent triangle with exactly the same indices
            bool hasTriangleSameWinding( const Node::Triangle& triangle, int& triIndex );

            // Return true if this edge has a parent triangle with the same indices (in any permutation) as the given one
            bool hasTriangleIgnoreWinding( const Node::Triangle& triangle, int& triIndex );

            hkUint32 m_endpointIndex;

            // Parent triangles of edge
            hkArray<Node::Triangle> m_triangles;
            hkArray<hkUint32> m_triangleIndices;

            int m_numIncoming;
            int m_numOutgoing;

            bool m_nonManifold;
            bool m_inconsistentWinding;
            bool m_processed;
        };

        hkUint32 m_vertexIndex;
        hkArray<Edge> m_edges;

        void addEdge( hkUint32 endpointIndex, const Node::Triangle& triangle, hkUint32 triangleIndex, bool incoming = false );
        _Ret_maybenull_ Edge* findEdge( hkUint32 endpointIndex );

        bool checkForNonManifoldGeometry() const;
        void warnAboutInconsistentWinding( int e ) const;
    };

    /// Gets inner sphere for a set of planes
    /// initialCenter should be set to any point inside the shape formed by the planes (for example, average of the vertices)
    /// Returns false if initial center is invalid (outside of the shape) or planes don't form a convex shape
    static hkResult HK_CALL calculateInnerSphere(const hkVector4* planes, const int numPlanes, hkVector4Parameter initialCenter, hkSphere& sphereOut);

    /// Deflected sphere cast against plane constraints.
    /// Returns nearest point to the target position.
    static void HK_CALL deflectedSphereCast(const hkVector4* planes, const int numPlanes, const hkSphere& initialSphere, hkVector4Parameter targetPosition, int maxIterations, hkVector4& positionOut);
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
