// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/GeometryUtilities/hkGeometryUtilities.h>

#include <Common/GeometryUtilities/Misc/hkGeometryUtils.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/Algorithm/Collide/1AxisSweep/hk1AxisSweep.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>
#include <Common/Base/Math/Vector/hkIntVector.h>
#include <Common/Base/Math/Vector/hkVector4Util.h>
#include <Common/Internal/GeometryProcessing/hkGeometryProcessing.h>
#include <Common/Internal/GeometryProcessing/ConvexHull/hkgpConvexHull.h>
#include <Common/Base/Thread/Concurrency/hkConcurrency.h>
#include <Common/Base/Algorithm/Sort/hkMortonSort.h>

#define DEBUG_LOG_IDENTIFIER "common.geoutils"
#include <Common/Base/System/Log/hkLog.hxx>

hkGeometryUtils::GridInput::GridInput(int numVertsX, hkVector4Parameter up)
{
    m_stepX.set(1,0,0);
    m_stepY.setCross( m_stepX, up);
    m_numVertsX = numVertsX;
    m_numVertsY = numVertsX;

    centerGrid();
}

void hkGeometryUtils::GridInput::centerGrid()
{
    hkReal x = -0.5f * (m_numVertsX-1);
    hkReal y = -0.5f * (m_numVertsY-1);

    m_origin.setMul( m_stepX, hkSimdReal::fromFloat(x) );
    m_origin.addMul( m_stepY, hkSimdReal::fromFloat(y) );
}

void hkGeometryUtils::appendGrid( const GridInput& input, hkGeometry& out, int defaultMaterial )
{
    hkVector4 posX = input.m_origin;
    int iB = out.m_vertices.getSize();
    out.m_vertices.reserve( out.m_vertices.getSize() + input.m_numVertsX * input.m_numVertsY );
    out.m_triangles.reserve( out.m_triangles.getSize() + 2 * (input.m_numVertsX-1) * (input.m_numVertsY-1) );
    for ( int x=0; x<input.m_numVertsX; x++ )
    {
        hkVector4 posY = posX;
        for ( int y=0; y<input.m_numVertsY; y++ )
        {
            out.m_vertices.expandByUnchecked(1)[0] = posY;
            if ( x>0 && y >0)
            {
                int yf = input.m_numVertsY;
                int i = iB + y + x*yf;
                out.m_triangles.expandByUnchecked(1)->set( i   -0, i -1, i-yf-1, defaultMaterial );
                out.m_triangles.expandByUnchecked(1)->set( i-yf-1, i-yf, i   -0, defaultMaterial );
            }
            posY.add(input.m_stepY);
        }
        posX.add(input.m_stepX);
    }
}

void hkGeometryUtils::appendRandomConvex( int numVertices, hkReal minRadius, hkReal maxRadius, int randomSeed, hkGeometry& geomOut, int material )
{
    hkLocalArray<hkVector4> vertices(numVertices);
    hkSimdReal minRadSqrd = hkSimdReal::fromFloat( minRadius * minRadius );
    hkPseudoRandomGenerator random(randomSeed);
    {
        for(int i = 0; i < numVertices; i++)
        {
            hkVector4 vec; random.getRandomVector11(vec);
            vec.mul( hkSimdReal::fromFloat(maxRadius) );
            vec.setClampedToMaxLength( vec, hkSimdReal::fromFloat(maxRadius) );
            if ( vec.lengthSquared<3>() < minRadSqrd)
            {
                vec.normalize<3>();
                vec.mul( hkSimdReal::fromFloat(minRadius));
            }
            vertices.pushBackUnchecked( vec );
        }
    }
    hkgpConvexHull hull;
    hull.build(vertices.begin(), vertices.getSize() );
    hull.generateGeometry(hkgpConvexHull::SOURCE_VERTICES, geomOut, 0);

    if (material != -1 )
    {
        for (int i =0; i < geomOut.m_triangles.getSize(); i++)
        {
            geomOut.m_triangles[i].m_material = material;
        }
    }
}

template<typename T>
inline void hkGeometryUtils::Node::Triangle::_sort3(T& a, T& b, T& c)
{
    if (b < a) hkAlgorithm::swap(b, a);
    if (c < b) hkAlgorithm::swap(c, b);
    if (b < a) hkAlgorithm::swap(b, a);
}

hkGeometryUtils::Node::Triangle::Triangle(hkUint32 a, hkUint32 b, hkUint32 c)
{
    m_indices[0] = a; m_sortedIndices[0] = a;
    m_indices[1] = b; m_sortedIndices[1] = b;
    m_indices[2] = c; m_sortedIndices[2] = c;
    _sort3(m_sortedIndices[0], m_sortedIndices[1], m_sortedIndices[2]);
}

hkGeometryUtils::Node::Node( hkUint32 vertexIndex ) : m_vertexIndex(vertexIndex) {}

hkGeometryUtils::Node::Node(const hkGeometryUtils::Node& other)
{
    m_vertexIndex = other.m_vertexIndex;
    m_edges = other.m_edges;
}

hkGeometryUtils::Node& hkGeometryUtils::Node::operator= (const hkGeometryUtils::Node& other)
{
    m_vertexIndex = other.m_vertexIndex;
    m_edges = other.m_edges;
    return *this;
}

hkGeometryUtils::Node::Edge::Edge(hkUint32 endpointIndex, const Triangle& triangle, hkUint32 triangleIndex) :

m_endpointIndex(endpointIndex),
m_numIncoming(0),
m_numOutgoing(0),
m_nonManifold(false),
m_inconsistentWinding(false),
m_processed(false)

{
    m_triangles.pushBack(triangle);
    m_triangleIndices.pushBack(triangleIndex);
}

hkGeometryUtils::Node::Edge::Edge(const Edge& other)
{
    m_endpointIndex = other.m_endpointIndex;
    m_triangles = other.m_triangles;
    m_triangleIndices = other.m_triangleIndices;
    m_numIncoming = other.m_numIncoming;
    m_numOutgoing = other.m_numOutgoing;
    m_nonManifold = other.m_nonManifold;
    m_inconsistentWinding = other.m_inconsistentWinding;
    m_processed = other.m_processed;
}

hkGeometryUtils::Node::Edge& hkGeometryUtils::Node::Edge::operator= (const hkGeometryUtils::Node::Edge& other)
{
    m_endpointIndex = other.m_endpointIndex;
    m_triangles = other.m_triangles;
    m_triangleIndices = other.m_triangleIndices;
    m_numIncoming = other.m_numIncoming;
    m_numOutgoing = other.m_numOutgoing;
    m_nonManifold = other.m_nonManifold;
    m_inconsistentWinding = other.m_inconsistentWinding;
    m_processed = other.m_processed;
    return *this;
}

bool hkGeometryUtils::Node::Edge::hasTriangleSameWinding(const Triangle& triangle, int& triIndex)
{
    triIndex = -1;
    for (int t=0; t<m_triangles.getSize(); ++t)
    {
        if ( ( triangle.m_indices[0] == m_triangles[t].m_indices[0] ) &&
             ( triangle.m_indices[1] == m_triangles[t].m_indices[1] ) &&
             ( triangle.m_indices[2] == m_triangles[t].m_indices[2] ) )
        {
            triIndex = (int)m_triangleIndices[t];
            return true;
        }
    }
    return false;
}

bool hkGeometryUtils::Node::Edge::hasTriangleIgnoreWinding(const Triangle& triangle, int& triIndex)
{
    triIndex = -1;
    for (int t=0; t<m_triangles.getSize(); ++t)
    {
        if ( ( triangle.m_sortedIndices[0] == m_triangles[t].m_sortedIndices[0] ) &&
             ( triangle.m_sortedIndices[1] == m_triangles[t].m_sortedIndices[1] ) &&
             ( triangle.m_sortedIndices[2] == m_triangles[t].m_sortedIndices[2] ) )
        {
            triIndex = (int)m_triangleIndices[t];
            return true;
        }
    }
    return false;
}

_Ret_maybenull_ hkGeometryUtils::Node::Edge* hkGeometryUtils::Node::findEdge(hkUint32 endpointIndex)
{
    for (int e=0; e<m_edges.getSize(); ++e)
    {
        if (m_edges[e].m_endpointIndex == endpointIndex) return &m_edges[e];
    }
    return HK_NULL;
}

void hkGeometryUtils::Node::addEdge(hkUint32 endpointIndex, const Triangle& triangle, hkUint32 triangleIndex, bool incoming)
{
    hkGeometryUtils::Node::Edge* edge = findEdge(endpointIndex);
    if (edge)
    {
        if (incoming) edge->m_numIncoming++;
        else edge->m_numOutgoing++;

        // Flag edge as having inconsistent winding if it comes in or out of this node more than once
        if ( edge->m_numIncoming>1 || edge->m_numOutgoing>1) edge->m_inconsistentWinding = true;

        edge->m_triangleIndices.pushBack(triangleIndex);
        edge->m_triangles.pushBack(triangle);

        // Flag edge as non-manifold if more than 2 triangles share it
        if (edge->m_triangleIndices.getSize()>2) edge->m_nonManifold = true;
    }
    else
    {
        hkGeometryUtils::Node::Edge e(endpointIndex, triangle, triangleIndex);
        if (incoming) e.m_numIncoming++;
        else e.m_numOutgoing++;
        m_edges.pushBack(e);
    }
}

bool hkGeometryUtils::Node::checkForNonManifoldGeometry() const
{
    for (int ei=0; ei<m_edges.getSize(); ++ei)
    {
        const hkGeometryUtils::Node::Edge& edge = m_edges[ei];
        if (edge.m_nonManifold) return false;
    }
    return true;
}

void hkGeometryUtils::Node::warnAboutInconsistentWinding(int e) const
{
    const hkGeometryUtils::Node::Edge& edge = m_edges[e];
    if (edge.m_inconsistentWinding && edge.m_triangles.getSize()>1)
    {
        HK_WARN_ALWAYS(0xabba1daf, "Edge (" << edge.m_endpointIndex << "," << m_vertexIndex << ") has inconsistent winding in triangles " << edge.m_triangleIndices[0] << "and" << edge.m_triangleIndices[1] << ".");
    }
}


/******************************   hkGeometryUtils::weldVertices  ******************************/
namespace weldVerticesVirtualNs
{
    struct VertexRef
    {
        HK_INLINE   bool operator<(const VertexRef& other) const { return m_value < other.m_value; }
        hkSimdReal  m_value;
        int         m_index;
    };
}

using namespace weldVerticesVirtualNs;

int HK_CALL hkGeometryUtils::weldVerticesVirtual (_In_ const IVertices* vertices, hkArray<int>& remap, hkReal thr)
{
    const hkSimdReal threshold = hkSimdReal::fromFloat(thr);
    hkSimdReal maxDistanceSquared; maxDistanceSquared.setMul(threshold,threshold);
    const int           numVertices = vertices->getNumVertices();
    hkArray<VertexRef>  sortedVertices(numVertices);
    int                 numUnique = 0;
    remap.setSize(numVertices);
    for (int i=0; i<numVertices; i++)
    {
        hkVector4   x; vertices->getVertex(i, x);
        sortedVertices[i].m_index   =   i;
        sortedVertices[i].m_value   =   x.getComponent<0>();
    }

    hkAlgorithm::explicitStackQuickSort(sortedVertices.begin(), numVertices);

    for(int i=0; i<numVertices; ++i)
    {
        VertexRef&  vri = sortedVertices[i];

        if(vri.m_index < 0) continue;

        remap[vri.m_index]  =   vri.m_index;

        hkVector4   xi; vertices->getVertex(vri.m_index,xi);
        for(int j=i+1; j<numVertices; ++j)
        {
            VertexRef&  vrj = sortedVertices[j];

            if(vrj.m_index < 0) continue;
            if((vrj.m_value - vri.m_value) > threshold) break;

            hkVector4   xj; vertices->getVertex(vrj.m_index,xj);
            if(xi.distanceToSquared(xj).isLessEqual(maxDistanceSquared))
            {
                if(vertices->isWeldingAllowed(vrj.m_index, vri.m_index))
                {
                    remap[vrj.m_index]  =   vri.m_index;
                    vrj.m_index         =   -1;
                }
            }
        }
        vri.m_index = -1;
        ++numUnique;
    }

    return numUnique;
}

void HK_CALL hkGeometryUtils::weldVertices( struct hkGeometry& meshGeometry, hkReal threshold )
{
    hkArray<int> vertexRemap;
    hkArray<int> triangleRemap;
    hkArray<hkVector4> uniqueVerts;
    hkArray<hk1AxisSweep::AabbInt> sweepAabbs( meshGeometry.m_vertices.getSize() + 4 );
    hkArray<hkRadixSort::SortData32> sortData( meshGeometry.m_vertices.getSize() + 4 );
    hkArray<hk1AxisSweep::AabbInt> sortedAabbs( meshGeometry.m_vertices.getSize() + 4 );
    _weldVertices( meshGeometry, threshold, false, vertexRemap, triangleRemap, uniqueVerts, sweepAabbs, sortData, sortedAabbs );
}

void HK_CALL hkGeometryUtils::weldVertices( struct hkGeometry& meshGeometry, hkReal threshold, hkBool keepVertexOrder, hkArray<int>& vertexRemapOut, hkArray<int>& triangleRemapOut )
{
    hkArray<hkVector4> uniqueVerts;
    hkArray<hk1AxisSweep::AabbInt> sweepAabbs( meshGeometry.m_vertices.getSize() + 4 );
    hkArray<hkRadixSort::SortData32> sortData( meshGeometry.m_vertices.getSize() + 4 );
    hkArray<hk1AxisSweep::AabbInt> sortedAabbs( meshGeometry.m_vertices.getSize() + 4 );
    _weldVertices(meshGeometry, threshold, keepVertexOrder, vertexRemapOut, triangleRemapOut, uniqueVerts, sweepAabbs, sortData, sortedAabbs );
}

namespace
{
    // Weld the vertices inplace returning the number of new verts
    void hkWeldVertices( const hkArray<hkVector4>& vertsIn, hkArray<hkVector4>& uniqueVerts, hkArray<int>& remap, hkArrayBase<hk1AxisSweep::AabbInt>& vertAabbs, hkArrayBase<hkRadixSort::SortData32>& sortData,hkArrayBase<hk1AxisSweep::AabbInt>& tempAabbs, hkReal tol )
{
        const hkSimdReal tolerance = hkSimdReal::fromFloat(tol);
        const int numVerts = vertsIn.getSize();
        remap.setSize( numVerts );

        // Create AABBs for each vertex
        HK_ASSERT_NO_MSG(0x4be7d0ae, vertAabbs.getCapacity() >= numVerts + 4);
    {
            const hkSimdReal halfTol = tolerance * hkSimdReal_Half;
            vertAabbs.setSizeUnchecked(numVerts + 4);
            for (int v = 0; v < numVerts; v++)
        {
                hkAabb aabb;
                aabb.m_min.setSub( vertsIn[v], halfTol );
                aabb.m_max.setAdd( vertsIn[v], halfTol );
                vertAabbs[v].set(aabb, v);
            }

            hk1AxisSweep::AabbInt terminator; terminator.setInvalid(); terminator.setEndMarker();
            // I can do this because I made space for 4 terminators above
            hk1AxisSweep::AabbInt* end = vertAabbs.begin() + numVerts;
            end[0] = terminator;
            end[1] = terminator;
            end[2] = terminator;
            end[3] = terminator;
        }

        // Sort
        hk1AxisSweep::sortAabbs( vertAabbs.begin(), numVerts, sortData, tempAabbs );

        // Find all unique verts
        hkSimdReal tolSqrd; tolSqrd.setMul(tolerance, tolerance);
        for (int current=0; current < numVerts; current++)
        {
            const hkUint32 currentKey = vertAabbs[current].getKey();

            // This vertex is a duplicate
            if (currentKey == 0xFFFFFFFF)
                continue;

            // Store this vertex it is unique
            const hkVector4& currentPos = vertsIn[ currentKey ];
            remap[ currentKey ] = uniqueVerts.getSize();
            uniqueVerts.pushBack( currentPos );

            // For each overlapping AABB
            for ( int potential = current+1;
                  (potential < numVerts) && ( vertAabbs[potential].m_min[0] <= vertAabbs[current].m_max[0] );
                  potential++ )
            {
                const hkUint32 potentialKey = vertAabbs[potential].getKey();

                // Skip duplicates
                if (potentialKey == 0xFFFFFFFF)
                continue;

                const hkVector4& potentialPos = vertsIn[ potentialKey ];

                if ( currentPos.distanceToSquared( potentialPos ).isLessEqual(tolSqrd) )
                {
                    // Mark vertex as a duplicate
                    remap[ potentialKey ] = remap[ currentKey ];
                    vertAabbs[potential].getKey() = 0xFFFFFFFF;
                }
            }
        }
    }

    bool hkTriangle_isDegenerate( const hkGeometry::Triangle& a )
    {
        return (a.m_a == a.m_b) || (a.m_a == a.m_c) || (a.m_b == a.m_c);
    }
}

/*
static inline bool _sameTriangleIgnoringWinding(const hkGeometry::Triangle& triA, const hkGeometry::Triangle& triB)
{
        // Some assumptions we are making re. hkGeometryTriangle
        HK_ASSERT_NO_MSG(0xd52ff74, hkUlong(&(triA.m_a)) == hkUlong(&triA));
        HK_ASSERT_NO_MSG(0x64b63cb4, hkUlong(&(triA.m_b))-hkUlong(&(triA.m_a)) == hkUlong(sizeof(int)));

    const int* tA = reinterpret_cast<const int*> (&triA);
    const int* tB = reinterpret_cast<const int*> (&triB);

    // We assume neither triangle is degenerate
    for (int i=0; i<3; ++i)
    {
        bool found = false;
        for (int j=0; j<3; ++j)
        {
            if (tA[i]==tB[j]) found = true;
        }
        if (!found) return false;
    }

    return true;
}
*/

static inline bool _sameTriangleConsideringWinding(const hkGeometry::Triangle& triA, const hkGeometry::Triangle& triB)
{

    {
        // Some assumptions we are making re. hkGeometryTriangle
        HK_ASSERT_NO_MSG(0xd52ff74, hkUlong(&(triA.m_a)) == hkUlong(&triA));
        HK_ASSERT_NO_MSG(0x64b63cb4, hkUlong(&(triA.m_b))-hkUlong(&(triA.m_a)) == hkUlong(sizeof(int)));
    }

    const int* tA = reinterpret_cast<const int*> (&triA);
    const int* tB = reinterpret_cast<const int*> (&triB);

    // If triangle A is {0, 1, 2}, then any of
    //      {0, 1, 2}
    //      {1, 2, 0}
    //      {2, 0, 1}
    // is considered the "same" triangle
    return ((tA[0] == tB[0]) && (tA[1] == tB[1]) && (tA[2] == tB[2]))
        || ((tA[0] == tB[1]) && (tA[1] == tB[2]) && (tA[2] == tB[0]))
        || ((tA[0] == tB[2]) && (tA[1] == tB[0]) && (tA[2] == tB[1])) ;
}

hkUint64 hashTriangle( const hkGeometry::Triangle& t1 )
{
    const int mask = (1<<21) - 1;
    hkUint64 x = t1.m_a & (mask);
    hkUint64 y = t1.m_b & (mask);
    hkUint64 z = t1.m_c & (mask);

    return (x<<42) | (y<<21) | z ;
}

struct HK_EXPORT_COMMON triInfo
{
    hkUint64 m_key;
    hkInt32 m_index;

};

struct HK_EXPORT_COMMON sortByKey
{
public:

    HK_INLINE hkBool32 operator() ( const triInfo& t1, const triInfo& t2 )
    {
        // Sort by key first, then index is the tie-breaker
        return (t1.m_key < t2.m_key) || ((t1.m_key == t2.m_key) && (t1.m_index < t2.m_index));
    }
};

struct HK_EXPORT_COMMON sortByIndex
{
public:

    HK_INLINE hkBool32 operator() ( const triInfo& t1, const triInfo& t2 )
    {
        return t1.m_index < t2.m_index;
    }
};

hkResult HK_CALL hkGeometryUtils::removeDuplicateTrianglesFast(struct hkGeometry& meshGeometry, hkArray<hkGeometry::Triangle>& newTriangles )
{
    int size = meshGeometry.m_triangles.getSize();
    if (size == 0)
    {
        return HK_SUCCESS;
    }

    hkArray<triInfo>::Temp sortArray(size);
    if (sortArray.begin() == HK_NULL)
    {
        return HK_FAILURE;
    }

    // Go through the triangles, sort the indices, and overwrite the material with the actual index
    for (int t=0; t<meshGeometry.m_triangles.getSize(); t++)
            {
        // Bubble sort!
        hkGeometry::Triangle triangle = meshGeometry.m_triangles[t];
        if (triangle.m_a > triangle.m_b)    { hkAlgorithm::swap( triangle.m_a, triangle.m_b ); }
        if (triangle.m_a > triangle.m_c)    { hkAlgorithm::swap( triangle.m_a, triangle.m_c ); }
        if (triangle.m_b > triangle.m_c)    { hkAlgorithm::swap( triangle.m_b, triangle.m_c ); }

        HK_ASSERT_NO_MSG(0x53429a52, triangle.m_a <= triangle.m_b);
        HK_ASSERT_NO_MSG(0x53429a52, triangle.m_a <= triangle.m_c);
        HK_ASSERT_NO_MSG(0x53429a52, triangle.m_b <= triangle.m_c);

        triInfo& entry = sortArray[t];

        entry.m_key = hashTriangle(triangle);
        entry.m_index = t;
            }

    // Sort by key. This will clump potentially identical triangles together
    hkSort( sortArray.begin(), sortArray.getSize(), sortByKey() );

    // Walk backwards, removing duplicates
    for (int t=sortArray.getSize()-1; t>=1; t--)
    {
        const hkUint64 key1 = sortArray[t].m_key;

        int lastMatch = -1;
        // Find the first triangle that matches this one
        for (int t2 = t-1; t2>=0 && key1 == sortArray[t2].m_key; t2--)
            {
            hkGeometry::Triangle& tri1 = meshGeometry.m_triangles[ sortArray[t].m_index ];
            hkGeometry::Triangle& tri2 = meshGeometry.m_triangles[ sortArray[t2].m_index ];

            if ( _sameTriangleConsideringWinding( tri1, tri2 ) )
                {
                lastMatch = t2;
            }
                }

        // If we found a match, remove it
        if (lastMatch != -1)
                {
            sortArray.removeAt(t);
                }
            }

    // Resort by original index
    hkSort( sortArray.begin(), sortArray.getSize(), sortByIndex() );

    newTriangles.clear();

    for (int t=0; t<sortArray.getSize(); t++)
    {
        int originalIndex = sortArray[t].m_index;
        newTriangles.pushBack( meshGeometry.m_triangles[ originalIndex ] );
    }

    meshGeometry.m_triangles = newTriangles;
    return HK_SUCCESS;
        }

void HK_CALL hkGeometryUtils::optimizeMemoryLayout( hkGeometry& geometry, _Inout_opt_ hkArray<int>* vertexMap, _Inout_opt_ hkArray<int>* triangleMap )
{
    // Generate primitives from triangles and initialize tMap.
    hkArray<hkVector4>::Temp primitives; primitives.setSize( geometry.m_triangles.getSize() );
    hkConcurrency::parallelFor( primitives.getSize(), [&geometry, &primitives]( int triangleIndex )
    {
        const hkGeometry::Triangle& t = geometry.m_triangles[ triangleIndex ];
        hkAabb aabb; aabb.setFromTriangle( geometry.m_vertices[ t.m_a ], geometry.m_vertices[ t.m_b ], geometry.m_vertices[ t.m_c ] );
        hkVector4& prim = primitives[ triangleIndex ]; aabb.getCenter( prim ); prim.setInt24W( triangleIndex );
    } );

    // Sort.
    hkAlgorithm::Morton::sort( primitives );

    // Generate output and re-index vertices.
    hkArray<int> vMap( geometry.m_vertices.getSize(), -1 );
    hkArray<int> tMap( geometry.m_triangles.getSize() );
    hkArray<hkGeometry::Triangle> triangles( geometry.m_triangles.getSize() );
    hkArray<hkVector4> vertices( geometry.m_vertices.getSize() );
    int vertexIndex = 0;
    for ( int primitiveIndex = 0; primitiveIndex < primitives.getSize(); ++primitiveIndex )
    {
        const int triangleIndex = primitives[ primitiveIndex ].getInt24W();

        triangles[ primitiveIndex ] = geometry.m_triangles[ triangleIndex ];
        tMap[ triangleIndex ] = primitiveIndex;

        int* indices = &triangles[ primitiveIndex ].m_a;
        for ( int i = 0; i < 3; ++i )
        {
            const int j = indices[ i ];
            if ( vMap[ j ] == -1 )
            {
                vMap[ j ] = vertexIndex;
                vertices[ vertexIndex++ ] = geometry.m_vertices[ j ];
            }
            indices[ i ] = vMap[ j ];
        }
    }

    // Exchange vertex and triangles buffers.
    geometry.m_triangles.swap( triangles );
    geometry.m_vertices.swap( vertices );

    // Set maps.
    if ( vertexMap ) vertexMap->swap( vMap );
    if ( triangleMap ) triangleMap->swap( tMap );
}

void HK_CALL hkGeometryUtils::_weldVertices( struct hkGeometry& meshGeometry, hkReal threshold, hkBool keepVertexOrder, hkArray<int>& vertexRemapOut, hkArray<int>& triangleRemapOut, hkArray<hkVector4>& uniqueVerts, hkArrayBase<hk1AxisSweep::AabbInt>& sweepAabbs, hkArrayBase<hkRadixSort::SortData32>& sortData, hkArrayBase<hk1AxisSweep::AabbInt>& sortedAabbs )
{
    const int originalNumVertices = meshGeometry.m_vertices.getSize();
    const int originalNumTriangles = meshGeometry.m_triangles.getSize();

    // Delete any duplicate vertices, building the map from old to new indices
    {
        // Initialize the index map
        vertexRemapOut.setSize( originalNumVertices );
        hkWeldVertices( meshGeometry.m_vertices, uniqueVerts, vertexRemapOut, sweepAabbs, sortData, sortedAabbs, threshold );

        if (keepVertexOrder)
        {
            // HCL-1174
            hkArray< hkVector4 > orderedVerts; orderedVerts.reserve(uniqueVerts.getSize());
            hkArray< int > extraMap;
            extraMap.setSize(uniqueVerts.getSize(),-1);

            for (int i=0; i<vertexRemapOut.getSize(); ++i)
            {
                const int originalIndex = i;
                const int remapedIndex = vertexRemapOut[i];
                int newIndex = extraMap [remapedIndex];
                if (newIndex == -1) // new vert
                {
                    newIndex = orderedVerts.getSize();
                    extraMap [remapedIndex] = newIndex;
                    orderedVerts.pushBack(uniqueVerts[remapedIndex]);
                }
                vertexRemapOut[originalIndex] = newIndex;
                }

            HK_ASSERT_NO_MSG(0x36b08746, orderedVerts.getSize() == uniqueVerts.getSize());

            meshGeometry.m_vertices = orderedVerts;
            }
        else
        {
            meshGeometry.m_vertices = uniqueVerts;
        }
    }

    // Remap each triangle's vertex indices
    {
        for( int i=0; i < originalNumTriangles; ++i )
        {
            hkGeometry::Triangle& triangle = meshGeometry.m_triangles[i];
            triangle.m_a = vertexRemapOut[triangle.m_a];
            triangle.m_b = vertexRemapOut[triangle.m_b];
            triangle.m_c = vertexRemapOut[triangle.m_c];
        }
    }

    triangleRemapOut.setSize( originalNumTriangles );

    // Copy down nondegenerate triangles in place
    {
        hkGeometry::Triangle* nonDegenerate = meshGeometry.m_triangles.begin();
        for( int i=0; i<originalNumTriangles; ++i )
        {
            const hkGeometry::Triangle& triangle = meshGeometry.m_triangles[i];

            int remappedIndex = -1; // -1 implies the triangle is degenerate

            if ( !hkTriangle_isDegenerate( triangle ) )
            {
                // Copy if not degenerate
                remappedIndex = (int) ( nonDegenerate - meshGeometry.m_triangles.begin() );
                *nonDegenerate++ = triangle;
            }

            triangleRemapOut[i] = remappedIndex;

        }
        meshGeometry.m_triangles.setSize( (int) ( nonDegenerate - meshGeometry.m_triangles.begin() ) );
    }
}



/******************************   hkGeometryUtils::removeDuplicateTriangles  ******************************/


void HK_CALL hkGeometryUtils::removeDuplicateTriangles (struct hkGeometry& meshGeometry, hkArray<int>& triangleMapOut, bool ignoreWinding)
{
    hkArray<hkGeometry::Triangle> newTriangles;
    hkArray<Node>::Temp nodes;

    for (hkUint32 vi=0; vi<(hkUint32)meshGeometry.m_vertices.getSize(); ++vi)
    {
        Node node(vi);
        nodes.pushBack(node);
        }

    int numDuplicates = 0;

    if (ignoreWinding)
    {
        for (hkUint32 ti=0; ti<(hkUint32)meshGeometry.m_triangles.getSize(); ++ti)
        {
            hkGeometry::Triangle& geomTriangle = meshGeometry.m_triangles[ti];
            hkGeometryUtils::Node::Triangle tri(geomTriangle.m_a, geomTriangle.m_b, geomTriangle.m_c);

            Node::Edge* AB = nodes[geomTriangle.m_a].findEdge(geomTriangle.m_b);
            Node::Edge* BC = nodes[geomTriangle.m_b].findEdge(geomTriangle.m_c);
            Node::Edge* CA = nodes[geomTriangle.m_c].findEdge(geomTriangle.m_a);

            int triIndex;
            if ( (AB && AB->hasTriangleIgnoreWinding(tri, triIndex)) ||
                 (BC && BC->hasTriangleIgnoreWinding(tri, triIndex)) ||
                 (CA && CA->hasTriangleIgnoreWinding(tri, triIndex)) )
            {
                HK_ASSERT_NO_MSG(0x5c6b5be9, triIndex>=0);
                triangleMapOut.pushBack(triIndex);
                numDuplicates++;
            }
            else
            {
                int indices[3] = {geomTriangle.m_a, geomTriangle.m_b, geomTriangle.m_c};
                int newTriangleIndex = newTriangles.getSize();

                Node::Triangle triangle(geomTriangle.m_a, geomTriangle.m_b, geomTriangle.m_c);

                for (int a=0; a<3; ++a)
            {
                    int b = (a+1) % 3;
                    hkUint32 va = indices[a];
                    hkUint32 vb = indices[b];

                    nodes[va].addEdge(vb, triangle, newTriangleIndex, false);
                    nodes[vb].addEdge(va, triangle, newTriangleIndex, true);
                }

                triangleMapOut.pushBack(newTriangleIndex);
                newTriangles.pushBack(meshGeometry.m_triangles[ti]);
                }
            }
        }

    else
    {
        for (hkUint32 ti=0; ti<(hkUint32)meshGeometry.m_triangles.getSize(); ++ti)
        {
            hkGeometry::Triangle& geomTriangle = meshGeometry.m_triangles[ti];
            hkGeometryUtils::Node::Triangle tri(geomTriangle.m_a, geomTriangle.m_b, geomTriangle.m_c);

            Node::Edge* AB = nodes[geomTriangle.m_a].findEdge(geomTriangle.m_b);
            Node::Edge* BC = nodes[geomTriangle.m_b].findEdge(geomTriangle.m_c);
            Node::Edge* CA = nodes[geomTriangle.m_c].findEdge(geomTriangle.m_a);

            int triIndex;
            if ( (AB && AB->hasTriangleSameWinding(tri, triIndex)) ||
                 (BC && BC->hasTriangleSameWinding(tri, triIndex)) ||
                 (CA && CA->hasTriangleSameWinding(tri, triIndex)) )
        {
                HK_ASSERT_NO_MSG(0x3aaf0b78, triIndex>=0);
                triangleMapOut.pushBack(triIndex);
                numDuplicates++;
            }
            else
            {
                int indices[3] = {geomTriangle.m_a, geomTriangle.m_b, geomTriangle.m_c};
                int newTriangleIndex = newTriangles.getSize();

                Node::Triangle triangle(geomTriangle.m_a, geomTriangle.m_b, geomTriangle.m_c);

                for (int a=0; a<3; ++a)
                {
                    int b = (a+1) % 3;
                    hkUint32 va = indices[a];
                    hkUint32 vb = indices[b];

                    nodes[va].addEdge(vb, triangle, newTriangleIndex, false);
                    nodes[vb].addEdge(va, triangle, newTriangleIndex, true);
                }

                triangleMapOut.pushBack(newTriangleIndex);
                newTriangles.pushBack(meshGeometry.m_triangles[ti]);
                }
            }
        }

    if (numDuplicates>0)
    {
        Log_Info( "Removed {} duplicate triangles out of a total of {} triangles.", numDuplicates, meshGeometry.m_triangles.getSize() );
    }

    meshGeometry.m_triangles = newTriangles;
}


/******************************   debug display geometry  ******************************/


static const hkGeometry::Triangle s_boxTriangle[] = { {0, 1, 2, -1}, {0, 2, 3, -1}, {4, 6, 5, -1}, {4, 7, 6, -1}, {0, 4, 5, -1}, {0, 5, 1, -1}, {2, 6, 7, -1}, {2, 7, 3, -1}, {2, 1, 5, -1}, {2, 5, 6, -1}, {0, 3, 7, -1}, {0, 7, 4, -1}, };
static const hkVector4ComparisonMask::Mask s_boxOffsetMasks[] = { hkVector4ComparisonMask::MASK_NONE, hkVector4ComparisonMask::MASK_X,  hkVector4ComparisonMask::MASK_XY,  hkVector4ComparisonMask::MASK_Y,
                                                                  hkVector4ComparisonMask::MASK_Z,    hkVector4ComparisonMask::MASK_XZ, hkVector4ComparisonMask::MASK_XYZ, hkVector4ComparisonMask::MASK_YZ };

void hkGeometryUtils::appendBox( hkVector4Parameter center, hkVector4Parameter halfExtents, hkGeometry& geometryOut, int material )
{
    hkArray<hkVector4> verts;
    hkArray<hkGeometry::Triangle> tris;

    hkVector4 halfExtentsAbs; halfExtentsAbs.setAbs(halfExtents);

    int nExistingVerts = verts.getSize();
    verts.setSize(nExistingVerts + 8);

    hkVector4Comparison mask;
    hkVector4 pos;

    for (int i = 0; i < 8; ++i)
    {
        mask.set(s_boxOffsetMasks[i]);
        pos.setFlipSign(halfExtentsAbs, mask);
        pos.add(center);
        verts[nExistingVerts + i] = pos;
    }

    for(int t = 0; t < HK_COUNT_OF(s_boxTriangle); t++)
    {
        hkGeometry::Triangle& tri = tris.expandOne();
        tri.m_material = material;
        tri.set(s_boxTriangle[t].m_a + nExistingVerts, s_boxTriangle[t].m_b + nExistingVerts, s_boxTriangle[t].m_c + nExistingVerts, material);
    }

    hkGeometry builtGeom;
    builtGeom.m_vertices.swap( verts );
    builtGeom.m_triangles.swap( tris );
    geometryOut.appendGeometry( builtGeom );
}

void HK_CALL hkGeometryUtils::appendCapsule(const hkVector4& start, const hkVector4& end, hkReal radius, int heightSamples, int thetaSamples, const hkTransform& transform, hkGeometry& geometryOut)
{

    const int phiSamples = thetaSamples >> 1;

    hkArray<hkVector4> verts;

    // Create "transform" from start, end.
    hkTransform capsuleToLocal;

    const hkVector4& zAxis = hkVector4::getConstant<HK_QUADREAL_0010>();
    hkVector4 axis;
    axis.setSub(end, start);
    hkSimdReal height = axis.length<3>();
    if(height.isGreaterZero())
    {
        axis.normalize<3>();

        hkSimdReal axisDot; axisDot.setAbs(axis.dot<3>(zAxis));
        if(axisDot < hkSimdReal::fromFloat(1.0f - 1e-5f))
        {
            hkRotation rotation;
            hkVector4& col0 = rotation.getColumn(0);
            hkVector4& col1 = rotation.getColumn(1);
            hkVector4& col2 = rotation.getColumn(2);

            col2 = axis;
            col1.setCross( col2, zAxis);
            col1.normalize<3>();
            col0.setCross( col1, col2 );
            capsuleToLocal.setRotation(rotation);
        }
        else
        {
            capsuleToLocal.setIdentity();
        }
    }
    else
    {
        capsuleToLocal.setRows4( hkVector4::getConstant<HK_QUADREAL_0010>(), hkVector4::getConstant<HK_QUADREAL_0100>(), -hkVector4::getConstant<HK_QUADREAL_1000>(), hkVector4::getConstant<HK_QUADREAL_0001>() );
    }

    // Now recenter
    {
        hkVector4 toCentre;
        toCentre.setAdd(start, end);
        toCentre.mul(hkSimdReal_Half);
        capsuleToLocal.setTranslation(toCentre);
    }

    // We'll sweep around the axis of the deflector, from top to bottom, using the original
    // sample directions and data to define the vertices. We'll tessellate in the obvious way.
    // N.B. Top and bottom vertices are added to cap the object.

    int i,j;

    hkVector4 vert;

    hkVector4 bottomInGeom; bottomInGeom.setMul(zAxis,-height*hkSimdReal_Half);
    hkVector4 topInGeom; topInGeom.setNeg<4>(bottomInGeom);
    hkVector4 axisInGeom;
    axisInGeom.setSub(topInGeom, bottomInGeom);
    hkVector4 axisNInGeom = zAxis;
    const hkVector4& normalInGeom = hkVector4::getConstant<HK_QUADREAL_1000>();
    hkVector4 binormalInGeom; binormalInGeom.setMul(hkVector4::getConstant<HK_QUADREAL_0100>(),hkSimdReal_Minus1);

    // top capsule = phiSamples (segments) = phiSamples + 1 vert rings but top ring is vert so phiSamples * thetaSamples + 1 verts
    // This contains the top ring of the cylinder, top cap = phiSamples rings of faces
    // bottom capsule = phiSamples (segments) = phiSamples + 1 rings but bottom ring is vert so phiSamples * thetaSamples + 1 verts
    // This contains the bottom ring of the cylinder, bottom cap = phiSamples rings of faces
    // cylinder body = heightSamples (segments) = heightSamples + 1 vert rings but bottom and top caps already create 2 so (heightSamples -1) * thetaSamples verts
    // cylinder body = heightSamples rings of faces
    // total number of face rings = 2 * phiSamples + heightSamples
    verts.reserveExactly(2 * phiSamples * thetaSamples + 2 + (heightSamples-1) * thetaSamples);

    //
    // GET TOP VERTICES
    //
    const hkSimdReal radiusSr = hkSimdReal::fromFloat(radius);

    vert.setMul(zAxis,height*hkSimdReal_Half+radiusSr);
    vert._setTransformedPos(capsuleToLocal, vert);
    verts.pushBack(vert);

    const hkSimdReal invPhiSamples = hkSimdReal::fromInt32(phiSamples).reciprocal() * hkSimdReal_PiOver2;
    const hkSimdReal invThetaSamples = hkSimdReal::fromInt32(thetaSamples).reciprocal() * hkSimdReal_TwoPi;
    for (i = phiSamples-1 ; i >= 0; i--)
    {
        hkQuaternion qTop; qTop.setAxisAngle(binormalInGeom, hkSimdReal::fromInt32(i) * invPhiSamples);
        hkVector4 topElevation;
        topElevation._setRotatedDir(qTop, normalInGeom);

        for (j = 0; j < thetaSamples; j++)
        {
            hkQuaternion rotationTop; rotationTop.setAxisAngle(axisNInGeom, hkSimdReal::fromInt32(j) * invThetaSamples);
            hkVector4 topDirection;
            topDirection._setRotatedDir(rotationTop, topElevation);

            vert.setAddMul(topInGeom, topDirection, radiusSr);

            vert._setTransformedPos(capsuleToLocal, vert);

            //push back the rest of the vertices
            verts.pushBack(vert);

        }
    }

    //
    // GET MIDDLE VERTICES
    //
    const hkSimdReal invHeightSamples = hkSimdReal::fromInt32(heightSamples).reciprocal();
    for (j = heightSamples-1; j > 0; j--)
    {

        for (i = 0; i < thetaSamples; i++)
        {
        //
        // Calculate direction vector for this angle
        //

            hkQuaternion q; q.setAxisAngle(axisNInGeom, hkSimdReal::fromInt32(i) * invThetaSamples);
            hkVector4 direction;
            direction._setRotatedDir(q, normalInGeom);

            hkVector4 s;
            s.setAddMul(bottomInGeom, axisInGeom, hkSimdReal::fromInt32(j) * invHeightSamples);

            vert.setAddMul(s, direction, radiusSr);

            vert._setTransformedPos(capsuleToLocal, vert);

            verts.pushBack(vert);

        }
    }

    //
    // GET BOTTOM VERTICES
    //
    for (i = 0; i < phiSamples; i++)
    {
        hkQuaternion qBottom; qBottom.setAxisAngle(binormalInGeom, -hkSimdReal::fromInt32(i) * invPhiSamples);
        hkVector4 bottomElevation;
        bottomElevation._setRotatedDir(qBottom, normalInGeom);

        for (j = 0; j < thetaSamples; j++)
        {
            hkQuaternion rotationBottom; rotationBottom.setAxisAngle(axisNInGeom, hkSimdReal::fromInt32(j) * invThetaSamples);
            hkVector4 bottomDirection;
            bottomDirection._setRotatedDir(rotationBottom, bottomElevation);

            vert.setAddMul(bottomInGeom, bottomDirection, radiusSr);
            vert._setTransformedPos(capsuleToLocal, vert);
            verts.pushBack(vert);
        }
    }

    vert.setMul(zAxis,-(height*hkSimdReal_Half+radiusSr));
    vert._setTransformedPos(capsuleToLocal, vert);

        // Push back bottom vertex
    verts.pushBack(vert);

    // Transform all the points by m_transform.
    
    
    
    {
        for ( int vi = 0; vi < verts.getSize(); ++vi )
        {
            verts[vi]._setTransformedPos(transform, verts[vi]);
        }
    }

    //
    // CONSTRUCT FACE DATA
    //

    // Right, num samples AROUND axis is thetaSamples.

    // First off, we have thetaSamples worth of faces connected to the top
    hkGeometry::Triangle tr;
    hkArray<hkGeometry::Triangle> tris;
    tr.m_material=-1;

    int currentBaseIndex = 1;
    for (i = 0; i < thetaSamples; i++)
    {
        tr.m_a = 0;
        tr.m_b = currentBaseIndex + i;
        tr.m_c = currentBaseIndex + (i+1)%(thetaSamples);

        tris.pushBack(tr);
    }

    // Next we have phi-1 + height + phi-1 lots of thetaSamples*2 worth of faces connected to the previous row
    for(j = 0; j < 2*(phiSamples-1) + heightSamples; j++)
    {
        for (i = 0; i < thetaSamples; i++)
        {
            tr.m_a = currentBaseIndex + i;
            tr.m_b = currentBaseIndex + thetaSamples + i;
            tr.m_c = currentBaseIndex + thetaSamples + (i+1)%(thetaSamples);

            tris.pushBack(tr);

            tr.m_b = currentBaseIndex + i;
            tr.m_a = currentBaseIndex + (i+1)%(thetaSamples);
            tr.m_c = currentBaseIndex + thetaSamples + (i+1)%(thetaSamples);

            tris.pushBack(tr);

        }
        currentBaseIndex += thetaSamples;
    }

    // Finally, we have thetaSamples worth of faces connected to the bottom
    for (i = 0; i < thetaSamples; i++)
    {
        tr.m_b = currentBaseIndex + i;
        tr.m_a = currentBaseIndex + (i+1)%(thetaSamples);
        tr.m_c = currentBaseIndex + thetaSamples;

        tris.pushBack(tr);
    }

    hkGeometry builtGeom;
    builtGeom.m_vertices.swap( verts );
    builtGeom.m_triangles.swap( tris );
    geometryOut.appendGeometry(builtGeom);
}


void HK_CALL hkGeometryUtils::appendTaperedCapsule(const hkVector4& start, const hkVector4& end, hkReal startRadius, hkReal endRadius, int heightSamples, int thetaSamples, const hkTransform& transform, hkGeometry& geometryOut)
{
    hkSimdReal bigRadius, smallRadius;
    hkVector4 big, small;


    if (startRadius > endRadius)
    {
        big = start;
        small = end;
        bigRadius.setFromFloat(startRadius);
        smallRadius.setFromFloat(endRadius);
    }
    else
    {
        big = end;
        small = start;
        bigRadius.setFromFloat(endRadius);
        smallRadius.setFromFloat(startRadius);
    }

    const int phiSamples = thetaSamples >> 1;
    hkArray<hkVector4> verts;

    hkVector4 axis;
    axis.setSub(big, small);
    hkSimdReal height = axis.normalizeWithLength<3>() + hkSimdReal_Eps;

    const hkVector4& xAxis = hkVector4::getConstant<HK_QUADREAL_1000>();
    const hkVector4& yAxis = hkVector4::getConstant<HK_QUADREAL_0100>();
    const hkVector4& zAxis = hkVector4::getConstant<HK_QUADREAL_0010>();

    // Create "transform" from start, end.
    hkTransform capsuleToLocal;
    {
        if(height > (hkSimdReal_Eps+hkSimdReal_Eps))
        {
            axis.normalize<3>();

            hkSimdReal axisDot; axisDot.setAbs(axis.dot<3>(zAxis));
            if(axisDot < hkSimdReal::fromFloat(1.0f - 1e-5f))
            {
                hkRotation rotation;
                hkVector4& col0 = rotation.getColumn(0);
                hkVector4& col1 = rotation.getColumn(1);
                hkVector4& col2 = rotation.getColumn(2);

                col2 = axis;
                col1.setCross( col2, zAxis);
                col1.normalize<3>();
                col0.setCross( col1, col2 );
                capsuleToLocal.setRotation(rotation);
            }
            else
            {
                if (axis.dot<3>(zAxis).isLessZero())
                {
                    // Capsule oriented along -z
                    hkRotation rotation;
                    rotation.setAxisAngle(xAxis, hkSimdReal_Pi);
                    capsuleToLocal.setRotation(rotation);
                }
                else
                {
                    capsuleToLocal.setIdentity();
                }
            }
        }
        else
        {
            capsuleToLocal.setIdentity();
        }

        hkVector4 center;
        center.setAdd(start, end);
        center.mul(hkSimdReal_Half);
        capsuleToLocal.setTranslation(center);
    }

    hkVector4 localBig;   localBig.setMul(zAxis, height*hkSimdReal_Half);
    hkVector4 localSmall; localSmall.setNeg<4>(localBig);

    verts.reserveExactly(2 * phiSamples * thetaSamples + 2 + (heightSamples-1) * thetaSamples);

    hkVector4 vert;

    //
    // GET BIG SPHERE VERTICES
    //
    vert.setMul(zAxis, height*hkSimdReal_Half + bigRadius);
    vert._setTransformedPos(capsuleToLocal, vert);
    verts.pushBack(vert);

    hkSimdReal sinTheta; sinTheta.setClamped( (bigRadius - smallRadius) / height, hkSimdReal_Minus1, hkSimdReal_1 );
    hkSimdReal cosTheta = (hkSimdReal_1 - sinTheta*sinTheta).sqrt();
    hkSimdReal theta = hkVector4Util::aSin(sinTheta);

    hkSimdReal invPhiSamples; invPhiSamples.setReciprocal(hkSimdReal::fromInt32(phiSamples));
    hkSimdReal invThetaSamples; invThetaSamples.setReciprocal(hkSimdReal::fromInt32(thetaSamples)); invThetaSamples.mul(hkSimdReal_TwoPi);
    for (int i = phiSamples-1 ; i >= 0; i--)
    {
        hkSimdReal frac = hkSimdReal::fromInt32(i) * invPhiSamples;
        hkSimdReal angle = -theta * (hkSimdReal_1 - frac) + hkSimdReal_PiOver2 * frac;

        hkQuaternion qTop; qTop.setAxisAngle(yAxis, -angle);
        hkVector4 topElevation;
        topElevation._setRotatedDir(qTop, xAxis);

        for (int j = 0; j < thetaSamples; j++)
        {
            hkQuaternion rotationTop; rotationTop.setAxisAngle(zAxis, hkSimdReal::fromInt32(j) * invThetaSamples);
            hkVector4 topDirection;
            topDirection._setRotatedDir(rotationTop, topElevation);

            vert.setAddMul(localBig, topDirection, bigRadius);
            vert._setTransformedPos(capsuleToLocal, vert);
            verts.pushBack(vert);
        }
    }

    //
    // GET CONE VERTICES
    //
    hkSimdReal sMin = -smallRadius * sinTheta;
    hkSimdReal sMax = height - bigRadius * sinTheta;

    hkSimdReal rMin = smallRadius * cosTheta;
    hkSimdReal rMax = bigRadius * cosTheta;

    hkSimdReal invHeightSamples; invHeightSamples.setReciprocal(hkSimdReal::fromInt32(heightSamples));
    for (int j = heightSamples-1; j > 0; j--)
    {
        for (int i = 0; i < thetaSamples; i++)
        {
            //
            // Calculate direction vector for this angle
            //
            hkQuaternion q; q.setAxisAngle(zAxis, hkSimdReal::fromInt32(i) * invThetaSamples);
            hkVector4 direction;
            direction._setRotatedDir(q, xAxis);

            hkSimdReal axisFrac = hkSimdReal::fromInt32(j) * invHeightSamples;

            hkSimdReal s = (hkSimdReal_1 - axisFrac) * sMin + axisFrac * sMax;

            hkVector4 coneZ;
            coneZ.setAddMul(localSmall, zAxis, s);

            hkSimdReal rad = (hkSimdReal_1 - axisFrac) * rMin + axisFrac * rMax;

            vert.setAddMul(coneZ, direction, rad);
            vert._setTransformedPos(capsuleToLocal, vert);
            verts.pushBack(vert);
        }
    }

    //
    // GET SMALL SPHERE VERTICES
    //
    for (int i = 0; i < phiSamples; i++)
    {
        hkSimdReal frac = hkSimdReal::fromInt32(i) * invPhiSamples;
        hkSimdReal angle = theta * (hkSimdReal_1 - frac) + hkSimdReal_PiOver2 * frac;

        hkQuaternion qBottom; qBottom.setAxisAngle(yAxis, angle);
        hkVector4 bottomElevation;
        bottomElevation._setRotatedDir(qBottom, xAxis);

        for (int j = 0; j < thetaSamples; j++)
        {
            hkQuaternion rotationBottom; rotationBottom.setAxisAngle(zAxis, hkSimdReal::fromInt32(j) * invThetaSamples);
            hkVector4 bottomDirection;
            bottomDirection._setRotatedDir(rotationBottom, bottomElevation);

            vert.setAddMul(localSmall, bottomDirection, smallRadius);
            vert._setTransformedPos(capsuleToLocal, vert);
            verts.pushBack(vert);
        }
    }

    vert.setMul(zAxis, -(height*hkSimdReal_Half + smallRadius));
    vert._setTransformedPos(capsuleToLocal, vert);

    // Push back bottom vertex
    verts.pushBack(vert);

    {
        for ( int vi = 0; vi < verts.getSize(); ++vi )
        {
            verts[vi]._setTransformedPos(transform, verts[vi]);
        }
    }
    //
    // CONSTRUCT FACE DATA
    //

    // Num. samples AROUND axis is thetaSamples.
    // First off, we have thetaSamples worth of faces connected to the top
    hkGeometry::Triangle tr;
    hkArray<hkGeometry::Triangle> tris;
    tr.m_material=-1;

    int currentBaseIndex = 1;
    for (int i = 0; i < thetaSamples; i++)
    {
        tr.m_a = 0;
        tr.m_b = currentBaseIndex + i;
        tr.m_c = currentBaseIndex + (i+1)%(thetaSamples);
        tris.pushBack(tr);
    }

    // Next we have phi-1 + height + phi-1 lots of thetaSamples*2 worth of faces connected to the previous row
    for(int j = 0; j < 2*(phiSamples-1) + heightSamples; j++)
    {
        for (int i = 0; i < thetaSamples; i++)
        {
            tr.m_a = currentBaseIndex + i;
            tr.m_b = currentBaseIndex + thetaSamples + i;
            tr.m_c = currentBaseIndex + thetaSamples + (i+1)%(thetaSamples);
            tris.pushBack(tr);

            tr.m_b = currentBaseIndex + i;
            tr.m_a = currentBaseIndex + (i+1)%(thetaSamples);
            tr.m_c = currentBaseIndex + thetaSamples + (i+1)%(thetaSamples);
            tris.pushBack(tr);

        }
        currentBaseIndex += thetaSamples;
    }

    // Finally, we have thetaSamples worth of faces connected to the bottom
    for (int i = 0; i < thetaSamples; i++)
    {
        tr.m_b = currentBaseIndex + i;
        tr.m_a = currentBaseIndex + (i+1)%(thetaSamples);
        tr.m_c = currentBaseIndex + thetaSamples;
        tris.pushBack(tr);
    }

    hkGeometry builtGeom;
    builtGeom.m_vertices.swap( verts );
    builtGeom.m_triangles.swap( tris );
    geometryOut.appendGeometry(builtGeom);
}

void HK_CALL hkGeometryUtils::appendCylinder(const hkVector4& top, const hkVector4& bottom, hkReal radius, int numSides, int numHeightSegments, hkGeometry& geometryOut)
{
    hkGeometry builtGeometry;
    // Generate points in the order: bottom, top & then around the axis
    {
        // center point on the base of the cylinder
        const hkVector4& base = bottom;
        // axis lies along OZ + the length of a single heightSegment
        hkVector4 axis;
        axis.setSub(top, bottom);
        hkSimdReal ihs; ihs.setFromFloat(1.0f / numHeightSegments);
        axis.mul( ihs );

        hkVector4 unitAxis = axis;
        unitAxis.normalize<3>();
        //( 0.0f, 0.0f, 1.0f );
        // vector perpendicular to the axis of the clinder; choose OX
        hkVector4 radiusVec;
        {
            hkVector4 tmp;
            if ( hkMath::fabs(unitAxis(0)) > 0.5f )
            {
                tmp = hkVector4::getConstant(HK_QUADREAL_0100);
            }
            else if ( hkMath::fabs(unitAxis(1)) > 0.5f )
            {
                tmp = hkVector4::getConstant(HK_QUADREAL_0010);
            }
            else //if ( hkMath::fabs(unitAxis(2)) > 0.5f )
            {
                tmp = hkVector4::getConstant(HK_QUADREAL_1000);
            }

            radiusVec.setCross(unitAxis, tmp);
            radiusVec.normalize<3>();
            hkSimdReal sradius; sradius.setFromFloat(radius);
            radiusVec.mul(sradius);
        }
        // quaternion used to rotate radius to generate next set of (axis aligned) points
        hkQuaternion stepQuat; stepQuat.setAxisAngle( unitAxis, HK_REAL_PI * 2.0f / numSides );
        hkQuaternion currQuat; currQuat.setAxisAngle( unitAxis, 0 );

        *(builtGeometry.m_vertices.expandBy(1)) = base;
        {
            hkVector4 axisEnd;
            hkSimdReal hSeg; hSeg.setFromFloat(hkReal(numHeightSegments));
            axisEnd.setMul(hSeg, axis);
            axisEnd.add(base);
            *(builtGeometry.m_vertices.expandBy(1)) = axisEnd;
        }

        // Generate all points on the side of the cylinder
        for (int s = 0; s < numSides; ++s)
        {
            hkVector4 point;
            {
                // Set start position at the bottom of the cylinder
                hkVector4 modRadius;
                modRadius.setRotatedDir(currQuat, radiusVec);
                point.setAdd(base, modRadius);
            }

            // loop for numHeightSegments and add one extra ending after the loop
            for (int h = 0; h < numHeightSegments; ++h)
            {
                *(builtGeometry.m_vertices.expandBy(1)) = point;
                point.add(axis);
            }
            *(builtGeometry.m_vertices.expandBy(1)) = point;

            hkQuaternion tmp = currQuat;
            currQuat.setMul(tmp, stepQuat);
        }
        // The above loop does one unneeded quat multiplication at the end.
    }

    //
    // Generate triangle indices: Draw triangles in a right-hand axes space in CCW direction
    //
    {
        // Store number of vertices in every column along the cylinder axis
        const int vertColLen = numHeightSegments + 1;
        const int allVertsCnt = 2 + numSides * (1 + numHeightSegments);
        // Store indices to the first two triangles;
        // then increase all by the same number to get following triangles
        int indices[4] = { 2+0,
            2+vertColLen,
            2+1,
            2+vertColLen+1
        };

        // Generate trinagles on cylinder side
        for (int s = 0; s < numSides; ++s)
        {
            for (int h = 0; h < numHeightSegments; ++h)
            {
                builtGeometry.m_triangles.expandBy(1)->set(indices[0], indices[1], indices[2]);
                builtGeometry.m_triangles.expandBy(1)->set(indices[2], indices[1], indices[3]);
                ++indices[0];
                ++indices[1];
                ++indices[2];
                ++indices[3];
            }
            // Increase indices again, since we have one less triangles/segments then vertices in a column
            ++indices[0];
            ++indices[1];
            ++indices[2];
            ++indices[3];
            if (indices[3] >= allVertsCnt)
            {
                indices[1] += 2 - allVertsCnt;
                indices[3] += 2 - allVertsCnt;
            }
        }

        // Generate both bases
        for (int base = 0; base < 2; ++base)
        {
            int start = 2 + (base ? numHeightSegments : 0);
            for (int s = 0; s < numSides; ++s)
            {
                int next = start + vertColLen;
                if (next >= allVertsCnt)
                {
                    next += 2 - allVertsCnt;
                }

                if (base)
                {
                    builtGeometry.m_triangles.expandBy(1)->set(base, start, next );
                }
                else
                {
                    builtGeometry.m_triangles.expandBy(1)->set(base, next, start );
                }
                start = next;
            }
        }
    }

    geometryOut.appendGeometry(builtGeometry);
}

void HK_CALL hkGeometryUtils::appendPlane(const hkVector4& normal, const hkVector4& perpToNormal, const hkVector4& center, const hkVector4& extents, hkGeometry& geometryOut)
{
    hkGeometry builtGeometry;
    builtGeometry.m_vertices.setSize(5);
    builtGeometry.m_triangles.setSize(4);

    hkVector4 otherDir;
    otherDir.setCross(normal, perpToNormal);

    hkVector4 offset1;
    offset1.setMul(extents, otherDir);

    hkVector4 offset2;
    offset2.setMul(extents, perpToNormal);

    builtGeometry.m_vertices[0].setAdd(center, offset1);
    builtGeometry.m_vertices[0].add(offset2);

    builtGeometry.m_vertices[1].setAdd(center, offset1);
    builtGeometry.m_vertices[1].sub(offset2);

    builtGeometry.m_vertices[2].setSub(center, offset1);
    builtGeometry.m_vertices[2].add(offset2);

    builtGeometry.m_vertices[3].setSub(center, offset1);
    builtGeometry.m_vertices[3].sub(offset2);

    builtGeometry.m_vertices[4] = center;

    builtGeometry.m_triangles[0].set(2,4,3);
    builtGeometry.m_triangles[1].set(0,4,2);
    builtGeometry.m_triangles[2].set(1,4,0);
    builtGeometry.m_triangles[3].set(3,4,1);
    geometryOut.appendGeometry(builtGeometry);
}

void HK_CALL hkGeometryUtils::appendBone(const hkVector4& parentPos, const hkVector4& childPos, hkGeometry& geometryOut)
{
    hkGeometry builtGeometry;
    hkArray<hkVector4>& verts = builtGeometry.m_vertices;
    hkArray<hkGeometry::Triangle>& tris = builtGeometry.m_triangles;

    verts.setSize(6);
    tris.setSize(8);

    verts[0] = parentPos;
    verts[1] = childPos;
    hkVector4 bdir; bdir.setSub(verts[1], verts[0]);
    hkSimdReal blen = bdir.length<3>();
    bdir.mul( hkSimdReal::fromFloat(.2f) );
    hkSimdReal thickness; thickness.setMin(blen * hkSimdReal::fromFloat(0.1f), hkSimdReal::fromFloat(0.25f));

    hkVector4 r, u;
    {
        hkVector4 dir = bdir;
        dir.normalize<3>();
        hkVector4Util::calculatePerpendicularNormalizedVectors<true>(dir, r, u);
        u.mul(thickness);
        r.mul(thickness);
    }

    hkVector4 c; c.setAdd(verts[0], bdir);
    verts[2].setAdd(c, r);
    verts[3] = verts[2];
    verts[2].sub(u);
    verts[3].add(u);
    verts[4].setSub(c, r);
    verts[5] = verts[4];
    verts[4].sub(u);
    verts[5].add(u);

    static int indices[] = { 0, 4, 2,   0, 5, 4,    0, 3, 5,    0, 2, 3,    1, 3, 2,    1, 2, 4,    1, 4, 5,    1, 5, 3 };
    for (int ti=0; ti < 8; ++ti)
    {
        tris[ti].set(indices[ti*3], indices[ti*3+1], indices[ti*3+2]);
    }

    geometryOut.appendGeometry(builtGeometry);
}

void HK_CALL hkGeometryUtils::triangulateTetrahedron( const hkVector4* positionsAndDistance, hkGeometry& geom )
{

    // Compute type.
    int         type = 0;
    const hkVector4* x= positionsAndDistance;

    if ( x[0].getW().isLessZero() ) type |= 1;
    if ( x[1].getW().isLessZero() ) type |= 2;
    if ( x[2].getW().isLessZero() ) type |= 4;
    if ( x[3].getW().isLessZero() ) type |= 8;

    //
    static const int cases[16][4][2] = {
    {{-1,-1},{-1,-1},{-1,-1},{-1,-1}}, {{0,1},{0,2},{0,3},{-1,-1}},
    {{1,0},{1,3},{1,2},{-1,-1}}, {{0,2},{0,3},{1,3},{1,2}},
    {{2,0},{2,1},{2,3},{-1,-1}}, {{0,3},{0,1},{2,1},{2,3}},
    {{1,0},{1,3},{2,3},{2,0}}, {{3,0},{3,1},{3,2},{-1,-1}},
    {{3,0},{3,2},{3,1},{-1,-1}}, {{0,1},{0,2},{3,2},{3,1}},
    {{1,2},{1,0},{3,0},{3,2}}, {{2,0},{2,3},{2,1},{-1,-1}},
    {{2,0},{2,1},{3,1},{3,0}}, {{1,0},{1,2},{1,3},{-1,-1}},
    {{0,1},{0,3},{0,2},{-1,-1}}, {{-1,-1},{-1,-1},{-1,-1},{-1,-1}} };

    int         indices[4];
    int         numVertices = 0;
    for ( int i = 0; i < 4 && cases[type][i][0] >= 0; ++i )
    {
        int j0 = cases[type][i][0];
        int j1 = cases[type][i][1];

        hkVector4 a = positionsAndDistance[j0];
        hkVector4 b = positionsAndDistance[j1];

        indices[numVertices++] = geom.m_vertices.getSize();

        hkSimdReal d0 = a.getW();
        hkSimdReal d1 = b.getW();
        hkSimdReal diffInv; diffInv.setReciprocal( d0 - d1 );
        hkVector4   vtx = a * (d0 * diffInv) - b * (d1 * diffInv );

        geom.m_vertices.pushBack( vtx );
    }

    if ( numVertices )
    {
        const int numTriangles = numVertices - 2;
        for ( int i = 0; i < numTriangles; ++i )
        {
            hkGeometry::Triangle& t = geom.m_triangles.expandOne();
            t.set( indices[0], indices[2 + i], indices[1 + i], 0 );
        }
    }
}


static void _subdivideRecursive( hkGeometry& geometry, int triIndex, const hkReal radius, int recursionsLeft )
{
    if( recursionsLeft == 0 )
    {
        return;
    }

    // allocate space for new vertices and triangles
    int nV = geometry.m_vertices.getSize();
    int nT = geometry.m_triangles.getSize();
    geometry.m_vertices.expandBy( 3 );
    geometry.m_triangles.expandBy( 3 );

    hkGeometry::Triangle& t = geometry.m_triangles[ triIndex ];

    // set vertex data
    {
        hkVector4& v1 = geometry.m_vertices[ t.m_a ];
        hkVector4& v2 = geometry.m_vertices[ t.m_b ];
        hkVector4& v3 = geometry.m_vertices[ t.m_c ];
        hkVector4& v12 = geometry.m_vertices[ nV ];
        hkVector4& v23 = geometry.m_vertices[ nV+1 ];
        hkVector4& v31 = geometry.m_vertices[ nV+2 ];
        hkSimdReal sradius; sradius.setFromFloat(radius);
        v12.setAdd( v1, v2 ); v12.normalize<3>(); v12.mul( sradius );
        v23.setAdd( v2, v3 ); v23.normalize<3>(); v23.mul( sradius );
        v31.setAdd( v3, v1 ); v31.normalize<3>(); v31.mul( sradius );
    }

    // set triangle data
    {
        hkGeometry::Triangle& t1 = geometry.m_triangles[ nT ];
        hkGeometry::Triangle& t2 = geometry.m_triangles[ nT+1 ];
        hkGeometry::Triangle& t3 = geometry.m_triangles[ nT+2 ];
        t1.m_a = t.m_a; t1.m_b = nV;    t1.m_c = nV+2;
        t2.m_a = nV;    t2.m_b = t.m_b; t2.m_c = nV+1;
        t3.m_a = nV+2;  t3.m_b = nV+1;  t3.m_c = t.m_c;
        t.m_a = nV;     t.m_b = nV+1;   t.m_c = nV+2;
    }

    // recurse
    {
        _subdivideRecursive( geometry, nT, radius, recursionsLeft-1 );
        _subdivideRecursive( geometry, nT+1, radius, recursionsLeft-1 );
        _subdivideRecursive( geometry, nT+2, radius, recursionsLeft-1 );
        _subdivideRecursive( geometry, triIndex, radius, recursionsLeft-1 );
    }
}


static void HK_CALL buildIcosahedron( hkGeometry& geomOut, hkSimdRealParameter r )
{
    // Create an Icosahedron
    {
        const hkSimdReal cx = hkSimdReal::fromFloat(0.525731112119133606f);
        const hkSimdReal x = r * cx;
        const hkSimdReal cz = hkSimdReal::fromFloat(0.850650808352039932f);
        const hkSimdReal z = r * cz;
        const hkSimdReal zero = hkSimdReal::getConstant<HK_QUADREAL_0>();
        int i;


        hkArray<hkVector4>& v = geomOut.m_vertices;
        v.setSize( 12 );
        i=0;
        v[i++].set( -x, zero, z, zero);
        v[i++].set( x, zero, z, zero );
        v[i++].set( -x, zero, -z , zero);
        v[i++].set( x, zero, -z, zero );
        v[i++].set( zero, z, x, zero);
        v[i++].set( zero, z, -x, zero );
        v[i++].set( zero, -z, x, zero );
        v[i++].set( zero, -z, -x, zero );
        v[i++].set( z, x, zero , zero);
        v[i++].set( -z, x, zero, zero );
        v[i++].set( z, -x, zero, zero );
        v[i++].set( -z, -x, zero, zero );

        hkArray<hkGeometry::Triangle>& t = geomOut.m_triangles;
        t.setSize( 20 );
        i=0;
        t[i++].set( 1,4,0 );
        t[i++].set( 4,9,0 );
        t[i++].set( 4,5,9 );
        t[i++].set( 8,5,4 );
        t[i++].set( 1,8,4 );
        t[i++].set( 1,10,8 );
        t[i++].set( 10,3,8 );
        t[i++].set( 8,3,5 );
        t[i++].set( 3,2,5 );
        t[i++].set( 3,7,2 );
        t[i++].set( 3,10,7 );
        t[i++].set( 10,6,7 );
        t[i++].set( 6,11,7 );
        t[i++].set( 6,0,11 );
        t[i++].set( 6,1,0 );
        t[i++].set( 10,1,6 );
        t[i++].set( 11,0,9 );
        t[i++].set( 2,11,9 );
        t[i++].set( 5,2,9 );
        t[i++].set( 11,2,7 );
    }
}

void hkGeometryUtils::appendSphere( hkVector4_ pos, hkReal radius, int numSubdivisions, hkGeometry& geomOut, int material )
{
    

    hkSimdReal r = hkSimdReal::fromFloat(radius);

    hkGeometry geom;

    buildIcosahedron(geom, r);

    // Subdivide each triangle a bunch of times
    const int numTris = geom.m_triangles.getSize();
    for( int i=0; i<numTris; ++i )
    {
        _subdivideRecursive(geom, i, r.getReal(), numSubdivisions );
    }

    const int numVerts = geom.m_vertices.getSize();
    for (int i = 0; i < numVerts; ++i)
    {
        geom.m_vertices[i].add( pos );
    }

    if (material != -1 )
    {
        for (int i =0; i < geom.m_triangles.getSize(); i++)
        {
            geom.m_triangles[i].m_material = material;
        }
    }

    geomOut.appendGeometry( geom );
}

static inline hkSimdReal _determinant3x3 (hkVector4Parameter col0, hkVector4Parameter col1, hkVector4Parameter col2)
{
    hkVector4 r0; r0.setCross( col1, col2 );

    return col0.dot<3>(r0);
}

/*static*/ void HK_CALL hkGeometryUtils::computeVolume (const hkGeometry& geometry, hkSimdReal& volume)
{
    volume.setZero();

    const hkArray<hkVector4>& verts = geometry.m_vertices;
    const hkArray<hkGeometry::Triangle>& tris = geometry.m_triangles;

    for(int i=0; i < geometry.m_triangles.getSize(); i++) // for each triangle
    {

        volume.add( _determinant3x3(verts[tris[i].m_a],verts[tris[i].m_b],verts[tris[i].m_c]) );
    }

    volume.mul(hkSimdReal_Inv6); // since the determinant give 6 times tetra volume
}

//
//  Computes the AABB of the given geometry

void HK_CALL hkGeometryUtils::computeAabb(const hkGeometry& geomIn, hkAabb& aabbOut)
{
    hkAabb aabb;    aabb.setEmpty();

    const hkVector4* HK_RESTRICT vptr = geomIn.m_vertices.begin();
    for (int k = geomIn.m_vertices.getSize() - 1; k >= 0; k--)
    {
        aabb.includePoint(vptr[k]);
    }

    aabbOut = aabb;
}

void HK_CALL hkGeometryUtils::computeAabbFromTriangles( const hkGeometry& geomIn, hkAabb& aabbOut )
{
    hkAabb aabb;    aabb.setEmpty();
    for (int t=0; t<geomIn.m_triangles.getSize(); t++)
    {
        hkVector4 triVerts[3];
        geomIn.getTriangle(t, triVerts);
        aabb.includePoint(triVerts[0]);
        aabb.includePoint(triVerts[1]);
        aabb.includePoint(triVerts[2]);
    }

    aabbOut = aabb;
}


/// Transform a geometry
void HK_CALL hkGeometryUtils::transformGeometry (const hkMatrix4& transform, hkGeometry& geometryInOut)
{
    for (int v=0; v<geometryInOut.m_vertices.getSize(); ++v)
    {
        transform.transformPosition (geometryInOut.m_vertices[v], geometryInOut.m_vertices[v]);
    }
}

void HK_CALL hkGeometryUtils::transformGeometry(const hkTransform& transform, hkGeometry& geometryInOut)
{
    for(int v = 0; v<geometryInOut.m_vertices.getSize(); ++v)
    {
        geometryInOut.m_vertices[v].setTransformedPos(transform, geometryInOut.m_vertices[v]);
    }
}

void HK_CALL hkGeometryUtils::appendGeometry( const hkGeometry& input, hkGeometry& geomInOut )
{
    int vertexOffset = geomInOut.m_vertices.getSize();
    geomInOut.m_vertices.append( input.m_vertices.begin(), input.m_vertices.getSize() );
    hkGeometry::Triangle* newTris = geomInOut.m_triangles.expandBy( input.m_triangles.getSize() );

    for (int i=0, n = input.m_triangles.getSize(); i<n; i++)
    {
        newTris[i] = input.m_triangles[i];
        newTris[i].m_a += vertexOffset;
        newTris[i].m_b += vertexOffset;
        newTris[i].m_c += vertexOffset;
    }
}



void HK_CALL hkGeometryUtils::quantize( hkGeometry& mesh, int resolution )
{
    hkAabb aabb; aabb.setEmpty();

    for (int v=0; v < mesh.m_vertices.getSize(); v++)
    {
        aabb.includePoint( mesh.m_vertices[v] );
    }

    // Add a small (hardcoded) tolerance for the
    // <ng.todo.aa> Review this quantization for:
    // 1. Error for large values (ask DG)
    // 2. Issues with ranges being different (hence quantization grid being different) in adjacent navmesh sections.
    // 3. Hardcoded #bits is 16
    const hkSimdReal tol = hkSimdReal::fromFloat( hkReal(0.01f) );
    aabb.m_min.setSub( aabb.m_min,tol );
    aabb.m_max.setAdd( aabb.m_max,tol );

    hkVector4 range; range.setSub( aabb.m_max , aabb.m_min );

    hkVector4 domain; domain.setAll( hkSimdReal::fromInt32(resolution - 1) );
    hkVector4 stepSize; stepSize.setDiv( range, domain );
    hkVector4 invStepSize; invStepSize.setDiv( domain, range );

    for (int v=0; v < mesh.m_vertices.getSize(); v++)
    {
        hkVector4 vec;
        vec.setSub( mesh.m_vertices[v], aabb.m_min );
        vec.mul( invStepSize );

        // Convert
        hkIntVector c; 
        c.setConvertF32toU32( vec );
        c.convertU32ToF32( vec );

        // Clamp vec
        vec.setClamped( vec, hkVector4::getZero(), domain );

        vec.mul( stepSize );

        mesh.m_vertices[v].setAdd( vec, aabb.m_min );
    }
}

hkResult HK_CALL hkGeometryUtils::getGeometryInsideAabb( const hkGeometry& geometryIn, hkGeometry& geometryOut, const hkAabb& aabb, GetGeometryInsideAabbMode mode /*= MODE_COPY_DATA*/ )
{
    hkArray<int> vertexRemap;
    return getGeometryInsideAabb(geometryIn, geometryOut, aabb, vertexRemap, mode);
}

hkResult HK_CALL hkGeometryUtils::getGeometryInsideAabb( const hkGeometry& geometryIn, hkGeometry& geometryOut, const hkAabb& aabb, hkArray<int>& vertexRemap, GetGeometryInsideAabbMode mode /*= MODE_COPY_DATA*/ )
{
    vertexRemap.clear();
    hkResult vertexRemapRes = vertexRemap.reserve( geometryIn.m_vertices.getSize() );
    if (vertexRemapRes.isFailure())
    {
        return HK_FAILURE;
    }

    vertexRemap.setSize( geometryIn.m_vertices.getSize(), -1);

    int totalTriangles = 0;
    int totalVertices = 0;

    for (int t=0; t<geometryIn.m_triangles.getSize(); t++)
    {
        const hkGeometry::Triangle& tri = geometryIn.m_triangles[t];
        hkVector4 verts[3];
        geometryIn.getTriangle(t, verts);

        hkAabb triAabb;
        triAabb.setFromTriangle(verts[0], verts[1], verts[2]);

        if( !aabb.overlaps(triAabb) )
        {
            continue;
        }

        totalTriangles++;
        const int* triIndices = &tri.m_a;

        // For each triangle vertex, check if we've seen it before
        // If not, update the map and add a vertex to the output vertex array.
        for (int i=0; i<3; i++)
        {
            int index = triIndices[i];
            if (vertexRemap[index] == -1)
            {
                vertexRemap[index] = totalVertices;
                totalVertices++;

                if (mode == MODE_COPY_DATA)
                {
                    geometryOut.m_vertices.pushBack( verts[i] );
                }
            }
        }

        if (mode == MODE_COPY_DATA)
        {
            hkGeometry::Triangle& newTri = geometryOut.m_triangles.expandOne();
            newTri.m_a = vertexRemap[tri.m_a];
            newTri.m_b = vertexRemap[tri.m_b];
            newTri.m_c = vertexRemap[tri.m_c];
            newTri.m_material = tri.m_material;
        }
    }

    if (mode == MODE_PRESIZE_ARRAYS)
    {
        hkResult triRes = geometryOut.m_triangles.reserve( totalTriangles );
        if (triRes.isFailure())
            return HK_FAILURE;

        hkResult vertRes = geometryOut.m_vertices.reserve( totalVertices );
        if (vertRes.isFailure())
            return HK_FAILURE;
    }

    return HK_SUCCESS;
}


void HK_CALL hkGeometryUtils::appendMinkowskiSum(
    const hkTransform& tA, const hkStridedVertices& vertsA,
    const hkTransform& tB, const hkStridedVertices& vertsB,
    hkGeometry& geometryOut )
{
    const int a = vertsA.getSize();
    const int b = vertsB.getSize();

    hkLocalArray<hkVector4> pointsMk( a * b );
    for(int i = 0; i < a; i++)
    {
        hkVector4 pointA; vertsA.getVertex( i, pointA );
        pointA._setTransformedPos(tA, pointA);
        for(int j = 0; j < b; j++)
        {
            hkVector4 pointB; vertsB.getVertex(j, pointB);
            pointB._setTransformedPos(tB, pointB);

            hkVector4 pointMk = pointA-pointB;
            pointsMk.pushBackUnchecked(pointMk);
        }
    }
    int material = 1;

    hkgpConvexHull hull;
    hull.build(pointsMk.begin(), pointsMk.getSize());
    hull.generateGeometry(hkgpConvexHull::SOURCE_VERTICES, geometryOut, material);
}

// Maximum number of velocities that can be active in the inner sphere algorithm
const int maxNumVelocities = 4;

static HK_INLINE hkVector4 calculateResultingVelocity(hkVector4Parameter velocity1, hkVector4Parameter velocity2)
{
    hkVector4 velocity = velocity1 + velocity2;
    hkSimdReal dotProd = velocity.dot3(velocity1);

    const hkSimdReal tolerance = hkSimdReal::fromFloat(0.001f);
    if (dotProd.isLessEqual(tolerance))
    {
        return hkVec4_0;
    }
    else
    {
        velocity.mul(dotProd.reciprocal());
    }

    return velocity;
}

static HK_INLINE hkVector4 getVelocityFromMatrix(const hkMatrix3& m, hkSimdRealParameter det)
{
    hkVector4 pointVel;
    pointVel._setRotatedDir(m, hkVec4_1);
    pointVel.mul(det.reciprocal());

    return pointVel;
}

static hkVector4 calculateResultingVelocity(hkInplaceArray<hkVector4, maxNumVelocities>& velocitiesInOut)
{
    HK_ASSERT(0xA9D96B4A, velocitiesInOut.getSize() > 0 && velocitiesInOut.getSize() < maxNumVelocities, "Calculating resulting velocity works only for 1-3 velocities!");

    int numVels = velocitiesInOut.getSize();
    if (numVels == 1)
    {
        return velocitiesInOut[0];
    }
    else if (numVels == 2)
    {
        return calculateResultingVelocity(velocitiesInOut[0], velocitiesInOut[1]);
    }
    else
    {
        hkMatrix3 m;
        hkVector4& r1 = m.getColumn<0>();
        hkVector4& r2 = m.getColumn<1>();
        hkVector4& r3 = m.getColumn<2>();

        r1.setCross(velocitiesInOut[1], velocitiesInOut[2]);
        r2.setCross(velocitiesInOut[2], velocitiesInOut[0]);
        r3.setCross(velocitiesInOut[0], velocitiesInOut[1]);

        hkSimdReal det = r1.dot<3>(velocitiesInOut[0]);
        hkSimdReal absDet; absDet.setAbs(det);

        const hkSimdReal tolerance = hkSimdReal::fromFloat(0.00001f);
        if (absDet.isLess(tolerance))
        {
            return hkVec4_0;
        }
        else
        {
            return getVelocityFromMatrix(m, det);
        }
    }
}

static int findNextPlane(
    const hkArray<hkVector4>& workingPlanes,
    const hkArray<bool>& visitedPlanes,
    hkVector4Parameter currentCenter,
    hkVector4Parameter currentVelocity,
    hkSimdReal& timeOut)
{
    // Iterate through all the planes and find which one will touch the point next
    hkSimdReal minTime = hkSimdReal_Max;
    int planeIndex = -1;

    const hkSimdReal tolerance = hkSimdReal::fromFloat(0.00001f);
    for (int i = 0; i < workingPlanes.getSize(); i++)
    {
        // If the plane was already touched, just ignore it
        if (visitedPlanes[i])
        {
            continue;
        }

        // Relative velocity of the point and plane
        hkSimdReal relVel = currentVelocity.dot3(workingPlanes[i]) + hkSimdReal_1;

        if (relVel.isLess(tolerance))
        {
            // Center is not approaching the plane, just skip
            continue;
        }
        else
        {
            // Distance of point to the plane
            hkSimdReal distance = currentCenter.dot4(-workingPlanes[i]);
            if (distance.isLess(-tolerance))
            {
                // Negative distance, center is outside the shape
                planeIndex = -1;
                break;
            }

            // Time needed for point and plane to meet
            hkSimdReal tempTime = distance * relVel.reciprocal();
            if (tempTime.isLess(minTime))
            {
                minTime = tempTime;
                planeIndex = i;
            }
        }
    }

    timeOut = minTime;
    return planeIndex;
}

static int findNextTouchingPlane(
    const hkArray<hkVector4>& workingPlanes,
    const hkArray<bool>& visitedPlanes,
    hkVector4Parameter currentCenter,
    hkVector4& currentVelocityInOut,
    hkInplaceArray<hkVector4, maxNumVelocities>& velocitiesInOut,
    hkSimdReal& timeOut)
{
    hkSimdReal minTime = hkSimdReal_Max;
    int planeIndex = -1;

    if (velocitiesInOut.getSize() < 4)
    {
        planeIndex = findNextPlane(workingPlanes, visitedPlanes, currentCenter, currentVelocityInOut, minTime);
    }
    else
    {
        int velToDrop = -1;
        hkVector4 velocity = hkVec4_0;
        minTime = -hkSimdReal_Max;

        // Try to drop velocities 0, 1 or 2
        // (we never drop the last velocity since we have just hit that plane)
        for (int i = 0; i < velocitiesInOut.getSize() - 1; i++)
        {
            hkInplaceArray<hkVector4, maxNumVelocities> vels;
            for (int j = 0; j < velocitiesInOut.getSize(); j++)
            {
                if (i != j)
                {
                    vels.pushBackUnchecked(velocitiesInOut[j]);
                }
            }

            hkVector4 resultingVelocity = calculateResultingVelocity(vels);
            if (resultingVelocity.dot<3>(velocitiesInOut[i]).isGreater(hkSimdReal_1))
            {
                hkSimdReal tempMinTime = hkSimdReal_Max;
                int tempPlaneIndex = findNextPlane(workingPlanes, visitedPlanes, currentCenter, resultingVelocity, tempMinTime);

                if (tempPlaneIndex != -1 && tempMinTime.isGreater(minTime))
                {
                    planeIndex = tempPlaneIndex;
                    minTime = tempMinTime;

                    // Set resulting velocity and remember the one that is not needed
                    velocity = resultingVelocity;
                    velToDrop = i;
                }
            }
        }

        if (planeIndex != -1)
        {
            velocitiesInOut.removeAt(velToDrop);
        }

        currentVelocityInOut = velocity;
    }

    timeOut = minTime;
    return planeIndex;
}

static HK_INLINE void shrinkPlanes(hkArray<hkVector4>& workingPlanes, hkSimdRealParameter time)
{
    // Iterate through all the planes
    // and shrink them for the amount of time they can travel
    for (int i = 0; i < workingPlanes.getSize(); i++)
    {
        hkSimdReal distanceFromOrigin = workingPlanes[i].getComponent<3>();
        distanceFromOrigin.add(time);
        workingPlanes[i].setComponent<3>(distanceFromOrigin);
    }
}

static HK_INLINE hkResult hkGeometryUtils_step(
    hkArray<hkVector4>& workingPlanes,
    hkArray<bool>& visitedPlanes,
    hkVector4& centerInOut,
    hkVector4& velocityInOut,
    hkInplaceArray<hkVector4, maxNumVelocities>& velocitiesInOut)
{
    // Perform one step of the algorithm by finding the next touching plane,
    // moving the center with that plane and calculating the resulting velocity afterwards

    hkSimdReal minTime = hkSimdReal_Max;
    int planeIndex = findNextTouchingPlane(workingPlanes, visitedPlanes, centerInOut, velocityInOut, velocitiesInOut, minTime);

    // Can't hit another plane
    if (planeIndex == -1)
    {
        // If we hit less than 3 planes, it's an error
        if (velocitiesInOut.getSize() < 3)
        {
            return HK_FAILURE;
        }
        // If we hit at least 3 planes, then we are done
        else
        {
            velocityInOut = hkVec4_0;
            return HK_SUCCESS;
        }
    }

    // Mark plane as visited
    visitedPlanes[planeIndex] = true;

    // Move the point to the new position
    centerInOut.addMul(velocityInOut, minTime);
    centerInOut.setComponent<3>(hkSimdReal_1);

    // Shrink the planes
    shrinkPlanes(workingPlanes, minTime);

    // New velocity of the point (only for up to 3 planes)
    // If there are 4 planes we will try all combinations anyway
    velocitiesInOut.pushBackUnchecked(-workingPlanes[planeIndex]);
    if (velocitiesInOut.getSize() < 4)
    {
        velocityInOut = calculateResultingVelocity(velocitiesInOut);
    }

    return HK_SUCCESS;
}

hkResult HK_CALL hkGeometryUtils::calculateInnerSphere(
    const hkVector4* planes,
    const int numPlanes,
    hkVector4Parameter initialCenter,
    hkSphere& sphereOut)
{
    // At least 4 planes form a shape
    HK_ASSERT(0x7347D12F, numPlanes >= 4, "At least 4 planes form a shape!");

    // Initialize an array of non const planes and indicators which planes were already used
    hkArray<hkVector4> workingPlanes(numPlanes);
    workingPlanes.copy(workingPlanes.begin(), planes, numPlanes);
    hkArray<bool> visitedPlanes(numPlanes, false);

    // Set initial center
    hkVector4 center = initialCenter;
    center.setComponent<3>(hkSimdReal_1);

    // Initialize array of velocities and current velocity
    hkInplaceArray<hkVector4, 4> velocities;
    hkVector4 velocity = hkVec4_0;

    // Allow only one more plane hit after the initial 4
    bool triedToDropOnePlane = false;
    while (!triedToDropOnePlane)
    {
        if (velocities.getSize() == 4)
        {
            triedToDropOnePlane = true;
        }

        // Step the algorithm and get the success indicator
        HK_RETURN_IF_FAILED( hkGeometryUtils_step(workingPlanes, visitedPlanes, center, velocity, velocities) );

        // If resulting velocity is zero, it's a sign that we should wrap up and calculate the radius of the sphere
        if (velocity.allExactlyEqualZero<3>())
        {
            break;
        }
    }

    // Find min distance of the center to planes (radius of the sphere)
    hkSimdReal minDistance = hkSimdReal_Max;
    for (int i = 0; i < workingPlanes.getSize(); i++)
    {
        minDistance.setMin(minDistance, -center.dot4(planes[i]));
    }

    if (minDistance < -hkSimdReal_Eps)
    {
        return HK_FAILURE;
    }

    minDistance.setMax(minDistance, hkSimdReal_Eps);

    sphereOut.setPositionAndRadius(center, minDistance);
    return HK_SUCCESS;
}

void HK_CALL hkGeometryUtils::deflectedSphereCast(const hkVector4* planes, const int numPlanes, const hkSphere& initialSphere, hkVector4Parameter targetPosition, int maxIterations, hkVector4& positionOut)
{
    // used to prevent cast point to land on plane
    const hkSimdReal c_planeConvexRadius = hkSimdReal::fromFloat(0.0001);

    hkVector4 initialPosition = initialSphere.getPosition();
    hkSimdReal radius = initialSphere.getRadiusSimdReal() - c_planeConvexRadius;

    // calc planes indented for the cast sphere radius
    hkArray<hkVector4> indentedPlanes;  indentedPlanes.setSize(numPlanes);
    for (int i = 0; i < numPlanes; i++)
    {
        hkSimdReal dist = planes[i].dot4xyz1(initialPosition);
        hkSimdReal offset;  offset.setFlipSign(radius, dist.greaterZero());
        indentedPlanes[i].setXYZ_W(planes[i], planes[i].getW() + offset);
    }

    hkVector4 position = initialPosition;
    hkVector4 dir = targetPosition - position;
    hkSimdReal maxCastDist = dir.length<3>();
    hkVector4 dirN = dir;   dirN.normalize<3>();

    for (int i = 0; i < maxIterations; i++)
    {
        // stop if further cast is not possible
        if (maxCastDist <= hkSimdReal_0)
        {
            break;
        }

        hkSimdReal castDist = maxCastDist;
        hkVector4 plane = hkVec4_0;

        // get the closest plane and distance
        for (int p = 0; p < numPlanes; p++)
        {
            hkSimdReal t = hkSimdReal_Max;  // set to max if line and plane are parallel
            hkGeometryProcessing::intersectLineWithPlane(position, position + dirN, indentedPlanes[p], t);

            if (t < castDist && t >= hkSimdReal_0)
            {
                castDist = t;
                plane = indentedPlanes[p];
            }
        }

        // do not cast point all the way, let the small distance to plane remain
        const hkSimdReal den = plane.dot<3>(dirN);
        if (HK_VERY_LIKELY(!den.isEqualZero()))
        {
            hkSimdReal len;
            len.setDiv<hkMathAccuracyMode::HK_ACC_FULL, hkMathDivByZeroMode::HK_DIV_IGNORE>(c_planeConvexRadius, den);
            len.setAbs(len);
            castDist.setMax(castDist - len, hkSimdReal_0);
        }

        if (!castDist.isEqualZero())
        {
            // move the point to next position
            position.setAddMul(position, dirN, castDist);
        }

        // stop if whole sphere can be moved to goal position
        if (castDist >= maxCastDist)
        {
            break;
        }

        // update direction
        hkSimdReal planeVel = dir.dot<3>(plane);
        dir.setSubMul(dir, plane, planeVel);
        dirN = dir; dirN.normalize<3>();

        if (!dirN.isNormalized<3>())
        {
            break;
        }

        // update max cast distance
        hkGeometryProcessing::squaredDistanceFromPointToInfLine(targetPosition, position, position + dirN, maxCastDist);
    }

    positionOut = position;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
