// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY     : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/GeometryUtilities/Mesh/hkMeshVertexBuffer.h>

class hkMeshSystem;
class hkBitField;

/// A utility to help in the processing of vertices and vertex buffers
class HK_EXPORT_COMMON hkMeshVertexBufferUtil
{
    public:
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE, hkMeshVertexBufferUtil);

        struct Thresholds
        {
            Thresholds():
                m_positionThreshold(hkReal(1e-5f)),
                m_normalThreshold(hkReal(1e-3f)),
                m_colorThreshold(hkReal(1) / hkReal(255)),
                m_otherThreshold(hkReal(1e-5f)),
                m_texCoordThreshold(hkReal(1e-6f))
            {
            }
            hkReal m_positionThreshold;
            hkReal m_normalThreshold;
            hkReal m_colorThreshold;
            hkReal m_otherThreshold;
            hkReal m_texCoordThreshold;
        };

        enum TransformFlag
{
            TRANSFORM_NORMALIZE = 0x1,                      ///< Normalize 'normal' components (normal, tangent etc)
            TRANSFORM_PRE_NEGATE = 0x2,                     ///< Negate normal and binormal before transform
            TRANSFORM_POST_NEGATE = 0x4,                    ///< Negate normal and binormal after transform
        };

        typedef hkMeshVertexBuffer::LockedVertices LockedVertices;
        typedef LockedVertices::Buffer Buffer;
        typedef hkVertexFormat::Element Element;

            /// Extract the buffer as 4xfloat32
        static hkResult HK_CALL getElementVectorArray(const Buffer& buffer, _Out_writes_(4 * numVertices) hkFloat32* out, int numVertices);

            /// Extract the buffer as 4xfloat32
        static hkResult HK_CALL getIndexedElementVectorArray(const Buffer& buffer, _In_reads_(numVertices) const int* indices, _Out_writes_(numVertices * 4) hkFloat32* out, int numVertices);

            /// Get components and store in a 4xfloat32 array (helper function - calls getElementVectorArray with the appropriate buffer)
        static hkResult HK_CALL getElementVectorArray(const LockedVertices& lockedVertices, int bufferIndex, _Out_writes_(4 * lockedVertices.m_numVertices) hkFloat32* dst);
        static hkResult HK_CALL getElementVectorArray(const LockedVertices& lockedVertices, int bufferIndex, const hkBitField& verticesToRetrieve, _Out_writes_(_Inexpressible_(4 * verticesToRetrieve.getSize())) hkFloat32* dst);

            /// Set components held in buffer from an array of 4xhkFloat32s
        static hkResult HK_CALL setElementVectorArray(const LockedVertices::Buffer& buffer, _In_reads_(4 * numVertices) const hkFloat32* src, int numVertices);

            /// Set components from a hkVector4 array (helper function - calls setElementVectorArray with the appropriate buffer)
        static hkResult HK_CALL setElementVectorArray(const LockedVertices& lockedVertices, int bufferIndex, _In_reads_(4 * lockedVertices.m_numVertices)  const hkFloat32* src);
        static hkResult HK_CALL setElementVectorArray(const LockedVertices& lockedVertices, int bufferIndex, int dstStartVertex, _In_reads_(4 * numVertices) const hkFloat32* src, int numVertices);

            /// The elements being extracted must have been previously been locked with lock/partialLock.
        static hkResult HK_CALL getElementIntArray(const LockedVertices& lockedVertices, int elementIndex, _Out_writes_(_Inexpressible_()) int* dst);
        static hkResult HK_CALL getElementIntArray(const LockedVertices& lockedVertices, int elementIndex, const hkBitField& verticesToRetrieve, _Out_writes_(_Inexpressible_()) int* dst, int dstMaxSize = -1);

            /// Sets the elements of an array doing conversions from ints as needed. Converts only integral types.
            /// The elements being set must have previously have been locked with lock/partialLock.
        static hkResult HK_CALL setElementIntArray(const LockedVertices& lockedVertices, int elementIndex, _In_reads_(_Inexpressible_()) const int* src);
        static hkResult HK_CALL setElementIntArray(const LockedVertices& lockedVertices, int elementIndex, int dstStartVertex, _In_reads_(_Inexpressible_()) const int* src, int numVertices);

            /// Copy the elements of src to dst of elementSize with the appropriate striding
        static void HK_CALL stridedCopy(_In_reads_bytes_(_Inexpressible_()) const void* srcIn, int srcStride, _Out_writes_bytes_(_Inexpressible_()) void* dstIn, int dstStride, int elementSize, int numVertices);

            /// Zero the strided destination data
        static void HK_CALL stridedZero(_Out_writes_bytes_(_Inexpressible_()) void* dstIn, int dstStride, int elementSize, int numVertices);

            /// Copy vertex data from srcVertices to dstVertices
        static void HK_CALL copy(const LockedVertices& srcVertices, const LockedVertices& dstVertices);

            /// Copy the data from one buffer to the other - they must be the same type (m_type + m_numValues the same)
        static void HK_CALL copy(const Buffer& srcBuffer, const Buffer& dstBuffer, int numVertices);

            /// Copy/convert all of the src buffers into the destination buffers.
        static void HK_CALL convert(const LockedVertices& srcVertices, const LockedVertices& dstVertices);

            /// Copys/converts all members of src into dst
        static void HK_CALL convert(_Inout_ hkMeshVertexBuffer* src, _Inout_ hkMeshVertexBuffer* dst);

            /// Copy and convert as necessary
        static void HK_CALL convert(const Buffer& srcBuffer, const Buffer& dstBuffer, int numVertices);

            /// Finds out if the locked vertex data is contiguous
        static hkBool HK_CALL isContiguous(const LockedVertices& srcVertices, _Outptr_ void** start, int& dataSize);

            /// Partition the format into shared and non shared (instanced) elements
        static void HK_CALL partitionVertexFormat(const hkVertexFormat& format, hkVertexFormat& sharedFormat, hkVertexFormat& instanceFormat);

            /// Get array component, converting to hkVector4
        static hkResult HK_CALL getElementVectorArray(_Inout_ hkMeshVertexBuffer* vertexBuffer, hkVertexFormat::ComponentUsage usage, int subUsage, hkArray<hkVector4>& vectorsOut);

            /// Transform the content of the buffer
        static void HK_CALL transform(const Buffer& srcBuffer, const hkMatrix4& transform, int transformFlags, int numVertices);

            /// Transform the content of the vertex buffer
        static hkResult HK_CALL transform(_Inout_ hkMeshVertexBuffer* buffer, const hkMatrix4& transform, int transformFlags);

            /// Produces a new vertex buffer which is a concatenation of the input buffers. If only 1 buffer is input the original buffer is returned with an extra ref count.
        static _Ret_maybenull_ hkMeshVertexBuffer* concatVertexBuffers(_Inout_ hkMeshSystem* system, _Inout_updates_(numBuffers) hkMeshVertexBuffer** buffers, int numBuffers);

            /// Compares two buffers - all values must be equal within the threshold to return true.
        static hkBool32 HK_CALL isBufferDataEqual(const Buffer& bufferA, const Buffer& bufferB, hkSimdRealParameter threshold);

            /// Compare - handle data as a normal (use dot products difference from 1 as measure of error)
        static hkBool32 HK_CALL isBufferNormalDataEqual(const Buffer& bufferA, const Buffer& bufferB, hkSimdRealParameter threshold);

            /// Compares an array of buffer data. Uses the thresholds structure to determine what is equal based on type
        static hkBool32 HK_CALL isBufferDataEqual(_In_reads_(numBuffers) const Buffer* a, _In_reads_(numBuffers) const Buffer* b, int numBuffers, const Thresholds& thresholds);

            /// Performs a linear interpolation of components. Will normalize 'normal' type components.
            /// If interp is 0, a is output, 1 then b is output. I.e., its a * (1-interp) + b * interp
        static void HK_CALL interpolate(const hkVertexFormat::Element& element, _In_reads_bytes_(_Inexpressible_()) const void* a, _In_reads_bytes_(_Inexpressible_()) const void* b, hkSimdRealParameter interp, _Out_writes_bytes_(_Inexpressible_()) void* dst);

            /// Returns true if the buffer contains positions, normals, tangents and / or bitangents.
        static bool HK_CALL bufferIsSkinnable(_In_ const hkMeshVertexBuffer* vertexBuffer);

            /// Returns true if the buffer contains blend weights and indices.
        static bool HK_CALL bufferHasWeights(_In_ const hkMeshVertexBuffer* vertexBuffer);

            /// Merges the source vertex format into the destination vertex format
        static void HK_CALL mergeVertexFormat(hkVertexFormat& dstFormat, const hkVertexFormat& srcFormat);

            /// Computes the most fitting vertex format, that will be able to store all data in the given source vertex formats
        static void HK_CALL computeMostFittingVertexFormat(hkVertexFormat& dstFormat, _In_reads_(numSourceFormats) const hkVertexFormat* srcFormats, int numSourceFormats);

        /// Clones the given vertex buffer
        static _Ret_notnull_ hkMeshVertexBuffer* HK_CALL cloneBuffer(_Inout_ hkMeshSystem* meshSystem, _Inout_ hkMeshVertexBuffer* srcVb);

};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
