// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/GeometryUtilities/hkGeometryUtilities.h>
#include <Common/GeometryUtilities/Mesh/Utils/TextureAtlasUtil/hkTextureAtlasUtil.h>
#include <Common/Base/Container/BitField/hkBitField.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>

//
//  Utility functions

namespace hkTextureAtlasImpl
{
    //
    //  Types

    typedef hkTextureAtlasUtil::Texture     Texture;
    typedef hkTextureAtlasUtil::Placement   Placement;

    //
    //  Converts an int value to the largest power of 2

    static HK_INLINE int HK_CALL convertToPow2(int x)
    {
        int logX = 31 - hkMath::countLeadingZeros<hkUint32>(x);
        if ( x > (1 << logX) )
        {
            logX++;
        }
        return (1 << logX);
    }

    //
    //  Atlas size comparator

    static HK_INLINE int HK_CALL atlasLess(const Texture& tA, const Texture& tB)
    {
        const int areaA = tA.m_width * tA.m_height;
        const int areaB = tB.m_width * tB.m_height;
        return (areaA < areaB);
    }

    //
    //  Placement comparator

    static HK_INLINE int HK_CALL placementLess(const Placement& pA, const Placement& pB)
    {
        return (pA.m_textureId < pB.m_textureId);
    }

    //
    //  Item to be packed into the atlas

    struct Item
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_SCENE_DATA, hkTextureAtlasImpl::Item);

        /// Initializes the item. It will rotate it such that the width is always >= height
        HK_INLINE void set(int w, int h, int id)
        {
            if ( h > w )    {   m_w = h;    m_h = w;    m_rot = 90; }
            else            {   m_w = w;    m_h = h;    m_rot = 0;  }
            m_id = id;
        }

        /// Initializes the item to the source item rotated by 90 degrees
        HK_INLINE void rotate()
        {
            hkMath::swap(m_w, m_h);
            m_rot = 90 - m_rot;
        }

        /// Comparator
        static HK_INLINE int HK_CALL less(const Item& iA, const Item& iB)
        {
            const int areaA = iA.m_w * iA.m_h;
            const int areaB = iB.m_w * iB.m_h;
            if ( areaA == areaB )
            {
                const int minA = hkMath::min2(iA.m_w, iA.m_h);
                const int minB = hkMath::min2(iB.m_w, iB.m_h);
                return minA > minB;
            }

            return (areaA > areaB);
        }

        int m_w;        ///< Width.
        int m_h;        ///< Height.
        int m_rot;      ///< Rotation angle, either 0 or 90 degrees.
        int m_id;       ///< Texture Id.
    };

    //
    //  Point

    struct Point
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_SCENE_DATA, hkTextureAtlasImpl::Point);

        /// Comparator
        static HK_INLINE int HK_CALL less(const Point& pA, const Point& pB)
        {
            return   (pA.m_binId < pB.m_binId) ||
                    ((pA.m_binId == pB.m_binId) && (pA.m_y < pB.m_y)) ||
                    ((pA.m_binId == pB.m_binId) && (pA.m_y == pB.m_y) && (pA.m_x < pB.m_x));
        }

        int m_x;
        int m_y;
        int m_binId;
    };

    //
    //  A bin

    class Bin
    {
        public:

            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_SCENE_DATA, hkTextureAtlasImpl::Bin);

            enum
            {
                OCCUPIED_PIXEL  = 0,
                FREE_PIXEL      = 1,
            };

        public:

            /// Opens the bin
            void open(int w, int h);

            /// Returns true if the given item can be placed at the given position
            bool canPlaceItem(int x, int y, const Item& texItem) const;

            /// Returns the score of placing the given item at the given position
            int getPlacementScore(int x, int y, const Item& texItem) const;

            /// Places the given item at the given position
            void placeItem(int x, int y, const Item& texItem);

            /// Returns the candidate positions for placing an item
            void getPlacementCandidates(hkArray<Point>& candidatesOut) const;

            /// Returns the packed area
            HK_INLINE int getPackedArea() const {   return m_packedArea;    }

        protected:

            hkBitField m_img;   ///< The atlas occupancy map.
            int m_width;        ///< Atlas width
            int m_height;       ///< Atlas height
            int m_pitch;        ///< The pitch
            int m_packedArea;   ///< The packed area
    };

    //
    //  Class that fits some rectangles into an atlas

    class AtlasBuilder
    {
        public:

            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_SCENE_DATA, hkTextureAtlasImpl::AtlasBuilder);

            enum
            {
                OCCUPIED_PIXEL  = 0,
                FREE_PIXEL      = 1,
            };

        public:

            /// Constructor
            HK_INLINE AtlasBuilder(int w, int h)
            :   m_binWidth(w)
            ,   m_binHeight(h)
            {}

        public:

            /// Attempts to fit all given textures into the bin. Returns the number of textures fit.
            void fitTextures(_In_reads_(numTextures) const Texture* texturesIn, int numTextures, _Inout_updates_(numTextures) Placement* placementsOut);

        protected:

            /// Attempts to place the given item into a bin.
            void placeItem(Item& texItem, Point& placementPosOut);

        protected:

            int m_binWidth;         ///< Atlas width
            int m_binHeight;        ///< Atlas height
            hkArray<Bin> m_bins;    ///< The open bins
    };

    //
    //  Opens the bin

    void Bin::open(int w, int h)
    {
        m_width     = w;
        m_height    = h;
        m_pitch     = m_width + 2;
        m_packedArea= 0;

        // Allocate occupancy map. We want a 1 pixel border around the image
        // to simplify the computations
        const int numBits = m_pitch * (h + 2);
        m_img.resize(0, numBits);

        // Everything is occupied
        m_img.assignAll<OCCUPIED_PIXEL>();

        // Mark the non-borders as free
        for (int y = 0; y < m_height; y++)
        {
            const int startBit = (y + 1) * m_pitch + 1;
            m_img.assignRange(startBit, m_width, FREE_PIXEL);
        }
    }

    //
    //  Functor that finds candidate points for texture placement

    struct FunFindCandidates
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_SCENE_DATA, hkTextureAtlasImpl::FunFindCandidates);

        HK_INLINE FunFindCandidates(const hkBitField& imgBits, int pitch, hkArray<Point>& candidates)
        :   m_img(imgBits)
        ,   m_candidates(candidates)
        ,   m_pitch(pitch)
        {}

        HK_INLINE void operator()(int bitIdx)
        {
            // For the point to be a candidate, its N and W neighbours must be occupied
            const int bitIdxW = bitIdx - 1;         // The West neighbour index
            const int bitIdxN = bitIdx - m_pitch;   // The North neighbour index

            if ( (m_img.get(bitIdxN) == AtlasBuilder::OCCUPIED_PIXEL) &&
                 (m_img.get(bitIdxW) == AtlasBuilder::OCCUPIED_PIXEL) )
            {
                // This is a candidate!
                Point& p = m_candidates.expandOne();

                // Compute image coordinates
                p.m_y       = bitIdx / m_pitch;
                p.m_x       = bitIdx - p.m_y * m_pitch;
            }
        }

        const hkBitField& m_img;
        hkArray<Point>& m_candidates;
        const int m_pitch;
    };

    //
    //  Returns true if the given item can be placed at the given position

    bool Bin::canPlaceItem(int startX, int startY, const Item& texItem) const
    {
        // Some quick sanity checks
        if ( ((startX - 1) + texItem.m_w > m_width) ||
             ((startY - 1) + texItem.m_h > m_height) )
        {
            return false;   // Out of bin bounds!
        }

        // We'll count the number of free pixels along each line. If the number == width, we're good
        for (int y = texItem.m_h - 1; y >= 0; y--)
        {
            const int bitStart = (y + startY) * m_pitch + startX;
            const int bitCount = m_img.bitCount(bitStart, texItem.m_w);
            HK_ASSERT_NO_MSG(0x40cd9b7a, bitCount <= texItem.m_w);

            if ( bitCount < texItem.m_w )
            {
                return false;
            }
        }

        return true;
    }

    //
    //  Returns the score of placing the given item at the given position

    int Bin::getPlacementScore(int startX, int startY, const Item& texItem) const
    {
        int score = 0;

        // Count the number of occupied bits along the perimeter
        int topRowBit = (startY - 1) * m_pitch + startX;
        int botRowBit = (startY + texItem.m_h) * m_pitch + startX;
        for (int x = texItem.m_w - 1; x >= 0; x--)
        {
            const int b0 = m_img.get(topRowBit + x);
            const int b1 = m_img.get(botRowBit + x);
            score += (b0 + b1);
        }

        int leftColBit  = startY * m_pitch + startX - 1;
        int rightColBit = startY * m_pitch + startX + texItem.m_w;
        for (int y = texItem.m_h - 1; y >= 0; y--, leftColBit += m_pitch, rightColBit += m_pitch)
        {
            const int b0 = m_img.get(leftColBit);
            const int b1 = m_img.get(rightColBit);
            score += (b0 + b1);
        }

        HK_COMPILE_TIME_ASSERT(OCCUPIED_PIXEL == 0);
        return ((texItem.m_w + texItem.m_h) << 1) - score;
    }

    //
    //  Places the given item at the given position

    void Bin::placeItem(int startX, int startY, const Item& texItem)
    {
        for (int y = texItem.m_h - 1; y >= 0; y--)
        {
            const int startBit = (startY + y) * m_pitch + startX;
            m_img.assignRange(startBit, texItem.m_w, OCCUPIED_PIXEL);
        }

        m_packedArea += texItem.m_w * texItem.m_h;
    }

    //
    //  Returns the candidate positions for placing an item

    void Bin::getPlacementCandidates(hkArray<Point>& candidatesOut) const
    {
        FunFindCandidates fun(m_img, m_pitch, candidatesOut);
        const int numBits = m_img.getSize();
        m_img.forEach<hkBitFieldLoop::ASCENDING, (hkBitFieldBit::Type)FREE_PIXEL>(0, numBits, fun);
    }

    //
    //  Attempts to place the given item into a bin

    void AtlasBuilder::placeItem(Item& texItem, Point& placementPosOut)
    {
        hkArray<Point> candidates;

        // Find all candidate placement points
        const int numBins = m_bins.getSize();
        for (int bi = 0; bi < numBins; bi++)
        {
            const int cStart = candidates.getSize();
            m_bins[bi].getPlacementCandidates(candidates);

            for (int ci = candidates.getSize() - 1; ci >= cStart; ci--)
            {
                candidates[ci].m_binId = bi;
            }
        }

        // Sort candidates
        hkSort(candidates.begin(), candidates.getSize(), Point::less);

        // Evaluate the placement score for each candidate point
        int bestScore = -1;
        int bestCandidate = -1;
        int bestRot = texItem.m_rot;
        int bestPackedArea = 0;
        const int numCandidates = candidates.getSize();
        for (int ci = 0; ci < numCandidates; ci++)
        {
            // Try to place the original item
            const Point& pt = candidates[ci];
            const Bin& bin  = m_bins[pt.m_binId];

            if ( bin.canPlaceItem(pt.m_x, pt.m_y, texItem) )
            {
                // Compute placement score
                const int score         = bin.getPlacementScore(pt.m_x, pt.m_y, texItem);
                const int packedArea    = bin.getPackedArea();
                if ( (bestScore < score) || ((bestScore == score) && (bestPackedArea < packedArea)) )
                {
                    bestScore       = score;
                    bestCandidate   = ci;
                    bestRot         = texItem.m_rot;
                    bestPackedArea  = packedArea;
                }
            }

            // Try to place the rotated item
            if ( texItem.m_w != texItem.m_h )
            {
                Item rotItem = texItem;
                rotItem.rotate();

                if ( bin.canPlaceItem(pt.m_x, pt.m_y, rotItem) )
                {
                    // Compute placement score
                    const int score         = bin.getPlacementScore(pt.m_x, pt.m_y, rotItem);
                    const int packedArea    = bin.getPackedArea();
                    if ( (bestScore < score) || ((bestScore == score) && (bestPackedArea < packedArea)) )
                    {
                        bestScore       = score;
                        bestCandidate   = ci;
                        bestRot         = rotItem.m_rot;
                        bestPackedArea  = packedArea;
                    }
                }
            }
        }

        // Place the item in the best candidate position
        if ( texItem.m_rot != bestRot )
        {
            texItem.rotate();
        }

        // If we failed to place the texture, open a new bin
        if ( bestCandidate < 0 )
        {
            // Add a new candidate
            bestCandidate   = candidates.getSize();
            Point& pt       = candidates.expandOne();
            pt.m_binId      = m_bins.getSize();
            pt.m_x          = 1;
            pt.m_y          = 1;

            // Open the bin
            m_bins.expandOne().open(m_binWidth, m_binHeight);
        }

        // Write output
        {
            placementPosOut = candidates[bestCandidate];
            Bin& bin        = m_bins[placementPosOut.m_binId];
            bin.placeItem(placementPosOut.m_x, placementPosOut.m_y, texItem);
        }
    }

    //
    //  Attempts to fit all given textures into the bin

    void AtlasBuilder::fitTextures(_In_reads_(numTextures) const Texture* texturesIn, int numTextures, _Inout_updates_(numTextures) Placement* placementsOut)
    {
        // Compute the number of initial bins
        int L = 1;
        {
            int sumArea = 0;
            for (int k = numTextures - 1; k >= 0; k--)
            {
                const Texture& t = texturesIn[k];
                sumArea += t.m_width * t.m_height;
            }

            const int binArea = m_binWidth * m_binHeight;
            L = hkMath::max2((int)(hkReal(sumArea) / hkReal(binArea) + 0.5f), 1);
            HK_ASSERT_NO_MSG(0x7910d24a, L * binArea >= sumArea);
        }

        // Open the initial bins
        m_bins.setSize(L);
        for (int bi = L - 1; bi >= 0; bi--)
        {
            m_bins[bi].open(m_binWidth, m_binHeight);
        }

        // Initialize the items
        hkArray<Item> items;
        {
            items.setSize(numTextures);

            for (int k = 0; k < numTextures; k++)
            {
                const Texture& t = texturesIn[k];
                items[k].set(t.m_width, t.m_height, k);
            }

            // Sort the items descending by area. Break the ties by min(w, h)
            hkSort(items.begin(), numTextures, Item::less);
        }

        // Place each item
        for (int k = 0; k < numTextures; k++)
        {
            Item& texItem = items[k];
            Point texPos;

            placeItem(texItem, texPos);

            // Write output
            Placement& p    = placementsOut[k];
            p.m_rotation    = (hkUint8)texItem.m_rot;
            p.m_textureId   = texItem.m_id;
            p.m_atlasId     = (hkUint16)texPos.m_binId;
            p.m_x           = texPos.m_x - 1;
            p.m_y           = texPos.m_y - 1;
        }
    }
};

//
//  Estimates the size of the atlas map that could fit the given textures

void HK_CALL hkTextureAtlasUtil::estimateAtlasSize(_In_reads_(numTextures) const Texture* texturesIn, int numTextures, int& atlasWidthOut, int& atlasHeightOut)
{
    // Compute total area and max sizes
    int totalArea = 0;
    int maxW = HK_INT32_MIN;
    int maxH = HK_INT32_MIN;
    for (int ti = numTextures - 1; ti >= 0; ti--)
    {
        const Texture& tex  = texturesIn[ti];
        const int texArea   = tex.m_width * tex.m_height;
        totalArea += texArea;
        maxW = hkMath::max2(maxW, tex.m_width);
        maxH = hkMath::max2(maxH, tex.m_height);
    }

    // We want an atlas size that has a larger area than the total area
    const hkReal w  = hkMath::sqrt(hkReal(totalArea));
    int atlasW      = hkMath::max2(maxW, (int)(w + 0.5f));
    atlasW          = hkMath::max2(8, hkTextureAtlasImpl::convertToPow2(atlasW));

    // Compute the height
    const hkReal h  = hkReal(totalArea) / hkReal(atlasW);
    int atlasH      = hkMath::max2(maxH, (int)(h + 0.5f));
    atlasH          = hkMath::max2(8, hkTextureAtlasImpl::convertToPow2(atlasH));

    // Write output
    atlasWidthOut = atlasW;
    atlasHeightOut = atlasH;
}

//
//  Arranges the given textures in an atlas.
//  Based on "Algorithms for Two-Dimensional Bin Packing And Assignment Problems" PhD thesis, pag 52.

hkResult HK_CALL hkTextureAtlasUtil::arrangeTextures(_In_reads_(numTextures) const Texture* texturesIn, int numTextures, _Inout_updates_(numTextures) Placement* placementsOut, int& atlasWidthOut, int& atlasHeightOut, int maxAtlasWidth, int maxAtlasHeight)
{
    // Convert max sizes to a power of 2
    maxAtlasWidth   = hkTextureAtlasImpl::convertToPow2(maxAtlasWidth);
    maxAtlasHeight  = hkTextureAtlasImpl::convertToPow2(maxAtlasHeight);

    // Estimate initial atlas size
    int atlasW, atlasH;
    estimateAtlasSize(texturesIn, numTextures, atlasW, atlasH);
    if ( (atlasW > maxAtlasWidth) || (atlasH > maxAtlasHeight) )
    {
        return HK_FAILURE;
    }

    // We'll try to fit the textures multiple times in increasingly larger atlases
    // Compute the number of tries now and sort them by area
    hkArray<Texture> atlasSizes;
    {
        for (int w = atlasW; w <= maxAtlasWidth; w *= 2)
        {
            for (int h = atlasH; h <= maxAtlasHeight; h *= 2)
            {
                Texture& t  = atlasSizes.expandOne();
                t.m_width   = w;
                t.m_height  = h;
            }
        }
        hkSort(atlasSizes.begin(), atlasSizes.getSize(), hkTextureAtlasImpl::atlasLess);
    }

    // Try to fit our textures in increasingly larger atlases.
    // Stop on first success
    const int numTries = atlasSizes.getSize();
    for (int ti = 0; ti < numTries; ti++)
    {
        const Texture& atlasSize = atlasSizes[ti];
        hkTextureAtlasImpl::AtlasBuilder builder(atlasSize.m_width, atlasSize.m_height);
        builder.fitTextures(texturesIn, numTextures, placementsOut);

        // Check if we have created more than an atlas
        int k = numTextures - 1;
        for (; k >= 0; k--)
        {
            if ( placementsOut[k].m_atlasId > 0 )
            {
                break;
            }
        }

        // Check whether k < 0 i.e.  all atlas Ids are == 0.
        if ( k < 0 )
        {
            // Sort the placements by texture Id for convenience
            hkSort(placementsOut, numTextures, hkTextureAtlasImpl::placementLess);
            atlasWidthOut   = atlasSize.m_width;
            atlasHeightOut  = atlasSize.m_height;
            return HK_SUCCESS;
        }
    }

    // Failed to fit the textures in the atlas
    return HK_FAILURE;
}

//
//  Debug self-test.

hkResult HK_CALL hkTextureAtlasUtil::selfTest()
{
    const int numTex = 12;

    // Setup the textures
    hkArray<Texture> texIn;     texIn.setSize(numTex);
    hkArray<Placement> posOut;  posOut.setSize(numTex);
    texIn[0].set(4, 6);     // 24
    texIn[1].set(4, 4);     // 16
    texIn[2].set(8, 3);     // 24
    texIn[3].set(4, 3);     // 12
    texIn[4].set(4, 3);     // 12
    texIn[5].set(4, 3);     // 12
    texIn[6].set(1, 3);     // 3
    texIn[7].set(6, 2);     // 12
    texIn[8].set(2, 2);     // 4
    texIn[9].set(9, 2);     // 18
    texIn[10].set(9, 2);    // 18
    texIn[11].set(3, 1);    // 3

    // Fit textures in the first bin.
    hkTextureAtlasImpl::AtlasBuilder builder(10, 8);
    builder.fitTextures(texIn.begin(), numTex, posOut.begin());
    hkSort(posOut.begin(), numTex, hkTextureAtlasImpl::placementLess);

#define TEST_TEX(i, x, y, r, aid)\
    if ( (posOut[i].m_textureId != i)   ||\
         (posOut[i].m_x != x)           ||\
         (posOut[i].m_y != y)           ||\
         (posOut[i].m_rotation != r)    ||\
         (posOut[i].m_atlasId != aid) ) \
    {\
        return HK_FAILURE;\
    }

    //      texId   x   y,  rot,    atlasId
    TEST_TEX(0,     0,  0,  90,     0);
    TEST_TEX(1,     6,  0,  0,      0);
    TEST_TEX(2,     0,  0,  90,     1);
    TEST_TEX(3,     3,  3,  0,      1);
    TEST_TEX(4,     7,  0,  90,     1);
    TEST_TEX(5,     3,  0,  0,      1);
    TEST_TEX(6,     9,  4,  0,      1);
    TEST_TEX(7,     3,  6,  0,      1);
    TEST_TEX(8,     7,  4,  0,      1);
    TEST_TEX(9,     0,  4,  0,      0);
    TEST_TEX(10,    0,  6,  0,      0);
    TEST_TEX(11,    9,  4,  90,     0);

#undef TEST_TEX

    return HK_SUCCESS;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
