// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

//
//  Transforms the given array of entries into bone weights / indices

template <typename WeightType, typename IndexType>
void HK_CALL hkSkinningUtil::computeBoneIndicesAndWeights(  const hkArray<Entry>& entries, hkReal maxD, int numEntriesPerVertex,
                                                            WeightType* HK_RESTRICT weightsOut, IndexType* HK_RESTRICT indicesOut)
{
    const hkReal maxDistance2       = (maxD * maxD);
    const hkReal invMaxDistance2    = maxDistance2 ? (1.0f / maxDistance2) : 0.0f;
    const int numEntries            = entries.getSize();

    for (int i = 0; i < numEntries; i += numEntriesPerVertex)
    {
        const Entry* cur = entries.begin() + i;

        int usedWeights = 0;
        hkReal sum = 0.0f;

        for (int j = 0; j < numEntriesPerVertex; j++)
        {
            if (cur[j].m_index >= 0)
            {
                const hkReal curDist    = cur[j].m_distanceSquared;
                const hkReal e          = (curDist < maxDistance2) ? (1 - (curDist * invMaxDistance2)) : 0.0f;
                const hkReal dist       = hkMath::max2(0.001f, e);  // To avoid division by zero if all are on the 'edge' of the bounding sphere

                indicesOut[i + j]   = (IndexType)(cur[j].m_index);
                weightsOut[i + j]   = WeightType(dist);
                sum += dist;
                usedWeights++;
            }
            else
            {
                indicesOut[i + j] = (IndexType)((j == 0) ? 0 : cur[0].m_index);     // use an already existing bone instead of bone 0 when possible to reduce numerical issues
                weightsOut[i + j] = WeightType(0);
            }
        }

        if ( usedWeights > 0 )
        {
            int maxWeightInd = -1;
            WeightType maxWeight = 0;
            WeightType weightSum = 0;

            const hkReal recipSum = 1.0f / sum;
            for (int j = 0; j < usedWeights; j++)
            {
                weightsOut[i + j] *= WeightType(recipSum);

                // update max weight
                if  ( weightsOut[i + j] > maxWeight )
                {
                    maxWeight = weightsOut[i + j];
                    maxWeightInd = j;
                }
                weightSum += weightsOut[i + j];
            }

            // Makes sure the weights exactly sum to 1
            {
                const WeightType diff = 1.0f - weightSum;
                weightsOut[i + maxWeightInd] += diff;
            }
        }
        else
        {
            // None of the bones were in range... I could try and find the closest bone...
            // But for now I'll just attach to the first bone
            weightsOut[i] = WeightType(1);
        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
