// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY     : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/GeometryUtilities/hkGeometryUtilities.h>
#include <Common/GeometryUtilities/Mesh/Utils/FindVertexWeightsUtil/hkFindVertexWeightsUtil.h>
#include <Common/GeometryUtilities/Mesh/Utils/MeshSectionLockSet/hkMeshSectionLockSet.h>
#include <Common/GeometryUtilities/Mesh/Utils/MeshShapeUtil/hkMeshShapeUtil.h>
#include <Common/GeometryUtilities/Mesh/Utils/VertexBufferUtil/hkMeshVertexBufferUtil.h>
#include <Common/GeometryUtilities/Mesh/Utils/PrimitiveUtil/hkMeshPrimitiveUtil.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>
#include <Common/Base/Container/LocalArray/hkLocalBuffer.h>
#include <Common/Base/Math/Vector/hkIntVector.h>

void hkSkinningUtil_findMatrixIndexRange(const hkMeshVertexBuffer::LockedVertices::Buffer& bufferIn, int numVertices, int& minIndexOut, int& maxIndexOut)
{
    hkMeshVertexBuffer::LockedVertices::Buffer buffer = bufferIn;
    const int numValues = buffer.m_element.m_numValues;

    int maxIndex = *(const hkUint8*)buffer.m_start;
    int minIndex = maxIndex;

    for (int i = 0; i < numVertices; i++)
    {
        const hkUint8* values = (const hkUint8*)buffer.m_start;

        for (int j = 0; j < numValues; j++)
        {
            int index = int(values[j]);

            if (index > maxIndex)
            {
                maxIndex = index;
            }
            if (index < minIndex)
            {
                minIndex = index;
            }
        }

        buffer.next();
    }
    maxIndexOut = maxIndex;
    minIndexOut = minIndex;
}

/* static */hkResult hkSkinningUtil::findMatrixIndexRange(hkMeshVertexBuffer* vertexBuffer, int& minIndexOut, int& maxIndexOut)
{
    hkVertexFormat vertexFormat;
    vertexBuffer->getVertexFormat(vertexFormat);

    int elementIndex = vertexFormat.findElementIndex(hkVertexFormat::USAGE_BLEND_MATRIX_INDEX, 0);
    if (elementIndex < 0)
    {
        return HK_FAILURE;
    }

    // Lets do a lock and maybe a conversion
    hkVertexFormat::Element& srcElement = vertexFormat.m_elements[elementIndex];

    // I need to lock this
    hkMeshVertexBuffer::LockInput lockInput;
    hkMeshVertexBuffer::PartialLockInput partialLockInput;
    partialLockInput.m_elementIndices[0] = elementIndex;
    partialLockInput.m_lockFlags[0] = hkMeshVertexBuffer::ACCESS_READ;
    partialLockInput.m_numLockFlags = 1;

    hkMeshVertexBuffer::LockedVertices lockedVertices;

    hkMeshVertexBuffer::LockResult result = vertexBuffer->partialLock(lockInput, partialLockInput, lockedVertices);
    if (result != hkMeshVertexBuffer::RESULT_SUCCESS || lockedVertices.m_numVertices <= 0)
    {
        return HK_FAILURE;
    }

    hkResult res = HK_SUCCESS;
    // Okay if the type is UINT8 we can just run tru them
    if (srcElement.m_dataType == hkVertexFormat::TYPE_UINT8)
    {
        hkSkinningUtil_findMatrixIndexRange(lockedVertices.m_buffers[0], lockedVertices.m_numVertices, minIndexOut, maxIndexOut);
    }
    else
    {
        hkLocalBuffer<hkUint8> indices(lockedVertices.m_numVertices * srcElement.m_numValues);

        hkMeshVertexBuffer::LockedVertices::Buffer dstBuffer;
        dstBuffer.m_start = indices.begin();
        dstBuffer.m_stride = srcElement.m_numValues;
        dstBuffer.m_element.set(hkVertexFormat::USAGE_BLEND_MATRIX_INDEX, hkVertexFormat::TYPE_UINT8, srcElement.m_numValues, 0, 0);

        // Do the conversion
        hkMeshVertexBufferUtil::convert(lockedVertices.m_buffers[0], dstBuffer, lockedVertices.m_numVertices);

        // Work out the range
        hkSkinningUtil_findMatrixIndexRange(dstBuffer, lockedVertices.m_numVertices, minIndexOut, maxIndexOut);
    }

    vertexBuffer->unlock(lockedVertices);

    return res;
}

/* static */ hkResult hkSkinningUtil::findMatrixIndexRange(hkMeshShape* meshShape, int& minIndexOut, int& maxIndexOut)
{
    hkMeshSectionLockSet sectionSet;
    sectionSet.addMeshSections(meshShape, hkMeshShape::ACCESS_VERTEX_BUFFER);

    hkArray<hkMeshVertexBuffer*> uniqueBuffers;
    sectionSet.findUniqueVertexBuffers(uniqueBuffers);

    int shapeMinIndex = 0x7fffffff;
    int shapeMaxIndex = -1;

    const int numBuffers = uniqueBuffers.getSize();
    for (int i = 0; i < numBuffers; i++)
    {
        hkMeshVertexBuffer* buffer = uniqueBuffers[i];

        int minIndex,maxIndex;
        hkResult res = findMatrixIndexRange(buffer, minIndex, maxIndex);
        if (res.isFailure())
        {
            continue;
        }

        if (minIndex < shapeMinIndex)
        {
            shapeMinIndex = minIndex;
        }
        if (maxIndex > shapeMaxIndex)
        {
            shapeMaxIndex = maxIndex;
        }
    }

    minIndexOut = shapeMinIndex;
    maxIndexOut = shapeMaxIndex;

    return (shapeMinIndex <= shapeMaxIndex) ? HK_SUCCESS : HK_FAILURE;
}

/* static */void hkSkinningUtil::findDistances(const hkArray<hkVector4>& vertices, const hkArray<hkVector4>& boneCenters, int maxDistances, hkReal maxDistance, hkArray<Entry>& entriesOut)
{
    const int numVertices = vertices.getSize();
    const int numEntries = maxDistances * numVertices;
    entriesOut.setSize(numEntries);

    const int numBones = boneCenters.getSize();

    hkReal maxDistanceSq = maxDistance * maxDistance;

    Entry* entries = entriesOut.begin();
    for (int i = 0; i < numVertices; i++, entries += maxDistances)
    {
        const hkVector4& pos = vertices[i];

        for (int j = 0; j < maxDistances; j++)
        {
            entries[j].m_index = -1;
            entries[j].m_distanceSquared = maxDistanceSq;
        }

        hkReal maxDistSq = maxDistanceSq;
        int numValues = 0;

        for (int j = 0; j < numBones; j++)
        {
            // Work out the distance
            hkVector4 diff; diff.setSub(boneCenters[j], pos);

            hkReal distSq = diff.lengthSquared<3>().getReal();

            if (distSq < maxDistSq)
            {
                numValues = hkFindVertexWeightsUtil::insertEntry(entries, numValues, maxDistances, distSq, j);
                maxDistSq = hkFindVertexWeightsUtil::getMaxValue(entries, maxDistances);
            }
        }
    }
}

void hkSkinningUtil::extractBoneCenters(const hkArray<hkMatrix4>& bones, hkArray<hkVector4>& boneCenters)
{
    const int numBones = bones.getSize();
    boneCenters.setSize(numBones);

    for (int i = 0; i < numBones; i++)
    {
        boneCenters[i] = bones[i].getColumn<3>();
    }
}

int hkSkinningUtil::findNumBoneIndices(const hkVertexFormat& vertexFormat)
{
    int matrixIndexElementIndex = vertexFormat.findElementIndex( hkVertexFormat::USAGE_BLEND_MATRIX_INDEX, 0);

    if (matrixIndexElementIndex < 0 )
    {
        return 0;
    }

    const hkVertexFormat::Element& element = vertexFormat.m_elements[matrixIndexElementIndex];
    return element.m_numValues;
}

hkResult hkSkinningUtil::calculateBoneIndicesAndWeights(const hkArray<hkVector4>& boneCenters, hkMeshVertexBuffer* vertexBuffer, hkReal maxDistance)
{
    const int numVertices = vertexBuffer->getNumVertices();

    // We need the vertex positions
    hkLocalArray<hkVector4> positions(numVertices);
    hkMeshVertexBufferUtil::getElementVectorArray(vertexBuffer, hkVertexFormat::USAGE_POSITION, 0, positions);

    hkVertexFormat vertexFormat;
    vertexBuffer->getVertexFormat(vertexFormat);

    const int numValues = findNumBoneIndices(vertexFormat);

    if (numVertices <= 0)
    {
        HK_ASSERT_NO_MSG(0x32432, !"Bone weights/transform indices not present");
        return HK_FAILURE;
    }

    hkArray<Entry> entries;
    findDistances(positions, boneCenters, numValues, maxDistance, entries);

    return setSkinningValues(entries, vertexBuffer, maxDistance);
}

hkResult hkSkinningUtil::setSkinningValues(const hkArray<Entry>& entries, hkMeshVertexBuffer* vertexBuffer, hkReal maxDistance)
{
    const int numVertices = vertexBuffer->getNumVertices();

    hkVertexFormat vertexFormat;
    vertexBuffer->getVertexFormat(vertexFormat);

    int matrixIndexElementIndex = vertexFormat.findElementIndex( hkVertexFormat::USAGE_BLEND_MATRIX_INDEX, 0);
    int blendWeightsIndex = vertexFormat.findElementIndex( hkVertexFormat::USAGE_BLEND_WEIGHTS, 0);
    if ( blendWeightsIndex < 0)
    {
        blendWeightsIndex = vertexFormat.findElementIndex( hkVertexFormat::USAGE_BLEND_WEIGHTS_LAST_IMPLIED, 0);
    }

    if (matrixIndexElementIndex < 0 || blendWeightsIndex < 0)
    {
        HK_ASSERT_NO_MSG(0x32432, !"Bone weights/transform indices not present");
        return HK_FAILURE;
    }

    const hkVertexFormat::Element& element = vertexFormat.m_elements[matrixIndexElementIndex];
    int numValues = element.m_numValues;


    hkLocalBuffer<hkFloat32> weights(numVertices * numValues);
    hkLocalBuffer<hkUint8> boneIndices(numVertices * numValues);

    HK_ASSERT_NO_MSG(0x32432, numVertices * numValues == entries.getSize());
    computeBoneIndicesAndWeights(entries, maxDistance, numValues, weights.begin(), boneIndices.begin());

    {
        hkMeshVertexBuffer::LockInput lockInput;
        hkMeshVertexBuffer::PartialLockInput partialLock;

        partialLock.m_numLockFlags = 2;
        partialLock.m_elementIndices[0] = matrixIndexElementIndex;
        partialLock.m_lockFlags[0] = hkMeshVertexBuffer::ACCESS_WRITE | hkMeshVertexBuffer::ACCESS_WRITE_DISCARD;
        partialLock.m_elementIndices[1] = blendWeightsIndex;
        partialLock.m_lockFlags[1] = hkMeshVertexBuffer::ACCESS_WRITE | hkMeshVertexBuffer::ACCESS_WRITE_DISCARD;

        hkMeshVertexBuffer::LockedVertices::Buffer srcWeights;
        srcWeights.m_start = weights.begin();
        srcWeights.m_stride = sizeof(hkFloat32) * numValues;
        srcWeights.m_element.set(hkVertexFormat::USAGE_BLEND_WEIGHTS, hkVertexFormat::TYPE_FLOAT32, numValues);

        hkMeshVertexBuffer::LockedVertices::Buffer srcBoneIndices;
        srcBoneIndices.m_start = boneIndices.begin();
        srcBoneIndices.m_stride = sizeof(hkUint8) * numValues;
        srcBoneIndices.m_element.set(hkVertexFormat::USAGE_BLEND_WEIGHTS, hkVertexFormat::TYPE_UINT8, numValues);

        hkMeshVertexBuffer::LockedVertices lockedVertices;
        hkMeshVertexBuffer::LockResult lockRes = vertexBuffer->partialLock(lockInput, partialLock, lockedVertices);
        if (lockRes != hkMeshVertexBuffer::RESULT_SUCCESS)
        {
            return HK_FAILURE;
        }

        hkMeshVertexBufferUtil::convert(srcBoneIndices, lockedVertices.m_buffers[0], numVertices);
        hkMeshVertexBufferUtil::convert(srcWeights, lockedVertices.m_buffers[1], numVertices);

        vertexBuffer->unlock(lockedVertices);
    }

    return HK_SUCCESS;
}


/* static */hkBool hkSkinningUtil::isSkinnedVertexFormat(const hkVertexFormat& vertexFormat)
{
    hkBool hasBoneIndices = false;
    hkBool hasWeights = false;
    for (int i = 0; i < vertexFormat.m_numElements; i++)
    {
        const hkVertexFormat::Element& element = vertexFormat.m_elements[i];
        // We are only interested in sub usage 0
        if (element.m_subUsage != 0)
        {
            continue;
        }
        if (element.m_usage == hkVertexFormat::USAGE_BLEND_MATRIX_INDEX)
        {
            hasBoneIndices = true;
        }
        else if (element.m_usage == hkVertexFormat::USAGE_BLEND_WEIGHTS ||
            element.m_usage == hkVertexFormat::USAGE_BLEND_WEIGHTS_LAST_IMPLIED)
        {
            hasWeights = true;
        }
    }

    return hasBoneIndices && hasWeights;
}

/* static */hkMeshVertexBuffer* hkSkinningUtil::createSkinnedVertexBuffer(hkMeshSystem* meshSystem, hkMeshVertexBuffer* srcVertexBuffer, int numWeights)
{
    hkVertexFormat srcFormat;
    srcVertexBuffer->getVertexFormat(srcFormat);

    hkVertexFormat dstFormat;

    for (int i = 0; i < srcFormat.m_numElements; i++)
    {
        const hkVertexFormat::Element& srcElement = srcFormat.m_elements[i];
        if (srcElement.m_usage == hkVertexFormat::USAGE_BLEND_MATRIX_INDEX ||
            srcElement.m_usage == hkVertexFormat::USAGE_BLEND_WEIGHTS ||
            srcElement.m_usage == hkVertexFormat::USAGE_BLEND_WEIGHTS_LAST_IMPLIED)
        {
            // Ignore
        }
        else
        {
            dstFormat.addElement(srcElement);
        }
    }

    dstFormat.addElement(hkVertexFormat::USAGE_BLEND_WEIGHTS, hkVertexFormat::TYPE_FLOAT32, numWeights);
    dstFormat.addElement(hkVertexFormat::USAGE_BLEND_MATRIX_INDEX, hkVertexFormat::TYPE_UINT8, numWeights);

    dstFormat.makeCanonicalOrder();

    hkVertexFormat finalFormat;
    meshSystem->findSuitableVertexFormat(dstFormat, finalFormat);

    hkMeshVertexBuffer* dstVertexBuffer = meshSystem->createVertexBuffer(finalFormat, srcVertexBuffer->getNumVertices());

    // Convert all the src stuff over
    hkMeshVertexBufferUtil::convert(srcVertexBuffer, dstVertexBuffer);

    return dstVertexBuffer;
}

/* static */hkMeshVertexBuffer* hkSkinningUtil::ensureSkinnedVertexBuffer(hkMeshSystem* meshSystem, hkMeshVertexBuffer* srcVertexBuffer, int numWeights)
                {
    hkVertexFormat vertexFormat;
    srcVertexBuffer->getVertexFormat(vertexFormat);
    if (isSkinnedVertexFormat(vertexFormat))
    {
        srcVertexBuffer->addReference();
        return srcVertexBuffer;
    }
    else
    {
        return createSkinnedVertexBuffer(meshSystem, srcVertexBuffer, numWeights);
    }
}

/* static */hkBool hkSkinningUtil::isSkinnedShape(hkMeshShape* meshShape)
{
    hkMeshSectionLockSet sectionSet;
    sectionSet.addMeshSections(meshShape, hkMeshShape::ACCESS_VERTEX_BUFFER);

    hkArray<hkMeshVertexBuffer*> uniqueBuffers;
    sectionSet.findUniqueVertexBuffers(uniqueBuffers);

    const int numVertexBuffers = uniqueBuffers.getSize();
    for (int i = 0; i < numVertexBuffers; i++)
    {

        hkMeshVertexBuffer* buffer = uniqueBuffers[i];

        hkVertexFormat vertexFormat;
        buffer->getVertexFormat(vertexFormat);

        if (!isSkinnedVertexFormat(vertexFormat))
        {
            return false;
        }
    }
    return true;
}

/* static */hkResult hkSkinningUtil::setSkinningValues(hkMeshShape* meshShape, const hkArray<hkVector4>& boneCenters, hkReal maxDistance)
{
    hkMeshSectionLockSet sectionSet;
    sectionSet.addMeshSections(meshShape, hkMeshShape::ACCESS_VERTEX_BUFFER | hkMeshShape::ACCESS_INDICES);

    hkArray<hkMeshVertexBuffer*> uniqueBuffers;
    sectionSet.findUniqueVertexBuffers(uniqueBuffers);

    const int numUniqueBuffers = uniqueBuffers.getSize();
    for (int i = 0; i< numUniqueBuffers; i++)
    {
        hkMeshVertexBuffer* buffer = uniqueBuffers[i];
        hkVertexFormat vertexFormat;
        buffer->getVertexFormat(vertexFormat);

        int numBones = findNumBoneIndices(vertexFormat);

        if (numBones > 0)
                {
            hkResult res = calculateBoneIndicesAndWeights(boneCenters, buffer, maxDistance);
            if (res.isFailure())
                    {
                return res;
            }
        }
                    }

    return HK_SUCCESS;
                }

/* static */hkMeshShape* hkSkinningUtil::createEmptySkinnedShape(hkMeshSystem* meshSystem, const hkMeshShape* meshShape, int numWeights)
                {
    hkMeshSectionLockSet sectionSet;
    sectionSet.addMeshSections(meshShape, hkMeshShape::ACCESS_VERTEX_BUFFER);

    hkArray<hkMeshVertexBuffer*> uniqueVertexBuffers;
    sectionSet.findUniqueVertexBuffers(uniqueVertexBuffers);
    hkArray<hkMeshVertexBuffer*> uniqueSkinnedVertexBuffers;
    hkPointerMap<hkMeshVertexBuffer*, hkMeshVertexBuffer*> bufferMap;

    hkBool bufferReplaced = false;
    const int numBuffers = uniqueVertexBuffers.getSize();
    for (int i = 0; i < numBuffers; i++)
    {
        hkMeshVertexBuffer* buffer = uniqueVertexBuffers[i];

        hkVertexFormat vertexFormat;
        buffer->getVertexFormat(vertexFormat);

        hkMeshVertexBuffer* skinnedBuffer;
        if (hkSkinningUtil::isSkinnedVertexFormat(vertexFormat))
        {
            buffer->addReference();
            skinnedBuffer = buffer;
        }
        else
                {
            skinnedBuffer = hkSkinningUtil::createSkinnedVertexBuffer(meshSystem, buffer, numWeights);
            bufferReplaced = true;
                }

        uniqueSkinnedVertexBuffers.pushBack(skinnedBuffer);
        bufferMap.insert(buffer, skinnedBuffer);
            }

    hkMeshShape* skinnedMeshShape;
    if (bufferReplaced)
    {
        skinnedMeshShape = hkMeshShapeUtil::replaceShapeVertexBuffers(meshSystem, meshShape, bufferMap);
        }
    else
    {
        meshShape->addReference();
        skinnedMeshShape = const_cast<hkMeshShape*>(meshShape);
    }

    hkReferencedObject::removeReferences(uniqueSkinnedVertexBuffers.begin(), uniqueSkinnedVertexBuffers.getSize());
    return skinnedMeshShape;
}

//
//  Utility functions

namespace hkSkinningUtilImpl
{
    struct SkinInfo
    {
        HK_DECLARE_CLASS(SkinInfo, New);

        /// Initializes the skin info
        void init(hkMeshVertexBuffer* vb)
        {
            m_vb = vb;

            // Clear the visited bits
            const int numVerts = vb->getNumVertices();
            m_visitedVerts.resize(0, numVerts);
            m_visitedVerts.assignAll<0>();

            // Find the skinning data
            hkVertexFormat vfmt;        vb->getVertexFormat(vfmt);
            const int boneIndicesFmtIdx = vfmt.findElementIndex(hkVertexFormat::USAGE_BLEND_MATRIX_INDEX, 0);
            const int boneWeightsFmtIdx = hkMath::max2(vfmt.findElementIndex(hkVertexFormat::USAGE_BLEND_WEIGHTS, 0), vfmt.findElementIndex(hkVertexFormat::USAGE_BLEND_WEIGHTS_LAST_IMPLIED, 0));

            if ( boneIndicesFmtIdx >= 0 )
            {
                // Lock the vb
                hkMeshVertexBuffer::LockInput lockInput;
                hkMeshVertexBuffer::LockedVertices lockedVerts;
                vb->lock(lockInput, lockedVerts);

                // Get skinning data if any
                m_indices.setSize(numVerts << 2);   vb->getElementIntArray(lockedVerts, boneIndicesFmtIdx, m_indices.begin());
                m_weights.setSize(numVerts << 2);   vb->getElementVectorArray(lockedVerts, boneWeightsFmtIdx, m_weights.begin());

                vb->unlock(lockedVerts);
            }
        }

        hkMeshVertexBuffer* m_vb;
        hkBitField m_visitedVerts;
        hkArray<int> m_indices;
        hkArray<hkFloat32> m_weights;
    };

    /// Returns the skin info that matches the given vertex buffer
    static int findSkinInfo(const hkMeshVertexBuffer* vbIn, const hkArray<SkinInfo>& skinInfos)
    {
        for (int k = skinInfos.getSize() - 1; k >= 0; k--)
        {
            if ( skinInfos[k].m_vb == vbIn )
            {
                return k;
            }
        }

        return -1;
    }
}

//
//  Computes the bone indices used by the given mesh

void HK_CALL hkSkinningUtil::computeUsedBones(const hkMeshShape* meshShapeIn, hkBitField& usedBonesOut)
{
    // Types
    typedef hkSkinningUtilImpl::SkinInfo            SkinInfo;
    typedef hkMeshPrimitiveUtil::PrimitiveProvider  PrimitiveProvider;

    // Clear output
    usedBonesOut.clearStorage();
    if ( !meshShapeIn )
    {
        return;
    }

    hkMeshSectionLockSet sectionSet;
    sectionSet.addMeshSections(meshShapeIn, hkMeshShape::ACCESS_VERTEX_BUFFER | hkMeshShape::ACCESS_INDICES);

    // Get the skin infos
    hkArray<SkinInfo> skinInfos;
    {
        hkArray<hkMeshVertexBuffer*> uniqueBuffers;
        sectionSet.findUniqueVertexBuffers(uniqueBuffers);

        const int numVbs = uniqueBuffers.getSize();
        skinInfos.setSize(numVbs);

        for (int k = numVbs - 1; k >= 0; k--)
        {
            skinInfos[k].init(uniqueBuffers[k]);
        }
    }

    // Get the bones used by each section
    hkArray<int> boneIndices;
    for (int si = sectionSet.getNumSections() - 1; si >= 0; si--)
    {
        const hkMeshSection& section = sectionSet.getSection(si);

        // Locate its skin info
        const int sInfoIdx = hkSkinningUtilImpl::findSkinInfo(section.m_vertexBuffer, skinInfos);
        if (  sInfoIdx >= 0 )
        {
            const SkinInfo& sInfo = skinInfos[sInfoIdx];

            if ( sInfo.m_indices.getSize() )
            {
                PrimitiveProvider pReader(section.m_indices, section.m_primitiveType, section.m_indexType);

                // Copy and re-index indices
                boneIndices.clear();
                boneIndices.append(sInfo.m_indices);
                hkMeshShapeUtil::mapSectionBoneIndices(section, boneIndices);

                for (int pi = 0; pi < section.m_numPrimitives; pi++)
                {
                    pReader.readPrimitive();
                    const int vtxIdx[3] = { pReader.m_a, pReader.m_b, pReader.m_c   };

                    // Process the 3 verts
                    for (int vi = 0; vi < 3; vi++)
                    {
                        // Load bone indices & weights
                        hkIntVector bi;     bi.load<4>((const hkUint32*)&boneIndices[vtxIdx[vi] << 2]);
                        hkVector4 bw;       bw.load<4, HK_IO_NATIVE_ALIGNED>(&sInfo.m_weights[vtxIdx[vi] << 2]);

                        // Set the bones as used
                        for (int k = 3; k >= 0; k--)
                        {
                            if ( bw.getComponent<0>().isGreater(hkSimdReal_Eps) )
                            {
                                const int boneIdx       = bi.getComponent<0>();
                                const int numOldBones   = usedBonesOut.getSize();
                                const int numNewBones   = hkMath::max2(numOldBones, boneIdx + 1);

                                // Resize bit-field if necessary
                                if ( numNewBones > numOldBones )
                                {
                                    usedBonesOut.resize(0, numNewBones);
                                    usedBonesOut.assignRange(numOldBones, numNewBones - numOldBones, 0);
                                }

                                usedBonesOut.set(boneIdx);
                            }

                            // Rotate all
                            bw.setPermutation<hkVectorPermutation::YZWX>(bw);
                            bi.setPermutation<hkVectorPermutation::YZWX>(bi);
                        }
                    }
                }
            }
        }
    }
}

//
//  Remaps the bone indices used by the given mesh

hkMeshShape* HK_CALL hkSkinningUtil::remapBones(hkMeshSystem* meshSystem, const hkMeshShape* meshShapeIn, hkArray<int>& newFromOldBoneIndex)
{
    // Types
    typedef hkSkinningUtilImpl::SkinInfo            SkinInfo;
    typedef hkMeshPrimitiveUtil::PrimitiveProvider  PrimitiveProvider;
    typedef hkRefPtr<hkMeshVertexBuffer>            VbPtr;

    hkMeshSectionLockSet sectionSet;
    sectionSet.addMeshSections(meshShapeIn, hkMeshShape::ACCESS_VERTEX_BUFFER | hkMeshShape::ACCESS_INDICES);
    const int numSections = sectionSet.getNumSections();

    // Get the skin infos
    hkArray<SkinInfo> skinInfos;
    {
        hkArray<hkMeshVertexBuffer*> uniqueBuffers;
        sectionSet.findUniqueVertexBuffers(uniqueBuffers);

        const int numVbs = uniqueBuffers.getSize();
        skinInfos.setSize(numVbs);

        for (int k = numVbs - 1; k >= 0; k--)
        {
            skinInfos[k].init(uniqueBuffers[k]);
        }
    }

    // Collect the mesh infos
    hkArray<hkMeshSectionCinfo> cInfos;
    cInfos.setSize(numSections);

    for (int si = numSections - 1; si >= 0; si--)
    {
        const hkMeshSection& srcSection = sectionSet.getSection(si);
        hkMeshSectionCinfo& dstSection  = cInfos[si];

        dstSection.m_material           = srcSection.m_material;
        dstSection.m_primitiveType      = srcSection.m_primitiveType;
        dstSection.m_numPrimitives      = srcSection.m_numPrimitives;
        dstSection.m_indexType          = srcSection.m_indexType;
        dstSection.m_indices            = srcSection.m_indices;
        dstSection.m_vertexStartIndex   = srcSection.m_vertexStartIndex;
        dstSection.m_transformIndex     = srcSection.m_transformIndex;

        bool hasRemapped = false;
        if ( const hkMeshBoneIndexMapping* srcMap = srcSection.m_boneMatrixMap )
        {
            const int numBonesInMap = srcMap->m_mapping.getSize();
            if ( numBonesInMap )
            {
                hkMeshBoneIndexMapping& dstMap = dstSection.m_boneMatrixMap;
                dstMap.m_mapping.setSize(numBonesInMap);
                hasRemapped = true;

                for (int k = numBonesInMap - 1; k >= 0; k--)
                {
                    const int srcIdx    = srcMap->m_mapping[k];
                    dstMap.m_mapping[k] = hkInt16(newFromOldBoneIndex[srcIdx]);
                }
            }
        }

        // See if we were able to remap using the bone index mapping
        if ( !hasRemapped )
        {
            // We need to apply the remap to the vertex buffer
            // Locate the skin info
            const int sInfoIdx = hkSkinningUtilImpl::findSkinInfo(srcSection.m_vertexBuffer, skinInfos);
            if ( sInfoIdx >= 0 )
            {
                SkinInfo& sInfo = skinInfos[sInfoIdx];

                if ( sInfo.m_indices.getSize() )
                {
                    PrimitiveProvider pReader(srcSection.m_indices, srcSection.m_primitiveType, srcSection.m_indexType);

                    for (int pi = 0; pi < srcSection.m_numPrimitives; pi++)
                    {
                        pReader.readPrimitive();
                        const int vtxIndices[3] = { pReader.m_a, pReader.m_b, pReader.m_c   };

                        // Process the 3 verts
                        for (int vi = 0; vi < 3; vi++)
                        {
                            const int vtxIdx = vtxIndices[vi];

                            if ( !sInfo.m_visitedVerts.get(vtxIdx) )
                            {
                                // Mark vertex as visited
                                sInfo.m_visitedVerts.set(vtxIdx);

                                // Load bone indices & weights
                                hkIntVector bi;     bi.load<4>((const hkUint32*)&sInfo.m_indices[vtxIdx << 2]);
                                hkVector4 bw;       bw.load<4, HK_IO_NATIVE_ALIGNED>(&sInfo.m_weights[vtxIdx << 2]);

                                // Re-index
                                for (int k = 3; k >= 0; k--)
                                {
                                    if ( bw.getComponent<0>().isGreater(hkSimdReal_Eps) )
                                    {
                                        const int oldBoneIdx = bi.getComponent<0>();
                                        const int newBoneIdx = newFromOldBoneIndex[oldBoneIdx];
                                        bi.setComponent<0>(newBoneIdx);
                                    }

                                    // Rotate all
                                    bw.setPermutation<hkVectorPermutation::YZWX>(bw);
                                    bi.setPermutation<hkVectorPermutation::YZWX>(bi);
                                }

                                // Save bone indices
                                bi.store<4>((hkUint32*)&sInfo.m_indices[vtxIdx << 2]);
                            }
                        }
                    }
                }
            }
        }
    }

    // Clone the vertex buffers
    const int numVbs        = skinInfos.getSize();
    hkArray<VbPtr> dstVbs;  dstVbs.setSize(numVbs);

    for (int k = 0; k < numVbs; k++)
    {
        const SkinInfo& sInfo = skinInfos[k];

        if ( sInfo.m_visitedVerts.findFirstSet() < 0 )
        {
            // No vertex was changed, can use the original buffer
            dstVbs[k] = sInfo.m_vb;
        }
        else
        {
            // Bone indices were changed, clone the original buffer
            hkMeshVertexBuffer* newVb = sInfo.m_vb->clone();
            dstVbs[k].setAndDontIncrementRefCount(newVb);

            // Write the new indices
            hkVertexFormat vfmt;        newVb->getVertexFormat(vfmt);
            const int boneIndicesFmtIdx = vfmt.findElementIndex(hkVertexFormat::USAGE_BLEND_MATRIX_INDEX, 0);

            if ( boneIndicesFmtIdx >= 0 )
            {
                // Lock the vb
                hkMeshVertexBuffer::LockInput lockInput;
                hkMeshVertexBuffer::LockedVertices lockedVerts;
                newVb->lock(lockInput, lockedVerts);

                // Write
                newVb->setElementIntArray(lockedVerts, boneIndicesFmtIdx, sInfo.m_indices.begin());

                newVb->unlock(lockedVerts);
            }
        }
    }

    // Set the vbs in the cInfos
    for (int si = numSections - 1; si >= 0; si--)
    {
        const hkMeshSection& srcSection = sectionSet.getSection(si);
        hkMeshSectionCinfo& dstSection  = cInfos[si];

        const int sInfoIdx = hkSkinningUtilImpl::findSkinInfo(srcSection.m_vertexBuffer, skinInfos);
        if ( sInfoIdx >= 0 )
        {
            dstSection.m_vertexBuffer = dstVbs[sInfoIdx];
        }
    }

    // Create the new mesh
    hkMeshShape* ret = meshSystem->createShape(cInfos.begin(), numSections);
    return ret;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
