// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY     : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/GeometryUtilities/Mesh/hkMeshShape.h>

/// This utility is designed to help the construction of mesh sections
///
/// Specifically it hold all of the indices needed to construct a shape, as well as the hkMeshSectionCinfos
///
/// Usage is of the form of calling 'startMeshSection' at the start of a new mesh section. You can optionally pass
/// in the vertexBuffer and material associated with the section. Next multiple addTriangleIndices or addUnindexed
/// calls can be made - and these will be accumulated in internally buffers. Once all of the indices have been
/// added for the section a call to endMeshSection can be called.
///
/// Before calling endMeshSection - if the vertexBuffer or the material haven't been set - they can be set with
/// setVertexBuffer or setMaterial. It is invalid to have not set this data either in startMeshSection or with these methods
/// before the endMeshSection call.
///
/// Once all of the mesh sections have been added the hkMeshSectionCinfo array needed for construction can be retrieved
/// by calling 'getSections' and the number of sections from getNumSections
///
/// A hkMeshSectionBuilder can be emptied of its contents by calling 'clear' - allowing any internal allocation used for
/// building to be potentially reused.
///
/// \sa hkMeshMaterial hkMeshShape
class HK_EXPORT_COMMON hkMeshSectionBuilder
{
    public:
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_SCENE_DATA, hkMeshSectionBuilder );

            /// Dtor
        ~hkMeshSectionBuilder();

            /// Remove everything
        void clear();

            /// Start a new primitive
        void startMeshSection(_Inout_opt_ hkMeshVertexBuffer* vertexBuffer = HK_NULL, _Inout_opt_ hkMeshMaterial* material = HK_NULL);

            /// Add indices to the current primitive
        hkResult concatPrimitives(hkMeshSection::PrimitiveType primType, _In_reads_(numIndices) const hkUint16* indices, int numIndices, int indexBase = 0);

            /// Add indices to the current primitive
        hkResult concatPrimitives(hkMeshSection::PrimitiveType primType, _In_reads_(numIndices) const hkUint32* indices, int numIndices, int indexBase = 0);

            /// Add non indexed - this will add a new mesh section
        hkResult concatUnindexed(hkMeshSection::PrimitiveType primType, int vertexStartIndex, int numIndices);

            /// End the current primitive
        void endMeshSection();

            /// Can only be performed once in start/end - and must not have been set up at start
        void setVertexBuffer(_Inout_ hkMeshVertexBuffer* vertexBuffer);

            /// Can only be performed once in start/end - and must not have been set up at start
        void setMaterial(_Inout_ hkMeshMaterial* material);

            /// Get an array of primitives
        HK_INLINE _Ret_notnull_ const hkMeshSectionCinfo* getSections() const { return m_sections.begin(); }

            /// Get the number of primitives
        HK_INLINE int getNumSections() const { return m_sections.getSize(); }

        static bool HK_CALL canConcatPrimitives(hkMeshSection::PrimitiveType b, hkMeshSection::PrimitiveType a);

    protected:

        hkUint32* _addIndices32(int numIndices);
        hkUint16* _addIndices16(int numIndices);
        bool _canConcatPrimitive(hkMeshSection::PrimitiveType primType);
        void _makeIndices32();
        void _concatIndices(int vertexStartIndex, int numIndices);
        void _concatIndices(_In_reads_(numIndices) const hkUint16* srcIndices, int numIndices, int indexBase = 0);
        void _concatIndices(_In_reads_(numIndices) const hkUint32* srcIndices, int numIndices, int indexBase = 0);
        hkResult _makeConcatable(hkMeshSection::PrimitiveType primType);
        hkBool _isIndexed() const;
        hkResult _concatPrimitives(hkMeshSection::PrimitiveType primType, _In_reads_(numIndices) const hkUint16* indices, int numIndices, int indexBase);
        hkResult _concatPrimitives(hkMeshSection::PrimitiveType primType, _In_reads_(numIndices) const hkUint32* indices, int numIndices, int indexBase);

        static bool HK_CALL _needsIndices32(_In_reads_(numIndices) const hkUint16* srcIndices, int numIndices, int indexBase);

        int m_indexBase16;                                  
        int m_indexBase32;                                  

        hkArray<hkMeshSectionCinfo> m_sections;             ///< The sections produced
        hkArray<hkUint16> m_indices16;                        ///< Triangle indices
        hkArray<hkUint32> m_indices32;                      ///<
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
