// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/GeometryUtilities/Mesh/hkMeshTexture.h>

/// Utility class that computes per-vertex dominant displacements to be used in displacement mapping.
class HK_EXPORT_COMMON hkDisplacementMappingUtil
{
    public:

        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_SCENE_DATA, hkDisplacementMappingUtil);

        typedef hkMeshTexture::RawBufferDescriptor Descriptor;

    public:

        /// Per-vertex dominant info
        class DominantInfo
        {
            public:

                HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_SCENE_DATA, hkDisplacementMappingUtil::DominantInfo);

            public:

                /// Constructor
                HK_INLINE DominantInfo();

                /// Retrieves the normal
                HK_INLINE void getNormal(hkVector4& nrmOut) const;

                /// Sets the displacement normal. The normal is assumed normalized and will be encoded to just 2 components, .xy
                HK_INLINE void setNormal(hkVector4Parameter n);

                /// Gets / sets the displacement amount, i.e. a factor in [0, 1] used to scale the displacement normal
                HK_INLINE const hkSimdReal getDisplacement() const;
                HK_INLINE void setDisplacement(hkSimdRealParameter d);

                /// Gets / sets the interpolation factor between the two displacement maps
                HK_INLINE const hkSimdReal getBlendFactor() const;
                HK_INLINE void setBlendFactor(hkSimdRealParameter f);

                /// Returns the uv for the i-th map
                HK_INLINE void getUv(int mapIdx, hkVector4& uvOut) const;

                /// Sets the displacement map uv
                template <int I>
                HK_INLINE void setUv(hkVector4Parameter uv);

                /// Sets the displacement map uv
                HK_INLINE void setUv(int mapIdx, hkVector4Parameter uv);

                /// Gets the displacement vector
                HK_INLINE void getDisplacementVector(hkVector4& dOut) const;

                /// Sets the displacement vector (i.e. n = normalize(vD), d = length(vD)). The normal is considered null if its length is under the given tol.
                HK_INLINE void setDisplacementVector(hkVector4Parameter vD, hkSimdRealParameter tol);

                /// Permutes the displacement maps
                HK_INLINE void permuteDisplacementMaps();

                /// Loads the values from a buffer
                HK_INLINE void load(_In_reads_bytes_(32) const void* buffer);

            protected:

                /// Displacement map uv.
                hkVector4f m_data0; ///< [uv0.x, uv0.y, uv1.x, uv1.y]

                /// Normalized displacement direction in .xyz.
                hkVector4f m_data1; ///< [normal.x, normal.y, mapBlendFactor, displacementAmount]
        };

        /// Dominants buffer
        class HK_EXPORT_COMMON DominantsBuffer
        {
            public:

                HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_SCENE_DATA, hkDisplacementMappingUtil::DominantsBuffer);

            public:

                /// Constructor
                DominantsBuffer();

                /// Destructor
                virtual ~DominantsBuffer();

            public:

                /// Allocates a buffer for N dominants
                void alloc(int numDominants);

                /// Creates and returns the texture
                _Ret_notnull_ hkMeshTexture* realize(_Inout_ hkMeshSystem* meshSystem);

                /// Returns the I-th element
                HK_INLINE DominantInfo& operator[](int i);
                HK_INLINE const DominantInfo& operator[](int i) const;

            protected:

                hkUint8* m_data;                    ///< Buffer data
                hkRefPtr<hkMeshTexture> m_texture;  ///< The texture
        };

    public:

        /// Returns the displacement map set on the given material
        static _Ret_maybenull_ hkMeshTexture* HK_CALL getDisplacementMap(_In_opt_ const hkMeshMaterial* mtl);

        /// Returns the dominants map set on the given material
        static _Ret_maybenull_ hkMeshTexture* HK_CALL getDominantsMap(_In_opt_ const hkMeshMaterial* mtl);

        /// Sets the given dominants map on the given material
        static void HK_CALL setDominantsMap(_Inout_ hkMeshMaterial* mtl, _In_opt_ hkMeshTexture* dominantsMap);

        /// Sets the I-th displacement map on the given material to the given texture
        static void HK_CALL setDisplacementMap(int i, _Inout_ hkMeshMaterial* mtl, _In_opt_ hkMeshTexture* displacementMap);

        /// Clones a material without preserving the name
        static _Ret_notnull_ hkMeshMaterial* HK_CALL duplicateMaterial(_Inout_ hkMeshSystem* meshSystem, _In_ const hkMeshMaterial* src);
};

#include <Common/GeometryUtilities/Mesh/Utils/DisplacementMappingUtil/hkDisplacementMappingUtil.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
