// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY     : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/GeometryUtilities/Mesh/hkMeshVertexBuffer.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>

    /// A memory implementation of a hkMeshVertexBuffer
class HK_EXPORT_COMMON hkMemoryMeshVertexBuffer : public hkMeshVertexBuffer
{
    public:
        HK_DECLARE_CLASS(hkMemoryMeshVertexBuffer, Reflect, New, Version(2));

            /// Default Ctor
        hkMemoryMeshVertexBuffer();
            /// Ctor
        hkMemoryMeshVertexBuffer(const hkVertexFormat& format, int numVertices);
            /// Assignment
        hkMemoryMeshVertexBuffer& operator= (const hkMemoryMeshVertexBuffer& rhs);

            /// Dtor
        virtual ~hkMemoryMeshVertexBuffer();

            // hkMeshVertexBuffer implementation
        virtual void getVertexFormat( hkVertexFormat& formatOut ) const HK_OVERRIDE { formatOut = m_format; }

            // hkMeshVertexBuffer implementation
        virtual int getNumVertices() const HK_OVERRIDE { return m_numVertices; }

            // hkMeshVertexBuffer implementation
        virtual LockResult lock( const LockInput& input, LockedVertices& lockedVerticesOut ) override;

            // hkMeshVertexBuffer implementation
        virtual LockResult partialLock( const LockInput& input, const PartialLockInput& partialInput, LockedVertices& lockedOut) override;

            // hkMeshVertexBuffer implementation
        virtual void getElementVectorArray(const LockedVertices& lockedVertices, int elementIndex, _Out_writes_(4 * lockedVertices.m_numVertices) hkFloat32* compData) override;

            // hkMeshVertexBuffer implementation
        virtual void setElementVectorArray(const LockedVertices& lockedVertices, int elementIndex, _In_reads_(4 * lockedVertices.m_numVertices) const hkFloat32* compData) override;

            // hkMeshVertexBuffer implementation
        virtual void getElementIntArray(const LockedVertices& lockedVertices, int elementIndex, _Out_writes_(_Inexpressible_()) int* compData) override;

            // hkMeshVertexBuffer implementation
        virtual void setElementIntArray(const LockedVertices& lockedVertices, int elementIndex, _In_reads_(_Inexpressible_()) const int* compData) override;

            // hkMeshVertexBuffer implementation
        virtual void unlock( const LockedVertices& lockedVertices ) override;

            /// hkMeshVertexBuffer implementation
        virtual _Ret_notnull_ hkMeshVertexBuffer* clone() override;

            /// hkMeshVertexBuffer implementation
        virtual bool isSharable()  override { return m_isSharable; }

            /// Get the buffer (avoids having to do a lock)
        void getLockedVerticesBuffer(int elementIndex, LockedVertices::Buffer& buffer);

            /// Get a locked vertex buffer structure for a range of vertices. (avoids doing a lock)
        void getLockedVertices( int startVertex, int numVertices, LockedVertices& lockedVerticesOut );

            /// Set a different vertex format - will set up as having zero vertices
        void setVertexFormat(const hkVertexFormat& format);
            /// Set up the amount of vertices - will zero the memory
        void setNumVerticesAndZero(int numVertices);

            /// Use an external buffer for data storage. bufferSize must be equal to the vertex stride * numVertices.
            /// The buffer must stay in scope as long as the hkMemoryVertexBuffer exists
        void useExternalBuffer(_Out_writes_bytes_(bufferSize) void* data, int numVertices, int bufferSize);

            /// Get the vertex data
        _Ret_notnull_ hkUint8* getVertexData() { return m_memory.begin(); }
            /// Get the vertex data
        _Ret_notnull_ const hkUint8* getVertexData() const { return m_memory.begin(); }

            /// Get the vertex stride
        int getVertexStride() const { return m_vertexStride; }

            /// Add a single vertex
        _Ret_notnull_ hkUint8* expandOne() { m_numVertices++; return m_memory.expandBy(m_vertexStride); }
            /// Add multiple vertices
        _Ret_notnull_ hkUint8* expandBy(int numVertices) { m_numVertices += numVertices; return m_memory.expandBy(numVertices * m_vertexStride); }

            /// Calculates the offset to each component, as well as the total size of a vertex
        static int HK_CALL calculateElementOffsets(const hkVertexFormat& format, int offsets[hkVertexFormat::MAX_ELEMENTS]);
            /// Calculate offset of specific component
        static int HK_CALL calculateElementOffset(const hkVertexFormat& format, hkVertexFormat::ComponentUsage usage, int subUsage);

        void afterReflectNew();

    protected:

            /// Bytes swaps the memory buffer if the vertex buffer was serialized on a machine with a different endian
        void handleEndian();

    protected:


        struct hkVertexFormat m_format;                             ///< The vertex format
        int m_elementOffsets[hkVertexFormat::MAX_ELEMENTS];         ///< Offsets from start for each element type
        hkArray<hkUint8> m_memory;                                  ///< The storage for the vertices
        int m_vertexStride;                                         ///< The amount of bytes between one vertex to the next

        int m_numReadLocks          HK_ATTR(hk::Reflect(false));    ///< Number of times lock have been called with read flag
        int m_numWriteLocks         HK_ATTR(hk::Reflect(false));    ///< Number of times lock have been called with write flag (max 1)
        hkCriticalSection m_lock    HK_ATTR(hk::Reflect(false));    ///< Safe lock for multi thread access

        int m_numVertices;                                          ///< Total number of vertices

        hkBool m_isBigEndian;                                       ///< If true, was saved on a big endian machine, false other wise. Used to byte swap after serialization.

        hkBool m_isSharable;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
