// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY     : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
//HK_HAVOK_ASSEMBLY_EXCLUDE_FILE

#pragma once

#include <Common/GeometryUtilities/Mesh/hkMeshShape.h>
#include <Common/GeometryUtilities/Mesh/hkMeshBody.h>
#include <Common/GeometryUtilities/Mesh/IndexedTransformSet/hkIndexedTransformSet.h>

class hkMeshSystem;


/// A simple default implementation of hkMeshShape, which could be used as a default implementation for compound shapes
///
/// When using an hkMeshSystem, there is the method 'createCompoundShape' which creates a new shape out of a combination
/// of pre-existing shapes. Each of the contained shapes is controlled via a unique transform in the transform set.
///
/// The default compound mesh shape, provides a simple way of implementing this functionality, in that it will use the
/// functionality that has already been implemented in for normal shapes on hkMeshSystem to emulate the compound shape
/// result.
///
/// In order to make this work in an implementation of hkMeshSystem, when a body is constructed from a hkDefaultCompoundMeshShape,
/// a hkDefaultCompoundMeshBody should be created. This will then manufacture hkMeshBodys for all of the contained hkMeshShapes.
/// When a hkDefaultCompoundMeshBody is added to a hkMeshSystem, the addToSystem method should be called. When it is removed the
/// addFromSystem should be called. These methods will add and remove all of the contained hkMeshBodys respectively.
///
/// In an implementation of hkMeshSystem - identification of the shape or bodies type so an appropriate code route can be
/// taken can be achieved by using the getDynamicType method.
///
/// \sa hkMeshShape
class HK_EXPORT_COMMON hkDefaultCompoundMeshShape : public hkMeshShape
{
    public:

        HK_DECLARE_CLASS(hkDefaultCompoundMeshShape, New, Reflect, BypassCtor);

            // Ctor
        hkDefaultCompoundMeshShape(_In_reads_(numShapes) const hkMeshShape*const * shapes, _In_reads_opt_(numShapes) const hkMatrix4* transforms, int numShapes);

            // Dtor
        virtual ~hkDefaultCompoundMeshShape();

            // hkMeshShape implementation
        virtual int getNumSections() const;

            // hkMeshShape implementation
        virtual void lockSection(int sectionIndex, hkUint8 accessFlags, hkMeshSection& sectionOut) const;

            // hkMeshShape implementation
        virtual void unlockSection(const hkMeshSection& section) const;

            /// Create a list of shapes (ref'd) for each transform index
        static void HK_CALL createTransformIndexedShapeList(_Inout_ hkMeshSystem* meshSystem, _In_reads_(numSections) const hkMeshSectionCinfo* sectionsIn, int numSections, hkArray<const hkMeshShape*>& shapesOut);

            /// Returns true if all of the sections have a transform index.
        static hkBool HK_CALL hasIndexTransforms(_In_reads_(numSections) const hkMeshSectionCinfo* sections, int numSections);

            /// Will create a compound shape, made up of shapes which have the same transform index
        static _Ret_notnull_ hkDefaultCompoundMeshShape* HK_CALL createTransformIndexedShape(_Inout_ hkMeshSystem* meshSystem, _In_reads_(numSections) const hkMeshSectionCinfo* sectionsIn, int numSections);

    public:
        //
        // Members
        //

        struct HK_EXPORT_COMMON MeshSection
        {
            HK_DECLARE_CLASS(MeshSection, New, Reflect);
            int m_shapeIndex;
            int m_sectionIndex;
        };

        hkArray<const hkMeshShape*> m_shapes;   ///< The child shapes
        hkArray<hkMatrix4> m_defaultChildTransforms;     ///< An optional array to transform from child shape to this shape
        hkArray<MeshSection> m_sections;        ///< The mesh sections
};



/// A simple implementation of hkMeshBody - which simulates a compound body using multiple contained bodies.
///
/// This class should be used with the hkDefaultCompoundMeshShape class. It holds the hkMeshBody objects that
/// are constructed from the contained shapes from the hkDefaultCompoundMeshShape.
///
/// To make function correctly in an implementation of hkMeshSystem, the method addToSystem and removeFromSystem
/// should be called when the hkDefaultCompoundMeshBody is added or removed from the world. Use the getDynamicType
/// method to identity the type of object in the implementation of addBody/removeBody on the hkMeshSystem.
///
/// For an implementation of this - look at the hkgMeshSystem implementation.
///
/// \sa hkMeshBody
class HK_EXPORT_COMMON hkDefaultCompoundMeshBody : public hkMeshBody
{
    public:
        HK_DECLARE_CLASS(hkDefaultCompoundMeshBody, New, Reflect, BypassCtor);

            /// Ctor
        hkDefaultCompoundMeshBody(_Inout_ hkMeshSystem* meshSystem, _In_ const hkDefaultCompoundMeshShape* shape, const hkMatrix4& transform, _In_opt_ const hkIndexedTransformSetCinfo* transformSetCinfo);

            // Dtor
        virtual ~hkDefaultCompoundMeshBody();

            // hkMeshBody implementation
        virtual _Ret_notnull_ const hkMeshShape* getMeshShape() const HK_OVERRIDE;

            // hkMeshBody implementation
        virtual void getTransform( hkMatrix4& transform ) const HK_OVERRIDE;

            // hkMeshBody implementation
        virtual void setTransform(const hkMatrix4& matrix) HK_OVERRIDE;

            // hkMeshBody implementation
        virtual hkResult setPickingData(int id, void* data) HK_OVERRIDE;

            // hkMeshBody implementation
        virtual _Ret_maybenull_ hkMeshVertexBuffer* getVertexBuffer(int sectionIndex) HK_OVERRIDE;

            // hkMeshBody implementation
        virtual int getNumIndexedTransforms() HK_OVERRIDE { return m_transformSet ? m_transformSet->getNumMatrices() : 0; }

            // hkMeshBody implementation
        virtual void setIndexedTransforms(int startIndex, _In_reads_(numMatrices) const hkMatrix4* matrices, int numMatrices) HK_OVERRIDE;

            // hkMeshBody implementation
        virtual void getIndexedTransforms(int startIndex, _Out_writes_(numMatrices) hkMatrix4* matrices, int numMatrices) HK_OVERRIDE { HK_ASSERT_NO_MSG(0x423432, m_transformSet); m_transformSet->getMatrices(startIndex, matrices, numMatrices); }

            // hkMeshBody implementation
        virtual void getIndexedInverseTransforms(int startIndex, _In_reads_(numMatrices) hkMatrix4* matrices, int numMatrices) HK_OVERRIDE { HK_ASSERT_NO_MSG(0x34243206, m_transformSet); m_transformSet->getInverseMatrices(startIndex, matrices, numMatrices); }

            // hkMeshBody implementation
        virtual void completeUpdate() HK_OVERRIDE;
        virtual void completeUpdate(const hkMatrix4& transform) HK_OVERRIDE;

            // Set inverse transforms
        virtual void setIndexedInverseTransforms(int startIndex, _In_reads_(numMatrices) const hkMatrix4* matrices, int numMatrices);

            // Ensure inverse transforms
        virtual void ensureInverseTransforms();

        void addToSystem(_Inout_opt_ hkMeshSystem* meshSystem);

        void removeFromSystem(_Inout_opt_ hkMeshSystem* meshSystem);

        const hkArray<hkMeshBody*>& getChildBodies() const { return m_bodies; }

    protected:

        void _update();


        //
        // Members
        //
    protected:

        // With a compound shape, there are multiple contained shapes, otherwise there is just one
        hkArray<hkMeshBody*> m_bodies;

        hkMatrix4 m_transform;

        // The shape this has been created from
        const hkDefaultCompoundMeshShape* m_shape;

        hkRefPtr<hkIndexedTransformSet> m_transformSet;
        //
        hkBool m_transformIsDirty;
        hkBool m_transformSetUpdated;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
