// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Compat/hkCompat.h>
#include <Common/Base/Reflect/Version/hkReflectVersionPatcher.h>
#include <Common/Base/Reflect/Version/hkModifiableTypeSet.h>
#include <Common/Base/Reflect/Visitor/hkReflectVisitor.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Compat/Common/Serialize/Data/hkDataObject.h>
#include <Common/Base/Reflect/Builder/hkTypeBuilder.h>
#include <Common/Base/Reflect/Builder/hkRecordLayout.h>
#include <Common/Base/Reflect/TypeReg/hkTypeReg.h>
#include <Common/Base/Config/hkOptionalComponent.h>
#include <Common/Base/Serialize/Core/hkSerializeCore.h>
#include <Common/Base/Reflect/TypeVm/hkTypeVmFastCopyInterpreter.h>
#include <Common/Base/Reflect/TypeVm/hkTypeVmCompiler.h>
#include <Common/Base/Reflect/TypeVm/hkTypeVmCompilerPasses.h>
#include <Common/Base/Serialize/Detail/hkIndexedBundle.h>

#define DEBUG_LOG_IDENTIFIER "ver.CompatPatcher"
#include <Common/Base/System/Log/hkLog.hxx>

hkUint32 hkHashValue(const hkDataObject::VarOrExtra& o) { return o.getId(); }

namespace
{
    hkReflect::DataFieldDecl findFieldInPatchedType(const hkReflect::RecordType* versioningTargetRecord, const char* name)
    {
        hkReflect::FieldDecl f = versioningTargetRecord->findField(name, false);
        if (!f)
        {
            hkStringBuf addedFieldName("#");
            addedFieldName.append(name);
            f = versioningTargetRecord->findField(addedFieldName, false);
        }

        HK_ASSERT_NO_MSG(0x387c0e6f, f); // Member should already be in the patching type

        // Strip off the '#'
        hkReflect::TypeDetail::setFieldName(f, name);
        return f.asDataField();
    }

    using namespace hkSerialize;
    using namespace hkReflect;

    template<typename T>
    void grow(hkArray<T>& a, int i, const T& fill=T())
    {
        a.setSize(hkMath::max2(i + 1, a.getSize()), fill);
    }

    struct WorkBundle : public hkSerialize::Bundle
    {
        WorkBundle() : m_allocator(hkMemHeapAllocator()) {}

        void init(const hkArrayView<Bundle::Item>& allItems)
        {
            
            m_extras.pushBack(VarN::invalid());
            m_singleVars.setSize(allItems.getSize());

            int lastVar = -1;
            for (int i = 0; i < allItems.getSize(); ++i)
            {
                const Bundle::Item& item = allItems[i];
                if (item.isVarN())
                {
                    m_extras.pushBack(item.varn());
                }
                else
                {
                    lastVar = hkMath::max2(lastVar, i);
                    m_singleVars[i] = item;

                    
                    if (item.isNote() && !allItems[item.getAnnotated()].var().isValid())
                    {
                        lastVar = hkMath::max2(lastVar, item.getAnnotated());
                        m_singleVars[item.getAnnotated()].m_extra = i;
                    }
                }
            }
            HK_ASSERT_NO_MSG(0x389a7fb, lastVar < m_singleVars.getSize());
            m_singleVars.setSize(lastVar + 1);
        }

        Var var(int i) { return m_singleVars[i].var(); }
        VarN extra(int i) { return m_extras[i]; }

        void setVar(VarId vid, const Var& v)
        {
            grow(m_singleVars, vid);
            m_singleVars[vid].m_addr = v.getAddress();
            m_singleVars[vid].m_type = v.getType();
        }
        void setVarN(int i, const VarN& v)
        {
            VarN zero = VarN::fromArray(HK_NULL, HK_NULL, 0);
            grow(m_extras, i, zero);
            m_extras[i] = v;
        }

        int numVars() const { return m_singleVars.getSize(); }
        int numExtras() const { return m_extras.getSize(); }

        int getAnnotated(VarId id) const { return m_singleVars[id].getAnnotated(); }

        virtual hkReflect::Var getContents() const HK_OVERRIDE
        {
            return m_singleVars.getSize() ? m_singleVars[1].var() : hkReflect::Var();
        }

        virtual int getItems(hkArray<Item>& out) const HK_OVERRIDE
        {
            out.clear();
            out.reserve(m_singleVars.getSize() + m_extras.getSize() - 1);
            out.append(m_singleVars);
            for (int i = 1; i < m_extras.getSize(); ++i)
            {
                out.expandOne().set(m_extras[i].getAddress(), m_extras[i].getType(), Item::VARN, m_extras[i].getCount());
            }
            return out.getSize();
        }

        virtual hkReflect::Var getNoteOnPointer(const hkReflect::PointerVar& ptr) const HK_OVERRIDE
        {
            hkUint32 index = *(hkUint32Le*)ptr.getAddress();
            const Bundle::Item& item = m_singleVars[index];
            if (item.getAnnotated())
            {
                return item.var();
            }
            else if (!item.var().isValid() && item.m_extra != 0)
            {
                
                return m_singleVars[item.m_extra].var();
            }
            return hkReflect::Var();
        }

        void setType(int tid, const hkReflect::Type* t)
        {
            if (m_types.getSize() <= tid)
            {
                m_types.setSize(tid + 1);
            }
            m_types[tid] = t;
        }

        void clear()
        {
            m_allocator.clear();
        }

        hkArray<Bundle::Item> m_singleVars;
        hkArray<VarN> m_extras;
        hkTransientAllocator m_allocator;
    };
}

class CompatPatcher : public hkReflect::Version::PatcherInterface2014
{
public:
    HK_DECLARE_CLASS(CompatPatcher, New);

    
    struct TrackedType
    {
        struct ObjectRef
        {
            void* m_data;
            int m_numElems;
            int m_stride;


            hkDataObject::VarOrExtra m_id;

            ObjectRef() : m_id(0, hkDataObject::VAR) {}
            void set(void* d, int numElems, int stride, int varId, int extraId) { m_data = d; m_numElems = numElems; m_stride = stride; m_id.set(varId == -1 ? extraId : varId, (varId == -1) ? hkDataObject::EXTRA : hkDataObject::VAR); }
            void set(void* d, int numElems, int stride, hkDataObject::VarOrExtra id) { m_data = d; m_numElems = numElems; m_stride = stride; m_id.set(id); }
        };

        // List of all the single objects of this type
        hkArray<ObjectRef> objectList;
    };

    virtual ~CompatPatcher()
    {
        delete m_arrayImpl;
        delete m_pointerImpl;
    }

    hkVersionedRecordType* findType(const char* name, int version)
    {
        hkVersionedRecordType* versionedType = m_typeSet.findType(name, version);
        HK_ASSERT_NO_MSG(0x4907056f, versionedType);
        return versionedType;
    }

    int addObject(hkReflect::Var object)
    {
        for (int objectIndex = 0; objectIndex < m_bundle.numVars(); objectIndex++)
        {
            hkReflect::Var obj = m_bundle.var(objectIndex);
            if ((obj.getType() == object.getType()) && (obj.getAddress() == object.getAddress()))
            {
                return objectIndex;
            }
        }
        HK_ASSERT_NO_MSG(0x2d2edea9, object.getType());
        HK_ASSERT_NO_MSG(0x25d1e881, object.getType()->getKind() != 0);

        const int newId = m_bundle.numVars();
        m_bundle.setVar(newId, object);
        m_bundleVarHasBeenCopied.pushBack(1);
        HK_ASSERT_NO_MSG(0x2d06ba63, m_bundleVarHasBeenCopied.getSize() == m_bundle.numVars());

        addObjectToTrackingMap(object, 1, 0, newId, -1);
        return newId;
    }


    int newObject(const char* name)
    {
        
        const int typeVersion = m_currentPatchDependencies.getVersion(name);
        hkVersionedRecordType* vrt = m_typeSet.findType(name, typeVersion);
        hkReflect::RecordType* generatedRecord = vrt->getCompletedType()->asRecord();
        HK_ASSERT_NO_MSG(0x213bcbdb, generatedRecord);

        void* data = m_bundle.m_allocator.blockAlloc(generatedRecord->getSizeOf());
        hkString::memSet(data, 0, generatedRecord->getSizeOf());
        hkReflect::Var handle(data, generatedRecord);
        handle.writeDefault();

        HK_ASSERT_NO_MSG(0x42adfd76, generatedRecord->getKind());

        const int newId = m_bundle.numVars();
        hkReflect::Var newVar(data, generatedRecord);
        m_bundle.setVar(newId, newVar);
        m_bundleVarHasBeenCopied.pushBack(1);
        HK_ASSERT_NO_MSG(0x39fa970a, m_bundleVarHasBeenCopied.getSize() == m_bundle.numVars());
        addObjectToTrackingMap(newVar, 1, 0, newId, -1);

        return newId;
    }

    // These perform the mapping from pointers in the source objects to object indices in the versioning objects
    int getVarIndexFromSourceAddress(void* sourceAddress)
    {
        const int sz = m_bundle.numVars();
        hkPointerMap<void*, int>::Iterator it = m_indexFromAddressMap.findOrInsertKey(sourceAddress, sz);
        int index = m_indexFromAddressMap.getValue(it);

        if (index == m_bundle.numVars())
        {
            // Resizes the array and inserts a placeholder. We still need
            // to add it to the tracking map once it is completed
            const int newId = m_bundle.numVars();
            m_bundle.setVar(newId, hkReflect::Var());
            m_bundleVarHasBeenCopied.pushBack(0);
            HK_ASSERT_NO_MSG(0x3685f4b4, m_bundleVarHasBeenCopied.getSize() == m_bundle.numVars());
        }
        return index;
    }

    int getArrayIndexFromSourceAddress(void* sourceAddress)
    {
        const int sz = m_bundle.numExtras();
        hkPointerMap<void*, int>::Iterator it = m_indexFromAddressMap.findOrInsertKey(sourceAddress, sz);
        int index = m_indexFromAddressMap.getValue(it);

        if (index == m_bundle.numExtras())
        {
            // Resizes the array and inserts a placeholder. We still need
            // to add it to the tracking map once it is completed
            m_bundle.setVarN(m_bundle.numExtras(), hkSerialize::VarN::invalid());
            m_bundleExtraHasBeenCopied.pushBack(0);
            HK_ASSERT_NO_MSG(0x2eeadc5e, m_bundleExtraHasBeenCopied.getSize() == m_bundle.numExtras());
        }
        return index;
    }

    hkVersionedRecordType* getVersionedRecord(const hkReflect::Type* sourceType)
    {
        return m_typeSet.versionedTypeFromSourceType(sourceType);
    }

    const void* pointerFromId(int id) 
    {
        // !!!
        return m_bundle.var(id).getAddress();
    }

    int getObjectListIndex(const hkReflect::Type* type)
    {
        while (type->isDecorator())
        {
            type = type->getParent();
        }
        HK_ASSERT_NO_MSG(0xe8db3ca, type);
        hkPointerMap<void*, int>::Iterator mapIterator = m_trackingListIndexFromTypeMap.findOrInsertKey(type, m_trackingList.getSize());
        const int retrievedIndex = m_trackingListIndexFromTypeMap.getValue(mapIterator);
        if (retrievedIndex == m_trackingList.getSize())
        {
            m_trackingList.expandOne();
        }
        return retrievedIndex;
    }

    void addObjectToTrackingMap(hkReflect::Var firstObject, int numObjects, int stride, int varId, int extraId)
    {
        HK_ASSERT_NO_MSG(0x5ce8e6d4, (varId == -1) || (extraId == -1));
        if (numObjects > 0)
        {
            //printf("TRACKING ARRAY[O] at %x (%d) (%d)\n", arrayHandle.getDataPointer(), arrayHandle.getSize(), arrayHandle.getElemSize());
            const hkReflect::Type* subType = firstObject.getType();

            int listIndex = getObjectListIndex(subType);
            TrackedType& trackingItem = m_trackingList[listIndex];
            trackingItem.objectList.expandOne().set(firstObject.getAddress(), numObjects, stride, varId, extraId);

            if (const hkReflect::RecordType* recordType = subType->asRecord())
            {
                {
                    DLOG("Adding {0} x {6} to tracking, size {1}/{2} at {3}, li {4}, ai {5}", firstObject.getType()->getName(), firstObject.getType()->getSizeOf(), stride, firstObject.getAddress(), listIndex, trackingItem.objectList.getSize(), numObjects);
                }

                for( hkReflect::DeclIter<hkReflect::DataFieldDecl> fieldIterator(recordType); fieldIterator.advance(); )
                {
                    const hkReflect::DataFieldDecl f = fieldIterator.current();
                    hkReflect::Var fieldVar(hkAddByteOffset(firstObject.getAddress(), f.getOffset()), f.getType());
                    addObjectToTrackingMap(fieldVar, numObjects, stride, varId, extraId);
                }
            }
        }
    }

    void removeObjectFromTrackingMap(hkReflect::Var firstObject)
    {
        int listIndex = getObjectListIndex(firstObject.getType());
        if (listIndex >= 0)
        {
            TrackedType& trackingItem = m_trackingList[listIndex];
            for (int i = 0; i < trackingItem.objectList.getSize(); ++i)
            {
                if (trackingItem.objectList[i].m_data == firstObject.getAddress())
                {
                    trackingItem.objectList.removeAt(i);
                    return;
                }
            }
            HK_ASSERT(0x28ea7eb4, 0, "Trying to remove a non-tracked object");
        }
    }

    hkDataWorld& getWorld() const
    {
        return const_cast<hkDataWorld&>(m_world);
    }

    struct VersioningArrayImpl : public hkReflect::Detail::ArrayImpl
    {
        HK_DECLARE_CLASS(VersioningArrayImpl, New);

        // Need to support subtype for homogenous arrays?? Or do we care??
        VersioningArrayImpl(CompatPatcher& parent) : m_parent(parent)
        {
        }

        ~VersioningArrayImpl()
        {
        }

        virtual hkResult getValue(const void* arrAddr, const hkReflect::ArrayType* arrType, _Out_ hkReflect::ArrayValue* val) const HK_OVERRIDE
        {
            if (arrAddr)
            {
                const hkUint32 id = *reinterpret_cast<const hkUint32Le*>(arrAddr);
                if (id > 0)
                {
                    if (m_parent.m_bundle.extra(id).getAddress())
                    {
                        VarN o = m_parent.m_bundle.extra(id);
                        *val = hkReflect::ArrayValue(o.getAddress(), o.getCount(), o.getType());
                        return HK_SUCCESS;
                    }
                }
            }
            *val = hkReflect::ArrayValue();
            return HK_SUCCESS;
        }

        virtual hkResult setNumElements(void* arrAddr, const hkReflect::ArrayType* arrType, int newSize) const HK_OVERRIDE
        {
            hkUint32 id = 0;
            if (arrAddr)
            {
                id = *reinterpret_cast<const hkUint32Le*>(arrAddr);
                HK_ASSERT_NO_MSG(0x427b0574, m_parent.m_bundleExtraHasBeenCopied[id]); 
            }
            int oldElementCount = 0;
            void* oldData = HK_NULL;
            if (id)
            {
                m_parent.removeObjectFromTrackingMap(m_parent.m_bundle.extra(id).getFirst());
                oldElementCount = m_parent.m_bundle.extra(id).getCount();
                oldData = m_parent.m_bundle.extra(id).getFirst().getAddress();
            }

            
            HK_ASSERT_NO_MSG(0x63a5b94f, !arrType->getSubType()->asVoid());
            const int elementSize = arrType->getSubType()->getSizeOf();

            void* newData = m_parent.m_bundle.m_allocator.blockAlloc(elementSize * newSize);

            hkString::memSet(hkAddByteOffset(newData, elementSize * oldElementCount), 0, (newSize > oldElementCount) ? (elementSize * (newSize - oldElementCount)) : 0);

            m_parent.addObjectToTrackingMap(hkReflect::Var(newData, arrType->getSubType()), newSize, elementSize, -1, id);

            if (id)
            {
                hkString::memCpy(newData, oldData, (newSize > oldElementCount ? oldElementCount : newSize) * elementSize);

                m_parent.m_bundle.setVarN(id, hkSerialize::VarN::fromArray(newData, arrType->getSubType(), newSize));
            }
            else
            {
                const int newId = m_parent.m_bundle.numExtras();
                m_parent.m_bundle.setVarN(newId, hkSerialize::VarN::fromArray(newData, arrType->getSubType(), newSize));
                m_parent.m_bundleExtraHasBeenCopied.pushBack(1);
                HK_ASSERT_NO_MSG(0x758cd329, m_parent.m_bundleExtraHasBeenCopied.getSize() == m_parent.m_bundle.numExtras());

                *reinterpret_cast<hkUint32Le*>(arrAddr) = newId; // Stash the id here so the array impl can find it
            }

            return HK_SUCCESS;
        }

        virtual AllocResult allocateElements(void* arrAddr, const hkReflect::ArrayType* arrType, hkReflect::QualType elemType, int newSize) const HK_OVERRIDE
        {
            return setNumElements(arrAddr, arrType, newSize).isSuccess() ? ALLOC_SUCCESS : ALLOC_FAILURE;
        }

        CompatPatcher& m_parent;
    };

    struct VersioningPointerImpl : public hkReflect::Detail::PointerImpl
    {
        HK_DECLARE_CLASS(VersioningPointerImpl, New);

        VersioningPointerImpl(CompatPatcher& parent) : m_parent(parent)
        {
        }

        virtual hkResult setValue(void* self, const hkReflect::PointerType* type, const hkReflect::Var& var) const HK_OVERRIDE
        {
            HK_ASSERT_NO_MSG(0x3a40bcac, 0); // Not supported
            return HK_FAILURE;
        }

        virtual hkResult getValue(const void* self, const hkReflect::PointerType* type, _Out_ hkReflect::Var* val) const HK_OVERRIDE
        {
            hkUint32 id = 0;
            if (self)
            {
                id = *reinterpret_cast<const hkUint32Le*>(self);
            }

            if (id)
            {
                if (int annotated = m_parent.m_bundle.getAnnotated(id))
                {
                    *val = m_parent.m_bundle.var(annotated);
                    return HK_SUCCESS;
                }
                else
                {
                    *val = m_parent.m_bundle.var(id);
                    return HK_SUCCESS;
                }
            }
            *val = hkReflect::Var();
            return HK_SUCCESS;
        }

        virtual void* queryInterfaceImpl(const hkReflect::Type* type, const hkReflect::Var& self) const HK_OVERRIDE
        {
            if (type->extendsOrEquals<hkSerialize::Detail::IdFromPointer>())
            {
                if (self)
                {
                    hkUint32 id = *reinterpret_cast<const hkUint32Le*>(self.getAddress());
                    if (id)
                    {
                        return reinterpret_cast<hkSerialize::Detail::IdFromPointer*>(self.getAddress());
                    }
                }
            }
            return HK_NULL;
        }


        CompatPatcher& m_parent;
    };


    HK_INLINE static char* checkInternalOffset(void* object, int offset, const hkReflect::Type* objectType)
    {
        HK_ASSERT_NO_MSG(0x4cf1769e, (offset >= 0) && (offset <= objectType->getSizeOf()));
        return hkAddByteOffset<char>(reinterpret_cast<char*>(object), offset);
    }

    HK_INLINE static void setDefaultForNewMember(void* object, const hkReflect::DataFieldDecl member, const void* value, const hkReflect::Type* objectType)
    {
        HK_ASSERT_NO_MSG(0x54ca4ea0, value);
        const hkReflect::Type* memberType = member.getType();
        void* ptr = hkAddByteOffset<char>(reinterpret_cast<char*>(object), member.getOffset());
        // cannot memcpy directly because the actual type of the default value might be different from the field type
        if (const hkReflect::BoolType* b = memberType->asBool())
        {
            hkReflect::BoolVar dst(ptr, b);
            dst.setValue(*reinterpret_cast<const hkInt64*>(value) != 0);
        }
        else if (const hkReflect::IntType* i = memberType->asInteger())
        {
            hkReflect::IntVar dst(ptr, i);
            dst.setValue(*reinterpret_cast<const hkInt64*>(value));
        }
        else if (const hkReflect::FloatType* f = memberType->asFloat())
        {
            hkReflect::FloatVar dst(ptr, f);

            dst.setValue(*reinterpret_cast<const hkReal*>(value));
        }
        else
        {
            HK_ASSERT_NO_MSG(0x2b1be3b8, memberType->asPointer() || memberType->asString() || (memberType->asArray() && (memberType->asArray()->getFixedCount() > 0)));
            hkString::memCpy(ptr, value, member.getType()->getSizeOf());
        }
    }

    int getIndexForNoteOnPointer(hkReflect::PointerVar ptr)
    {
        const hkSerialize::Bundle* toSearch;
        if (ptr.getImpl() == m_pointerImpl)
        {
            // Versioned object, look in current bundle
            toSearch = &m_bundle;
        }
        else
        {
            toSearch = m_bundleIn;
            // Original object, look in original bundle.
            if (ptr.getType()->getTypeWorld() == this)
            {
                // We cannot pass the Var as it is because the Impl does not correspond to the original one: unwrap it
                auto chainedImpl = static_cast<const ForwardingPointerImpl*>(ptr.getImpl())->m_chainedImpl;
                ptr = hkReflect::PointerVar(ptr.getAddress(), ptr.getType(), chainedImpl);
            }
        }

        if (hkReflect::Var noteVar = toSearch->getNoteOnPointer(ptr))
        {
            return getVarIndexFromSourceAddress(noteVar.getAddress());
        }
        return 0;
    }

    class NewVersioningInterpreter : public hkTypeVm::FastCopyInterpreter
    {
    public:
        HK_INLINE hkResult execPointer(hkReflect::PointerVar dst, hkReflect::PointerVar src, int dstStride, int srcStride, int numElems)
        {
            for (int i = 0; i < numElems; i++, src = incrementAddress(src, srcStride), dst = incrementAddress(dst, dstStride))
            {
                // Copy the old pointer value to the new object
                hkReflect::Var oldVar = src.getValue();

                //DLOG("PTR {0} at {1} (from {2})\n", oldVar, dst, src);

                void* addr = oldVar.getAddress();
                if (int noteIndex = m_parent->getIndexForNoteOnPointer(src))
                {
                    *reinterpret_cast<hkUint32Le*>(dst.getAddress()) = noteIndex; // Stash the value here so that the pointer impl can see it and get to the object
                }
                else if (addr)
                {
                    int newIndex = m_parent->getVarIndexFromSourceAddress(addr);
                    *reinterpret_cast<hkUint32Le*>(dst.getAddress()) = newIndex; // Stash the value here so that the pointer impl can see it and get to the object
                }
                else
                {
                    *reinterpret_cast<hkUint32Le*>(dst.getAddress()) = 0;
                }
            }
            return HK_SUCCESS;
        }

        HK_INLINE hkResult execCString(hkReflect::StringVar dst, hkReflect::StringVar src)
        {
            return dst.setValue(src.getValue());
        }


        HK_INLINE hkResult execArray(hkReflect::ArrayVar dst, hkReflect::ArrayVar src)
        {
            void* addr = src.getDataPointer();
            const int newId = m_parent->getArrayIndexFromSourceAddress(addr);
            //DLOG("Array reloc {0} -> {1} at {2} (from {3})", addr, newId, dst, src);

            *reinterpret_cast<hkUint32Le*>(dst.getAddress()) = newId;// Stash the value here so that the array impl can see it and get to the object
            return HK_SUCCESS;
        }

        CompatPatcher* m_parent;
    };

    void findAllPatchedObjects(hkVersionedRecordType* patchingObject, hkArray<TrackedType::ObjectRef>::Temp& allPatchedObjects)
    {
        // Here we copy any objects that are being versioned
        int listIndex = -1;

        const hkReflect::Type* sourceType = patchingObject->getSourceType();
        const hkReflect::Type* convertedType = patchingObject->getCompletedType();
        if (!m_trackingListIndexFromTypeMap.get(convertedType, &listIndex).isSuccess())
        {
            // This is new, we need to scan and discover any objects we need to copy
            listIndex = m_trackingList.getSize();
            m_trackingListIndexFromTypeMap.insert(convertedType, listIndex);
            m_trackingList.expandOne();

            // Now we only copy the input objects that we are ACTUALLY PATCHING
            // BUT we could be patching a member of a larger class. Need to do the tracking thing earlier...

            for (int i = 0; i < m_bundle.numVars(); i++)
            {
                hkReflect::Var thisVar = m_bundle.var(i);
                // Source Type was set as a direct copy of the input vars so this is safe
                if (thisVar.isValid() && (thisVar.getType().get() == sourceType))
                {
                    // May not be a record
                    // These are not real objects, don't need a type impl allocate
                    const int sizeToAllocate = convertedType->getSizeOf();
                    void* dstObjectPtr = m_bundle.m_allocator.blockAlloc(sizeToAllocate);
                    hkString::memSet(dstObjectPtr, 0, sizeToAllocate);

                    const hkTypeVm::Program* program = m_compiler.compile(sourceType, convertedType);
                    HK_ASSERT_NO_MSG(0x6977d65, program); // Compile failed!
                    NewVersioningInterpreter newMachine;
                    newMachine.m_parent = this;

                    void* dstPtr = dstObjectPtr;
                    const void* srcPtr = thisVar.getAddress();

                    HK_VERIFY(0x2589303, hkTypeVm::FastCopyInterpreter::execN(newMachine, program,
                        dstPtr, sizeToAllocate,
                        srcPtr, sourceType->getSizeOf(),
                        sizeToAllocate, 0, 1).isSuccess(), "TypeVM exec failed" );

                    addObjectToTrackingMap(hkReflect::Var(dstObjectPtr, convertedType), 1, 0, i, -1);
                    HK_ASSERT_NO_MSG(0x59c762dd, m_bundleVarHasBeenCopied[i] == 0);
                    m_bundleVarHasBeenCopied[i] = 1;

                    m_bundle.setVar(i, hkReflect::Var(dstPtr, convertedType));
                }
            }

            for (int i = 0; i < m_bundle.numExtras(); i++)
            {
                hkReflect::Var thisVar = m_bundle.extra(i).getFirst();
                const int numElements = m_bundle.extra(i).getCount();
                // Source Type was set as a direct copy of the input vars so this is safe
                if (thisVar.isValid() && (thisVar.getType().get() == sourceType))
                {
                    // May not be a record
                    // These are not real objects, don't need a type impl allocate
                    const int sizeToAllocate = patchingObject->getCompletedType()->getSizeOf() * numElements;
                    void* dstObjectPtr = m_bundle.m_allocator.blockAlloc(sizeToAllocate);
                    hkString::memSet(dstObjectPtr, 0, sizeToAllocate);

                    const hkTypeVm::Program* program = m_compiler.compile(sourceType, convertedType);
                    HK_ASSERT_NO_MSG(0x1221876f, program); // Compile failed!
                    NewVersioningInterpreter newMachine;
                    newMachine.m_parent = this;

                    void* dstPtr = dstObjectPtr;
                    const void* srcPtr = thisVar.getAddress();

                    if (numElements)
                    {
                        hkTypeVm::FastCopyInterpreter::execN(newMachine, program,
                            dstPtr, sizeToAllocate,
                            srcPtr, sourceType->getSizeOf() * numElements,
                            convertedType->getSizeOf(), sourceType->getSizeOf(), numElements);
                    }

                    addObjectToTrackingMap(hkReflect::Var(dstObjectPtr, convertedType), numElements, convertedType->getSizeOf(), -1, i);
                    HK_ASSERT_NO_MSG(0x4d96eef4, m_bundleExtraHasBeenCopied[i] == 0);
                    m_bundleExtraHasBeenCopied[i] = 1;

                    m_bundle.setVarN(i, hkSerialize::VarN::fromArray(dstPtr, convertedType, numElements));
                }
            }
        }

        if (listIndex >= 0)
        {
            const TrackedType& trackingItem = m_trackingList[listIndex];

            // reserve memory for all the objects
            int totalElems = trackingItem.objectList.getSize();
            for (int arrayIndex = 0; arrayIndex < trackingItem.objectList.getSize(); arrayIndex++)
            {
                totalElems += trackingItem.objectList[arrayIndex].m_numElems;
            }
            allPatchedObjects.reserve(totalElems);

            // add single objects
            for (int objectIndex = 0; objectIndex < trackingItem.objectList.getSize(); objectIndex++)
            {
                void* arrayPtr = trackingItem.objectList[objectIndex].m_data;
                const int arraySize = trackingItem.objectList[objectIndex].m_numElems;
                const int arrayStride = trackingItem.objectList[objectIndex].m_stride;
                for (int i = 0; i < arraySize; i++, arrayPtr = hkAddByteOffset(arrayPtr, arrayStride))
                {
                    void* thisObject = arrayPtr;
                    hkArray<BackLinkInfo::Link> links;
                    getAllBacklinks(trackingItem.objectList[objectIndex].m_id, links);
                    for (int j = 0; j < links.getSize(); j++)
                    {
                        if (links[j].m_assignedTo == hkReflect::Var(thisObject, patchingObject->getCompletedType()))
                        {
                            thisObject = links[j].m_copiedFrom.getAddress();
                        }
                    }

                    allPatchedObjects.expandOne().set(thisObject, 1, 0, trackingItem.objectList[objectIndex].m_id);
                }
                // In theory this could happen but it seems unlikely. Useful for debugging
                //HK_ASSERT_NO_MSG(0x11183e93, hkUlong(allPatchedObjects.back().m_data) != 0xd0d0d0d0d0d0d0d0);
            }
        }
    }

    hkReflect::Var getBackLink(const hkReflect::Var& object, hkDataObject::VarOrExtra id)
    {
        const int linkListIndex = m_linkIndexFromSrcId.getWithDefault(id, -1);
        if (linkListIndex >= 0)
        {
            for (int linkIndex = 0; linkIndex < m_backlinks[linkListIndex].m_links.getSize(); linkIndex++)
            {
                if (m_backlinks[linkListIndex].m_links[linkIndex].m_assignedTo == object)
                {
                    return m_backlinks[linkListIndex].m_links[linkIndex].m_copiedFrom;
                }
            }
        }

        return hkReflect::Var();
    }

    // This is the information stored when we create a backlink from a patch function (a.b = c)
    struct BackLinkInfo
    {
        struct Link
        {
            Link() {}
            Link(const hkReflect::Var& to, const hkReflect::Var& from) : m_assignedTo(to), m_copiedFrom(from) {}

            hkReflect::Var m_assignedTo;
            hkReflect::Var m_copiedFrom;
        };
        hkArray<Link> m_links;
    };

    hkArray<BackLinkInfo> m_backlinks;
    hkHashMap<hkDataObject::VarOrExtra, int> m_linkIndexFromSrcId;


    void setBackLink(const hkReflect::Var& object, hkReflect::Var newAddress, hkDataObject::VarOrExtra thisId, hkDataObject::VarOrExtra assigningToId)
    {
        int index;
        if (m_linkIndexFromSrcId.get(thisId, &index).isSuccess())
        {
            // We need to set both of these so that pointers to the new type will still match the reference to the old one
            // We need the type because the object and its first member are different and could both have links
            m_backlinks[index].m_links.pushBack(BackLinkInfo::Link(object, newAddress));
            m_backlinks[index].m_links.pushBack(BackLinkInfo::Link(hkReflect::Var(object.getAddress(), newAddress.getType()), newAddress));
        }
        else
        {
            const int newLinkIndex = m_backlinks.getSize();
            m_backlinks.expandOne().m_links.pushBack(BackLinkInfo::Link(object, newAddress));
            m_backlinks.back().m_links.pushBack(BackLinkInfo::Link(hkReflect::Var(object.getAddress(), newAddress.getType()), newAddress));
            m_linkIndexFromSrcId.insert(thisId, newLinkIndex);
        }
    }

    void getAllBacklinks(hkArray<BackLinkInfo::Link>& valuesOut)
    {
        for (int index = 0; index < m_backlinks.getSize(); index++)
        {
            valuesOut.append(m_backlinks[index].m_links);
        }
    }

    void getAllBacklinks(hkDataObject::VarOrExtra thisId, hkArray<BackLinkInfo::Link>& valuesOut)
    {
        int index;
        if (m_linkIndexFromSrcId.get(thisId, &index).isSuccess())
        {
            valuesOut.append(m_backlinks[index].m_links);
        }
    }

    
    
    class ContainsRecordVisitor : public hkReflect::TypeVisitor < ContainsRecordVisitor, bool >
    {
    public:
        bool visit(const hkReflect::RecordType* r)
        {
            return true;
        }

        bool visit(const hkReflect::ArrayType* a)
        {
            if (a->getSubType())
            {
                return dispatch(a->getSubType());
            }
            return false;
        }

        bool visit(const hkReflect::VoidType* r) { return false; }
        bool visit(const hkReflect::ValueType* r) { return false; }
        bool visit(const hkReflect::PointerType* r) { return false; }
    };

    static bool s_containsRecord(const hkReflect::Type* type)
    {
        return ContainsRecordVisitor().dispatch(type);
    }

    // This is an interval tree of the addresses the backlinks cover
    // This allows us to work out the correct order to apply them in
    // so that nested links are resolved correctly
    struct Link
    {
        Link() : m_completelyBelow(0), m_completelyAbove(0) {}
        hkUlong m_start;
        hkUlong m_end;
        hkArray<int> m_linksToDoBeforeThisOne;
        Link* m_completelyBelow;
        Link* m_completelyAbove;
        hkArray<Link*> m_overlapping;

        void insertIntoTree(Link* link)
        {
            Link* self = this;
            while(true)
            {
                int diff = self->compareTo(link);
                if (diff == 0)
                {
                    self->m_overlapping.pushBack(link);
                    return;
                }
                else if (diff < 0)
                {
                    if (self->m_completelyBelow)
                    {
                        self = self->m_completelyBelow;
                    }
                    else
                    {
                        self->m_completelyBelow = link;
                        return;
                    }
                }
                else
                {
                    if (self->m_completelyAbove)
                    {
                        self = self->m_completelyAbove;
                    }
                    else
                    {
                        self->m_completelyAbove = link;
                        return;
                    }
                }
            }
        }

        int compareTo(Link* b) const
        {
            if (b->m_end < m_start) { return -1; } // b completely below this
            if (b->m_start > m_end) { return 1; } // b completely above this
            return 0; // overlapping
        }

        void addDependencyForAllThatOverlap(hkUlong address, int indexToAdd)
        {
            Link* self = this;
            bool resolved = false;
            while (!resolved)
            {
                if (address < self->m_start)
            {
                    if (self->m_completelyBelow)
                {
                        self = self->m_completelyBelow;
                }
                    else
                    {
                        resolved = true;
            }
                }
                else if (address > self->m_end)
            {
                    if (self->m_completelyAbove)
                {
                        self = self->m_completelyAbove;
                }
                    else
                    {
                        resolved = true;
            }
                }
            else
            {
                    resolved = true;
                // Overlapping, need to test all in the array
                // First, check ourself
                    self->addDependencyIfRequired(address, indexToAdd);
                    for (int i = 0; i < self->m_overlapping.getSize(); i++)
                {
                        self->m_overlapping[i]->addDependencyIfRequired(address, indexToAdd);
                    }
                }
            }
        }

        void addDependencyIfRequired(hkUlong address, int indexToAdd)
        {
            if ((m_start <= address) && (m_end >= address))
            {
                m_linksToDoBeforeThisOne.pushBack(indexToAdd);
            }
        }

    };

    // These are all of the backlinks within an object. Some of them may be nested, if so we need to do the interior ones first: AABBBAA -- BCB -- C, need to copy C->B, then B->A
    void applyBacklinks(const hkArray<BackLinkInfo::Link>& linksIn)
    {
        hkArray<Link> links(linksIn.getSize());

        for (int linkInIndex = 0; linkInIndex < linksIn.getSize(); linkInIndex++)
        {
            const BackLinkInfo::Link& thisLinkIn = linksIn[linkInIndex];
            links[linkInIndex].m_start = (hkUlong)thisLinkIn.m_copiedFrom.getAddress();
            links[linkInIndex].m_end = links[linkInIndex].m_start + thisLinkIn.m_copiedFrom.getType()->getSizeOf() - 1;
        }

        for (int i = 1; i < links.getSize(); i++)
        {
            links[0].insertIntoTree(&links[i]);
        }

        for (int linkInIndex = 0; linkInIndex < links.getSize(); linkInIndex++)
        {
            const BackLinkInfo::Link& thisLinkIn = linksIn[linkInIndex];

            const hkUlong thisLinkTo = (hkUlong)thisLinkIn.m_assignedTo.getAddress();

            links[0].addDependencyForAllThatOverlap(thisLinkTo, linkInIndex);
        }

        // In almost all cases there are no overlaps and this will
        // just run straight through all the links
        for (int i = 0; i < links.getSize(); i++)
        {
            if ((links[i].m_start > 0) && (links[i].m_linksToDoBeforeThisOne.isEmpty()))
            {
                const hkReflect::Var& rec = linksIn[i].m_assignedTo;
                const hkReflect::Var& objectReference = linksIn[i].m_copiedFrom;
                const void* srcAddress = objectReference.getAddress();
                void* dstAddress = rec.getAddress();

                const hkReflect::Type* srcType = objectReference.getType();
                const hkReflect::Type* dstType = rec.getType();

                const int srcSize = srcType->getSizeOf();
                const int dstSize = dstType->getSizeOf();

                DLOG("RESOLVE Link Field of {0} from {1} to {2}", rec, objectReference, rec.getAddress());

                const hkTypeVm::Program* program = m_compiler.compile(srcType, dstType);
                HK_ASSERT_NO_MSG(0x64416a6, program); // Compile failed!
                NewVersioningInterpreter newMachine;
                newMachine.m_parent = this;
                {
                    hkTypeVm::FastCopyInterpreter::execN(newMachine, program,
                        dstAddress, dstSize,
                        srcAddress, srcSize,
                        0, 0, 1);
                }

                links[i].m_start = 0;
                for (int j = 0; j < links.getSize(); j++)
                {
                    for (int k = 0; k < links[j].m_linksToDoBeforeThisOne.getSize(); k++)
                    {
                        if (links[j].m_linksToDoBeforeThisOne[k] == i)
                        {
                            links[j].m_linksToDoBeforeThisOne.removeAt(k);
                        }
                    }
                }
                // Go back to the start
                i = 0;
            }
        }
    }

    void updateBacklinks(const hkSerialize::VarN& oldObjects, const hkSerialize::VarN& newObjects)
    {
        hkUlong oldStart = hkUlong(oldObjects.getFirst().getAddress());
        hkUlong oldEnd = oldStart + oldObjects.getCount() * oldObjects.getType()->getSizeOf();
        hkUlong newStart = hkUlong(newObjects.getFirst().getAddress());

        for (int index = 0; index < m_backlinks.getSize(); index++)
        {
            for (int linkIndex = 0; linkIndex < m_backlinks[index].m_links.getSize(); linkIndex++)
            {
                // Assigned to, copied from, id
                BackLinkInfo::Link& thisLink = m_backlinks[index].m_links[linkIndex];

                // Need to update both assigned to and copied from
                const hkUlong thisLinkTo = (hkUlong)thisLink.m_assignedTo.getAddress();
                const hkUlong thisLinkFrom = (hkUlong)thisLink.m_copiedFrom.getAddress();

                if ((thisLinkTo >= oldStart) && (thisLinkTo < oldEnd))
                {
                    thisLink.m_assignedTo.setAddress(hkAddByteOffset(thisLink.m_assignedTo.getAddress(), newStart - oldStart));
                }
                if ((thisLinkFrom >= oldStart) && (thisLinkFrom < oldEnd))
                {
                    thisLink.m_copiedFrom.setAddress(hkAddByteOffset(thisLink.m_copiedFrom.getAddress(), newStart - oldStart));
                }
            }
        }
    }

    void copyBackLinkedObjects()
    {
        hkArray<BackLinkInfo::Link> links;
        getAllBacklinks(links);
        applyBacklinks(links);
    }

    
    TypeId recursivelyAddTypes(hkHashMap<const hkReflect::Type*, TypeId>& tidFromType, const hkReflect::Type* t)
    {
        if (t)
        {
            while (t->isDecorator())
            {
                t = t->getParent();
            }
        }

        hkHashMap<const hkReflect::Type*, TypeId>::Iterator it = tidFromType.findOrInsertKey(t,-1);
        TypeId tid = tidFromType.getValue(it);
        if(tid == -1)
        {
            tid = m_bundle.types().getSize();
            m_bundle.setType(tid, t);
            tidFromType.setValue(it, tid);

            if (const hkReflect::RecordType* rt = t->asRecord())
            {
                for (int i = 0; i < rt->getNumFields(); i++)
                {
                    recursivelyAddTypes(tidFromType, rt->getField(i).getType()->getParent());
                }
            }
            else if (const hkReflect::ArrayType* at = t->asArray())
            {
                recursivelyAddTypes(tidFromType, at->getSubType());
            }
            else if (const hkReflect::PointerType* pt = t->asPointer())
            {
                recursivelyAddTypes(tidFromType, pt->getSubType());
            }
        }
        return tid;
    }

    class ForwardingPointerImpl : public hkReflect::Detail::PointerImpl
    {
    public:
        ForwardingPointerImpl(const hkReflect::Detail::PointerImpl* chainedImpl, CompatPatcher* parent) : m_chainedImpl(chainedImpl), m_parent(parent) {}

        /// Sets the pointer value from the given var.
        /// Returns false on failure (usually type incompatibility)
        virtual hkResult setValue(void* self, const hkReflect::PointerType* type, const hkReflect::Var& var) const HK_OVERRIDE
        {
            // Disallowed for now
            HK_ASSERT_NO_MSG(0xcd3e62f, 0);
            return HK_FAILURE;
        }
        /// Gets the pointed to object and type.
        virtual hkResult getValue(const void* self, const hkReflect::PointerType* type, _Out_ hkReflect::Var* val) const HK_OVERRIDE
        {
            hkReflect::Var chainedVal;
            m_chainedImpl->getValue(self, type, &chainedVal);

            if (void* addr = chainedVal.getAddress())
            {
                int index = m_parent->m_indexFromAddressMap.getWithDefault(addr, -1);

                m_parent->makeVarVersioningCopy(index);

                *val = m_parent->m_bundle.var(index);
                return HK_SUCCESS;
            }

            *val = hkReflect::Var();
            return HK_SUCCESS;

        }

        virtual void* queryInterfaceImpl(const hkReflect::Type* type, const hkReflect::Var& self) const HK_OVERRIDE
        {
            return m_chainedImpl->queryInterfaceImpl(type, self);
        }

        const hkReflect::Detail::PointerImpl* m_chainedImpl;
        CompatPatcher* m_parent;
    };

    class ForwardingArrayImpl : public hkReflect::Detail::ArrayImpl
    {
    public:
        ForwardingArrayImpl(const hkReflect::Detail::ArrayImpl* chainedImpl, CompatPatcher* parent) : m_chainedImpl(chainedImpl), m_parent(parent) {}

        virtual hkReflect::ContainerValue getContainerValue(const void* addr, const hkReflect::ContainerType* type) const HK_OVERRIDE
        {
            
            return m_chainedImpl->getContainerValue(addr, type);
        }
        ///
        virtual hkResult getValue(const void* arrAddr, const hkReflect::ArrayType* arrType, _Out_ hkReflect::ArrayValue* val) const HK_OVERRIDE
        {
            hkReflect::ArrayValue chainedVal;
            m_chainedImpl->getValue(arrAddr, arrType, &chainedVal);

            if (arrType->getFixedCount() > 0)
            {
                *val = chainedVal;
                return HK_SUCCESS;
            }
            else
            {
                int index = m_parent->m_indexFromAddressMap.getWithDefault(chainedVal.getAddress(), -1);

                m_parent->makeArrayVersioningCopy(index);
                hkSerialize::VarN newVal = m_parent->m_bundle.extra(index);
                *val = hkReflect::ArrayValue(newVal.getAddress(), newVal.getCount(), newVal.getType());
                return HK_SUCCESS;
            }
        }

        /// Resize the array
        virtual hkResult setNumElements(void* arrAddr, const hkReflect::ArrayType* arrType, int nelem) const HK_OVERRIDE
        {
            return m_chainedImpl->setNumElements(arrAddr, arrType, nelem);
        }

        /// Set the element type. Not all arrays support this operation.
        virtual AllocResult allocateElements(void* arrAddr, const hkReflect::ArrayType* arrType, hkReflect::QualType elemType, int nelem) const HK_OVERRIDE
        {
            return m_chainedImpl->allocateElements(arrAddr, arrType, elemType, nelem);
        }
        /// Default implementation replaces the elements in arrInOut with the elements in source if
        /// numToDel and numToInserts are equal, otherwise it returns HK_FAILURE.
        virtual hkResult spliceInto(void* arrAddr, const hkReflect::ArrayType* arrType, int index, int numToDel, const hkReflect::ArrayValue& val) const HK_OVERRIDE
        {
            return m_chainedImpl->spliceInto(arrAddr, arrType, index, numToDel, val);
        }

        const hkReflect::Detail::ArrayImpl* m_chainedImpl;
        CompatPatcher* m_parent;
    };

    hkReflect::Type* chainingCopyType(const hkReflect::Type* srcType)
    {
        if (hkReflect::Type* lookedUpType = m_chainedCopiedTypesFromSrc.getWithDefault(srcType, HK_NULL))
        {
            return lookedUpType;
        }

        // Just a dumb copy for now
        hkReflect::TypeBuilder tb;
        tb.beginShallowClone(srcType);
        tb.setTypeWorld(this);

        // Hide the native exactTypeOf. All Vars to these objects should be created exact anyway.
        tb.setItem<hkReflect::Opt::EXACT_TYPE>(HK_NULL);

        if (const hkReflect::PointerType* pt = srcType->asPointer())
        {
            ForwardingPointerImpl* newImpl = m_forwardingTypesAllocator.create<ForwardingPointerImpl>(pt->getImpl(), this);
            tb.setItem<hkReflect::Opt::IMPL>(newImpl);
            if (pt->getSubType())
            {
                tb.setItem<hkReflect::Opt::SUBTYPE>(chainingCopyType(pt->getSubType()));
            }
        }
        else if (const hkReflect::ArrayType* at = srcType->asArray())
        {
            ForwardingArrayImpl* newImpl = m_forwardingTypesAllocator.create<ForwardingArrayImpl>(at->getImpl(), this);
            tb.setItem<hkReflect::Opt::IMPL>(newImpl);
            if (at->getSubType())
            {
                tb.setItem<hkReflect::Opt::SUBTYPE>(chainingCopyType(at->getSubType()));
            }
        }



        if (const hkReflect::Type* pt = srcType->getParent())
        {
            tb.overrideParent(chainingCopyType(pt));
        }

        hkReflect::Type* ret = tb.allocate(&m_forwardingTypesAllocator);
        m_chainedCopiedTypesFromSrc.insert(srcType, ret);

        if (const hkReflect::RecordType* rt = srcType->asRecord())
        {
            const hkReflect::Detail::DeclsArray* srcFields = hkReflect::TypeDetail::localGetOptional<hkReflect::Opt::DECLS>(srcType);
            if (srcFields)
            {
                hkArray<const hkReflect::Type*> dstFields;
                for (int i = 0; i < srcFields->getNumFields(); i++)
                {
                    const hkReflect::FieldDecl srcField = srcFields->getField(i);
                    // size/align and offsets are computed after
                    if (srcField.getName() && srcField.isSerializable())
                    {
                        const hkReflect::Type* baseType = chainingCopyType(srcField.getType()->getParent()); // skip the field itself
                        const char* newName = hkString::strDup(srcField.getName(), m_forwardingTypesAllocator);
                        hkReflect::TypeBuilder fieldBuilder;
                        fieldBuilder.beginDerived(baseType);
                        fieldBuilder.setField(newName, srcField.getOffset(), 0);
                        fieldBuilder.setItem<hkReflect::Opt::DECL_CONTEXT>(ret);
                        hkReflect::Type* fieldType = fieldBuilder.allocate(&m_forwardingTypesAllocator);
                        dstFields.pushBack(fieldType);
                    }
                }
                hkReflect::Detail::DeclsArray* fieldsArray = hkReflect::Detail::DeclsArray::create(dstFields.begin(), dstFields.getSize(), 0, m_forwardingTypesAllocator);
                hkReflect::TypeDetail::localSetOptional<hkReflect::Opt::DECLS>(ret, fieldsArray);
            }
        }

        return ret;
    }

    void stripVarFromBundle(int index)
    {
        m_strippedVars.pushBack(m_bundle.var(index));
        DLOG("Stripping object of type {0} @ {1}", m_bundle.var(index).getType()->getName(), index);
        m_bundleVarHasBeenCopied[index] = -m_strippedVars.getSize();
        m_bundle.setVar(index, hkReflect::Var());
    }

    void stripExtraFromBundle(int index)
    {
        m_strippedExtras.pushBack(m_bundle.extra(index));
        DLOG("Stripping object block of type {0} @ {1}", m_bundle.extra(index).getType()->getName(), index);
        m_bundleExtraHasBeenCopied[index] = -m_strippedExtras.getSize();
        m_bundle.setVarN(index, hkSerialize::VarN::fromArray(HK_NULL, HK_NULL, 0));
    }

    hkResult makeVarVersioningCopy(int varIndex)
    {
        if (m_bundleVarHasBeenCopied[varIndex] || m_finishedModifyingObjects)
        {
            // This var was already copied
            return HK_FAILURE;
        }

        hkReflect::Var thisVar = m_bundle.var(varIndex);
        const hkReflect::Type* sourceType = thisVar.getType();
        const hkVersionedRecordType* vrt = m_typeSet.createType(HK_NULL, 0, sourceType);

        if (!vrt)
        {
            stripVarFromBundle(varIndex);
            return HK_FAILURE;
        }

        const hkReflect::Type* convertedType = vrt->getCompletedType();

        int listIndex = -1;
        if (!m_trackingListIndexFromTypeMap.get(convertedType, &listIndex).isSuccess())
        {
            // This is new, we need to scan and discover any objects we need to copy
            listIndex = m_trackingList.getSize();
            m_trackingListIndexFromTypeMap.insert(convertedType, listIndex);
            m_trackingList.expandOne();
        }

        const hkTypeVm::Program* program = m_compiler.compile(sourceType, convertedType);
        if (!program)
        {
            stripVarFromBundle(varIndex);
            return HK_FAILURE;
        }

        const int sizeToAllocate = convertedType->getSizeOf();
        void* dstObjectPtr = m_bundle.m_allocator.blockAlloc(sizeToAllocate);
        hkString::memSet(dstObjectPtr, 0, sizeToAllocate);

        NewVersioningInterpreter newMachine;
        newMachine.m_parent = this;

        void* dstPtr = dstObjectPtr;
        const void* srcPtr = thisVar.getAddress();

        {
            hkTypeVm::FastCopyInterpreter::execN(newMachine, program,
                dstPtr, sizeToAllocate,
                srcPtr, sourceType->getSizeOf(),
                sizeToAllocate, 0, 1);
        }

        addObjectToTrackingMap(hkReflect::Var(dstObjectPtr, convertedType), 1, 0, varIndex, -1);
        m_bundleVarHasBeenCopied[varIndex] = 1;

        m_bundle.setVar(varIndex, hkReflect::Var(dstPtr, convertedType));
        m_indexFromAddressMap.insert(dstPtr, varIndex); // We may see the new address while walking objects or pointers

        return HK_SUCCESS;
    }

    hkResult makeArrayVersioningCopy(int extraIndex)
    {
        if (m_bundleExtraHasBeenCopied[extraIndex] || m_finishedModifyingObjects)
        {
            return HK_FAILURE;
        }

        hkReflect::Var thisVar = m_bundle.extra(extraIndex).getFirst();
        const int numElements = m_bundle.extra(extraIndex).getCount();

        if (const hkReflect::Type* sourceType = thisVar.getType())
        {
            const hkReflect::Type* convertedTypeOut = sourceType; // We just keep simple types
            const hkReflect::Type** currentWritingType = &convertedTypeOut;

            while(sourceType->isDecorator())
            {
                hkReflect::TypeBuilder tb;
                tb.beginShallowClone(sourceType);

                hkReflect::Type* allocatedType;
                if (tb.allocate(&m_typeSet.getTypesAllocator(), &allocatedType).isFailure())
                {
                    Log_Error("Failed to allocate type from builder");
                    return HK_FAILURE;
                }

                *currentWritingType = allocatedType;
                currentWritingType =(const hkReflect::Type**) &(((hkUlong*)(allocatedType))[1]);
                sourceType = sourceType->getParent();
            }

            if (const hkReflect::RecordType* thisVarRecordType = sourceType->asRecord())
            {
                if (!thisVarRecordType->isDynamicType())
                {
                    const hkVersionedRecordType* vrt = m_typeSet.createType(HK_NULL, 0, sourceType);

                    if (!vrt)
                    {
                        stripExtraFromBundle(extraIndex);
                        return HK_FAILURE;
                    }

                    *currentWritingType = vrt->getCompletedType();
                }
            }
            else if (const hkReflect::PointerType* pt = sourceType->asPointer())
            {
                
                // We'll identify the pointer type by its name, so skip all typedefs
                //while (pt->isTypedef())
                //{
                //  pt = pt->getParent()->asPointer();
                //}

                hkReflect::BasicPointerType* allocated = m_typeSet.allocateFromTypes<hkReflect::BasicPointerType>();

                // The name of the pointer type needs to be preserved in order to find the native element type
                // when cloning a variant array to native - in this case, the element type cannot be inferred
                // from the array type.
                const hkReflect::Type* subType = pt->getSubType();
                const char* pointerTypeName = pt->getName();

                if (const hkReflect::RecordType* subRecordType = subType->asRecord())
                {
                    if (!subRecordType->isDynamicType())
                    {
                        if (const hkVersionedRecordType* vrt = m_typeSet.createType(HK_NULL, 0, subType))
                        {
                            subType = vrt->getCompletedType();
                        }
                    }
                }

                new((void*)allocated) hkReflect::BasicPointerType(subType, pointerTypeName); // Here we were passing through an unmodified type!!

                allocated->m_impl = m_pointerImpl;
                *currentWritingType = allocated;
            }
            else if (const hkReflect::StringType* st = sourceType->asString())
            {
                hkReflect::BasicStringType* allocated = m_typeSet.allocateFromTypes<hkReflect::BasicStringType>();
                const hkReflect::Format::StringValue stringFormat = st->getFormat();
                new((void*)allocated) hkReflect::BasicStringType(hkSizeOf(char*), (unsigned int)HK_ALIGN_OF(char*), &m_stringImpl, sourceType->getName(), stringFormat.isImmutable() );
                *currentWritingType = allocated;
            }

            if (!convertedTypeOut)
            {
                stripExtraFromBundle(extraIndex);
                return HK_FAILURE;
            }

            {
                int listIndex = -1;
                if (!m_trackingListIndexFromTypeMap.get(convertedTypeOut, &listIndex).isSuccess())
                {
                    // This is new, we need to scan and discover any objects we need to copy
                    listIndex = m_trackingList.getSize();
                    m_trackingListIndexFromTypeMap.insert(convertedTypeOut, listIndex);
                    m_trackingList.expandOne();
                }

                const hkTypeVm::Program* program = m_compiler.compile(sourceType, convertedTypeOut);
                if (!program)
                {
                    stripExtraFromBundle(extraIndex);
                    return HK_FAILURE;
                }

                const int sizeToAllocate = convertedTypeOut->getSizeOf() * numElements;
                void* dstObjectPtr = m_bundle.m_allocator.blockAlloc(sizeToAllocate);
                hkString::memSet(dstObjectPtr, 0, sizeToAllocate);

                NewVersioningInterpreter newMachine;
                newMachine.m_parent = this;

                void* dstPtr = dstObjectPtr;
                const void* srcPtr = thisVar.getAddress();

                if (numElements)
                {
                    hkTypeVm::FastCopyInterpreter::execN(newMachine, program,
                        dstPtr, sizeToAllocate,
                        srcPtr, sourceType->getSizeOf() * numElements,
                        convertedTypeOut->getSizeOf(), sourceType->getSizeOf(), numElements);
                }

                addObjectToTrackingMap(hkReflect::Var(dstObjectPtr, convertedTypeOut), numElements, convertedTypeOut->getSizeOf(), -1, extraIndex);
                HK_ASSERT_NO_MSG(0x1bb9d84d, m_bundleExtraHasBeenCopied[extraIndex] == 0);
                m_bundleExtraHasBeenCopied[extraIndex] = 1;

                m_bundle.setVarN(extraIndex, hkSerialize::VarN::fromArray(dstPtr, convertedTypeOut, numElements));
                m_indexFromAddressMap.insert(dstPtr, extraIndex); // We may see the new address while walking objects or pointers
            }
        }
        return HK_SUCCESS;
    }

    const hkReflect::Type* adjustReflectTypePointer(const hkReflect::Type* typePointer)
    {
        if (hkReflect::Decl asDecl = hkReflect::QualType(typePointer))
        {
            // Decl
            const hkReflect::Type* origContext = asDecl.getDeclContext();
            if (const hkReflect::Type* adjContext = adjustReflectTypePointer(origContext))
            {
                
                return adjContext->findDecl(asDecl.getName(), false).getType();
            }
            else
            {
                return HK_NULL;
            }
        }
        else if (bool patchesNeeded = m_typeSet.anyPatchesNeededFor(typePointer))
        {
            // Normal type
            if (typePointer->asRecord())
            {
                DLOG("Patching type pointer to {0}", typePointer->getName());
                const hkVersionedRecordType* vrt = m_typeSet.createType(HK_NULL, 0, typePointer);

                if (vrt)
                {
                    const hkReflect::Type* convertedType = vrt->getCompletedType();
                    return convertedType;
                }
                else
                {
                    return HK_NULL;
                }
            }
            else if (const hkReflect::PointerType* pt = typePointer->asPointer())
            {
                if (const hkReflect::Type* subType = adjustReflectTypePointer(pt->getSubType()))
                {
                    const hkReflect::Type* newPt = m_forwardingTypesAllocator.create<BasicPointerType>(subType, pt->getName());
                    return newPt;
                }
            }
        }
        return HK_NULL;
    }

    void applyRenameOperation(hkReflect::RecordType* versioningTargetRecord, const char* oldName, const char* newName)
    {
        const hkReflect::FieldDecl rf = findFieldInPatchedType(versioningTargetRecord, oldName);
        // Rename this member
        //LOG("%s applied rename %s->%s\n", patch->newName, memberRenamedPatch->oldName, memberRenamedPatch->newName);
        hkReflect::TypeDetail::setFieldName(rf, newName);
    }

    void applyMemberAddedOperation(hkVersionedRecordType* versionedRecordType, hkReflect::RecordType* versioningTargetRecord, const char* memberName, hkReflect::Version::LegacyType memberAddedType, const char* memberAddedTypeName, int memberAddedTuples, const void* defaultPtr, int memberAddedVersion)
    {
        HK_TIME_CODE_BLOCK("PATCH_MEMBER_ADDED", this);

        hkReflect::DataFieldDecl f = findFieldInPatchedType(versioningTargetRecord, memberName);
        // Defaults:
        if (defaultPtr)
        {
            HK_ASSERT_NO_MSG(0x6ddc95fe, (f.getOffset() >= 0) && ((f.getOffset() + f.getType()->getSizeOf()) <= versioningTargetRecord->getSizeOf()));
            hkArray<TrackedType::ObjectRef>::Temp allPatchedObjects;
            findAllPatchedObjects(versionedRecordType, allPatchedObjects);
            for (int objIndex = 0; objIndex < allPatchedObjects.getSize(); objIndex++)
            {
                setDefaultForNewMember(allPatchedObjects[objIndex].m_data, f, defaultPtr, versioningTargetRecord);
            }
        }
        if (memberAddedType == hkReflect::Version::TYPE_STRUCT)
        {
            hkArray<TrackedType::ObjectRef>::Temp allPatchedObjects;
            findAllPatchedObjects(versionedRecordType, allPatchedObjects);
            for (int objIndex = 0; objIndex < allPatchedObjects.getSize(); objIndex++)
            {
                hkReflect::Var handle(checkInternalOffset(allPatchedObjects[objIndex].m_data, f.getOffset(), versioningTargetRecord), f.getType());
                handle.writeDefault();
            }
        }
        if (memberAddedType == hkReflect::Version::TYPE_TUPLE_STRUCT)
        {
            // Tuples of structs need defaults...
            hkArray<TrackedType::ObjectRef>::Temp allPatchedObjects;
            findAllPatchedObjects(versionedRecordType, allPatchedObjects);

            const hkVersionedRecordType* typeInfo = m_typeSet.findType(memberAddedTypeName, memberAddedVersion);
            const hkReflect::Type* structType = typeInfo->getCompletedType()->asRecord();
            HK_ASSERT_NO_MSG(0x202b4b13, structType);

            for (int objIndex = 0; objIndex < allPatchedObjects.getSize(); objIndex++)
            {
                for (int tupleIndex = 0; tupleIndex < memberAddedTuples; tupleIndex++)
                {
                    hkReflect::Var handle(checkInternalOffset(allPatchedObjects[objIndex].m_data, f.getOffset() + tupleIndex * structType->getSizeOf(), versioningTargetRecord), structType);
                    handle.writeDefault();
                }
            }
        }
    }
    void applyMemberRemovedOperation(hkVersionedRecordType* versionedRecordType, hkReflect::RecordType* versioningTargetRecord, const char* memberName)
    {
        hkReflect::DataFieldDecl rf = findFieldInPatchedType(versioningTargetRecord, memberName);
        // Remove this member
        hkReflect::TypeDetail::setFieldName(rf, HK_NULL);

        if (const hkReflect::ArrayType* at = rf.getType()->asArray())
        {
            if (at->getFixedCount() == 0)
            {
                // If this is an array, we need to invalidate the array body as well
                hkArray<TrackedType::ObjectRef>::Temp allPatchedObjects;
                findAllPatchedObjects(versionedRecordType, allPatchedObjects);
                for (int objIndex = 0; objIndex < allPatchedObjects.getSize(); objIndex++)
                {
                    hkReflect::Var handle(checkInternalOffset(allPatchedObjects[objIndex].m_data, rf.getOffset(), versioningTargetRecord), rf.getType());
                    int id = *reinterpret_cast<hkUint32Le*>(handle.getAddress());
                    if (id)
                    {
                        DLOG("VER: Invalidating id {0} (was to {1})", id, at->getSubType()->getName() ? at->getSubType()->getName() : "");

                        // This array has come off the local allocator anyway
                        // Array contents may have been copied into a different object
                        // During a function patch so we can't always zero it. This is fine
                        // as long as the following stage knows not to pass it through if nothing
                        // references it
                        //m_bundle.setExtra(id, hkSerialize::VarN());

                        // Invalidate
                        *reinterpret_cast<hkUint32Le*>(handle.getAddress()) = 0;
                    }
                }
            }
        }
        else if (const hkReflect::PointerType* pt = rf.getType()->asPointer())
        {
            hkArray<TrackedType::ObjectRef>::Temp allPatchedObjects;
            findAllPatchedObjects(versionedRecordType, allPatchedObjects);
            for (int objIndex = 0; objIndex < allPatchedObjects.getSize(); objIndex++)
            {
                hkReflect::Var handle(checkInternalOffset(allPatchedObjects[objIndex].m_data, rf.getOffset(), versioningTargetRecord), rf.getType());
                DLOG_IF(int id = *reinterpret_cast<hkUint32Le*>(handle.getAddress()));
                DLOG("VER: Invalidating PTR id {0} (was to {1})", id, pt->getSubType()->getName() ? pt->getSubType()->getName() : "");
                *reinterpret_cast<hkUint32Le*>(handle.getAddress()) = 0;
            }
        }
    }

    // This is the external entry point
    virtual hkViewPtr<hkSerialize::Bundle> applyPatchesTo(hkViewPtr<hkSerialize::Bundle> bundleIn, hkReflect::Version::PatchSet* patchSet)
    {
        m_bundleIn = bundleIn;
        m_bundle.clear();

        // Make sure we always consider the oldest version we have encountered
        hkArrayView<const hkReflect::Version::PatchInfo*> patches = patchSet->getPatchList();

        for (int typeIndex = 0; typeIndex < bundleIn->types().getSize(); typeIndex++)
        {
            if (bundleIn->type(typeIndex) && bundleIn->type(typeIndex)->asRecord())
            {
                const hkReflect::RecordType* recType = bundleIn->type(typeIndex)->asRecord();
                if (!recType->isDynamicType())
                {
                    // We need to anchor the oldest versions encountered at the types in the file
                    // Do this BEFORE we create types as types may add todos to newer versions, but we
                    // need to create what is in the file
                    m_typeSet.addAllDependentTodos(recType);
                }
            }
        }

        {
            hkArray<Bundle::Item> allVars;
            bundleIn->getItems(allVars);
            m_bundle.init(allVars);

            // In case someone manages to get an empty bundle this far into the system
            m_bundleVarHasBeenCopied.setSize(hkMath::max2(m_bundle.m_singleVars.getSize(), 1), 0);
            m_bundleExtraHasBeenCopied.setSize(hkMath::max2(m_bundle.m_extras.getSize(), 1), 0);

            // Consider the nulls at the start to be already copied
            m_bundleVarHasBeenCopied[0] = 1;
            m_bundleExtraHasBeenCopied[0] = 1;

            // From now until we finish applying patches, we need to copy anything that is read
            m_finishedModifyingObjects = false;
            //HK_TIMER_BEGIN_LIST("hkReflect::Version::Patcher::doPatch", "Read from source");
            // Consume the source and copy the objects
            // We want to only copy the objects that are getting patched

            for (int typeIndex = 0; typeIndex < bundleIn->types().getSize(); typeIndex++)
            {
                if (bundleIn->type(typeIndex) && bundleIn->type(typeIndex)->asRecord())
                {
                    const hkReflect::RecordType* recType = bundleIn->type(typeIndex)->asRecord();
                    if (!recType->isDynamicType())
                    {
                        m_typeSet.createType(HK_NULL, 0, recType);
                    }
                }
            }

            // Set up the index from address map to resolve old to new pointers
            for (int i = 0; i < m_bundle.m_singleVars.getSize(); i++)
            {
                const hkReflect::Var& thisVar = m_bundle.var(i);
                if (const hkReflect::Type* sourceType = thisVar.getType())
                {
                    m_indexFromAddressMap.insert(thisVar.getAddress(), i);
                }
            }

            for (int i = 0; i < m_bundle.m_extras.getSize(); i++)
            {
                hkReflect::Var thisVar = m_bundle.m_extras[i].getFirst();
                if (const hkReflect::Type* sourceType = thisVar.getType())
                {
                    m_indexFromAddressMap.insert(thisVar.getAddress(), i);
                }
            }
        }

        m_typeSet.resolveAndFinalize(); // Let's just do this for now, we may not need it for every type

        for (int i = 0; i < m_bundle.numVars(); i++)
        {
            hkReflect::Var thisVar = m_bundle.var(i);
            // Need to check this first as the type types are records
            if (const hkReflect::Type* sourceType = thisVar.getType())
            {
                if (const hkReflect::Type* typePointer = hkReflect::Detail::asType( thisVar ))
                {
                    // A type pointer may need to be adjusted to point to the newest native version of the original type.

                    const hkReflect::Type* adjustedReflectTypePtr = adjustReflectTypePointer(typePointer);
                    if (adjustedReflectTypePtr)
                    {
                        m_bundle.setVar(i, hkReflect::Var(adjustedReflectTypePtr, sourceType));
                        m_bundleVarHasBeenCopied[i] = 1;
                    }
                    else
                    {
                        stripVarFromBundle(i);
                    }
                }
                else if ( const hkReflect::RecordType* thisVarRecordType = sourceType->asRecord() )
                {
                    bool patchesNeeded = m_typeSet.anyPatchesNeededFor(thisVarRecordType);
                    bool isConvertible = sourceType->isDynamicType() || m_typeSet.createType(HK_NULL, 0, sourceType);

                    if (!isConvertible)
                    {
                        stripVarFromBundle(i);
                    }
                    else if (thisVarRecordType && patchesNeeded)
                    {
                        makeVarVersioningCopy(i);
                    }
                    else
                    {
                        DLOG("No patches for {0} ({2}), copy opt @ {1}", sourceType->getName(), thisVar.getAddress(), sourceType->getVersion());

                        // Still need to copy the type, to set the dispatching pointer/array impls
                        // These chain to the old impls, run them and map the value back to new objects
                        hkReflect::Type* newType = chainingCopyType(thisVar.getType());

                        m_bundle.setVar(i, hkReflect::Var(thisVar.getAddress(), newType));
                    }
                }
            }
        }

        for (int i = 0; i < m_bundle.numExtras(); i++)
        {
            if (m_bundle.extra(i).getType() && (!m_bundle.extra(i).getType()->asString()))
            {
                hkReflect::Var thisVar = m_bundle.extra(i).getFirst();
                const int numElements = m_bundle.extra(i).getCount();
                if (const hkReflect::Type* sourceType = thisVar.getType())
                {
                    bool copyNeeded = false;
                    bool isConvertible = true;

                    if (const hkReflect::RecordType* thisVarRecordType = sourceType->asRecord())
                    {
                        if (!thisVarRecordType->isDynamicType())
                        {
                            copyNeeded = m_typeSet.anyPatchesNeededFor(thisVarRecordType);

                            if (!copyNeeded)
                            {
                                isConvertible = m_typeSet.createType(HK_NULL, 0, thisVarRecordType);
                            }
                        }
                    }
                    else if (const hkReflect::PointerType* pt = sourceType->asPointer())
                    {
                        copyNeeded = true;
                    }

                    if (!isConvertible)
                    {
                        stripExtraFromBundle(i);
                    }
                    else if (copyNeeded)
                    {
                        makeArrayVersioningCopy(i);
                    }
                    else
                    {
                        DLOG("[Array] No patches for {0} ({2}), copy opt @ {1}", thisVar.getType()->getName(), thisVar.getAddress(), thisVar.getType()->getVersion());

                        // No need to copy the object, just the type
                        hkReflect::Type* newType = chainingCopyType(thisVar.getType());

                        m_bundle.setVarN(i, hkSerialize::VarN::fromArray(thisVar.getAddress(), newType, numElements));
                    }
                }
            }
        }

        //// End


        //m_typeSet.resolveAndFinalize(); // Let's just do this for now, we may not need it for every type

        m_typeSet.setStartingVersions();

        for (int i = 0; i < patches.getSize(); i++)
        {
            const hkReflect::Version::PatchInfo* patch = patches[i];

            // Just skip new patches for now
            // Patches with no components are just renames so we need to run them
            if ((patch->numComponent > 0) && !patch->isCompatPatch())
            {
                continue;
            }

            hkArray<hkVersionedRecordType*>::Temp patchingRecordSet;

            DLOG("Applying patch {0} [{1}] to {2} [{3}]", patch->oldName, patch->oldVersion, patch->newName, patch->newVersion);

            // This will complete any types it needs to
            m_typeSet.getAllRecordTypes(patch->oldName, patch->oldVersion, patchingRecordSet);

            const hkVersionedRecordType* oldParent = HK_NULL;
            const hkVersionedRecordType* newParent = HK_NULL;

            // For each record we are patching
            for (int patchingRecordIndex = 0; patchingRecordIndex < patchingRecordSet.getSize(); patchingRecordIndex++)
            {
                hkVersionedRecordType* versionedRecordType = patchingRecordSet[patchingRecordIndex];
                hkReflect::RecordType* versioningTargetRecord = const_cast<hkReflect::RecordType*>(versionedRecordType->getCompletedType()->asRecord());
                hkPatchDependencies dependencies(patch);

                for (int componentIndex = 0; componentIndex < patch->numComponent; componentIndex++)
                {
                    const hkReflect::Version::PatchInfo::Component& component = patch->component[componentIndex];
                    if (const hkReflect::Version::PatchInfo::MemberAddedPatch* memberAddedPatch = component.asMemberAddedPatch())
                    {
                        const int memberVersionNumber = ((memberAddedPatch->type == hkReflect::Version::TYPE_FROMNAME) ? 0 : dependencies.getVersion(memberAddedPatch->typeName));

                        applyMemberAddedOperation(versionedRecordType, versioningTargetRecord, memberAddedPatch->name, memberAddedPatch->type, memberAddedPatch->typeName, memberAddedPatch->tuples, memberAddedPatch->defaultPtr, memberVersionNumber);
                    }
                    else if (const hkReflect::Version::PatchInfo::MemberRemovedPatch* memberRemovedPatch = component.asMemberRemovedPatch())
                    {
                        applyMemberRemovedOperation(versionedRecordType, versioningTargetRecord, memberRemovedPatch->name);
                    }
                    else if (const hkReflect::Version::PatchInfo::TypedMemberAddedPatch* typedMemberAddedPatch = component.asTypedMemberAddedPatch())
                    {
                        HK_TIME_CODE_BLOCK("PATCH_TYPED_MEMBER_ADDED", this);
                        hkReflect::DataFieldDecl f = findFieldInPatchedType(versioningTargetRecord, typedMemberAddedPatch->name);
                        // Defaults:
                        const hkReflect::Type* fieldType = f.getType();

                        hkArray<TrackedType::ObjectRef>::Temp allPatchedObjects;
                        findAllPatchedObjects(versionedRecordType, allPatchedObjects);
                        for (int objIndex = 0; objIndex < allPatchedObjects.getSize(); objIndex++)
                        {
                            hkReflect::Var handle(checkInternalOffset(allPatchedObjects[objIndex].m_data, f.getOffset(), versioningTargetRecord), fieldType);
                            handle.writeDefault();
                        }
                    }
                    else if (const hkReflect::Version::PatchInfo::TypedMemberRemovedPatch* typedMemberRemovedPatch = component.asTypedMemberRemovedPatch())
                    {
                        const hkReflect::FieldDecl rf = findFieldInPatchedType(versioningTargetRecord, typedMemberRemovedPatch->name);
                        // Remove this member
                        hkReflect::TypeDetail::setFieldName(rf, HK_NULL);
                    }
                    else if (const hkReflect::Version::PatchInfo::SetParentPatch* parentSetPatch = component.asSetParentPatch())
                    {
                        HK_TIME_CODE_BLOCK("PATCH_PARENT_SET", this);
                        if (parentSetPatch->oldParent)
                        {
                            // Delete the old parent members

                            if(!oldParent)
                            {
                                for (int parentIndex = 0; parentIndex < versionedRecordType->getNumParents(); parentIndex++)
                                {
                                    if(hkVersionedRecordType* p = versionedRecordType->getParent(parentIndex))
                                    {
                                        if (hkString::strCmp(p->getName(), parentSetPatch->oldParent) == 0)
                                        {
                                            oldParent = p;
                                        }
                                    }
                                }
                            }

                            HK_ASSERT_NO_MSG(0x62343de7, oldParent);

                            const hkReflect::Type* parentPatching = oldParent->getCompletedType();
                            const hkReflect::RecordType* parentPatchingObject = parentPatching->asRecord();
                            HK_ASSERT_NO_MSG(0x5d2f9355, parentPatchingObject);
                            for( hkReflect::DeclIter<hkReflect::DataFieldDecl> fieldIter( parentPatchingObject ); fieldIter.advance(); )
                            {
                                const hkReflect::FieldDecl f = fieldIter.current();
                                const char* fieldName = f.getName();
                                if (f.isSerializable() && fieldName && (fieldName[0] != '#')) // Don't remove members we haven't added yet
                                {
                                    hkReflect::FieldDecl rf = findFieldInPatchedType(versioningTargetRecord, fieldName);
                                    // Remove this member
                                    hkReflect::TypeDetail::setFieldName(rf, HK_NULL);
                                }
                            }

                            
                            const_cast<hkVersionedRecordType*>(oldParent)->removeChild(versionedRecordType);
                            versionedRecordType->removeParent(const_cast<hkVersionedRecordType*>(oldParent));
                        }
                        if (parentSetPatch->newParent)
                        {
                            if(!newParent)
                            {
                                // Set defaults for the new parent members

                                for (int parentIndex = 0; parentIndex < versionedRecordType->getNumParents(); parentIndex++)
                                {
                                    if(hkVersionedRecordType* p = versionedRecordType->getParent(parentIndex))
                                    {
                                        if (hkString::strCmp(p->getName(), parentSetPatch->newParent) == 0)
                                        {
                                            newParent = p;
                                        }
                                    }
                                }
                            }

                            HK_ASSERT_NO_MSG(0x3b6d648, newParent);
                            const hkReflect::Type* parentPatching = newParent->getCompletedType();
                            const hkReflect::RecordType* parentPatchingObject = parentPatching->asRecord();
                            HK_ASSERT_NO_MSG(0x276f0656, parentPatchingObject);
                            hkArray<TrackedType::ObjectRef>::Temp allPatchedObjects;
                            findAllPatchedObjects(versionedRecordType, allPatchedObjects);
                            for (int objIndex = 0; objIndex < allPatchedObjects.getSize(); objIndex++)
                            {
                                for( hkReflect::DeclIter<hkReflect::DataFieldDecl> ci(parentPatchingObject); ci.advance(); )
                                {
                                    // Get the member name from the parent class
                                    const hkReflect::FieldDecl& f = ci.current();
                                    if(f.getName())
                                    {
                                        // Get the offset of the member with the same name in our object
                                        hkReflect::DataFieldDecl rf = findFieldInPatchedType(versioningTargetRecord, f.getName());

                                        hkReflect::Var handle(checkInternalOffset(allPatchedObjects[objIndex].m_data, rf.getOffset(), versioningTargetRecord), f.getType());
                                        handle.writeDefault();
                                    }
                                }
                            }
                        }
                    }
                    else if (const hkReflect::Version::PatchInfo::DefaultChangedPatch* defaultChangedPatch = component.asDefaultChangedPatch())
                    {
                        HK_TIME_CODE_BLOCK("PATCH_MEMBER_DEFAULT_SET", this);

                        const hkReflect::FieldDecl rf = findFieldInPatchedType(versioningTargetRecord, defaultChangedPatch->name);

                        hkReflect::Type* newType = hkVersionedRecordType::copyAndAddOptional(rf.getType(), m_typeSet.getTypesAllocator(), hkReflect::Opt::DEFAULT, defaultChangedPatch->defaultPtr);
                        hkReflect::TypeDetail::setFieldType(versioningTargetRecord, defaultChangedPatch->name, newType);
                    }
                    else if (const hkReflect::Version::PatchInfo::DataObjectFunctionPatch* functionPatch = component.asDataObjectFunctionPatch()) // OLD FUNCTION
                    {
                        HK_TIME_CODE_BLOCK("PATCH_DATAOBJECT_FUNCTION", this);
                        // Apply function to all of the existing objects of this type. This can add objects to the world

                        // Need to find all objects of this type in the world, wrap them and apply the patch function
                        hkArray<TrackedType::ObjectRef>::Temp allPatchedObjects;
                        findAllPatchedObjects(versionedRecordType, allPatchedObjects);

                        // So that patch functions can get the correct versions
                        m_currentPatchDependencies.clear();
                        m_currentPatchDependencies.addPatch(patch);

                        for (int patchObjectIndex = 0; patchObjectIndex < allPatchedObjects.getSize(); patchObjectIndex++)
                        {
                            
                            hkDataObject wrappedDataObject(hkReflect::Var(allPatchedObjects[patchObjectIndex].m_data, versionedRecordType->getCompletedType()), this, allPatchedObjects[patchObjectIndex].m_id); // Needs wrapper impl here -- parent ptr is wrong
                            functionPatch->function(wrappedDataObject);

                            // Be careful --  the function may have added objects and reallocated any of the member's storage
                        }
                    }
                    else if (const hkReflect::Version::PatchInfo::MemberRenamedPatch* memberRenamedPatch = component.asMemberRenamedPatch())
                    {
                        applyRenameOperation(versioningTargetRecord, memberRenamedPatch->oldName, memberRenamedPatch->newName);
                    }
                    else if (const hkReflect::Version::PatchInfo::TypeChangedFunctionPatch* typeChangedPatch = component.asTypeChangedFunctionPatch())
                    {
                        hkStringBuf nameOld(typeChangedPatch->m_name);
                        nameOld.append("#OLD");
                        applyRenameOperation(versioningTargetRecord, typeChangedPatch->m_name, nameOld);
                        const int memberVersionNumber = dependencies.getVersion(typeChangedPatch->m_newTypeName);

                        applyMemberAddedOperation(versionedRecordType, versioningTargetRecord, typeChangedPatch->m_name, typeChangedPatch->m_newType, typeChangedPatch->m_newTypeName, typeChangedPatch->m_newTuples, HK_NULL, memberVersionNumber);

                        // Need to find all objects of this type in the world, wrap them and apply the patch function
                        hkArray<TrackedType::ObjectRef>::Temp allPatchedObjects;
                        findAllPatchedObjects(versionedRecordType, allPatchedObjects);
                        for (int patchObjectIndex = 0; patchObjectIndex < allPatchedObjects.getSize(); patchObjectIndex++)
                        {
                            hkDataObject wrappedDataObject(hkReflect::Var(allPatchedObjects[patchObjectIndex].m_data, versionedRecordType->getCompletedType()), this, allPatchedObjects[patchObjectIndex].m_id); // Needs wrapper impl here -- parent ptr is wrong
                            typeChangedPatch->m_function(wrappedDataObject, nameOld, typeChangedPatch->m_name);
                        }

                        applyMemberRemovedOperation(versionedRecordType, versioningTargetRecord, nameOld);
                    }
                    else if (component.asDependsPatch())
                    {
                        // Do nothing
                    }
                    else
                    {
                        // Unknown / invalid patch
                        HK_ASSERT_NO_MSG(0xfcf9b19, 0);
                    }
                }

                if ((patch->oldName && hkString::strCmp(versionedRecordType->getCompletedType()->getName(), patch->oldName) == 0))
                {
                    // This will apply the patch to derived types as well, but we don't want to rename them
                    hkReflect::TypeDetail::localSetOptional<hkReflect::Opt::VERSION>(versionedRecordType->getCompletedType(), patch->newVersion);
                }

                // Now do type rename
                if (patch->newName && patch->oldName && (hkString::strCmp(patch->oldName, patch->newName) != 0))
                {
                    // This will apply the patch to derived types as well, but we don't want to rename them
                    if (hkString::strCmp(versionedRecordType->getCompletedType()->getName(), patch->oldName) == 0)
                    {
                        versionedRecordType->renameType(patch->oldName, patch->newName);
                    }
                }
            }
        }

        // From now on, we no longer modify anything so we can read pass-through objects without needing to copy them
        m_finishedModifyingObjects = true;

        copyBackLinkedObjects();

        // Now we need to remove unreachable objects
        // This could be optional??

        // Push the workbundle values back into something we can return.
        {
            HK_ASSERT_NO_MSG(0x618d4126, m_bundle.types().getSize() == 0);
            hkHashMap<const hkReflect::Type*, TypeId> tidFromType;
            m_bundle.setType(0, HK_NULL);
            tidFromType.insert(HK_NULL, 0);
                // We need to set whichever types we used in the output bundle
            for(int vi = 0; vi < m_bundle.numVars(); vi++)
            {
                // Restore any vars we stripped out of the input
                if (m_bundleVarHasBeenCopied[vi] < 0)
                {
                    m_bundle.setVar(vi, m_strippedVars[(-m_bundleVarHasBeenCopied[vi]) - 1]);
                }

                Var v = m_bundle.var(vi);

                // Don't pass through objects that have been deleted
                if (v.getType() && (v.getType()->getVersion() == -1))
                {
                    m_bundle.setVar(vi, Var());
                }
                else
                {
                    recursivelyAddTypes(tidFromType, v.getType());
                }
            }

            for(int i = 0; i < m_bundle.numExtras(); i++)
            {
                // Restore any vars we stripped out of the input
                if (m_bundleExtraHasBeenCopied[i] < 0)
                {
                    m_bundle.setVarN(i, m_strippedExtras[(-m_bundleExtraHasBeenCopied[i]) - 1]);
                }

                VarN v = m_bundle.extra(i);

                // Don't pass through objects that have been deleted
                if (v.getType() && (v.getType()->getVersion() == -1))
                {
                    m_bundle.setVarN(i, VarN::invalid());
                }
                else
                {
                    recursivelyAddTypes(tidFromType, v.getType());
                }
            }
        }


        // Then we're done
        return &m_bundle;
    }

    HK_DETAIL_DIAG_MSVC_PUSH()
    // Disable warning, it is safe to use 'this' as long as it is not dereferenced
    HK_DETAIL_DIAG_MSVC_OFF(4355)

    CompatPatcher(const hkReflect::TypeReg* cb, hkReflect::Version::PatchSet& patches)
        : m_arrayImpl(new VersioningArrayImpl(*this))
        , m_pointerImpl(new VersioningPointerImpl(*this))
        , m_typeSet(patches, cb, hkMemHeapAllocator(), m_arrayImpl, m_arrayImpl, m_pointerImpl, &m_stringImpl)
        , m_stringImpl(&m_typeSet.getTypesAllocator())
        , m_patches(patches)
        , m_world(this)
        , m_forwardingTypesAllocator(hkMemHeapAllocator())
    {
        m_indexFromAddressMap.insert(HK_NULL, 0);

        hkTypeVm::addDefaultPasses(m_compiler);
        m_compiler.addPass<hkTypeVm::IntAndFloatConversionPass>();
        m_compiler.excludeNonSerializableFields = true;
        m_compiler.addInstrWithNoSource = false;
    }

    HK_DETAIL_DIAG_MSVC_POP()

    hkPointerMap<const void*, int> m_trackingListIndexFromTypeMap;
    hkReflect::Detail::ArrayImpl* m_arrayImpl;
    hkReflect::Detail::PointerImpl* m_pointerImpl;
    hkModifiableTypeSet m_typeSet; // AFTER arrayImpl
    hkReflect::Detail::HavokStringImpl m_stringImpl; // AFTER typeSet
    hkArray<TrackedType> m_trackingList;
    const hkReflect::Version::PatchSet& m_patches;

    hkViewPtr<hkSerialize::Bundle> m_bundleIn;
    WorkBundle m_bundle;

    hkArray<int> m_bundleVarHasBeenCopied;
    hkArray<int> m_bundleExtraHasBeenCopied;
    hkDataWorld m_world;
    hkPointerMap < void*, int > m_indexFromAddressMap;
    hkTypeVm::Compiler m_compiler;
    hkTransientAllocator m_forwardingTypesAllocator;
    hkPointerMap<const hkReflect::Type*, hkReflect::Type*> m_chainedCopiedTypesFromSrc;
    bool m_finishedModifyingObjects;
    hkArray<hkReflect::Var> m_strippedVars;
    hkArray<hkSerialize::VarN> m_strippedExtras;
    // This lives here so that the currently executing patch function can find it
    hkPatchDependencies m_currentPatchDependencies;
};


///////////////// hkDataObject

void hkDataObject::Value::addBackLinkIfRequired()
{
    hkReflect::Var objectReference = m_parent->getBackLink(m_object, m_id);
    if (objectReference)
    {
        DLOG_IF(hkReflect::Var oldObject = m_object);
        // Type doesn't change, only address
        m_object = objectReference;
        //DLOG("FOLLOW Link Field at {0} to {1}", oldObject, m_object);
    }
}

namespace
{
    class WrappedValueAssignmentVisitor : public hkReflect::TypeVisitor < WrappedValueAssignmentVisitor, void, const hkDataObject::Value& >
    {
    public:
        WrappedValueAssignmentVisitor(hkDataObject::Value& target) : m_target(target) {}

        void visit(const hkReflect::IntType* intType, const hkDataObject::Value& val)
        {
            if (intType->getFormat().equalsValue(hkReflect::Format::OfInt<hkUint64>::Value) || intType->getFormat().equalsValue(hkReflect::Format::OfInt<hkInt64>::Value))
            {
                m_target = val.asInt64();
            }
            else if (intType->getFormat().equalsValue(hkReflect::Format::OfInt<hkUint32>::Value))
            {
                m_target = val.asInteger<hkUint32>();
            }
            else
            {
                m_target = val.asInt();
            }
        }

        void visit(const hkReflect::FloatType* floatType, const hkDataObject::Value& val)
        {
            switch (floatType->getFormat().get())
            {
            case hkReflect::Format::OfFloat<hkDouble64>::Value:
            case hkReflect::Format::OfFloat<hkFloat32>::Value:
                m_target = val.asReal();
                break;
            case hkReflect::Format::OfFloat<hkHalf16>::Value:
                m_target = val.asHalf();
                break;

            default:
                HK_ASSERT_NO_MSG(0x79bd631d, 0);
            }
        }

        void visit(const hkReflect::PointerType* ptr, const hkDataObject::Value& v)
        {
            m_target = v.asObject();
        }

        void visit(const hkReflect::BoolType* b, const hkDataObject::Value& v)
        {
            if (v.getInternal().getType()->asInteger())
            {
                m_target = v.asInt();
            }
            else
            {
                m_target = v.asBool();
            }
        }

        void visit(const hkReflect::ArrayType* a, const hkDataObject::Value& v)
        {
            m_target = v.asArray();
        }

        void visit(const hkReflect::RecordType* r, const hkDataObject::Value& v)
        {
            m_target = v.asObject();
        }

        void visit(const hkReflect::StringType* r, const hkDataObject::Value& v)
        {
            m_target = v.asString();
        }

        void visit(const hkReflect::Type* other, const hkDataObject::Value&)
        {
            HK_ASSERT_NO_MSG(0x23e20b35, 0);
        }

        hkDataObject::Value& m_target;
    };
}

// Interface that looks like a hkDataObject but actually wraps a native-like versioning object
void hkDataObject::Value::operator=(const hkDataObject::Value& l)
{
    WrappedValueAssignmentVisitor assign(*this);
    assign.dispatch(m_object.getType(), l);
}

/// Assign 64-bit integer to value.
void hkDataObject::Value::operator=(hkInt64 v)
{
    hkReflect::IntVar dst(m_object);
    HK_ASSERT_NO_MSG(0x431e2ebb, dst.isValid());

    dst.setValue(v);
}

/// Assign 32-bit integer to value.
void hkDataObject::Value::operator=(int i)
{
    if (const hkReflect::IntType* it = m_object.getType()->asInteger())
    {
        hkReflect::IntVar dst(m_object);
        dst.setValue(i);
    }
    else if (const hkReflect::BoolType* boolType = m_object.getType()->asBool())
    {
        hkReflect::BoolVar dst(m_object);
        dst.setValue(i != 0);
    }
    else
    {
        HK_ASSERT_NO_MSG(0x241be78b, 0);
    }
}

/// Assign 32-bit integer to value.
void hkDataObject::Value::operator=(hkUint32 i)
{
    if (const hkReflect::IntType* it = m_object.getType()->asInteger())
    {
        hkReflect::IntVar dst(m_object);
        dst.setValue(i);
    }
    else if (const hkReflect::BoolType* boolType = m_object.getType()->asBool())
    {
        hkReflect::BoolVar dst(m_object);
        dst.setValue(i != 0);
    }
    else
    {
        HK_ASSERT_NO_MSG(0x241be78b, 0);
    }
}

/// Assign 32-bit float to value.
void hkDataObject::Value::operator=(float f)
{
    hkReflect::FloatVar dst(m_object);
    HK_ASSERT_NO_MSG(0x4bef673f, dst.isValid());
    dst.setValue(f);
}
void hkDataObject::Value::operator=(double d)
{
    hkReflect::FloatVar dst(m_object);
    HK_ASSERT_NO_MSG(0x4bef673f, dst.isValid());
    dst.setValue(d);
}
/// Assign bool to value.
void hkDataObject::Value::operator=(bool b)
{
    if (const hkReflect::BoolType* boolType = m_object.getType()->asBool())
    {
        hkReflect::BoolVar dst(m_object);
        dst.setValue(b);
    }
    else if (const hkReflect::IntType* intType = m_object.getType()->asInteger())
    {
        hkReflect::IntVar dst(m_object);
        dst.setValue(hkUint8(b));
    }
    else
    {
        HK_ASSERT_NO_MSG(0xc773c10, 0); // Unknown type
    }
}
/// Assign 16-bit float to value.
void hkDataObject::Value::operator=(hkHalf16 r)
{
    if (const hkReflect::IntType* it = m_object.getType()->asInteger())
    {
        hkUint16 u = *reinterpret_cast<hkUint16*>(&r);
        hkReflect::IntVar dst(m_object);
        dst.setValue(u);
    }
    else if (const hkReflect::FloatType* ft = m_object.getType()->asFloat())
    {
        hkReflect::FloatVar dst(m_object);
        dst.setValue(r.getReal());
    }
    else
    {
        HK_ASSERT_NO_MSG(0x17b880cf, 0); // Unknown type
    }
}
/// Assign c-style string to value.
void hkDataObject::Value::operator=(const char* s)
{
    hkReflect::StringVar dst(m_object);
    HK_ASSERT_NO_MSG(0x429f8e00, dst.isValid());
    dst.setValue(s);
}

/// Assign array or tuple to value.
void hkDataObject::Value::operator=(const Array& l)
{
    

    // Variable
    if (l.getVariableArrayObjectId() >= 0) // Correct check??
    {
        setVariableArrayObjectId(l.getVariableArrayObjectId());
    }
    else
    {
        HK_ASSERT_NO_MSG(0x61e74ab5, l.m_type->asArray());
        hkReflect::ArrayVar dstArray(m_object);
        HK_ASSERT_NO_MSG(0x287b1cb0, dstArray.isValid());

        const int oldSize = dstArray.getCount();
        const int newSize = l.getSize();

        // Retrack any objects that we are moving around
        if(oldSize && (dstArray.getType()->getFixedCount() == 0))
        {
            m_parent->removeObjectFromTrackingMap(dstArray);
        }
        dstArray.setArraySize(newSize);

        if (l.m_elemType->getSizeOf() == dstArray.getSubType()->getSizeOf())
        {
            hkMemUtil::memCpy(dstArray.getDataPointer(), l.m_data, l.m_elemType->getSizeOf() * newSize);
        }
        else
        {
            // Differently sized objects. We need a VM to convert, these should just be ints, floats
            // that have come from native and not patches. Base machine won't do pointers, arrays, anything complicated

            const hkTypeVm::Program* p = m_parent->m_compiler.compile(l.m_elemType, dstArray.getSubType());
            hkTypeVm::FastCopyInterpreter vm;
            hkTypeVm::FastCopyInterpreter::execN(vm, p,
                dstArray.getDataPointer(), dstArray.getSubType()->getSizeOf(),
                l.m_data, l.m_elemType->getSizeOf(), dstArray.getSubType()->getSizeOf(), l.m_elemType->getSizeOf(), newSize);
        }

        if(dstArray.getType()->getFixedCount() == 0)
        {
            m_parent->addObjectToTrackingMap(hkReflect::Var(dstArray.getDataPointer(), dstArray.getSubType()), newSize, dstArray.getSubType()->getSizeOf(), -1, -1); // TODO what is the extra Id here???
        }
    }
}
/// Assign object to value.
void hkDataObject::Value::operator=(const hkDataObject& o)
{
    HK_ASSERT_NO_MSG(0x36b5036b, m_object.getType()->asPointer() || m_object.getType()->asRecord());
    if (o.m_object.getAddress() != m_object.getAddress())
    {
        if (m_object.getType()->asPointer())
        {
            hkUlong id = o.m_object.getAddress() ? m_parent->addObject(o.m_object) : 0;
            // Add pointer patch
            *reinterpret_cast<hkUint32Le*>(m_object.getAddress()) = id;
            //DLOG("Wrapped TODO PTR {0} ({1} from {2})", id, m_object.getAddress(), o.m_object.getAddress());
        }
        else if (const hkReflect::RecordType* recType = m_object.getType()->asRecord())
        {
            // We assume reference behaviour when we assign to a struct
            // b = b'. b'.x = 1 -> b.x = 1
            const hkReflect::Type* srcType = o.m_object.getType();
            void* srcObject = HK_NULL;
            if (srcType->asPointer())
            {
                srcObject = const_cast<void*>(m_parent->pointerFromId(*reinterpret_cast<hkUint32Le*>(o.m_object.getAddress())));
                //DLOG("Wrapped TODO PTR {0} ({1}) ({2} from {3})", *(int*)o.m_object.getAddress(), srcObject, m_object.getAddress(), o.m_object.getAddress());
            }
            else if (srcType->asRecord())
            {
                srcObject = o.m_object.getAddress();
            }
            HK_ASSERT_NO_MSG(0x16e0a21a, srcObject);
            HK_ASSERT_NO_MSG(0x4701c85e, (m_object.getType()->getSizeOf() & (HK_POINTER_SIZE - 1)) == 0); // Will get strange alignment issues without this

#if defined(HK_DEBUG_SLOW)
            hkMemUtil::memSet(m_object.getAddress(), 0xd0, m_object.getType()->getSizeOf());
#endif
            // For old hkDataObjects this becomes a link to the new object
            m_parent->setBackLink(m_object, hkReflect::Var(srcObject, srcType), m_id, o.m_id);

            {
                DLOG("Link Field of {0} to {1}", m_object, srcObject);
            }
        }
    }
}

/// Assign HK_NULL object
void hkDataObject::Value::setNull()
{
    HK_ASSERT_NO_MSG(0x2e795b1b, m_object.getType()->asPointer());
    // Pointers are a special patch. 0 represents NULL
    *reinterpret_cast<hkUint32Le*>(m_object.getAddress()) = 0;
}

/// Get value as 32-bit float.
float hkDataObject::Value::asReal() const
{
    hkReflect::FloatVar src(m_object);
    HK_ASSERT_NO_MSG(0x18153471, src.isValid());
    return float(src.getValue());
}

hkHalf16 hkDataObject::Value::asHalf() const
{
    hkReflect::FloatVar src(m_object);
    HK_ASSERT_NO_MSG(0x108aeee8, src.isValid());

    hkHalf16 r; r.setReal<true>(hkReal(src.getValue()));
    return r;
}


/// Get value as c-style string.
const char* hkDataObject::Value::asString() const
{
    hkReflect::StringVar src(m_object);
    HK_ASSERT_NO_MSG(0x63a9a52d, src.isValid());
    return src.getValue();
}

/// Get value as array or tuple.
hkDataObject::Array hkDataObject::Value::asArray() const
{
    HK_ASSERT_NO_MSG(0x70d8f57b, m_object.getType()->asArray());
    return hkDataObject::Array(m_object.getAddress(), m_object.getType(), m_parent, m_id);
}

/// Get value as object.
hkDataObject hkDataObject::Value::asObject() const
{

    HK_ASSERT_NO_MSG(0x4d81d36f, m_object.getType()->asRecord() || m_object.getType()->asPointer());
    if (const hkReflect::RecordType* rt = m_object.getType()->asRecord())
    {
        
        return hkDataObject(m_object, m_parent, m_id);
    }
    else if (const hkReflect::PointerType* pointerType = m_object.getType()->asPointer())
    {
        const int objectId = *reinterpret_cast<hkUint32Le*>(m_object.getAddress());
        const hkReflect::Type* pointerSubType = pointerType->getSubType();
        const hkReflect::RecordType* pointerSubTypeRec = pointerSubType ? pointerSubType->asRecord() : HK_NULL;

        if (objectId && !pointerSubTypeRec)
        {
            if(const hkReflect::Type* typeFromVar = m_parent->m_bundle.var(objectId).getType())
            {
                pointerSubTypeRec = typeFromVar->asRecord();
            }
        }

        HK_ASSERT_NO_MSG(0x46026a5b, (objectId == 0) || (pointerSubTypeRec));
        if (objectId)
        {
            hkReflect::Var objectVar = m_parent->m_bundle.var(objectId);
            return hkDataObject(m_parent->m_bundle.var(objectId), m_parent, VarOrExtra(objectId, VAR));
        }
        if (pointerSubTypeRec)
        {
            return hkDataObject(hkReflect::Var(HK_NULL, pointerSubTypeRec), m_parent, VarOrExtra(0, VAR)); // Null pointers can still have a type
        }
    }
    return hkDataObject(hkReflect::Var(), m_parent, VarOrExtra(0, VAR));
}

const char* hkDataObject::Value::getTypeName() const
{

    const hkReflect::Type* type = m_object.getType();
    if (const hkReflect::PointerType* pointerType = type->asPointer())
    {
        type = pointerType->getSubType();
    }

    return type->getName();
}

void hkDataObject::Value::setVariableArrayObjectId(int id)
{
    *reinterpret_cast<hkUint32Le*>(m_object.getAddress()) = id;
}

////////////////////
// Array
////////////////////

hkSerialize::VarN hkDataObject::Array::getVariableArrayObjects() const
{
    const int id = getVariableArrayObjectId();
    if (id >= 0)
    {
        return m_parent->m_bundle.extra(id);
    }
    return hkSerialize::VarN::invalid();
}

void hkDataObject::Array::setVariableArrayObjects(hkSerialize::VarN newVal)
{
    const int id = getVariableArrayObjectId();
    if (id >= 0)
    {
        m_parent->m_bundle.setVarN(id, newVal);
    }
    else
    {
        // Not a variable array
        HK_ASSERT_NO_MSG(0x2250c94e, 0);
    }
}

int hkDataObject::Array::getVariableArrayObjectId() const
{
    if (m_type->asArray()->getFixedCount() == 0)
    {
        return *m_idPtr;
    }
    return -1;
}

void hkDataObject::Array::setVariableArrayObjectId(int id)
{
    HK_ASSERT_NO_MSG(0x450c923e, m_idPtr != HK_NULL);
    *m_idPtr = id;
}

hkDataObject::Array::Array(void* obj, const hkReflect::Type* type, class CompatPatcher* parent, hkDataObject::VarOrExtra id, const char* swizzledMemberName)
    : m_idPtr(HK_NULL)
    , m_type(type)
    , m_parent(parent)
    , m_id(id)
{
    if (const hkReflect::ArrayType* arrayType = m_type->asArray())
    {
        
        if (arrayType->getFixedCount() == 0)
        {
            m_idPtr = reinterpret_cast<hkUint32Le*>(obj);
            // If the array has contents, ensure they have been copied before we do anything
            if (hkUint32Le* idPtr = reinterpret_cast<hkUint32Le*>(m_idPtr))
            {
                if (static_cast<int>(*idPtr))
                {
                    m_parent->makeArrayVersioningCopy(*idPtr);
                }
            }
            hkSerialize::VarN objs = getVariableArrayObjects();
            m_elemType = objs.getType() ? objs.getType() : arrayType->getSubType();
            m_data = objs.getFirst().getAddress();
            m_size = objs.getCount();
        }
        else
        {
            hkReflect::ArrayVar handle(obj, arrayType);
            m_elemType = handle.getSubType();
            m_data = handle.getDataPointer();
            m_size = handle.getCount();

        }
    }

    if (swizzledMemberName)
    {
        const hkReflect::RecordType* subTypeRecordType = m_elemType->asRecord();
        HK_ASSERT_NO_MSG(0x6c7efc4, subTypeRecordType);

        const hkReflect::FieldDecl bf = subTypeRecordType->findField(swizzledMemberName, true);
        HK_ASSERT_NO_MSG(0x52fc8e21, bf);

        const hkReflect::DataFieldDecl f = bf.asDataField();
        HK_ASSERT_NO_MSG(0x52fc8e21, f);

        m_data = hkAddByteOffset(m_data, f.getOffset());
        m_elemType = f.getType();
        m_stride = subTypeRecordType->getSizeOf();
    }
    else
    {
        // Empty homogenous arrays don't have an element type
        m_stride = m_elemType ? m_elemType->getSizeOf() : 0;
    }
}


int hkDataObject::Array::getSize() const
{
    if (hkSerialize::VarN objs = getVariableArrayObjects())
    {
        HK_ASSERT(0x41989e5b, !m_type->asArray() || m_size == objs.getCount(), "Data array size has been changed externally");
        return m_size;
    }
    else
    {
        HK_ASSERT(0x41989e5b, !m_type->asArray() || m_size == hkReflect::ArrayVar(m_data, m_type->asArray()).getCount(), "Data array size has been changed externally");
        return m_size;
    }
}

void hkDataObject::Array::setSize(int newSize)
{
    if (newSize == m_size)
    {
        return;
    }

    // In-place array?
    if (getVariableArrayObjectId() < 0) // was getVariableArrayObjects == HK_NULL ???
    {
        if (newSize == m_size)
        {
            return;
        }

        // Can't resize a fixed-size array
        HK_ASSERT_NO_MSG(0x211b827a, 0);
    }
    else
    {
        const hkReflect::ArrayType* arrayType = m_type->asArray();
        HK_ASSERT_NO_MSG(0x279d6ab0, arrayType);

        const int extrasIndex = getVariableArrayObjectId();
        // Dodgy here
        if (extrasIndex == 0)
        {
            if (m_elemType->asRecord())
            {
                if (const hkVersionedRecordType* vrt = m_parent->m_typeSet.createType(HK_NULL, 0, m_elemType))
                {
                    m_elemType = vrt->getCompletedType();
                }
                else
                {
                    m_elemType = HK_NULL;
                    return;
                }

            }
            int newIndex = m_parent->m_bundle.numExtras();
            m_parent->m_bundle.setVarN(newIndex, hkSerialize::VarN::fromArray(HK_NULL, m_elemType, 0));

            m_parent->m_bundleExtraHasBeenCopied.pushBack(1);
            HK_ASSERT_NO_MSG(0x5eb02680, m_parent->m_bundleExtraHasBeenCopied.getSize() == m_parent->m_bundle.numExtras());

            setVariableArrayObjectId(newIndex);
        }
        else
        {
            // This is inefficient, we copy and then copy again
            m_parent->makeArrayVersioningCopy(extrasIndex);
        }

        hkSerialize::VarN oldObjects = getVariableArrayObjects();

        if ((m_size == 0) && (m_elemType == HK_NULL))
        {
            // This seems a bit wierd
            m_elemType = arrayType->getSubType();
            oldObjects = hkSerialize::VarN::fromArray(oldObjects.getFirst().getAddress(), m_elemType, oldObjects.getCount());
            m_stride = m_elemType ? m_elemType->getSizeOf() : 0;
        }

        // Retrack any objects that we are moving around
        if (m_size && (arrayType->getFixedCount() == 0))
        {
            if (oldObjects.getAddress())
            {
                // Always true now??
                if (m_parent->m_bundleExtraHasBeenCopied[extrasIndex]) // If it has not been copied, we were not tracking it anyway
                {
                    m_parent->removeObjectFromTrackingMap(oldObjects.getFirst());
                }
            }
        }

        const int elementSize = oldObjects.getType()->getSizeOf();

        m_data = m_parent->m_bundle.m_allocator.blockAlloc(elementSize * newSize); //objs->m_objectType->allocate(newSize).getAddress();
        hkString::memSet(hkAddByteOffset(m_data, elementSize * oldObjects.getCount()), 0, (newSize > oldObjects.getCount()) ? (elementSize * (newSize - oldObjects.getCount())) : 0);
        hkString::memCpy(m_data, oldObjects.getAddress(), (newSize > oldObjects.getCount() ? oldObjects.getCount() : newSize) * elementSize);

        hkSerialize::VarN newObjects = hkSerialize::VarN::fromArray(m_data, oldObjects.getType(), newSize);
        m_size = newSize;
        // We may have changed the type, need to re-set these
        m_stride = elementSize;
        m_elemType = newObjects.getType();

        setVariableArrayObjects(newObjects);

        if (oldObjects.getAddress())
        {
            m_parent->updateBacklinks(oldObjects, newObjects);
        }
        m_parent->addObjectToTrackingMap(newObjects.getFirst(), newSize, elementSize, -1, extrasIndex);
        // We have copied it now that we have resized the array, we had to allocate new storage
        const int id = getVariableArrayObjectId();
        m_parent->m_indexFromAddressMap.insert(m_data, id); // Needed??
    }
}

void hkDataObject::Array::reserve(int newSize)
{
    // New wrapped array can't reserve without allocating
}

void hkDataObject::Array::removeAt( int idx )
{
    HK_ASSERT(0x22440733, (idx >= 0) && (idx < getSize()), "Bad index to remove.");
    // Needs to do the same thing as setSize but tracking the elements moving
    // This is not efficient, it allocates a new block to remove one element

    const int newSize = getSize() - 1;

    // In-place array?
    HK_ASSERT_NO_MSG(0x931b827a, getVariableArrayObjectId() >= 0);  // Can't resize a fixed-size array

    const hkReflect::ArrayType* arrayType = m_type->asArray();
    HK_ASSERT_NO_MSG(0x279d6ab0, arrayType);

    const int extrasIndex = getVariableArrayObjectId();
    // This is inefficient, we copy and then copy again
    m_parent->makeArrayVersioningCopy(extrasIndex);

    hkSerialize::VarN oldObjects = getVariableArrayObjects();

    // Retrack any objects that we are moving around
    if (m_size && (arrayType->getFixedCount() == 0))
    {
        if (oldObjects.getAddress())
        {
            // Always true now??
            if (m_parent->m_bundleExtraHasBeenCopied[extrasIndex]) // If it has not been copied, we were not tracking it anyway
            {
                m_parent->removeObjectFromTrackingMap(oldObjects.getFirst());
            }
        }
    }

    const int elementSize = oldObjects.getType()->getSizeOf();

    m_data = m_parent->m_bundle.m_allocator.blockAlloc(elementSize * newSize);
    hkString::memSet(hkAddByteOffset(m_data, elementSize * oldObjects.getCount()), 0, (newSize > oldObjects.getCount()) ? (elementSize * (newSize - oldObjects.getCount())) : 0);

    if (idx > 0)
    {
        // old[0->idx-1] -> new [0->idx-1]
        hkString::memCpy(m_data, oldObjects.getAddress(), idx * elementSize);
        m_parent->updateBacklinks(hkSerialize::VarN::fromArray(oldObjects.getAddress(), oldObjects.getType(), idx), hkSerialize::VarN::fromArray(m_data, oldObjects.getType(), idx));
    }

    if (idx < (m_size - 1))
    {
        // old[idx+1->end] -> new [idx-end]
        hkString::memCpy(hkAddByteOffset(m_data, elementSize * idx), hkAddByteOffset(oldObjects.getAddress(), (idx + 1) * elementSize), (m_size - idx - 1) * elementSize);
        m_parent->updateBacklinks(hkSerialize::VarN::fromArray(hkAddByteOffset(oldObjects.getAddress(), (idx + 1) * elementSize), oldObjects.getType(), (m_size - idx - 1)), hkSerialize::VarN::fromArray(hkAddByteOffset(m_data, elementSize * idx), oldObjects.getType(), (m_size - idx - 1)));
    }

    hkSerialize::VarN newObjects = hkSerialize::VarN::fromArray(m_data, oldObjects.getType(), newSize);
    m_size = newSize;
    // We may have changed the type, need to re-set these
    m_stride = elementSize;
    m_elemType = newObjects.getType();

    setVariableArrayObjects(newObjects);

    m_parent->addObjectToTrackingMap(newObjects.getFirst(), newSize, elementSize, -1, extrasIndex);
    // We have copied it now that we have resized the array, we had to allocate new storage
    const int id = getVariableArrayObjectId();
    m_parent->m_indexFromAddressMap.insert(m_data, id); // Needed??
}

hkDataObject::Array hkDataObject::Array::swizzleObjectMember(const char* name) const
{
    return hkDataObject::Array(m_idPtr ? m_idPtr : m_data, m_type, m_parent, m_id, name);
}

hkDataObject::Class hkDataObject::Array::getClass() const
{
    if (hkSerialize::VarN objs = getVariableArrayObjects())
    {
        HK_ASSERT_NO_MSG(0x5c9a4192, m_type->asArray());
        const hkReflect::Type* elementType = objs.getType();
        HK_ASSERT_NO_MSG(0x5947b1db, elementType);

        return Class(elementType, m_parent);
    }
    else
    {
        const hkReflect::ArrayType* aType = m_type->asArray();
        HK_ASSERT_NO_MSG(0x5c9a4192, aType);
        hkReflect::ArrayVar handle(m_data, aType);
        const hkReflect::Type* elementType = handle.getSubType();
        HK_ASSERT_NO_MSG(0x5947b1db, elementType);

        return Class(elementType, m_parent);
    }
}

////////////////////
// End Array
////////////////////

hkBool32 hkDataObject::hasMember(const char* name) const
{
    return m_object.getType()->asRecord()->findField(name, false) != HK_NULL;
}

/// Get value of class member named 'name'.
hkDataObject::Value hkDataObject::operator[](const char* name)
{
    const hkReflect::RecordType* rec = m_object.getType()->asRecord();
    const hkReflect::DataFieldDecl f = findFieldInPatchedType(rec, name);
    return Value(hkReflect::Var(hkAddByteOffset(m_object.getAddress(), f.getOffset()), f.getType()), m_parent, m_id);
}

/// Get const value of class member named 'name'.
const hkDataObject::Value hkDataObject::operator[](const char* name) const
{
    const hkReflect::RecordType* rt = m_object.getType()->asRecord();
    const hkReflect::DataFieldDecl f = findFieldInPatchedType(rt, name);
    return Value(hkReflect::Var(hkAddByteOffset(m_object.getAddress(), f.getOffset()), f.getType()), m_parent, m_id);
}

hkDataObject::hkDataObject(hkReflect::Var object, CompatPatcher* parent, hkDataObject::VarOrExtra id)
    : m_object(object)
    , m_parent(parent)
    , m_id(id)
{
    // Check if we need to follow an object link
    if (m_object.getType() && m_object.getAddress() && m_object.getType()->asRecord() && m_parent->getVersionedRecord(m_object.getType()))
    {
        hkReflect::Var objectReference = m_parent->getBackLink(m_object, id);
        if (objectReference)
        {
            //DLOG_IF(void* oldObject = m_object.getAddress());
            m_object = objectReference;
            //DLOG("FOLLOW Link Field at {0} to {1}", oldObject, m_object);
        }
    }
}

hkDataObject::~hkDataObject()
{

}

hkDataObject::hkDataObject() : m_parent(HK_NULL), m_id(0, VAR)
{
}

hkDataObject::hkDataObject(const hkDataObject& o) : m_object(o.m_object), m_parent(o.m_parent), m_id(o.m_id)
{
}
void hkDataObject::operator=(const hkDataObject& o)
{
    m_object = o.m_object;
    m_parent = o.m_parent;
}

bool hkDataObject::operator==(const hkDataObject& o) const
{
    return (m_object.getAddress() == o.m_object.getAddress()) && (m_parent == o.m_parent);
}

class hkDataWorld& hkDataObject::getWorld() const
{
    return m_parent->getWorld();
}

hkDataWorld::hkDataWorld(CompatPatcher* parent) : m_parent(parent)
{
}

hkDataObject hkDataWorld::newObject(hkDataObject::Class t) const
{
    const int newId = m_parent->newObject(t.getName()); 
    hkReflect::Var newObj = m_parent->m_bundle.var(newId);

    return hkDataObject(newObj, m_parent, hkDataObject::VarOrExtra(newId, hkDataObject::VAR));
}

hkBool32 hkDataObject::isNull() const
{
    if (!m_object.getAddress())
    {
        return true;
    }
    if (m_object.getType()->asPointer() && (static_cast<int>(*reinterpret_cast<hkUint32Le*>(m_object.getAddress())) == 0))
    {
        return true;
    }
    return false;
}

class hkDataWorld* hkDataObject::Class::getWorld() const
{
    return &m_parent->getWorld();
}

hkDataObject::Class hkDataObject::Class::getParent() const
{
    return hkDataObject::Class(m_type->getParent(), m_parent);
}


hkDataObject::Class hkDataWorld::findClass(const char* name) const
{
    const int typeVersion = m_parent->m_currentPatchDependencies.getVersion(name);
    hkVersionedRecordType* type = m_parent->findType(name, typeVersion);

    if (type)
    {
        return hkDataObject::Class(type->getCompletedType(), m_parent);
    }
    else
    {
        return hkDataObject::Class(HK_NULL, m_parent);
    }
}

bool hkDataObject::Type::isPointer() const
{
    return m_type->asPointer() != HK_NULL;
}

bool hkDataObject::Type::isClass() const
{
    return m_type->asRecord() != HK_NULL;
}

bool hkDataObject::Type::isTuple() const
{
    return (m_type->asArray() != HK_NULL) && (m_type->asArray()->getFixedCount() > 0);
}

bool hkDataObject::Type::isArray() const
{
    return m_type->asArray() != HK_NULL;
}

hkDataObject::Type hkDataObject::Type::getParent()
{
    
    return Type(m_parent, m_parent->m_typeSet.getParent((hkReflect::Type*)m_type)->getCompletedType());
}

namespace
{
    bool fieldIsVisible(hkReflect::FieldDecl f)
    {
        return f.isSerializable() && f.getName();
    }

    int numVisibleFields(const hkReflect::RecordType* t)
    {
        int total = 0;
        for (int i = 0; i < t->getNumFields(); i++)
        {
            if (fieldIsVisible(t->getField(i)))
            {
                total++;
            }
        }
        return total;
    }
}

int hkDataObject::Class::getNumMembers() const
{
    int totalMembers = 0;

    const hkReflect::RecordType* t = m_type->asRecord();
    while (t)
    {
        totalMembers += numVisibleFields(t);
        t = t->getParentRecord();
    }
    return totalMembers;
}

void hkDataObject::Class::getMemberInfo(int index, MemberInfo& memberInfoOut) const
{
    const hkReflect::RecordType* t = m_type->asRecord();
    while (index >= numVisibleFields(t))
    {
        index -= numVisibleFields(t);
        t = t->getParentRecord();
    }

    // index < numVisibleFields(t)
    int fieldIndex = 0;
    hkReflect::FieldDecl f = t->getField(fieldIndex);
    while (!fieldIsVisible(f))
    {
        fieldIndex++;
        f = t->getField(fieldIndex);
    }

    while (index)
    {
        if (fieldIsVisible(f))
        {
            index--;
        }
        fieldIndex++;
        f = t->getField(fieldIndex);
    }

    HK_ASSERT_NO_MSG(0x59b00527, f);

    memberInfoOut.m_name = f.getName();
}

void hkDataObject::Class::getAllMemberInfos(hkArray<MemberInfo>& memberInfosOut) const
{
    for (const hkReflect::RecordType* t = m_type->asRecord(); t; t = t->getParentRecord())
    {
        hkArrayView<const hkReflect::FieldDecl> fields = t->getFields();

        for (int i = 0; i < fields.getSize(); i++)
        {
            if (fieldIsVisible(fields[i]))
            {
                memberInfosOut.expandOne().m_name = fields[i].getName();
            }
        }
    }
}


hkUlong hkDataObject::Class::getValueParameter(const char* name) const
{
    HK_ON_DEBUG(const hkReflect::RecordType* t = m_type->asRecord());
    HK_ASSERT_NO_MSG(0x26376e47, t);

    const hkReflect::Template* params = m_type->getTemplate();
    HK_ASSERT_NO_MSG(0x19f8b36a, params);


    for (int i = 0; i < params->getNumParams(); i++)
    {
        const hkReflect::Template::Parameter* thisParam = params->getParam(i);

        if (hkString::strCmp(thisParam->getName(), name) == 0)
        {
            const hkUlong ret = thisParam->getAsValue();
            return ret;
        }
    }

    HK_ASSERT_NO_MSG(0x7f2c740e, 0); // Not found
    return 0;
}

hkDataObject::Class::Class(const hkReflect::Type* type, class CompatPatcher* parent)
    : m_parent(parent)
    , m_type(type)
{
}




//////////////////



#ifndef HK_DYNAMIC_DLL
static
#endif
hkReflect::Version::PatcherInterface2014* s_createCompatPatcherImplementation(const hkReflect::TypeReg* cb, hkReflect::Version::PatchSet& patches)
{
    HK_OPTIONAL_COMPONENT_MARK_USED(hkCompatPatcher);
    return new CompatPatcher(cb, patches);
}

HK_OPTIONAL_COMPONENT_DEFINE(hkCompatPatcher, hkReflect::Version::PatcherInterface2014::s_createPatcher, s_createCompatPatcherImplementation);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
