// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Compat/hkCompat.h>
#include <Common/Compat/Common/Serialize/Data/hkDataObject.h>
#include <Common/Compat/Common/Serialize/Version/hkVersionPatchManager.h>
#include <Common/Base/KeyCode.h>
#include <Common/Base/Math/Vector/hkPackedVector3.h>

namespace HK_UNITY_ANONYMOUS_NAMESPACE
{
    static void copyArrayOfObjectMembers(hkDataObject& obj, const char* oldName, const char* newName)
    {
        hkDataArray srcArray = obj[oldName].asArray();
        hkDataArray dstArray = obj[newName].asArray();

        const int size = srcArray.getSize();
        if (size)
        {
            const hkDataClass objectClass = srcArray.getClass();
            dstArray.setSize(size);

            for (int j = 0; j < size; j++)
            {
                hkDataObject srcObj = srcArray[j].asObject();
                hkDataObject dstObj = dstArray[j].asObject();

                hkArray<hkDataClass::MemberInfo> memberInfos;
                objectClass.getAllMemberInfos(memberInfos);

                for (int i = 0; i < memberInfos.getSize(); i++)
                {
                    const char* memberName = memberInfos[i].m_name;

                    if (hkString::strCmp(memberName, "isExclusive") == 0)
                    {
                        // Work around some integer limit issues
                        dstObj[memberName] = int(srcObj[memberName].asInteger<hkUint32>());
                    }
                    else
                    {
                        dstObj[memberName] = srcObj[memberName];
                    }
                }
            }
        }
    }
}

static void hknpBody_3_to_4(hkDataObject& obj)
{
    //HK_WARN_ONCE(0x7c259ce4, "Versioning of hknpBody::m_aabb is not yet implemented" );
}

static void hknpShape_3_to_4(hkDataObject& obj)
{
    // Added m_dispatchType::DEBRIS
    int dispatchType = obj["dispatchType"].asInt();
    if( dispatchType > 0 )
    {
        obj["dispatchType"] = dispatchType + 1;
    }
}

static void hknpCompressedMeshShape_5_to_6(hkDataObject& obj)
{
    hkDataObject oldData = obj["data_old"].asObject();
    obj["data"] = oldData;
    obj["data_old"].setNull();
}

static void hknpCompressedHeightFieldShape_0_to_hknpCompressedHeightFieldShape_3(hkDataObject& obj)
{
    obj["includeShapeKeyInSdfContacts_new"] = obj["includeShapeKeyInSdfContacts"].asBool();
    obj["aabb_new"] = obj["aabb"].asObject();
    for (int i = 0; i < 4; i++)
    {
        obj["floatToIntScale_new"].asArray()[i] = obj["floatToIntScale"].asArray()[i].asReal();
        obj["intToFloatScale_new"].asArray()[i] = obj["intToFloatScale"].asArray()[i].asReal();
    }
    obj["intSizeX_new"] = obj["intSizeX"].asInt();
    obj["intSizeZ_new"] = obj["intSizeZ"].asInt();
    obj["numBitsX_new"] = obj["numBitsX"].asInt();
    obj["numBitsZ_new"] = obj["numBitsZ"].asInt();

    obj["minMaxTree_new"] = obj["minMaxTree"].asObject();
    obj["minMaxTreeCoarseness_new"] = obj["minMaxTreeCoarseness"].asInt();

    // Copy parent fields into temporaries
    obj["dispatchType_new"] = obj["dispatchType"].asInt();
    obj["convexRadius_new"] = obj["convexRadius"].asReal();
    obj["properties_new"] = obj["properties"].asObject();
    obj["userData_new"] = obj["userData"].asInt();
    obj["numShapeKeyBits_new"] = obj["numShapeKeyBits"].asInt();
    obj["flags_new"] = obj["flags"].asInt();
    obj["type_new"] = obj["type"].asInt();
    obj["edgeWeldingMap_new"] = obj["edgeWeldingMap"].asObject();
    obj["shapeTagCodecInfo_new"] = obj["shapeTagCodecInfo"].asInteger<hkUint32>();
    obj["materialTable_new"] = obj["materialTable"].asObject();
}

static void hknpCompressedHeightFieldShape_0_to_hknpCompressedHeightFieldShape_3_setFieldsFromParents(hkDataObject& obj)
{
    // We copy the parent fields as we can't just rename to them, they are
    // already present. If we don't do this, we could get the default
    // parent values
    obj["dispatchType"] = obj["dispatchType_new"].asInt();
    obj["convexRadius"] = obj["convexRadius_new"].asReal();
    obj["properties"] = obj["properties_new"].asObject();
    obj["userData"] = obj["userData_new"].asInt();
    obj["numShapeKeyBits"] = obj["numShapeKeyBits_new"].asInt();
    obj["flags"] = obj["flags_new"].asInt();
    obj["type"] = obj["type_new"].asInt();
    obj["edgeWeldingMap"] = obj["edgeWeldingMap_new"].asObject();
    obj["shapeTagCodecInfo"] = obj["shapeTagCodecInfo_new"].asInt();
    obj["materialTable"] = obj["materialTable_new"].asObject();
}

static void hknpCompressedHeightFieldShape_3_to_hknpHeightFieldShape_4(hkDataObject& obj)
{
    const hkDataWorld* world = obj.getClass().getWorld();
    hkDataClass geometryClass = world->findClass("hknpDefaultHeightFieldGeometry");
    hkDataObject geometryObj = world->newObject(geometryClass);

    geometryObj["storage"] = obj["storage"];
    geometryObj["shapeTags"] = obj["shapeTags"];
    geometryObj["triangleFlip"] = obj["triangleFlip"];
    geometryObj["resX"] = obj["intSizeX"].asInt() + 1;
    geometryObj["resZ"] = obj["intSizeZ"].asInt() + 1;
    geometryObj["heightScale"] = obj["scale"].asReal();
    geometryObj["heightOffset"] = obj["offset"].asReal();

    obj["geometry"] = geometryObj;
}

struct hknpMotionTypeCopy
{
    enum Enum
    {
        STATIC = 0,     ///< Body that will not change position or orientation unless manually set
        KEYFRAMED,      ///< Body with infinite mass and inertia
        DYNAMIC,        ///< Fully dynamic body
        NUM_TYPES
    };
};

struct hknpCollisionFlagsCopy
{
    enum Enum
    {
        IS_STATIC       = 1<<0,
        IS_DYNAMIC      = 1<<1,
        IS_KEYFRAMED    = 1<<2,
        ALL_INTERNALS = IS_STATIC | IS_DYNAMIC | IS_KEYFRAMED
    };
};

namespace
{
    hkDataObject getShapeMassProps(hkDataObject shape)
    {
        const int MASS_PROPERTIES_KEY = 0xf100;

        hkDataObject propContainer = shape["properties"].asObject();
        if(propContainer.isNull())
        {
            return hkDataObject();
        }

        hkDataArray props = propContainer["entries"].asArray();
        for(int p = 0; p < props.getSize(); p++)
        {
            if(props[p].asObject()["key"].asInt() == MASS_PROPERTIES_KEY)
            {
                return props[p].asObject()["object"].asObject();
            }
        }

        return hkDataObject();
    }

    hkDataObject createShapeMassProps(hkDataObject shape)
    {
        const int MASS_PROPERTIES_KEY = 0xf100;
        const hkDataWorld* world = shape.getClass().getWorld();

        // 1. No properties container
        hkDataObject propContainer = shape["properties"].asObject();
        if(propContainer.isNull())
        {
            hkDataClass propertiesContainerClass(world->findClass("hkRefCountedProperties"));
            propContainer = world->newObject(propertiesContainerClass);
            shape["properties"] = propContainer;
        }

        // 2. Properties container with null mass properties
        hkDataObject massProps;
        hkDataArray props = propContainer["entries"].asArray();
        for(int p = 0; p < props.getSize(); p++)
        {
            if(props[p].asObject()["key"].asInt() == MASS_PROPERTIES_KEY)
            {
                massProps = props[p].asObject()["object"].asObject();
                if(massProps.isNull())
                {
                    hkDataClass shapeMassPropsClass(world->findClass("hknpShapeMassProperties"));
                    massProps = world->newObject(shapeMassPropsClass);
                    props[p].asObject()["object"] = massProps;
                }
            }
        }

        // 3. Properties container with no mass properties
        if(massProps.isNull())
        {
            hkDataClass massPropsClass(world->findClass("hknpShapeMassProperties"));
            massProps = world->newObject(massPropsClass);

            hkDataClass entryClass(world->findClass("hkRefCountedProperties::Entry"));
            hkDataObject entry = world->newObject(entryClass);
            entry["key"] = MASS_PROPERTIES_KEY;
            entry["object"] = massProps;
            props.setSize(props.getSize() + 1);
            props[props.getSize() - 1] = entry;
        }

        return massProps;
    }
}

static void hknpPhysicsSystemData_0_to_1(hkDataObject& sys)
{
    hkDataArray oldInfos = sys["bodyCinfos_old"].asArray();
    hkDataArray bodyInfos = sys["bodyCinfos"].asArray();

    int numBodyCinfos = oldInfos.getSize();
    bodyInfos.setSize(numBodyCinfos);
    for(int i = 0; i < numBodyCinfos; i++)
    {
        const hkDataClass cinfoClass = oldInfos[i].asObject().getClass();
        hkDataObject srcObj = oldInfos[i].asObject();
        hkDataObject dstObj = bodyInfos[i].asObject();

        hkArray<hkDataClass::MemberInfo> memberInfos;
        cinfoClass.getAllMemberInfos(memberInfos);

        for (int j = 0; j < memberInfos.getSize(); j++)
        {
            const char* memberName = memberInfos[j].m_name;
            dstObj[memberName] = srcObj[memberName];
        }

        dstObj["attachedBody"] = -1;
    }

    hkDataArray motions = sys["motionCinfos"].asArray();
    hkArray<int> previousMotionUsers;
    previousMotionUsers.setSize(motions.getSize(), -1);

    for(int b = 0; b < numBodyCinfos; b++)
    {
        hkDataObject body = bodyInfos[b].asObject();
        body["flags"] = body["flags"].asInt() & ~hknpCollisionFlagsCopy::ALL_INTERNALS;

        int motionId = body["reservedMotionId"].asInt();
        if(motionId != HK_INT32_MAX)
        {
            body["attachedBody"] = previousMotionUsers[motionId];
            previousMotionUsers[motionId] = b;
        }

        if(motionId != HK_INT32_MAX)
        {
            hkDataObject motion = motions[motionId].asObject();

            body["motionPropertiesId"] = motion["motionPropertiesId"].asInt();
            body["reservedMotionId"] = HK_INT32_MAX;


            // Inverse mass
            hkReal invMass = motion["inverseMass"].asReal();
            hkReal mass = -1;
            if(invMass != 0)
            {
                mass = 1.0f / invMass;
                body["motionType"] = hknpMotionTypeCopy::DYNAMIC;
            }
            else
            {
                body["motionType"] = hknpMotionTypeCopy::KEYFRAMED;
            }
            body["mass"] = mass;

            hkDataObject bodyShape = body["shape"].asObject();
            if(bodyShape.isNull())
            {
                continue;
            }

            hkDataObject massProperties = getShapeMassProps(bodyShape);
            // If we don't have shape mass properties, create one from the motion
            if(massProperties.isNull())
            {
                massProperties = createShapeMassProps(bodyShape);
                hkDataObject compressedProps = massProperties["compressedMassProperties"].asObject();

                // Orientation
                hkQuaternionf motionOrientation = motion["orientation"].asQuaternionf();
                hkQuaternionf bodyOrientation = body["orientation"].asQuaternionf();
                hkQuaternionf majorAxisSpace;
                majorAxisSpace.setInverseMul(bodyOrientation, motionOrientation);

                // COM
                hkVector4f com;
                hkVector4f comW = motion["centerOfMassWorld"].asVector4f();
                hkVector4f bPos = body["position"].asVector4f();
                com.setSub(comW, bPos);
                com.setXYZ_W(com, hkSimdFloat32_Minus1);
                com.setRotatedInverseDir(motionOrientation, com);
                hkPackedVector3 packedCOM;
                packedCOM.pack(com);
                hkInt16* comData = reinterpret_cast<hkInt16*>(&packedCOM);
                for(int i = 0; i < 4; i++)
                {
                    compressedProps["centerOfMass"].asArray()[i] = comData[i];
                }

                // Inertia
                hkVector4f inertia;
                inertia.setReciprocal<HK_ACC_MID, HK_DIV_SET_ZERO>(motion["inverseInertiaLocal"].asVector4f());
                hkPackedVector3 packedInertia;
                packedInertia.pack(inertia);
                hkInt16* inertiaData = reinterpret_cast<hkInt16*>(&packedInertia);
                for(int i = 0; i < 4; i++)
                {
                    compressedProps["inertia"].asArray()[i] = inertiaData[i];
                }

                hkPackedUnitVector<4> packedMajorAxisSpace;
                packedMajorAxisSpace.pack(majorAxisSpace.m_vec);
                hkInt16* majorAxisSpaceData = reinterpret_cast<hkInt16*>(&packedMajorAxisSpace);
                for(int i = 0; i < 4; i++)
                {
                    compressedProps["majorAxisSpace"].asArray()[i] = majorAxisSpaceData[i];
                }

                compressedProps["mass"] = body["mass"].asReal();
                compressedProps["volume"] = -1.0f; // Set this to an invalid value; recompute it in the SDK.
            }
        }
    }
}

static void hknpHeightFieldShape_4_to_5(hkDataObject& obj)
{
    const hkDataWorld* world = obj.getClass().getWorld();
    hkDataClass shapeDataClass = world->findClass("hknpHeightFieldBoundingVolume");
    hkDataObject shapeDataObj = world->newObject(shapeDataClass);

    shapeDataObj["minMaxTree"] = obj["minMaxTree"];
    shapeDataObj["minLevel"] = obj["minMaxTreeCoarseness"];

    obj["boundingVolumeData"] = shapeDataObj;

    const int hknpCollisionDispatchType_COMPOSITE = 3;
    if (obj["dispatchType"].asInt() == hknpCollisionDispatchType_COMPOSITE)
        obj["markBorderEdgesForWelding"] = true;
}

static void hknpBody_5_to_6(hkDataObject& obj)
{
    // motionToBodyRotation changed type
    hkPackedUnitVector<4> motionToBodyRotation;
    motionToBodyRotation.m_vec[0] = (hkInt16)obj["motionToBodyRotation_old"].asArray()[0].asInt();
    motionToBodyRotation.m_vec[1] = (hkInt16)obj["motionToBodyRotation_old"].asArray()[1].asInt();
    motionToBodyRotation.m_vec[2] = (hkInt16)obj["motionToBodyRotation_old"].asArray()[2].asInt();
    motionToBodyRotation.m_vec[3] = (hkInt16)obj["motionToBodyRotation_old"].asArray()[3].asInt();

    hkVector4 vec4;
    motionToBodyRotation.unpack( &vec4 );

    hkQuaternionf quat;
    quat.m_vec.set( vec4 );
    quat.normalize();

    obj["motionToBodyRotation"] = quat;
}

static void patch_IdTohknpBodyId(hkDataObject& obj, const char* oldName, const char* newName)
{
    obj[newName].asObject()["serialAndIndex"] = obj[oldName].asInt();
}

static void patch_IdTohknpConstraintId(hkDataObject& obj, const char* oldName, const char* newName)
{
    obj[newName].asObject()["value"] = obj[oldName].asInt();
}

static void patch_pointerType_to_hkReferencedObject(hkDataObject& obj, const char* oldName, const char* newName)
{
    obj[newName] = obj[oldName].asObject();
    obj[oldName].setNull();
}

void HK_CALL registerNewPhysicsPatches_2014_2_5(hkVersionPatchManager& man)
{
#   define HK_PATCHES_FILE <Common/Compat/Patches/2014_2_5/hknpPatches_2014_2_5.hxx>
#   include <Common/Compat/Common/Serialize/Version/hkVersionPatchManager.cxx>
#   undef HK_PATCHES_FILE
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
