// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL !REFLECT
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#pragma once

#include <Common/Base/Config/hkConfigVersion.h>
#include <Common/Base/System/Error/hkDefaultError.h>
#include <Common/Base/System/Error/hkFilterError.h>
#include <Common/Base/Container/String/hkStringBuf.h>
#include <Common/Base/Config/hkConfigThread.h>


//
//  Test report function type

typedef hkBool(HK_CALL *hkTestReportFunctionType)(hkBool32 cond, const char* desc, const char* file, int line);

/// wrapper function to call g_hkTestReportFunction which allows to set a breakpoint
HK_EXPORT_COMMON hkBool HK_CALL hkTestReportFunction( hkBool32 cond, const char* desc, const char* file, int line );
HK_EXPORT_COMMON hkBool HK_CALL hkTestSuccess( hkResult res, const char* func, const char* file, int line );

/************* PUBLIC *******************/
#define HK_TEST(CONDITION)               hkTestReportFunction( static_cast<bool>(CONDITION), #CONDITION,  __FILE__, __LINE__)
#define HK_TEST1(CONDITION,DESCRIPTION)  hkTestReportFunction( static_cast<bool>(CONDITION), DESCRIPTION, __FILE__, __LINE__)

#define HK_TEST_SUCCESS(CONDITION) hkTestSuccess(CONDITION, #CONDITION, __FILE__, __LINE__)
#define HK_TEST_EQ(A,B) hkTestReportEq((A), (B), #A, #B, __FILE__, __LINE__)
#define HK_TEST_NE(A,B) hkTestReportNe((A), (B), #A, #B, __FILE__, __LINE__)

/// Test, where you can provide an extra value to be converted to string
#define HK_TEST2(CONDITION,DESCRIPTION)  do {   \
    char msgBuf[512];                               \
    hkOstream msg(msgBuf, sizeof(msgBuf), true);    \
    msg << #CONDITION  << " (" << DESCRIPTION << ')'; \
    hkTestReportFunction( (CONDITION), msgBuf, __FILE__, __LINE__); \
} while(0)

#define HK_TEST3(CONDITION,...) \
    do { \
        hkStringBuf strBuf; \
        strBuf.formatL( HK_VARARGS1_WRAP((__VA_ARGS__)) ); \
        hkTestReportFunction( (CONDITION), strBuf.cString(), __FILE__, __LINE__); \
    } while ( 0 )

#define HK_TEST_FLOAT_EQ(A, B, EPS, MSG) hkTestReportFunction( static_cast<bool>(hkMath::abs(static_cast<double>((A)-(B))) < (EPS)), MSG, __FILE__, __LINE__)

namespace hkTest
{
    struct TestStatementBase
    {
        virtual void testMe() = 0;
    };

    template<typename T>
    struct TestStatement : public TestStatementBase
    {
        TestStatement(T& t) : m_t(t) {}
        virtual void testMe() { m_t(); }
        T m_t;
    };
}

// Macro needed to prevent namespace errors caused by CodeWarrior #defining setjmp as ::std::__setjmp

// Macro used to test if an specific error message is raised during the execution of a statement and fail
// the test if it is not. This version will restore the program state after receiving the message, and will
// not execute any code after the message is raised.

#define HK_TEST_MESSAGE(MSG_TYPE, ID, STATEMENT, DESCRIPTION)  do { \
    hkFilterError filterError; \
    auto stmt = [&]() -> void { STATEMENT; }; \
    hkTest::TestStatement<decltype(stmt)> ts(stmt); \
    filterError.checkMessage(MSG_TYPE, ID, &ts); \
    HK_TEST2(filterError.wasMessageRaised(), DESCRIPTION); \
} while(0)

// Macro used to test if an specific error message is raised during the execution of a statement and fail
// the test if it is not. This version continue execution after the message is raised.
#define HK_TEST_MESSAGE_AND_CONTINUE(MSG_TYPE, ID, STATEMENT, DESCRIPTION)  do { \
    hkFilterError filterError; \
    auto stmt = [&]() -> void { STATEMENT; }; \
    hkTest::TestStatement<decltype(stmt)> ts(stmt); \
    filterError.checkMessageAndContinue(MSG_TYPE, ID, &ts); \
    HK_TEST2(filterError.wasMessageRaised(), DESCRIPTION); \
} while(0)


#define HK_TEST_ERROR(ID, STATEMENT)    HK_TEST_MESSAGE(hkError::MESSAGE_ERROR, ID, STATEMENT, "Error " #ID " not triggered.")
#define HK_TEST_ERROR2(ID, STATEMENT, DESCRIPTION)  HK_TEST_MESSAGE(hkError::MESSAGE_ERROR, ID, STATEMENT, DESCRIPTION)

#if !defined(HK_PLATFORM_CTR) && defined(HK_DEBUG)
#   define HK_TEST_ASSERT(ID, STATEMENT)    HK_TEST_MESSAGE(hkError::MESSAGE_ASSERT, ID, STATEMENT, "Assert " #ID " not triggered.")
#   define HK_TEST_ASSERT2(ID, STATEMENT, DESCRIPTION)  HK_TEST_MESSAGE(hkError::MESSAGE_ASSERT, ID, STATEMENT, DESCRIPTION)
#   define HK_TEST_ASSERT_AND_CONTINUE(ID, STATEMENT)   HK_TEST_MESSAGE_AND_CONTINUE(hkError::MESSAGE_ASSERT, ID, STATEMENT, "Assert " #ID " not triggered.")
#   define HK_TEST_ASSERT_AND_CONTINUE2(ID, STATEMENT, DESCRIPTION) HK_TEST_MESSAGE_AND_CONTINUE(hkError::MESSAGE_ASSERT, ID, STATEMENT, DESCRIPTION)
#   define HK_ON_TEST_ASSERT_ENABLED(STATEMENT) STATEMENT
#else   // no asserts in debug
#   define HK_TEST_ASSERT(ID, STATEMENT)
#   define HK_ON_TEST_ASSERT_ENABLED(STATEMENT)
#   define HK_TEST_ASSERT2(ID, STATEMENT, DESCRIPTION)
#   define HK_TEST_ASSERT_AND_CONTINUE(ID, STATEMENT)
#   define HK_TEST_ASSERT_AND_CONTINUE2(ID, STATEMENT, DESCRIPTION)
#endif

//
//  Test report function pointer

extern HK_EXPORT_COMMON hkTestReportFunctionType g_hkTestReportFunction;

template <class T, class S>
HK_ALWAYS_INLINE bool hkTestDiffer(const T& l, const S& r)
{
    return !hkTrait::EqualValues<T, S>::func(l, r);
}

template <class T, class S>
inline hkBool hkTestReportEq(const T& l, const S& r, const char* left, const char* right, const char* file, int line)
{
    if( hkTestDiffer(l,r) )
    {
        hkStringBuf sb;
        sb.formatL( HK_VARARGS1_WRAP(("left({}) == right({})", l, r)) );
        return hkTestReportFunction(false, sb.cString(), file, line);
    }
    return hkTestReportFunction(true, "Passed", file, line);
}

template <class T, class S>
inline hkBool hkTestReportNe(const T& l, const S& r, const char* left, const char* right, const char* file, int line)
{
    if( !hkTestDiffer(l,r) )
    {
        hkStringBuf sb;
        sb.formatL( HK_VARARGS1_WRAP(("left({}) != right({})", l, r)) );
        return hkTestReportFunction(false, sb.cString(), file, line);
    }
    return hkTestReportFunction(true, "Passed", file, line);
}


/************* INTERNAL USE ONLY *******************/
#include <Common/Base/System/Log/hkLog.h>

namespace UnitTest
{
        /// Sets m_stopMessage to true for a log Output, used for unit tests only.
    template <typename OUTPUT>
    struct LogStopT : public OUTPUT
    {
        LogStopT() : OUTPUT(hkLog::Level::Debug), m_stopMessage(true) {}
        hkLog::Output::Action put(const hkLog::Message& msg) HK_OVERRIDE
        {
            OUTPUT::put(msg);
            return m_stopMessage ? hkLog::Output::StopMessagePropagation : hkLog::Output::PropagateToNextOutput;
        }
        bool m_stopMessage;
    };

    namespace Raii
    {
            /// Captures the thread-local log output and stops it proceeding to the file & global logs.
        struct CaptureThreadLogOutput : LogStopT<hkLog::TextOutput>
        {
            CaptureThreadLogOutput() { m_hook.hook(this); }
            hkLog::Raii::ThreadHookOutput m_hook;
            bool hasText() const
            {
                return m_items.isEmpty() == false;
            }
            //Inherits char* TextOutput::getText();
            //Inherits bool LogStopT.m_stopMessage;
        };
    }
}

struct HK_EXPORT_COMMON hkTestEntry
{
    typedef int (HK_CALL *hkTestFunction)();

    HK_DECLARE_CLASS(hkTestEntry, NewOpaque);
    hkTestEntry(hkTestEntry::hkTestFunction func, const char* name, const char* category, const char* path);

    struct DontRegister {};
    hkTestEntry(hkTestEntry::hkTestFunction func, const char* name, const char* category, const char* path, DontRegister );

    hkTestFunction m_func;
    const char* m_name;
    const char* m_category;
    const char* m_path;

    hkTestEntry* m_next;
    static hkTestEntry* s_head;
};

// If a test isn't registering, check the Tests.h file in its project for errors.
#define HK_TEST_REGISTER(func, category, menu, path) hkTestEntry register##func( func, #func, category, menu path )

#define HK_REGRESSION_REPORT(NAME, TYPE, VALUE) HK_MULTILINE_MACRO_BEGIN                \
    char reportBuf[512];                                                                \
    hkErrStream ostr(reportBuf,sizeof(reportBuf));                                      \
    ostr << "[REGRESSION:" << NAME << ":" << TYPE << ":" << VALUE << "]";               \
    hkError::messageReport(0, reportBuf, HK_NULL, 0);                                   \
    HK_MULTILINE_MACRO_END

#define HK_REGRESSION_CHECK_RANGE(NAME, TYPE, VALUE, MINVALUE, MAXVALUE)    HK_MULTILINE_MACRO_BEGIN                                                                            \
    char reportBuf[512];                                                                                \
    hkErrStream ostr(reportBuf,sizeof(reportBuf));                                                      \
    if ((VALUE >= MINVALUE) && ( VALUE <= MAXVALUE ))                                                   \
{                                                                                                       \
    ostr << "[REGRESSION:" << NAME << ":" << TYPE << ":" << VALUE << "]";                               \
    hkError::messageReport(0, reportBuf, HK_NULL, 0);                                                   \
}                                                                                                       \
                                                                            else                        \
{                                                                                                       \
    ostr << "[REGRESSION FAILED:" << NAME << ":" << TYPE << ":" << VALUE << ": Range(" << MINVALUE << "," << MAXVALUE << ") ]"; \
    hkError::messageError(-1, reportBuf, __FILE__, __LINE__);                                           \
}                                                                                                       \
    HK_MULTILINE_MACRO_END

class HK_EXPORT_COMMON hkUnitTest
{
public:
    HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_DEMO, hkUnitTest);

    static hkUint32 randSeed;

    static hkUint32 HK_CALL rand();
    static hkReal HK_CALL rand01();
    static inline hkReal HK_CALL randRange(hkReal minv, hkReal maxv)
    {
        return minv + rand01() * (maxv-minv);
    }

#if (HK_CONFIG_THREAD == HK_CONFIG_MULTI_THREADED)
    static class hkJobQueue* s_jobQueue;
    static class hkTaskQueue* s_taskQueue;
    static class hkDefaultTaskQueue* s_defaultTaskQueue;    // if taskQueue is the default one this is set to taskQueue
    static class hkThreadPool* s_threadPool;
#endif

    /// KISS Based PRNG (http://www.cs.ucl.ac.uk/staff/d.jones/GoodPracticeRNG.pdf)
    /// The period about 2^121
    /// This means a current SandyBridge CPU can safely use it continuously for about a month
    struct HK_EXPORT_COMMON Prng
    {
        /// Ctor.
        Prng(hkUint32 seed = 123456789);

        /// Return the next unsigned int, range: [0,4294967295]
        hkUint32    nextUint32();

        /// Return the next signed int, range: [0,2147483647]
        hkInt32     nextInt32();

        /// Return the next single precision floating point number, range: [0,1]
        hkFloat32   nextFloat();

        /// Return the next double precision floating point number, range: [0,1]
        hkDouble64  nextDouble();

        /// Return the next hkReal, range: [0,1]
        hkReal      nextReal();

        /// Return the next single precision scalar, range: [0,1]
        void        nextSimdScalar(hkSimdFloat32& r);

        /// Return the next double precision scalar, range: [0,1]
        void        nextSimdScalar(hkSimdDouble64& r);

        /// Return the next single precision scalar, range: [-1,1]
        void        nextSimdScalar11(hkSimdFloat32& r);

        /// Return the next double precision scalar, range: [-1,1]
        void        nextSimdScalar11(hkSimdDouble64& r);

        /// Return the next single precision vector, range: [0,1] x 4
        void        nextVector(hkVector4f& v);

        /// Return the next double precision vector, range: [0,1] x 4
        void        nextVector(hkVector4d& v);

        /// Return the next single precision vector, range: [-1,1] x 4
        void        nextVector11(hkVector4f& v);

        /// Return the next double precision vector, range: [-1,1] x 4
        void        nextVector11(hkVector4d& v);

        /// Return the next single precision vector of unit length.
        void        nextUnitVector3(hkVector4f& v);

        /// Return the next double precision vector of unit length.
        void        nextUnitVector3(hkVector4d& v);

        /// Return the next single precision vector of unit length, pointing in opposite hemisphere as \a normal.
        void        nextUnitVector3(hkVector4f& v, hkVector4fParameter normal);

        /// Return the next double precision vector of unit length, pointing in opposite hemisphere as \a normal.
        void        nextUnitVector3(hkVector4d& v, hkVector4dParameter normal);

        /// Return the next single precision quaternion
        void        nextQuaternion(hkQuaternionf& v);

        /// Return the next double precision quaternion
        void        nextQuaternion(hkQuaterniond& v);

        /// Return the next single precision rotation matrix
        void        nextRotation(hkRotationf& r);

        /// Return the next double precision rotation matrix
        void        nextRotation(hkRotationd& r);

        /// Return the next 3d bary-center in single precision
        void        nextBaryCenter3D(hkVector4f& bc);

        /// Return the next 3d bary-center in double precision
        void        nextBaryCenter3D(hkVector4d& bc);

        /// Return the next single precision value as Gaussian deviate (ie. with mean 0 and standard deviation 1)
        hkFloat32   nextFloatGauss();

        /// Return the next double precision value as Gaussian deviate (ie. with mean 0 and standard deviation 1)
        hkDouble64  nextDoubleGauss();

        /// Return the next hkReal value as Gaussian deviate (ie. with mean 0 and standard deviation 1)
        hkReal      nextRealGauss();

        /// Returns a new PRNG.
        Prng        nextPrng();

        hkUint32    m_x, m_y, m_z, m_w, m_c;
    };
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
