// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/Thread/JobQueue/hkJobQueue.h>
#include <Common/Base/Math/Vector/hkVector4Util.h>


hkTestEntry* hkTestEntry::s_head;

hkTestEntry::hkTestEntry(hkTestEntry::hkTestFunction func, const char* name, const char* category, const char* path)
: m_next( s_head)
{
    m_func = func;
    m_name = name;
    m_category = category;
    m_path = path;
    s_head = this;
}

hkTestEntry::hkTestEntry(hkTestEntry::hkTestFunction func, const char* name, const char* category, const char* path, DontRegister )
: m_next( HK_NULL )
{
    m_func = func;
    m_name = name;
    m_category = category;
    m_path = path;
}

#if (HK_CONFIG_THREAD == HK_CONFIG_MULTI_THREADED)
    hkJobQueue* hkUnitTest::s_jobQueue = HK_NULL;
    hkTaskQueue* hkUnitTest::s_taskQueue = HK_NULL;
    hkDefaultTaskQueue* hkUnitTest::s_defaultTaskQueue = HK_NULL;
    hkThreadPool* hkUnitTest::s_threadPool = HK_NULL;
#endif

hkUint32 hkUnitTest::randSeed = 'h'+'a'+'v'+'o'+'k';

hkReal HK_CALL hkUnitTest::rand01()
{
    const hkUint32 a = 1103515245;
    const hkUint32 c = 12345;
    const hkUint32 m = hkUint32(-1) >> 1;
    randSeed = (a * randSeed + c ) & m;
    return (hkReal(randSeed) / hkReal(m));
}

hkUint32 HK_CALL hkUnitTest::rand()
{
    const hkUint32 a = 1103515245;
    const hkUint32 c = 12345;
    const hkUint32 m = hkUint32(-1) >> 1;
    randSeed = (a * randSeed + c ) & m;
    return randSeed;
}

// Prng ctor. Don't touch the numbers unless you are sure what you are doing!
hkUnitTest::Prng::Prng(hkUint32 seed)
    : m_x(seed)
    , m_y(234567891)
    , m_z(345678912)
    , m_w(456789123)
    , m_c(0)
{}

// Return the next unsigned int, range: [0,4294967295]
hkUint32    hkUnitTest::Prng::nextUint32()
{
    HK_ASSERT(0x53811bd7, m_y!=0, "Setting Prng::m_y to zero destroys its numerical properties.");

    m_y ^= (m_y<<5);
    m_y ^= (m_y>>7);
    m_y ^= (m_y<<22);
    const hkInt32 t = (hkInt32) (m_z + m_w + m_c);
    m_z =   m_w;
    m_c =   t < 0 ? 1 : 0;
    m_w =   (hkUint32) (t & 2147483647);
    m_x +=  1411392427;

    return m_x + m_y + m_w;
}

// Return the next signed int, range: [0,2147483647]
hkInt32     hkUnitTest::Prng::nextInt32()
{
    return (hkInt32) (nextUint32() >> 1);
}

// Return the next single precision floating point number, range: [0,1]
hkFloat32   hkUnitTest::Prng::nextFloat()
{
    return hkFloat32(nextUint32()) / 4294967296.0f;
}

// Return the next double precision floating point number, range: [0,1]
hkDouble64  hkUnitTest::Prng::nextDouble()
{
    hkUint32 a = nextUint32() >> 6; // upper 26 bits
    hkUint32 b = nextUint32() >> 5; // upper 27 bits
    return (hkDouble64(a) * 134217728.0 + hkDouble64(b)) / 9007199254740992.0;
}

// Return the next hkReal, range: [0,1]
hkReal      hkUnitTest::Prng::nextReal()
{
#if defined(HK_REAL_IS_DOUBLE)
    return nextDouble();
#else
    return nextFloat();
#endif
}

// Return the next single precision scalar, range: [0,1]
void        hkUnitTest::Prng::nextSimdScalar(hkSimdFloat32& r)
{
    r.setFromInt32(nextInt32()); // trim the sign
    const hkSimdFloat32 scale = hkSimdFloat32::fromFloat(1.0f / 2147483647.0f);
    r *= scale;
}

// Return the next double precision scalar, range: [0,1]
void        hkUnitTest::Prng::nextSimdScalar(hkSimdDouble64& r)
{
    hkSimdDouble64 a; a.setFromInt32(nextUint32() >> 6); // upper 26 bits
    hkSimdDouble64 b; b.setFromInt32(nextUint32() >> 5); // upper 27 bits
    const hkSimdDouble64 scale = hkSimdDouble64::fromFloat(1.0 / 9007199254740992.0);
    const hkSimdDouble64 offset = hkSimdDouble64::fromFloat(134217728.0);
    r.setAddMul(b, a, offset);
    r *= scale;
}

// Return the next single precision scalar, range: [-1,1]
void        hkUnitTest::Prng::nextSimdScalar11(hkSimdFloat32& r)
{
    nextSimdScalar(r);
    r += r;
    r -= hkSimdFloat32_1;
}

// Return the next double precision scalar, range: [-1,1]
void        hkUnitTest::Prng::nextSimdScalar11(hkSimdDouble64& r)
{
    nextSimdScalar(r);
    r += r;
    r -= hkSimdDouble64_1;
}

// Return the next single precision vector, range: [0,1] x 4
void        hkUnitTest::Prng::nextVector(hkVector4f& v)
{
    HK_ALIGN16(hkUint32 i_rnd[4]) = {0,0,0,0};
    for(int i=0; i<4; ++i)
    {
        i_rnd[i] = nextUint32();
    }
    hkIntVector iv; iv.load<4>(i_rnd);
    iv.setShiftRight32<1>(iv); // trim the sign
    iv.convertS32ToF32(v);
    const hkSimdFloat32 scale = hkSimdFloat32::fromFloat(1.0f / 2147483647.0f);
    v.mul(scale);
}

// Return the next double precision vector, range: [0,1] x 4
void        hkUnitTest::Prng::nextVector(hkVector4d& v)
{
    HK_ALIGN16(hkUint32 i_rndA[4]) = {0,0,0,0};
    HK_ALIGN16(hkUint32 i_rndB[4]) = {0,0,0,0};
    for(int i=0; i<4; ++i)
    {
        i_rndA[i] = nextUint32();
        i_rndB[i] = nextUint32();
    }
    hkIntVector a; a.load<4>(i_rndA); a.setShiftRight32<6>(a); // upper 26 bits
    hkIntVector b; b.load<4>(i_rndB); b.setShiftRight32<5>(b); // upper 27 bits
    hkVector4d av; a.convertS32ToF32(av);
    hkVector4d bv; b.convertS32ToF32(bv);
    const hkSimdDouble64 scale = hkSimdDouble64::fromFloat(1.0 / 9007199254740992.0);
    const hkSimdDouble64 offset = hkSimdDouble64::fromFloat(134217728.0);
    v.setAddMul(bv, av, offset);
    v.mul(scale);
}

// Return the next single precision vector, range: [-1,1] x 4
void        hkUnitTest::Prng::nextVector11(hkVector4f& v)
{
    nextVector(v);
    v.add(v);
    v.sub(hkVector4f::getConstant<HK_QUADREAL_1>());
}

// Return the next double precision vector, range: [-1,1] x 4
void        hkUnitTest::Prng::nextVector11(hkVector4d& v)
{
    nextVector(v);
    v.add(v);
    v.sub(hkVector4d::getConstant<HK_QUADREAL_1>());
}

// Return the next normal vector.
void        hkUnitTest::Prng::nextUnitVector3(hkVector4f& v)
{
    hkSimdFloat32 x; nextSimdScalar(x); x *= hkSimdFloat32_TwoPi;
    hkSimdFloat32 y; nextSimdScalar(y); y = hkVector4fUtil::aCos( hkSimdFloat32_1 - (y+y) );
    hkVector4f xy; xy.set(x,x,y,y);
    hkVector4f sc; hkVector4fUtil::sinCos(xy,sc);
    v.set( sc.getComponent<2>() * sc.getComponent<1>(), sc.getComponent<2>() * sc.getComponent<0>(), sc.getComponent<3>(), hkSimdFloat32_0);
}

// Return the next normal vector.
void        hkUnitTest::Prng::nextUnitVector3(hkVector4d& v)
{
    hkSimdDouble64 x; nextSimdScalar(x); x *= hkSimdDouble64_TwoPi;
    hkSimdDouble64 y; nextSimdScalar(y); y = hkVector4dUtil::aCos( hkSimdDouble64_1 - (y+y) );
    hkVector4d xy; xy.set(x,x,y,y);
    hkVector4d sc; hkVector4dUtil::sinCos(xy,sc);
    v.set( sc.getComponent<2>() * sc.getComponent<1>(), sc.getComponent<2>() * sc.getComponent<0>(), sc.getComponent<3>(), hkSimdDouble64_0);
}

// Return the next normal vector opposite the other
void        hkUnitTest::Prng::nextUnitVector3(hkVector4f& v, hkVector4fParameter normal)
{
    nextUnitVector3(v);
    v.setFlipSign(v, v.dot<3>(normal).lessZero());
}

// Return the next normal vector opposite the other
void        hkUnitTest::Prng::nextUnitVector3(hkVector4d& v, hkVector4dParameter normal)
{
    nextUnitVector3(v);
    v.setFlipSign(v, v.dot<3>(normal).lessZero());
}

// Return the next single precision quaternion
void        hkUnitTest::Prng::nextQuaternion(hkQuaternionf& v)
{
    hkVector4f axis; nextUnitVector3(axis);
    hkSimdFloat32 angle; nextSimdScalar(angle); angle *= hkSimdFloat32_TwoPi;
    v.setAxisAngle(axis, angle);
}

// Return the next double precision quaternion
void        hkUnitTest::Prng::nextQuaternion(hkQuaterniond& v)
{
    hkVector4d axis; nextUnitVector3(axis);
    hkSimdDouble64 angle; nextSimdScalar(angle);
    angle *= hkSimdDouble64_TwoPi;
    v.setAxisAngle(axis, angle);
}

// Return the next single precision rotation matrix
void        hkUnitTest::Prng::nextRotation(hkRotationf& r)
{
    hkQuaternionf q;
    nextQuaternion(q);
    r.set(q);
}

// Return the next double precision rotation matrix
void        hkUnitTest::Prng::nextRotation(hkRotationd& r)
{
    hkQuaterniond q;
    nextQuaternion(q);
    r.set(q);
}

// Return the next 3d bary-center.
void        hkUnitTest::Prng::nextBaryCenter3D(hkVector4f& bc)
{
    hkSimdFloat32 x; nextSimdScalar(x);
    x = x.sqrt<HK_ACC_FULL,HK_SQRT_SET_ZERO>(); // make it uniform in area
    hkSimdFloat32 y; nextSimdScalar(y);
    bc.set( hkSimdFloat32_1 - x, x * (hkSimdFloat32_1 - y), x * y, hkSimdFloat32_0);
}

// Return the next 3d bary-center.
void        hkUnitTest::Prng::nextBaryCenter3D(hkVector4d& bc)
{
    hkSimdDouble64 x; nextSimdScalar(x);
    x = x.sqrt<HK_ACC_FULL,HK_SQRT_SET_ZERO>(); // make it uniform in area
    hkSimdDouble64 y; nextSimdScalar(y);
    bc.set( hkSimdDouble64_1 - x, x * (hkSimdDouble64_1 - y), x * y, hkSimdDouble64_0);
}

// Generate gaussian deviate with mean 0 and stdev 1 (box-mueller transform)
hkFloat32   hkUnitTest::Prng::nextFloatGauss()
{
    hkFloat32 x, y, r;

    do {
        x = 2.0f * nextFloat() - 1.0f;
        y = 2.0f * nextFloat() - 1.0f;
        r = x * x + y * y;
    } while (r == 0.0f || r >= 1.0f);

    r = hkMath::sqrt((-2.0f * hkMath::log(r)) / r);

    return x * r;
}

// Generate gaussian deviate with mean 0 and stdev 1 (box-mueller transform)
hkDouble64  hkUnitTest::Prng::nextDoubleGauss()
{
    hkDouble64 x, y, r;

    do {
        x = 2.0 * nextDouble() - 1.0;
        y = 2.0 * nextDouble() - 1.0;
        r = x * x + y * y;
    } while (r == 0.0 || r >= 1.0);

    r = hkMath::sqrt((-2.0 * hkMath::log(r)) / r);

    return x * r;
}

// Generate gaussian deviate with mean 0 and stdev 1 (box-mueller transform)
hkReal      hkUnitTest::Prng::nextRealGauss()
{
#if defined(HK_REAL_IS_DOUBLE)
    return nextDoubleGauss();
#else
    return nextFloatGauss();
#endif
}

// Returns a new PRNG. Seeding it causes an assert in debug on the m_y check, but
// will just work in release.
hkUnitTest::Prng hkUnitTest::Prng::nextPrng()
{
    hkUnitTest::Prng prng;
    prng.m_x = nextUint32();
    while (0==m_y) m_y = nextUint32(); // y must not be zero
    prng.m_z = nextUint32();
    prng.m_w = nextUint32() % 698769068; // Should be less than 698769069
    prng.m_c = (prng.m_w + 1) & 1;

    // stir it a bit
    for(int i=0,n=int(nextUint32() % 67); i<n; ++i) prng.nextUint32(); // 67 next prime after 64
    return prng;
}


hkBool hkTestSuccess( hkResult res, const char* func, const char* file, int line )
{
    if ( res.isSuccess() )
    {
        return hkTestReportFunction( true, func, file, line );
    }
    hkStringBuf sb;
    sb.format( "{} returned {}", func, res );
    return hkTestReportFunction( false, sb.cString(), file, line );
}

static hkBool hkDefaultTestReportFunction(hkBool32 cond, const char* desc, const char* file, int line)
{
    if ( !cond )
    {
        HK_BREAKPOINT(0);
    }
    return bool(cond);
}

hkBool HK_CALL hkTestReportFunction( hkBool32 cond, const char* desc, const char* file, int line )
{
    if ( !cond )
    {
        cond = false; // dummy line so we can set breakpoints
    }
    return (*g_hkTestReportFunction)(cond, desc, file, line);
}


hkTestReportFunctionType g_hkTestReportFunction = hkDefaultTestReportFunction;

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
