// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/Types/hkWeakPtr.h>
#include <Common/Base/UnitTest/Types/WeakPtrTest.h>
#include <Common/Base/Serialize/hkSerialize.h>
#include <Common/Base/Container/Hash/hkHashSet.h>


int weakptr_test()
{
#if defined(HK_BUILDING_WITH_ENGINE)
    using hkWeakPtrTest::Foo;
    using hkWeakPtrTest::Bar;

    // Make sure all template methods compile
    {
        Foo* f = 0;
        hkWeakPtr<hkReferencedObject> wptr = f;
        wptr = f;
        hkWeakPtr<Foo> wptr2 = f;
        wptr2 = f;
    }
    {
        hkRefPtr<Foo> f;
        hkWeakPtr<hkReferencedObject> wptr = f;
        wptr = f;
        hkWeakPtr<Foo> wptr2 = f;
        wptr2 = f;
    }
    {
        hkWeakPtr<Foo> f;
        hkWeakPtr<hkReferencedObject> wptr = f;
        wptr = f;
        hkWeakPtr<Foo> wptr2 = f;
        wptr2 = f;
    }
    {
        hkWeakPtr<hkReferencedObject> wptr = HK_NULL;
        wptr = HK_NULL;
    }


    // Test null weak ptr
    {
        hkWeakPtr<hkReferencedObject> wptr;
        hkRefPtr<hkReferencedObject> acquiredObj = wptr.acquire();
        HK_TEST(acquiredObj == HK_NULL);

        hkReflect::PointerVar pv(&wptr);
        if(HK_TEST(pv))
        {
            HK_TEST(pv.isNull());
            HK_TEST(pv.getValue() == hkReflect::Var());
        }
    }

    // Test non-null weak ptr
    {
        hkRefPtr<hkReferencedObject> obj1 = hkRefNew<hkReferencedObject>(new hkReferencedObject);
        hkWeakPtr<hkReferencedObject> wptr(obj1);
        HK_TEST(obj1->getReferenceCount() == 1);

        {
            hkRefPtr<hkReferencedObject> acquiredObj = wptr.acquire();
            HK_TEST(acquiredObj == obj1);
        }

        hkReflect::PointerVar pv(&wptr);
        if(HK_TEST(pv))
        {
            HK_TEST(pv.isNull() == false);
            HK_TEST(pv.getValue() == obj1);
        }

        wptr = HK_NULL;

        {
            hkRefPtr<hkReferencedObject> acquiredObj = wptr.acquire();
            HK_TEST(acquiredObj == HK_NULL);
        }

        if(HK_TEST(pv))
        {
            HK_TEST(pv.isNull());
            HK_TEST(pv.getValue() == hkReflect::Var());
        }
    }

    // Test weak ptr to an object which dies
    {
        hkRefPtr<hkReferencedObject> obj1 = hkRefNew<hkReferencedObject>(new hkReferencedObject);
        hkWeakPtr<hkReferencedObject> wptr(obj1);
        HK_TEST(obj1->getReferenceCount() == 1);
        {
            hkRefPtr<hkReferencedObject> acquiredObj = wptr.acquire();
            HK_TEST(acquiredObj == obj1);
        }
        obj1 = HK_NULL;
        {
            hkRefPtr<hkReferencedObject> acquiredObj = wptr.acquire();
            HK_TEST(acquiredObj == HK_NULL);
        }

        // Test re-using the weak ptr to point to another object
        obj1 = hkRefNew<hkReferencedObject>(new hkReferencedObject);
        wptr.set(obj1);
        {
            hkRefPtr<hkReferencedObject> acquiredObj = wptr.acquire();
            HK_TEST(acquiredObj == obj1);
        }
    }

    // Very specific test to ensure hkWeakPtrDetail::StatusRegistry::setDead is re-entrant.
    {
        hkRefPtr<hkReferencedObject> obj1 = hkRefNew<hkReferencedObject>(new hkReferencedObject);
        hkWeakPtr<hkReferencedObject> wptr = obj1;
        hkWeakPtr<hkWeakPtrDetail::Status> wptrStatus = hkWeakPtrDetail::getStatus(wptr);
        hkWeakPtr<hkWeakPtrDetail::Status> wptrStatus2 = hkWeakPtrDetail::getStatus(wptrStatus);
        wptr = HK_NULL;
        wptrStatus = HK_NULL;
        wptrStatus2 = HK_NULL;
        obj1 = HK_NULL;
    }

    // Test setting a weak ptr using reflection
    {
        hkRefPtr<hkReferencedObject> obj1 = hkRefNew<hkReferencedObject>(new hkReferencedObject);
        hkWeakPtr<hkReferencedObject> wptr;
        hkReflect::PointerVar pv(&wptr);

        if(HK_TEST(pv))
        {
            HK_TEST(pv.setValue(obj1).isSuccess());

            HK_TEST(pv.isNull() == false);
            HK_TEST(pv.getValue() == obj1);

            {
                hkRefPtr<hkReferencedObject> acquiredObj = wptr.acquire();
                HK_TEST(acquiredObj == obj1);
            }

            pv.setValue(hkReflect::Var());

            HK_TEST(pv.isNull());
            HK_TEST(pv.getValue() == hkReflect::Var());

            {
                hkRefPtr<hkReferencedObject> acquiredObj = wptr.acquire();
                HK_TEST(acquiredObj == HK_NULL);
            }
        }
    }

    // Test serialization & cloning
    {
        hkArray<char> buffer;

        Bar obj(hkRefNew<Foo>(new Foo));
        HK_TEST(obj.m_weakPtr.unsafeAcquire() == obj.m_strongPtr);

        hkResult res = hkSerialize::Save().contentsPtr(&obj, &buffer);
        HK_TEST(res.isSuccess());

        hkScopedPtr<Bar> newObj(hkSerialize::Load().toObject<Bar>(buffer));
        if(HK_TEST(newObj))
        {
            HK_TEST(newObj->m_strongPtr != HK_NULL);
            HK_TEST(newObj->m_weakPtr.unsafeAcquire() == newObj->m_strongPtr);
        }

        hkScopedPtr<Bar> clonedObj(hkReflect::Cloner().cloneRecursive(&obj));
        if(HK_TEST(clonedObj))
        {
            HK_TEST(clonedObj->m_strongPtr != HK_NULL);
            HK_TEST(clonedObj->m_weakPtr.unsafeAcquire() == clonedObj->m_strongPtr);
        }
    }

    // Test const weak pointer
    {
        hkRefPtr<const hkReferencedObject> object = hkRefNew<hkReferencedObject>(new hkReferencedObject);
        hkWeakPtr<const hkReferencedObject> weakPtr(object);
        hkRefPtr<const hkReferencedObject> acquiredObject = weakPtr.acquire();
        HK_TEST(object.val() == acquiredObject.val());
    }

    // Test hashing hkWeakPtr
    {
        hkRefPtr<Foo> obj = hkRefNew<Foo>(new Foo);
        hkWeakPtr<Foo> wptr = obj;

        const hkUint32 wptrHash = hkHashValue(wptr);
        HK_TEST(wptrHash != hkHashValue(hkWeakPtr<Foo>()));

        obj = HK_NULL; // obj dies

        HK_TEST(wptrHash == hkHashValue(wptr)); // hash must still be the same

        wptr.acquire(); // make sure acquiring the pointer does not change the hash
        HK_TEST(wptrHash == hkHashValue(wptr)); // hash must still be the same
    }

    // Test hkWeakPtr in hkHash* containers
    {
        hkHashSet<hkWeakPtr<Foo>> hashSet;

        hkArray<hkRefPtr<Foo>> objects;
        hkArray<hkWeakPtr<Foo>> weakPtrs;
        for(int i = 0; i < 10; ++i)
        {
            objects.pushBack(hkRefNew<Foo>(new Foo));
            weakPtrs.pushBack(objects.back());

            HK_TEST(hashSet.contains(weakPtrs.back()) == false);
            hashSet.insert(weakPtrs.back());
            HK_TEST(hashSet.contains(weakPtrs.back()) == true);
        }

        // Make sure the matching hkWeakPtr is the correct one
        for(int i = 0; i < weakPtrs.getSize(); ++i)
        {
            auto it = hashSet.findElement(weakPtrs[i]);
            hkRefPtr<Foo> elt = hashSet.getElement(it).acquire();
            HK_TEST(elt == objects[i]);
        }

        objects.clear(); // kill all the objects

        // We must still be able to uniquely identify all the individual hkWeakPtr.
        for(int i = 0; i < weakPtrs.getSize(); ++i)
        {
            HK_TEST(hashSet.contains(weakPtrs[i]) == true);
        }

        HK_TEST(hashSet.contains(hkWeakPtr<Foo>()) == false);
    }

    {
        hkRefPtr<hkReferencedObject> obj1 = hkRefNew<hkReferencedObject>(new hkReferencedObject);
        HK_TEST(obj1->getReferenceCount() == 1);

        auto wptr = hkWeakPtr<hkReferencedObject>(obj1);

        {
            // Increments the refcount as required
            auto acquiredObj = wptr.acquire();
            (void)acquiredObj;
            HK_TEST(obj1->getReferenceCount() == 2);
        }
        HK_TEST(obj1->getReferenceCount() == 1);
    }
#endif
    return 0;
}

HK_TEST_REGISTER(weakptr_test, "Fast", "Common/Test/UnitTest/Base/", __FILE__);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
