// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/UnitTest/Types/PtrReferenceLogger.h>

#define HK_TEST_RETURN(exp) \
    if(!HK_TEST(exp)) { return 0; }

using namespace hkPtrLogging;

namespace
{
    HK_NEVER_INLINE
    bool logIsCorrect(const hkArray<PtrReferenceLogger::Event>& log, const PtrReferenceLogger::Event* expectedEvents, size_t expectedEventsCount)
    {
        if (log.getSize() != hkLosslessCast<int>(expectedEventsCount))
        {
            return false;
        }
        for (int i = 0; i < hkLosslessCast<int>(expectedEventsCount); ++i)
        {
            if (log[i] != expectedEvents[i])
            {
                return false;
            }
        }
        return true;
    }

    template<size_t LogSize>
    HK_ALWAYS_INLINE
    bool logIsCorrect(const hkArray<PtrReferenceLogger::Event>& log, const PtrReferenceLogger::Event(&expectedEvents)[LogSize])
    {
        return logIsCorrect(log, expectedEvents, LogSize);
    }

}  // namespace


int refPtrTest_main()
{
    {   // hk::makeRef should not leak resources
        hkArray<PtrReferenceLogger::Event> log;
        {
            hk::makeRef<PtrReferenceLogger>(log, PtrReferenceLogger::Value::A);
        }
        static HK_CONSTEXPR PtrReferenceLogger::Event expected[] =
        {
            PtrReferenceLogger::Event::NormallyConstructedWithRValue,
            PtrReferenceLogger::Event::ReferenceRemoved,
            PtrReferenceLogger::Event::Destructed,
        };
        HK_TEST_RETURN(logIsCorrect(log, expected));
    }
    {   // Assigning the result to hkRefPtr
        hkArray<PtrReferenceLogger::Event> log;
        {
            hkRefPtr<PtrReferenceLogger> ptr = hk::makeRef<PtrReferenceLogger>(log, PtrReferenceLogger::Value::A);
            HK_TEST_RETURN(ptr);
            HK_TEST_RETURN(ptr->m_value == PtrReferenceLogger::Value::A);
            HK_TEST_RETURN(ptr->getReferenceCount() == 1);
        }
        static HK_CONSTEXPR PtrReferenceLogger::Event expected[] =
        {
            PtrReferenceLogger::Event::NormallyConstructedWithRValue,
            PtrReferenceLogger::Event::ReferenceRemoved,
            PtrReferenceLogger::Event::Destructed,
        };
        HK_TEST_RETURN(logIsCorrect(log, expected));
    }
    {   // hk::makeRef should forward arguments
        hkArray<PtrReferenceLogger::Event> log;
        {
            hkRefPtr<PtrReferenceLogger> ptr = hk::makeRef<PtrReferenceLogger>(log, PtrReferenceLogger::Value::A);
            hkRefPtr<PtrReferenceLogger> ptrCopy = hk::makeRef<PtrReferenceLogger>(*ptr);
            hkRefPtr<PtrReferenceLogger> ptrMove = hk::makeRef<PtrReferenceLogger>(hk::move(*ptr));
            HK_TEST_RETURN(ptr && ptrCopy && ptrMove);  // Pointers are still valid

            HK_TEST_RETURN(ptrCopy->m_value == PtrReferenceLogger::Value::A);
            HK_TEST_RETURN(ptrMove->m_value == PtrReferenceLogger::Value::A);
            HK_TEST_RETURN(ptr->m_value == PtrReferenceLogger::Value::Nothing);
            HK_TEST_RETURN(ptr->getReferenceCount() == 1);
            HK_TEST_RETURN(ptrCopy->getReferenceCount() == 1);
            HK_TEST_RETURN(ptrMove->getReferenceCount() == 1);
        }
        static HK_CONSTEXPR PtrReferenceLogger::Event expected[] =
        {
            PtrReferenceLogger::Event::NormallyConstructedWithRValue,
            PtrReferenceLogger::Event::CopyConstructed,
            PtrReferenceLogger::Event::MoveConstructed,
            PtrReferenceLogger::Event::ReferenceRemoved,
            PtrReferenceLogger::Event::Destructed,
            PtrReferenceLogger::Event::ReferenceRemoved,
            PtrReferenceLogger::Event::Destructed,
            PtrReferenceLogger::Event::ReferenceRemoved,
            PtrReferenceLogger::Event::Destructed,
        };
        HK_TEST_RETURN(logIsCorrect(log, expected));
    }
    {   // Copy-constructing a hkRefPtr
        hkArray<PtrReferenceLogger::Event> log;
        {
            hkRefPtr<PtrReferenceLogger> ptr1 = hk::makeRef<PtrReferenceLogger>(log, PtrReferenceLogger::Value::A);
            hkRefPtr<PtrReferenceLogger> ptr2(ptr1);
            HK_TEST_RETURN(ptr1 && (ptr1->m_value == PtrReferenceLogger::Value::A));
            HK_TEST_RETURN(ptr2 && (ptr2->m_value == PtrReferenceLogger::Value::A));
            HK_TEST_RETURN(ptr1->getReferenceCount() == 2);
        }
        static HK_CONSTEXPR PtrReferenceLogger::Event expected[] =
        {
            PtrReferenceLogger::Event::NormallyConstructedWithRValue,
            PtrReferenceLogger::Event::ReferenceAdded,
            PtrReferenceLogger::Event::ReferenceRemoved,
            PtrReferenceLogger::Event::ReferenceRemoved,
            PtrReferenceLogger::Event::Destructed,
        };
        HK_TEST_RETURN(logIsCorrect(log, expected));
    }
    {   // Copy-assigning to empty hkRefPtr
        hkArray<PtrReferenceLogger::Event> log;
        {
            hkRefPtr<PtrReferenceLogger> ptr1 = hk::makeRef<PtrReferenceLogger>(log, PtrReferenceLogger::Value::A);
            hkRefPtr<PtrReferenceLogger> ptr2;
            ptr2 = ptr1;
            HK_TEST_RETURN(ptr1 && (ptr1->m_value == PtrReferenceLogger::Value::A));
            HK_TEST_RETURN(ptr2 && (ptr2->m_value == PtrReferenceLogger::Value::A));
            HK_TEST_RETURN(ptr1->getReferenceCount() == 2);
        }
        static HK_CONSTEXPR PtrReferenceLogger::Event expected[] =
        {
            PtrReferenceLogger::Event::NormallyConstructedWithRValue,
            PtrReferenceLogger::Event::ReferenceAdded,
            PtrReferenceLogger::Event::ReferenceRemoved,
            PtrReferenceLogger::Event::ReferenceRemoved,
            PtrReferenceLogger::Event::Destructed,
        };
        HK_TEST_RETURN(logIsCorrect(log, expected));
    }
    {   // Copy-assigning to non-empty hkRefPtr
        hkArray<PtrReferenceLogger::Event> log;
        {
            hkRefPtr<PtrReferenceLogger> ptr1 = hk::makeRef<PtrReferenceLogger>(log, PtrReferenceLogger::Value::A);
            hkRefPtr<PtrReferenceLogger> ptr2 = hk::makeRef<PtrReferenceLogger>(log, PtrReferenceLogger::Value::B);
            ptr2 = ptr1;
            HK_TEST_RETURN(ptr1 && (ptr1->m_value == PtrReferenceLogger::Value::A));
            HK_TEST_RETURN(ptr2 && (ptr2->m_value == PtrReferenceLogger::Value::A));
            HK_TEST_RETURN(ptr1->getReferenceCount() == 2);
        }
        static HK_CONSTEXPR PtrReferenceLogger::Event expected[] =
        {
            PtrReferenceLogger::Event::NormallyConstructedWithRValue,
            PtrReferenceLogger::Event::NormallyConstructedWithRValue,
            PtrReferenceLogger::Event::ReferenceAdded,
            PtrReferenceLogger::Event::ReferenceRemoved,
            PtrReferenceLogger::Event::Destructed,
            PtrReferenceLogger::Event::ReferenceRemoved,
            PtrReferenceLogger::Event::ReferenceRemoved,
            PtrReferenceLogger::Event::Destructed,
        };
        HK_TEST_RETURN(logIsCorrect(log, expected));
    }
    {   // Move-constructing a hkRefPtr
        hkArray<PtrReferenceLogger::Event> log;
        {
            hkRefPtr<PtrReferenceLogger> ptr1 = hk::makeRef<PtrReferenceLogger>(log, PtrReferenceLogger::Value::A);
            hkRefPtr<PtrReferenceLogger> ptr2(hk::move(ptr1));
            HK_TEST_RETURN(!ptr1);
            HK_TEST_RETURN(ptr2 && (ptr2->m_value == PtrReferenceLogger::Value::A));
            HK_TEST_RETURN(ptr2->getReferenceCount() == 1);
        }
        static HK_CONSTEXPR PtrReferenceLogger::Event expected[] =
        {
            PtrReferenceLogger::Event::NormallyConstructedWithRValue,
            PtrReferenceLogger::Event::ReferenceRemoved,
            PtrReferenceLogger::Event::Destructed,
        };
        HK_TEST_RETURN(logIsCorrect(log, expected));
    }
    {   // Move-assigning to empty hkRefPtr
        hkArray<PtrReferenceLogger::Event> log;
        {
            hkRefPtr<PtrReferenceLogger> ptr1 = hk::makeRef<PtrReferenceLogger>(log, PtrReferenceLogger::Value::A);
            hkRefPtr<PtrReferenceLogger> ptr2;
            ptr2 = hk::move(ptr1);
            HK_TEST_RETURN(!ptr1);
            HK_TEST_RETURN(ptr2 && (ptr2->m_value == PtrReferenceLogger::Value::A));
            HK_TEST_RETURN(ptr2->getReferenceCount() == 1);
        }
        static HK_CONSTEXPR PtrReferenceLogger::Event expected[] =
        {
            PtrReferenceLogger::Event::NormallyConstructedWithRValue,
            PtrReferenceLogger::Event::ReferenceRemoved,
            PtrReferenceLogger::Event::Destructed,
        };
        HK_TEST_RETURN(logIsCorrect(log, expected));
    }
    {   // Move-assigning to non-empty hkRefPtr
        hkArray<PtrReferenceLogger::Event> log;
        {
            hkRefPtr<PtrReferenceLogger> ptr1 = hk::makeRef<PtrReferenceLogger>(log, PtrReferenceLogger::Value::A);
            hkRefPtr<PtrReferenceLogger> ptr2 = hk::makeRef<PtrReferenceLogger>(log, PtrReferenceLogger::Value::B);
            ptr2 = hk::move(ptr1);
            HK_TEST_RETURN(!ptr1);
            HK_TEST_RETURN(ptr2 && (ptr2->m_value == PtrReferenceLogger::Value::A));
            HK_TEST_RETURN(ptr2->getReferenceCount() == 1);
        }
        static HK_CONSTEXPR PtrReferenceLogger::Event expected[] =
        {
            PtrReferenceLogger::Event::NormallyConstructedWithRValue,
            PtrReferenceLogger::Event::NormallyConstructedWithRValue,
            PtrReferenceLogger::Event::ReferenceRemoved,
            PtrReferenceLogger::Event::Destructed,
            PtrReferenceLogger::Event::ReferenceRemoved,
            PtrReferenceLogger::Event::Destructed,
        };
        HK_TEST_RETURN(logIsCorrect(log, expected));
    }
    {   // Move-assigning to self
        hkArray<PtrReferenceLogger::Event> log;
        {
            hkRefPtr<PtrReferenceLogger> ptr = hk::makeRef<PtrReferenceLogger>(log, PtrReferenceLogger::Value::A);
            auto moveAssigner = [](hkRefPtr<PtrReferenceLogger>& a, hkRefPtr<PtrReferenceLogger>& b)
            {
                a = hk::move(b);
            };

            moveAssigner(ptr, ptr);

            HK_TEST_RETURN(ptr);
            HK_TEST_RETURN(ptr->getReferenceCount() == 1);
        }
        static HK_CONSTEXPR PtrReferenceLogger::Event expected[] =
        {
            PtrReferenceLogger::Event::NormallyConstructedWithRValue,
            PtrReferenceLogger::Event::ReferenceRemoved,
            PtrReferenceLogger::Event::Destructed,
        };
        HK_TEST_RETURN(logIsCorrect(log, expected));
    }
    {   // ADL-friendly swap
        hkArray<PtrReferenceLogger::Event> log;
        {
            hkRefPtr<PtrReferenceLogger> ptr1 = hk::makeRef<PtrReferenceLogger>(log, PtrReferenceLogger::Value::A);
            hkRefPtr<PtrReferenceLogger> ptr2 = hk::makeRef<PtrReferenceLogger>(log, PtrReferenceLogger::Value::B);
            swap(ptr1, ptr2);
            HK_TEST_RETURN(ptr1 && (ptr1->m_value == PtrReferenceLogger::Value::B));
            HK_TEST_RETURN(ptr2 && (ptr2->m_value == PtrReferenceLogger::Value::A));
            HK_TEST_RETURN(ptr1->getReferenceCount() == 1);
            HK_TEST_RETURN(ptr2->getReferenceCount() == 1);
        }
        static HK_CONSTEXPR PtrReferenceLogger::Event expected[] =
        {
            PtrReferenceLogger::Event::NormallyConstructedWithRValue,
            PtrReferenceLogger::Event::NormallyConstructedWithRValue,
            PtrReferenceLogger::Event::ReferenceRemoved,
            PtrReferenceLogger::Event::Destructed,
            PtrReferenceLogger::Event::ReferenceRemoved,
            PtrReferenceLogger::Event::Destructed,
        };
        HK_TEST_RETURN(logIsCorrect(log, expected));
    }

    return 0;
}

HK_TEST_REGISTER(refPtrTest_main, "Fast", "Common/Test/UnitTest/Base/", __FILE__);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
