// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/Types/PropertyBagTest.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/Types/Properties/hkPropertyBag.h>
#include <Common/Base/Types/Properties/hkPropertyRegistry.h>
#include <Common/Base/Reflect/Util/hkReflectClone.h>

#include <Common/Base/Serialize/Core/hkSerializeCore.h>
#include <Common/Base/Serialize/Resource/hkObjectResource.h>

namespace
{
    // Use the same allocation scheme as recursive clone, but don't follow pointers.
    struct ShallowCopyCallback : public hkReflect::Detail::CloneOnHeap
    {
        virtual hkResult atPointer(hkReflect::PointerVar& dst, hkReflect::PointerVar& src, _Out_ hkReflect::Cloner::PointerAction::Enum* action) HK_OVERRIDE
        {
            const hkReflect::Type* srcType = src.getType();

            if ( srcType->getValueAttribute<hk::StrongRef>(false) )
            {
                *action = hkReflect::Cloner::PointerAction::CLONE; // Deep-clone strong pointers
                return HK_SUCCESS;
            }

            // Don't clone the pointer target.
            *action = hkReflect::Cloner::PointerAction::SKIP;

            // Share the pointer value.
            return hkReflect::Detail::getPlain(dst).setValue(src.getValue());
        }
    };
}

static const hkStaticPropertyDesc<int> staticPropA("intA");
static const hkStaticPropertyDesc<hkReferencedObject*> refObjPtrProp("unittest.ptr.static");
static const hkStaticPropertyDesc<hkReferencedObject> refObjValProp("unittest.value.static");
static const hkStaticPropertyDesc<hkReal> unnamedRealProp(HK_NULL);
static const hkStaticPropertyDesc<hkQTransform> qTransformProp("unittest.qtransform.static");
static const hkStaticPropertyDesc<PropertyBagTest::ComplexProperty> complexProp("unittest.complexProperty.static");

static int propertyBag_main()
{
    hkPropertyBag pBag;

    // Single static property, no name
    {

        hkReal realProp = 3;
        pBag.add(unnamedRealProp, realProp);

        // Get value
        {
            hkReal val = pBag.getWithDefault(unnamedRealProp);
            HK_TEST_EQ(realProp, val);
        }

        // Get value, and update
        {
            hkReal& ptrVal = pBag.getOrCreate(unnamedRealProp);
            HK_TEST_EQ(realProp, ptrVal);
            ptrVal = 4;
        }

        // Get value
        {
            hkReal* val = pBag.getIfExists(unnamedRealProp);
            HK_TEST(val != nullptr);
            HK_TEST_EQ(*val, 4);
        }
        pBag.remove(unnamedRealProp);
    }

    // Single dynamic property, no name
    {
        hkPropertyId dynRealProp = hkPropertyId::make<hkReal>(HK_NULL);

        hkReal realProp = 3;
        pBag.add(dynRealProp, &realProp);

        // Get value
        {
            hkReal val = pBag.getWithDefault<hkReal>(dynRealProp);
            HK_TEST_EQ(realProp, val);
        }

        // Get value, and update
        {
            hkReal* ptrVal = pBag.getIfExists<hkReal>(dynRealProp);
            if( HK_TEST(ptrVal) )
            {
                HK_TEST_EQ(realProp, *ptrVal);
                *ptrVal = 4;
            }
        }

        // Get value
        {
            hkReal val = pBag.get<hkReal>(dynRealProp);
            HK_TEST_EQ(val, 4);
        }
        pBag.remove(dynRealProp);
    }

    // Two properties of same type, different names
    {
        hkPropertyId propB = hkPropertyId::make<int>("intB");

        const int intPropA = 1;
        const int intPropB = 2;

        pBag.add(staticPropA, intPropA);
        pBag.add(propB, &intPropB);

        // non-existant property
        int pVal0 = pBag.getWithDefault<int>(hkPropertyId::make<int>(HK_NULL));
        HK_TEST_EQ(pVal0, 0);
        int pVal1 = pBag.getWithDefault<int>(hkPropertyId::make<int>(HK_NULL), 1);
        HK_TEST_EQ(pVal1, 1);

        int pValA = pBag.get(staticPropA);
        HK_TEST_EQ(intPropA, pValA);

        int pValB = pBag.get<int>(propB);
        HK_TEST_EQ(intPropB, pValB);

        pBag.remove(staticPropA);
        pBag.remove(propB);
    }

    // Referenced object. Adding it as a property should just increase its ref-count
    {
        hkPropertyId objProp = hkPropertyId::make<hkReferencedObject*>(HK_NULL);
        hkReferencedObject* obj = new hkReferencedObject();
        pBag.add(objProp, &obj);

        hkReferencedObject* pObj = pBag.get<hkReferencedObject*>(objProp);
        if( HK_TEST(pObj != HK_NULL) )
        {
            HK_TEST_EQ(pObj, obj);
            HK_TEST_EQ(obj->getReferenceCount(), 2);
        }

        pBag.remove(objProp);
        obj->removeReference();
    }

    // Non-referenced object POINTER
    {
        hkPropertyId qtmProp = hkPropertyId::make<hkQTransform*>(HK_NULL);
        hkQTransform* qtm = new hkQTransform();
        qtm->setIdentity();
        pBag.add(qtmProp, &qtm);

        hkQTransform* pQtm = pBag.get<hkQTransform*>(qtmProp);
        HK_TEST_EQ(pQtm, qtm);
        pBag.remove(qtmProp);
        delete qtm;
    }

    // Get or create
    {
        // copy a static QTransform VALUE
        {
            hkQTransform& qtmA = pBag.getOrCreate<hkQTransform>(qTransformProp);
            qtmA.setIdentity();

            pBag.remove(qTransformProp);
        }

        // copy a dynamic QTransform VALUE
        {
            hkPropertyId qtmProp = hkPropertyId::make<hkQTransform>("unittest.qtransform.dynamic");

            hkQTransform& qtmA = pBag.getOrCreate<hkQTransform>(qtmProp);
            qtmA.setIdentity();

            hkQTransform qtmB = pBag.get<hkQTransform>(qtmProp);
            //HK_TEST_EQ(qtmB, hkQTransform::getIdentity());
            HK_TEST(pBag.remove(qtmProp).isSuccess());
        }

        // copy an int VALUE
        {
            hkPropertyId intProp = hkPropertyId::make<int>("unittest.int.dynamic");
            int& iA = pBag.getOrCreate<int>(intProp);
            iA = 2;

            int iB = pBag.get<int>(intProp);
            HK_TEST_EQ(iA, iB);
            HK_TEST_EQ(iB, 2);

            HK_TEST(pBag.remove(intProp).isSuccess());
        }

        // static refobject POINTER
        {
            //pBag.getOrAllocate(refObjValProp); // should not compile - only pointer properties can be allocated

            HK_TEST( pBag.getWithDefault(refObjPtrProp) == nullptr);
            hkReferencedObject* oA = pBag.getOrAllocate(refObjPtrProp);
            hkReferencedObject* oB = pBag.get(refObjPtrProp);
            HK_TEST_EQ(oA, oB);
            HK_TEST_EQ(oA->getReferenceCount(), 1);

            HK_TEST( pBag.remove(refObjPtrProp).isSuccess() );
        }

        // dynamic refobject POINTER
        {
            hkPropertyId objProp = hkPropertyId::make<hkReferencedObject*>("unittest.refptr.dynamic");
            HK_TEST(pBag.getWithDefault<hkReferencedObject*>(objProp) == nullptr);
            hkReferencedObject* oA = pBag.getOrAllocate<hkReferencedObject*>(objProp);
            HK_TEST(oA);
            hkReferencedObject* oB = pBag.get<hkReferencedObject*>(objProp);
            HK_TEST_EQ(oA, oB);
            HK_TEST(pBag.remove(objProp).isSuccess());
        }

        // transform POINTER
        if(0) // need a unique_ptr to clean this up
        {
            hkPropertyId qtmProp = hkPropertyId::make<hkQsTransform*>(HK_NULL);
            HK_TEST( pBag.getWithDefault<hkQsTransform*>(qtmProp) == nullptr);
            hkQsTransform* oA = pBag.getOrAllocate<hkQsTransform*>(qtmProp);
            HK_TEST(oA);
            hkQsTransform* oB = pBag.get<hkQsTransform*>(qtmProp);
            HK_TEST_EQ(oA, oB);

            HK_TEST(pBag.remove(qtmProp).isSuccess());
        }
    }

    // Iteration

    {
        pBag.add(staticPropA, 100);
        pBag.add(unnamedRealProp, (hkReal)3.14f);
        for (hkPropertyBag::Entry entry : pBag)
        {

        }
    }

    // Clone and serialize
    {
        hkPropertyId objProp = hkPropertyId::make<hkReferencedObject*>(HK_NULL);
        hkPropertyId intProp = hkPropertyId::make<int>(HK_NULL);
        hkPropertyId intPropNonCloned = hkPropertyId::make<int>("IntNotCloned", hkPropertyFlags::NO_CLONE);
        hkPropertyId intPropNonSerialized = hkPropertyId::make<int>("IntNotSerialized", hkPropertyFlags::NO_SERIALIZE);
        hkPropertyId intPropNonTracked = hkPropertyId::make<int>("IntNotTracked", hkPropertyFlags::NO_TRACK);

        hkReferencedObject* objA = new hkReferencedObject();

        int intVal = 3;
        objA->properties().add(intProp, &intVal);

        int intValNonCloned = 5;
        objA->properties().add(intPropNonCloned, &intValNonCloned);

        int intValNonSerialized = 7;
        objA->properties().add(intPropNonSerialized, &intValNonSerialized);

        int intValNonTracked = 11;
        objA->properties().add(intPropNonTracked, &intValNonTracked);

        hkReferencedObject* objVal = new hkReferencedObject();
        objA->properties().add(objProp, &objVal);

        int& intA = objA->properties().getOrCreate<int>(intProp);
        HK_TEST_EQ(intA, intVal);

        int* ptrIntNonClonedA = objA->properties().getIfExists<int>(intPropNonCloned);
        HK_TEST_EQ(*ptrIntNonClonedA, intValNonCloned);

        int* ptrIntNonSerializedA = objA->properties().getIfExists<int>(intPropNonSerialized);
        HK_TEST_EQ(*ptrIntNonSerializedA, intValNonSerialized);

        int* ptrIntNonTrackedA = objA->properties().getIfExists<int>(intPropNonTracked);
        HK_TEST_EQ(*ptrIntNonTrackedA, intValNonTracked);

        hkReferencedObject* ptrObjA = objA->properties().get<hkReferencedObject*>(objProp);
        HK_TEST_EQ(ptrObjA, objVal);
        HK_TEST_EQ(objVal->getReferenceCount(), 2);

        // hkReflect.Cloner
        {
            ShallowCopyCallback clonerCallback;
            hkReflect::Var srcVar       (objA);
            hkReflect::Var dstVar       = hkReflect::Cloner().cloneVar(srcVar, clonerCallback);
            hkReferencedObject* objB    = dstVar.dynCast<hkReferencedObject>();

            // The cloner sets the ref count to zero
            objB->setReferenceCount(1);
            HK_TEST(objA != objB);

            int& intB = objB->properties().getOrCreate<int>(intProp);
            HK_TEST(&intA != &intB);
            HK_TEST_EQ(intB, intVal);

            int* ptrIntNonClonedB = objB->properties().getIfExists<int>(intPropNonCloned);
            HK_TEST(!ptrIntNonClonedB);

            int* ptrIntNonSerializedB = objB->properties().getIfExists<int>(intPropNonSerialized);
            HK_TEST(ptrIntNonSerializedA != ptrIntNonSerializedB);
            HK_TEST(*ptrIntNonSerializedB == intValNonSerialized);

            int* ptrIntNonTrackedB = objB->properties().getIfExists<int>(intPropNonTracked);
            HK_TEST(ptrIntNonTrackedA != ptrIntNonTrackedB);
            HK_TEST(*ptrIntNonTrackedB == intValNonTracked);

            hkReferencedObject* ptrObjB = objB->properties().get<hkReferencedObject*>(objProp);
            HK_TEST_EQ(ptrObjA, ptrObjB);

            objB->removeReference();
        }

        // Serialization must be the last of these tests, as it destroys the original object
        // and clears the property type cache to obtain realistic results.
        {
            hkArray<char> content; content.reserve( 1024 * 256 );
            hkSerialize::Save().contentsPtr(objA, &content);

            HK_TEST(objA->getReferenceCount() == 1);
            objA->removeReference();

            // Test the flags of one of the property IDs
            {
                hkPropertyId tmpId = hkPropertyId::make<int>("IntNotTracked", hkPropertyFlags::NO_TRACK);
                HK_TEST(!tmpId.desc()->isTracked());
            }

            hkArray<const hkReflect::Type*> typesToRemove;
            typesToRemove.pushBack(hkReflect::getType<int>());
            typesToRemove.pushBack(hkReflect::getType<hkReferencedObject>());

            hkPropertyRegistry::getInstance().removeTypes(typesToRemove);
            hkPropertyRegistry::getInstance().rebuildIdFromKey();

            // Test that this really removed the int property (and one with different flags has been re-created
            // (then clear again)
            {
                hkPropertyId tmpId = hkPropertyId::make<int>("IntNotTracked");
                HK_TEST(tmpId.desc()->isTracked());

                hkPropertyRegistry::getInstance().removeTypes(typesToRemove);
                hkPropertyRegistry::getInstance().rebuildIdFromKey();
            }


            hkReferencedObject* objC = hkSerialize::Load().toObject<hkReferencedObject>(hkArrayViewT::make(content.begin(), content.getSize()));
            HK_ASSERT_NO_MSG(0x7164a652, objC);

            // Since the cached property IDs have been cleared, we fetch them anew here, but using only
            // type and name. This means that if the property has not been created by deserialization, we get
            // one without any flags; otherwise, we should get the flags back that the property ID had when it
            // was serialized.
            hkPropertyId objProp2 = hkPropertyId::make<hkReferencedObject>(HK_NULL);
            hkPropertyId intProp2 = hkPropertyId::make<int>(HK_NULL);
            hkPropertyId intPropNonCloned2 = hkPropertyId::make<int>("IntNotCloned", hkPropertyFlags::NO_CLONE);
            // HK_TEST(!intPropNonCloned2.isCloneable());
            hkPropertyId intPropNonSerialized2 = hkPropertyId::make<int>("IntNotSerialized", hkPropertyFlags::NO_SERIALIZE);
            // HK_TEST(intPropNonCloned2.isSerialized()); // Test for true, as the prop ID should have been recreated by this call (and not by deserialization)
            hkPropertyId intPropNonTracked2 = hkPropertyId::make<int>("IntNotTracked", hkPropertyFlags::NO_TRACK);
            // HK_TEST(!intPropNonTracked2.isTracked());

            int& intC = objC->properties().getOrCreate<int>(intProp2);
            // This time we can't test (&intA != &intC) because objA has been freed.
            HK_TEST_EQ(intC, intVal);

            int* ptrIntNonClonedC = objC->properties().getIfExists<int>(intPropNonCloned2);
            HK_TEST(*ptrIntNonClonedC == intValNonCloned);

            int* ptrIntNonSerializedC = objC->properties().getIfExists<int>(intPropNonSerialized2);
            HK_TEST(!ptrIntNonSerializedC); // Moderately pointless, but not worng.

            int* ptrIntNonTrackedC = objC->properties().getIfExists<int>(intPropNonTracked2);
            HK_TEST(*ptrIntNonTrackedC == intValNonTracked);

            hkReferencedObject* ptrObjC = objC->properties().get<hkReferencedObject*>(objProp);
            HK_TEST(ptrObjC);

            objC->removeReference();
        }

        objVal->removeReference();
    }

    // COM-3783
    {
        // Clone a complex property which is part of a graph.
        hkPropertyId complexPropId = hkPropertyId::make<PropertyBagTest::ComplexProperty>(HK_NULL);

        hkRefPtr<PropertyBagTest::GraphObject> objA = hkRefNew<PropertyBagTest::GraphObject>(new PropertyBagTest::GraphObject());
        hkRefPtr<PropertyBagTest::GraphObject> objB = hkRefNew<PropertyBagTest::GraphObject>(new PropertyBagTest::GraphObject());
        hkRefPtr<PropertyBagTest::GraphObject> objC = hkRefNew<PropertyBagTest::GraphObject>(new PropertyBagTest::GraphObject());

        objA->m_child = objB;
        objB->m_child = objC;

        {
            PropertyBagTest::ComplexProperty prop;
            prop.m_object = objC;
            prop.m_parent = objA;

            objB->properties().add(complexPropId, &prop);
        }

        hkRefPtr<PropertyBagTest::GraphObject> cloneA = hkReflect::Cloner().cloneRecursive(objA.val());

        PropertyBagTest::ComplexProperty& prop = cloneA->m_child->properties().get<PropertyBagTest::ComplexProperty>(complexPropId);
        HK_TEST(prop.m_afterReflectNewCalled);
        HK_TEST(prop.m_parent == cloneA);
        HK_TEST(prop.m_object == cloneA->m_child->m_child);
    }

    return 0;
}

HK_TEST_REGISTER(propertyBag_main, "Fast", "Common/Test/UnitTest/Base/", __FILE__);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
