// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>

#include <Common/Base/Types/hkSignalSlots.h>
#include <Common/Base/Thread/TaskQueue/Default/hkDefaultTaskQueue.h>
#include <Common/Base/Thread/TaskQueue/hkTask.h>
#include <Common/Base/Thread/Pool/hkCpuThreadPool.h>
#include <Common/Base/System/Stopwatch/hkStopwatch.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>
#include <Common/Base/Thread/TaskQueue/hkTaskGraph.h>

#define DEBUG_LOG_IDENTIFIER "base.thread.taskqueue"
#include <Common/Base/System/Log/hkLog.hxx>


namespace
{
    // A task used in these tests
    struct TestTask : public hkTask
    {
        TestTask() : m_duration(0.001f), m_processCount(0) {}

        virtual void process( const hkTask::Input& ) HK_OVERRIDE
        {
            hkAtomic::exchangeAdd32( &m_processCount, 1 );

            // Fire the signal
            m_processSignal.fire( this );

            // Do some "work"
            if( m_duration > 0.0f )
            {
                hkStopwatch stopwatch;
                stopwatch.start();
                while( stopwatch.getElapsedSeconds() < m_duration ) {}
            }
        }

        hkReal m_duration;          // how long to ~work during each process()
        hkUint32 m_processCount;    // the total number of calls to process()

        HK_DECLARE_SIGNAL( ProcessSignal, hkSignal1<TestTask*> );
        ProcessSignal m_processSignal;
    };

    // A task which tests that it was processed exactly once
    struct UnaryTask : public TestTask
    {
        ~UnaryTask() { HK_TEST( m_processCount == 1 ); }
    };

    // A unary task which also tests that it is not running in parallel with another one
    struct SerialUnaryTask : public UnaryTask
    {
        SerialUnaryTask() { HK_TEST( s_numBeingProcessed == 0 ); }
        ~SerialUnaryTask() { HK_TEST( s_numBeingProcessed == 0 ); }

        virtual void process( const hkTask::Input& input ) HK_OVERRIDE
        {
            hkAtomic::exchangeAdd32( &s_numBeingProcessed, 1 );
            HK_TEST( s_numBeingProcessed == 1 );
            TestTask::process( input );
            hkAtomic::exchangeAdd32( &s_numBeingProcessed, (hkUint32)-1 );
        }

        static hkUint32 s_numBeingProcessed;
    };
    hkUint32 SerialUnaryTask::s_numBeingProcessed = 0;

    // A task which spawns another independent task and waits for it to be processed
    struct SpawnIndependentTask : public UnaryTask
    {
        SpawnIndependentTask() : m_successorHandle(HK_NULL), m_queue(HK_NULL) {}
        ~SpawnIndependentTask() { HK_TEST( m_successorTask.m_processCount == 1 ); }

        virtual void process( const hkTask::Input& input ) HK_OVERRIDE
        {
            m_queue->allocateHandles( &m_successorHandle, 1 );
            m_queue->initHandle( m_successorHandle, &m_successorTask );
            m_queue->submitHandles( &m_successorHandle, 1 );
            UnaryTask::process( input );
        }

        UnaryTask m_successorTask;
        hkTaskQueue::Handle m_successorHandle;
        hkTaskQueue* m_queue;
    };

    // A task which spawns another dependent task and tests that they were processed in serial
    struct SpawnDependentTask : public SerialUnaryTask
    {
        SpawnDependentTask() : m_successorHandle(HK_NULL), m_queue(HK_NULL) {}
        ~SpawnDependentTask() { HK_TEST( m_successorTask.m_processCount == 1 ); }

        virtual void process( const hkTask::Input& input ) HK_OVERRIDE
        {
            m_queue->allocateHandles( &m_successorHandle, 1 );
            m_queue->initHandle( m_successorHandle, &m_successorTask );
            m_queue->addDependency( m_myHandle, m_successorHandle );
            m_queue->submitHandles( &m_successorHandle, 1 );
            SerialUnaryTask::process( input );
        }

        hkTaskQueue::Handle m_myHandle;
        hkTaskQueue::Handle m_successorHandle;
        SerialUnaryTask m_successorTask;
        hkTaskQueue* m_queue;
    };

    // A task which spawns two child tasks, all with a common successor
    struct BranchTask : public TestTask
    {
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, BranchTask );

        BranchTask( int depth ) : m_depth(depth), m_myHandle(HK_NULL), m_successorHandle(HK_NULL)
        {
            for( int i=0; i<HK_COUNT_OF(m_children); ++i )
            {
                m_children[i] = HK_NULL;
            }
        }

        ~BranchTask()
        {
            for( int i=0; i<HK_COUNT_OF(m_children); ++i )
            {
                delete m_children[i];
            }
        }

        virtual void process( const hkTask::Input& input ) HK_OVERRIDE
        {
            if( m_depth > 0 )
            {
                const int numChildren = HK_COUNT_OF(m_children);
                hkTaskQueue::Handle handles[numChildren];
                input.m_taskQueue->allocateHandles( &handles[0], numChildren );
                for( int i=0; i<numChildren; ++i )
                {
                    m_children[i] = new BranchTask( m_depth - 1 );
                    m_children[i]->m_myHandle = handles[i];
                    m_children[i]->m_successorHandle = m_successorHandle;

                    input.m_taskQueue->initHandle( handles[i], m_children[i] );
                    input.m_taskQueue->addDependency( handles[i], m_successorHandle );
                }
                input.m_taskQueue->submitHandles( &handles[0], numChildren );
            }
            TestTask::process( input );
        }

        const int m_depth;
        hkTaskQueue::Handle m_myHandle;
        hkTaskQueue::Handle m_successorHandle;

        BranchTask* m_children[2];
    };

    // A logger which can be connected to task process() signals to track their order
    struct Logger
    {
        void onProcess( TestTask* task )
        {
            m_criticalSection.enter();
            m_processedTasks.pushBack( task );
            m_criticalSection.leave();
        }

        hkArray<TestTask*> m_processedTasks;
        hkCriticalSection m_criticalSection;
    };
}


static void testHandleProcessing( hkTaskQueue& queue )
{
    // Independent handles with distinct tasks
    for( int m=0; m<2; m++ )
    {
        const int numTasks = 32;
        const int usePriorities = (m==1);

        TestTask tasks[numTasks];
        hkTaskQueue::Handle handles[numTasks];
        queue.allocateHandles( handles, numTasks );

        Logger logger;
        for( int i=0; i<numTasks; ++i )
        {
            queue.initHandle( handles[i], &tasks[i], usePriorities ? hkTask::Priority::Enum(numTasks-i) : hkTask::Priority::MEDIUM );
            tasks[i].m_processSignal.subscribe( &logger, &Logger::onProcess, "Test" );
        }

        queue.submitHandles( handles, numTasks );
        queue.processUntilFinished( handles, numTasks );

        // Checks
        HK_TEST( logger.m_processedTasks.getSize() == numTasks );
        for( int i=0; i<numTasks; ++i )
        {
            HK_TEST2( tasks[i].m_processCount == 1, "Tasks must be processed exactly once" );
            if( !usePriorities )
            {
                //HK_TEST2( logger.m_processedTasks[i] == &tasks[i], "Tasks must be processed in FIFO order" );
            }
            if( usePriorities )
            {
                //HK_TEST2( logger.m_processedTasks[i] == &tasks[numTasks-i-1], "Tasks must be processed in order of priority" );
            }
        }
    }

    // Independent handles with shared tasks
    if(1)
    {
        const int numTasks = 32;
        const int numHandles = numTasks * 2;

        TestTask tasks[numTasks];
        hkTaskQueue::Handle handles[numHandles];
        queue.allocateHandles( handles, numHandles );

        Logger logger;
        for( int i=0; i<numTasks; ++i )
        {
            HK_TEST( tasks[i].m_processCount == 0 );
            queue.initHandle( handles[2*i], &tasks[i] );
            queue.initHandle( handles[2*i + 1], &tasks[i] );
            tasks[i].m_processSignal.subscribe( &logger, &Logger::onProcess, "Test" );
        }

        queue.submitHandles( handles, numHandles );
        queue.processUntilFinished( handles, numHandles );

        // Checks
        HK_TEST( logger.m_processedTasks.getSize() == numHandles );
        for( int i=0; i<numTasks; ++i )
        {
            HK_TEST2( tasks[i].m_processCount == 2, "Tasks must have been processed exactly twice"  );
            //HK_TEST2( logger.m_processedTasks[2*i] == &tasks[i], "Tasks must have been processed in FIFO order" );
            //HK_TEST2( logger.m_processedTasks[2*i + 1] == &tasks[i], "Tasks must have been processed in FIFO order" );
        }
    }
}


static void testDependencies( hkTaskQueue& scheduler )
{
    // Chains of dependent tasks, forwards and reverse:
    // 0-1-2-...-30-31 and 31-30-...-2-1-0
    for( int k=0; k<2; k++ )
    {
        const int numTasks = 32;
        const bool reverse = (k==1);

        SerialUnaryTask tasks[numTasks];
        hkTaskQueue::Handle handles[numTasks];
        scheduler.allocateHandles( handles, numTasks );

        Logger logger;
        for( int i=0; i<numTasks; ++i )
        {
            tasks[i].m_duration = 1e-3; // 1ms
            tasks[i].m_processSignal.subscribe( &logger, &Logger::onProcess, "Test" );
            scheduler.initHandle( handles[i], &tasks[i] );
            if( i > 0 )
            {
                if( !reverse )
                {
                    scheduler.addDependency( handles[i-1], handles[i] );
                }
                else
                {
                    scheduler.addDependency( handles[i], handles[i-1] );
                }
            }
        }

        scheduler.submitHandles( handles, numTasks );
        scheduler.processUntilFinished( handles, numTasks );

        // Checks
        HK_TEST( logger.m_processedTasks.getSize() == numTasks );
        for( int i=0; i<numTasks; ++i )
        {
            HK_TEST2( tasks[i].m_processCount == 1, "Tasks must have been processed exactly once" );
            if( !reverse )
            {
                HK_TEST2( logger.m_processedTasks[i] == &tasks[i], "Dependencies must be respected" );
            }
            else
            {
                HK_TEST2( logger.m_processedTasks[i] == &tasks[numTasks-i-1], "Dependencies must be respected" );
            }
        }
    }

    /* Web of dependent tasks:
         0   3
          \ /
           2
          / \
         1   4
    */
    {
        const int numTasks = 5;

        UnaryTask tasks[numTasks];
        hkTaskQueue::Handle handles[numTasks];
        scheduler.allocateHandles( handles, numTasks );

        Logger logger;
        for( int i=0; i<numTasks; ++i )
        {
            tasks[i].m_duration = 1e-3; // 1ms
            tasks[i].m_processSignal.subscribe( &logger, &Logger::onProcess, "Test" );
            scheduler.initHandle( handles[i], &tasks[i] );
        }
        scheduler.addDependency( handles[0], handles[2] );
        scheduler.addDependency( handles[1], handles[2] );
        scheduler.addDependency( handles[2], handles[3] );
        scheduler.addDependency( handles[2], handles[4] );

        scheduler.submitHandles( handles, numTasks );
        scheduler.processUntilFinished( handles, numTasks );

        // Check that they were processed in order
        HK_TEST( logger.m_processedTasks.getSize() == numTasks );
        HK_TEST( logger.m_processedTasks[0] == &tasks[0] || logger.m_processedTasks[0] == &tasks[1] );
        HK_TEST( logger.m_processedTasks[1] == &tasks[0] || logger.m_processedTasks[1] == &tasks[1] );
        HK_TEST( logger.m_processedTasks[2] == &tasks[2] );
        HK_TEST( logger.m_processedTasks[3] == &tasks[3] || logger.m_processedTasks[3] == &tasks[4] );
        HK_TEST( logger.m_processedTasks[4] == &tasks[3] || logger.m_processedTasks[4] == &tasks[4] );
    }
}

struct SpawnTaskInOtherQueue : public TestTask
{
    virtual void process( const hkTask::Input& input ) HK_OVERRIDE
    {
        hkDefaultTaskQueue queue;
        TestTask innerTask;

        hkTaskQueue::Handle h;
        queue.allocateHandles( &h, 1 );
        queue.initHandle( h, &innerTask );
        queue.setMultiplicity( h, 16, hkTask::MultiplicityMode::PERFORM_ALL_CALLS );
        queue.submitHandles( &h, 1 );
        queue.processUntilFinished( h );
        queue.freeHandles( &h, 1 );
        HK_TEST( innerTask.m_processCount == 16 );

        TestTask::process( input );
    }
};

static void testSpawning( hkTaskQueue& queue )
{
    // A single task which spawns another independent task
    // 0[1]
    if(1)
    {
        SpawnIndependentTask task;
        task.m_queue = &queue;

        hkTaskQueue::Handle h;
        queue.allocateHandles( &h, 1 );
        queue.initHandle( h, &task );
        queue.submitHandles( &h, 1 );

        // Wait for both, in turn
        queue.processUntilFinished( h );
        queue.processUntilFinished( task.m_successorHandle );
    }

    // A single task which spawns another dependent task
    // 0[-1]
    if(1)
    {
        SpawnDependentTask task;
        task.m_queue = &queue;

        queue.allocateHandles( &task.m_myHandle, 1 );
        queue.initHandle( task.m_myHandle, &task );
        queue.submitHandles( &task.m_myHandle, 1 );

        // Wait for both, in turn
        queue.processUntilFinished( task.m_myHandle );
        queue.processUntilFinished( task.m_successorHandle );
    }

    // A root task that spawns a tree of other tasks and waits for them all
    if(0)   
    {
        // This task recursively spawns a tree of other tasks
        BranchTask rootTask(4);
        queue.allocateHandles( &rootTask.m_myHandle, 1 );
        queue.initHandle( rootTask.m_myHandle, &rootTask );

        // This handle gives us something to wait for
        queue.allocateHandles( &rootTask.m_successorHandle, 1 );
        queue.initHandle( rootTask.m_successorHandle, HK_NULL );

        queue.addDependency( rootTask.m_myHandle, rootTask.m_successorHandle );
        queue.submitHandles( &rootTask.m_myHandle, 1 );
        queue.submitHandles( &rootTask.m_successorHandle, 1 );

        queue.processUntilFinished( rootTask.m_successorHandle );
    }

    // A task that spawns a task in another queue and waits for it (COM-4219)
    if(1)
    {
        SpawnTaskInOtherQueue task;

        hkTaskQueue::Handle h;
        queue.allocateHandles( &h, 1 );
        queue.initHandle( h, &task );
        queue.setMultiplicity( h, 16, hkTask::MultiplicityMode::PERFORM_ALL_CALLS );
        queue.submitHandles( &h, 1 );
        queue.processUntilFinished( h );

        HK_TEST( task.m_processCount == 16 );
    }
}

static void testSingleTinyTask( hkTaskQueue& queue )
{
    // Submit and wait for a single tiny task repeatedly
    TestTask tt;
    tt.m_duration = 0.0f;
    for( int i = 0; i < 1000; i++ )
    {
        hkTaskQueue::Handle handle;
        queue.allocateHandles(&handle, 1);
        queue.initHandle(handle, &tt);
        queue.submitHandles(&handle, 1);
        queue.processUntilFinished(handle);
        queue.freeHandles(&handle, 1);
    }
}

static void testManualLoop( hkTaskQueue& queue )
{
    hkDefaultTaskQueue& dt = (hkDefaultTaskQueue&)queue;
    const int numTasks = 64;
    UnaryTask tasks[numTasks];
    hkTaskQueue::Handle handles[numTasks];
    queue.allocateHandles( handles, numTasks );
    for( int i=0; i<numTasks; ++i )
    {
        queue.initHandle( handles[i], &tasks[i] );
    }
    queue.submitHandles( handles, numTasks );

    // Manually loop on this thread (while auto-looping on others)
    while( dt.processOnce() ) {}

    queue.processUntilFinished( handles, numTasks );
}



static void testMultiplicity1( hkTaskQueue& queue )
{
    struct TestMultiplicityTask: public hkTask
    {
        TestMultiplicityTask()
        {
            for( int i=0; i<HK_COUNT_OF(m_bits); i++ )
            {
                m_bits[i] = 0;
            }
        }
        void process( const Input& input )
        {
            m_bits[input.m_multiplicityIndex]++;
        }

        int m_bits[8];
    } task;

    {
        hkTaskQueue::Handle handle;
        queue.allocateHandles(&handle, 1);
        queue.initHandle(handle, &task);
        queue.setMultiplicity(handle, HK_COUNT_OF(task.m_bits), hkTask::MultiplicityMode::PERFORM_ALL_CALLS);
        queue.submitHandles(&handle, 1);
        queue.processUntilFinished(handle);
        queue.freeHandles(&handle, 1);
    }

    for( int i=0; i<HK_COUNT_OF(task.m_bits); i++ )
    {
        HK_TEST( task.m_bits[i] == 1 );
    }
}


static void testMultiplicity2( hkTaskQueue& queue )
{
    struct testMultiplicityTask : public hkTask
    {
        testMultiplicityTask() : m_counter(0) {}
        void process( const Input& input ) { m_counter++; }
        int m_counter;
    } task;

    // lets use the task graph
    {
        hkTaskGraph graph;
        graph.addTaskWithMultiplicity(&task, 1000, hkTask::MultiplicityMode::ABORT_ON_FIRST_FINISHED_TASK);
        queue.submitGraphAndWait( graph );
    }
    HK_TEST( task.m_counter < 1000 );
}



struct EulerPhiTask : public hkTask
{
    virtual void process(const hkTask::Input& in) HK_OVERRIDE
    {
        int n = m_number;

        if (n <= 1)
        {
            m_result = 0;
        }
        else
        {
            hkArray<hkTuple<int, int> > p;
            for (int i = 2; i <= n; ++i)
            {
                int j, q = 1;
                for (j = 0; n % i == 0; ++j)
                {
                    n /= i;
                    q *= i;
                }
                if (j > 0)
                {
                    p.pushBack(hkTupleT::make(i, q));
                }
            }

            if (p.getSize() > 1)
            {
                hkTaskGraph graph;
                hkArray<EulerPhiTask> childTasks; childTasks.setSize(p.getSize());
                for (int i = 0; i < p.getSize(); ++i)
                {
                    childTasks[i].m_number = p[i].m_1;
                    graph.addTask(&childTasks[i]);
                };
                in.m_taskQueue->submitGraphAndWait(graph);

                m_result = 1;
                for (int i = 0; i < p.getSize(); ++i){ m_result *= childTasks[i].m_result; }
            }
            else
            {
                HK_ASSERT_NO_MSG(0x7673da4, p.getSize() == 1);
                m_result = p[0].m_1 - p[0].m_1 / p[0].m_0;
            }
        }
        Log_Info( "phi {} is {}", m_number, m_result );
    }
    int m_number;
    int m_result;
} ;

static void test_testEulerPhi(hkTaskQueue& queue)
{
    const int n = 100;
    EulerPhiTask tasks[n];
    {
        hkTaskGraph graph;
        for (int i = 0; i < n; i++ )
        {
            tasks[i].m_number = n - i;
            graph.addTask(&tasks[i]);
        }
        queue.submitGraphAndWait(graph);
    }
}



// test simple batching
static void testBatch1( hkTaskQueue& queue )
{
    struct CheckMultiplicity: public hkTaskBatchProcessor
    {
        CheckMultiplicity() { m_bits = 0; m_result = 0; }

        void beginTask( const hkTask::Input& input, const BatchInfo& ) { m_bits |= 1<<input.m_multiplicityIndex; }
        void processBatch( const hkTask::Input& input, const BatchInfo&, int batchStart, int numItems ) { m_result |= 1<<batchStart; }

        int m_bits;
        hkUint32 m_result;
    } batchProcessor;

    // run
    queue.forEach( &batchProcessor, hkTaskBatchProcessor::BatchInfo( 32, 4 ) );

    // test
    //HK_TEST( batchProcessor.m_bits = 0xff );
    //HK_TEST( batchProcessor.m_result == 0xffffffff );
}



// test complicated batching:
// calculate the average of [0..15]*
static void testBatch2( hkTaskQueue& queue )
{
    struct BuildAverage : public hkTaskBatchProcessor
    {
        void prepare( const BatchInfo& bi, int numBatches )
        {
            m_localSum.setSize( numBatches, 0 );
        }

        void processBatch( const hkTask::Input& input, const BatchInfo &bi, int batchStart, int numItems )
        {
            for (int i = batchStart; i < batchStart+numItems; i++ )
            {
                m_localSum[input.m_multiplicityIndex] += (i & 15);
            }
        }
        hkInplaceArray<hkUint32, 16> m_localSum;
    } batchProcessor;

    // run
    const int numItems = 1024;
    queue.forEach( &batchProcessor, numItems );

    // test
    {
        int sum = 0;
        for (int i =0; i<batchProcessor.m_localSum.getSize(); i++ )
        {
            sum += batchProcessor.m_localSum[i];
        }
        HK_TEST( sum == (numItems/16) * 15 * 8 );
    }
}



//
//  Interruption test specific

namespace TaskInterruptionTest
{
    typedef hkTaskGraph::TaskId TaskId;
    typedef hkTaskQueue::Handle Handle;

    // The base class for the tasks used in this test
    struct BaseTask : public hkReferencedTask
    {
        BaseTask()
        :   m_wasProcessed(false)
        {}

        virtual bool isAbortable() const override
        {
            return true;
        }

        bool m_wasProcessed;
    };

    // An expensive task
    struct WorkerTask : public BaseTask
    {
        enum { NUM_ITER = 100000, };

        WorkerTask(hkPseudoRandomGenerator& rng)
        {
            rng.getRandomRotation(m_a);
        }

        // Computes the n-th power of a quaternion
        void process( const Input& input )
        {
            for (int k = NUM_ITER - 1; k >= 0; k--)
            {
                if ( hkTask::receivedAbortRequest(input.m_executionContext) )
                {
                    return;
                }

                hkQuaternion q;
                q.setMul(m_a, q);
                q.normalize();
                m_a = q;
            }
            m_wasProcessed = true;
        }
        hkQuaternion m_a;
    };

    // The cancel task
    struct CancelTask : public BaseTask
    {
        void process( const Input& input )
        {
            hkTaskQueue* taskQueue = input.m_taskQueue;
            taskQueue->abortHandles(&m_handle, 1);
            m_wasProcessed = true;
        }
        Handle m_handle;
    };

    //
    //  Test entry point

    static void run( hkTaskQueue& queue )
    {
        hkArray< hkRefPtr<BaseTask> > allTasks;

        // Create a binary tree of stages all depending on each other
        hkTaskGraph taskGraph;
        {
            hkArray<TaskId> prevTaskIds;
            hkArray<TaskId> crtTaskIds;

            hkPseudoRandomGenerator rng(13);

            // Create the first stage
            const int numInitialTasks = 20;
            prevTaskIds.setSize(numInitialTasks);
            for (int k = numInitialTasks - 1; k >= 0; k--)
            {
                BaseTask* t = new WorkerTask(rng);
                allTasks.expandOne().setAndDontIncrementRefCount(t);
                prevTaskIds[k] = taskGraph.addReferencedTask(t);
            }

            // Create the cancel task
            const int cancelTaskIdx     = 27;
            CancelTask* cancelTask      = new CancelTask();
            const TaskId cancelledTaskId(35);

            // Add the other stages
            int numTotalTasks = numInitialTasks;
            while ( prevTaskIds.getSize() > 1 )
            {
                crtTaskIds.clear();

                const int numPrevTasks = prevTaskIds.getSize();
                int k = 0;
                for (; k < numPrevTasks - 1; k += 2)
                {
                    // Add task
                    BaseTask* t = ( ++numTotalTasks == cancelTaskIdx ) ? static_cast<BaseTask*>(cancelTask) : new WorkerTask(rng);
                    allTasks.expandOne().setAndDontIncrementRefCount(t);
                    TaskId crtTaskIdAB = taskGraph.addReferencedTask(t);

                    // Set-up dependencies
                    crtTaskIds.pushBack(crtTaskIdAB);
                    taskGraph.addDependency(prevTaskIds[k], crtTaskIdAB);
                    taskGraph.addDependency(prevTaskIds[k + 1], crtTaskIdAB);
                }

                for (; k < numPrevTasks; k++)
                {
                    crtTaskIds.pushBack(prevTaskIds[k]);
                }

                prevTaskIds.swap(crtTaskIds);
            }

            // Set-up the cancel task
            cancelTask->m_handle = taskGraph.preallocateTaskQueueHandle(queue, cancelledTaskId);

            // Run the tasks
            queue.submitGraphAndWait(taskGraph);
        }

        // Print the task execution stats
#if 0
        Log_Info( "-----------Task Interruption Test--------------" );
        for (int k = allTasks.getSize() - 1; k >= 0; k--)
        {
            const BaseTask* task = allTasks[k];

            if ( task->m_wasProcessed ) { Log_Info( "Task {} was processed.", k );  }
            else                        { Log_Info( "Task {} was cancelled!", k );  }
        }
        Log_Info( "-----------------------------------------------" );
#endif
    }
}

int taskQueueTest_main()
{
    
    if( hkUnitTest::s_taskQueue == HK_NULL ) return 0;

    // Create a task queue
    hkTaskQueue& queue = *hkUnitTest::s_taskQueue;

    // test interruptions
    {
        TaskInterruptionTest::run( queue );
    }

    // Test base interface
    {
        testHandleProcessing( queue );
        testDependencies( queue );
        testSpawning( queue );
        testSingleTinyTask( queue );
    }

    // Test added functions in default Havok implementation
    {
        testManualLoop( queue );
    }

    // test multiplicity
    {
        testMultiplicity1( queue );
        testMultiplicity2( queue );
    }

    // test batching interface
    {
        testBatch1( queue );
        testBatch2( queue );
    }

    return 0;
}

HK_TEST_REGISTER(taskQueueTest_main, "Fast", "Common/Test/UnitTest/Base/", __FILE__);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
