// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/Container/String/hkUtf8.h>
#include <Common/Base/Container/String/hkUtf8Detail.h>
#include <Common/Base/Container/StringView/hkStringView.h>

static const char helloworld[] = "Hello world!\n";
static const char HELLOWORLD[] = "HELLO WORLD!\n";
static const char hello[] = "Hello";
static const char world[] = "world!\n";

static void hkarray_stringbuf_not_allowed()
{
    // uncommenting should produce a compile error
    //hkArray<hkStringBuf> sb0;
}

static void string_memset16_test()
{
    HK_ALIGN16(hkUint32 dst[1024 / 4]);
    HK_ALIGN16(hkUint32 src[   4]);

    src[0] = 0;
    src[1] = 1;
    src[2] = 2;
    src[3] = 3;

    // medium, gets unrolled on PlayStation(R)3
    hkString::memSet16<128> (dst, src);

    for (int i=0; i < (128 / 4); i+=4)
    {
        HK_TEST( dst[i+0] == 0 );
        HK_TEST( dst[i+1] == 1 );
        HK_TEST( dst[i+2] == 2 );
        HK_TEST( dst[i+3] == 3 );
    }

    src[0] = 10;
    src[1] = 11;
    src[2] = 12;
    src[3] = 13;

    // large, always falls back to loop
    hkString::memSet16<1024> (dst, src);

    for (int i=0; i < (1024 / 4); i+=4)
    {
        HK_TEST( dst[i+0] == 10 );
        HK_TEST( dst[i+1] == 11 );
        HK_TEST( dst[i+2] == 12 );
        HK_TEST( dst[i+3] == 13 );
    }
}

static void testSplitEqual(const char* src, const char*const* expected)
{
    hkStringBuf sb(src);
    hkArray<const char*>::Temp bits;
    sb.split('/', bits);
    int i;
    for( i = 0; i < bits.getSize(); ++i )
    {
        HK_TEST( expected[i] != HK_NULL );
        HK_TEST( hkString::strCmp(bits[i], expected[i]) == 0 );
    }
    HK_TEST(expected[i] == HK_NULL);
}

//#include <Common/Base/Fwd/hkcstdlib.h>
static void string_int24w()
{
#if 0 // currently fails
    hkVector4 v0;
    for( int i = 0; i < 1<<24; ++i )
    {
        v0.setInt24W(i);
        hkStringBuf sb;
        sb.printf("%g", v0(3));

        hkVector4 v1; v1(3) = (hkReal)strtod(sb.cString(), HK_NULL);
        HK_TEST2( v0(3) == v1(3), "Iteration" << i );
    }
#endif
}

static void testPathAppend( const char* expected, const char* orig, const char* a, const char* b)
{
    hkStringBuf sb = orig;
    sb.pathAppend(a,b);
    HK_TEST2( sb == expected, "PathAppend sb='"<<sb<<"' expected='"<<expected<<"'");
}

static void testPathChildname(const char* input, const char* expected)
{
    hkStringBuf sb = input;
    sb.pathChildname();
    HK_TEST2(sb == expected, "PathChildname sb='" << sb << "' expected='" << expected << "'");
}

static void testPathNormalize( const char* input, const char* expected)
{
    hkStringBuf sb = input;
    sb.pathNormalize();
    HK_TEST2( sb == expected, "PathNormalize input='"<<input<<"' sb='"<<sb<<"' expected='"<<expected<<"'");
}

static void testStringsEqual(const wchar_t* wideInput, const char* utf8Input)
{
    // test that wild converts to utf8
    hkStringPtr utf8Local = hkUtf8::Utf8FromWide(wideInput).cString();
    HK_TEST( hkString::strCmp(utf8Input, utf8Local) == 0 );

    // and vice versa
    hkUtf8::WideFromUtf8 wideHelper(utf8Input);
    HK_TEST( hkString::memCmp(wideInput, wideHelper.cString(), wideHelper.getLength() * sizeof(wchar_t) ) == 0 );
}

static void testUtf8Decode(const wchar_t* wideInput, const char* utf8Input)
{
    // test that utf8 decodes to wide
    hkUtf8::WideFromUtf8 wideHelper(utf8Input);
    HK_TEST( hkString::memCmp(wideInput, wideHelper.cString(), wideHelper.getLength() * sizeof(wchar_t) ) == 0 );
}

static void string_encoding_equal()
{
    //Visual Studio doesn't do string unicode constants so we use escapes
    testStringsEqual(L"\x3072\x307f\x3087 \x3074\x30b8", "\xe3\x81\xb2\xe3\x81\xbf\xe3\x82\x87 \xe3\x81\xb4\xe3\x82\xb8");

    // test bad encodings

    HK_TEST(hkUtf8::Detail::findFirstInvalidUtf8("bad \xff yeah?") == 4);
    HK_TEST(hkUtf8::Detail::findFirstInvalidUtf8("bad \xca *") == 4);
    HK_TEST(hkUtf8::Detail::findFirstInvalidUtf8("bad \xea\x8a *") == 4);
    HK_TEST(hkUtf8::Detail::findFirstInvalidUtf8("bad \xf7\x8a *") == 4);
    HK_TEST(hkUtf8::Detail::findFirstInvalidUtf8("bad \xfd\x8f\x8f *") == 4);
    HK_TEST(hkUtf8::Detail::findFirstInvalidUtf8("bad \xfd\x8f\x8f\xe3\x81\xb4 *") == 4);

    {
        wchar_t wbuf[100] = {};
        int n;
        n = hkUtf8::wideFromUtf8(HK_NULL, 0, "hello");
        HK_TEST( n == 6 );
        n = hkUtf8::wideFromUtf8(wbuf, 100, "hello");
        HK_TEST( n == 6 );
        HK_TEST( hkString::memCmp(wbuf, L"hello", 6*sizeof(wchar_t)) == 0);

        // only room for 1st char
        n = hkUtf8::wideFromUtf8(wbuf, 2, "<\xe3\x81\xb4>");
        HK_TEST( n == 4 );
        HK_TEST( wbuf[1] == 0 );
    }

    {
        char cbuf[100] = {};
        const wchar_t wsun[] = L"<\x65e5>";
        const char csun[] = "<\xe6\x97\xa5>";
        int n8 = hkUtf8::utf8FromWide(HK_NULL, 0, wsun);
        HK_TEST( n8 == sizeof(csun) ); //0x65e5 -> 0xE6 0x97 0xA5
        n8 = hkUtf8::utf8FromWide(cbuf, 100, wsun);
        HK_TEST( hkString::memCmp(cbuf, csun, sizeof(csun)) == 0 );
        // room for first but not enough room for 2nd char
        n8 = hkUtf8::utf8FromWide(cbuf, 3, wsun);
        HK_TEST( n8 == sizeof(csun) );
        HK_TEST( cbuf[0]=='<' );
        HK_TEST( cbuf[1]==0 );
    }

    // null string
    {
        hkUtf8::WideFromUtf8 wideHelper(HK_NULL);
        HK_TEST(wideHelper.cString() == HK_NULL);
        hkUtf8::Utf8FromWide utf8Helper((const wchar_t*)HK_NULL);
        HK_TEST(utf8Helper.cString() == HK_NULL);
    }
}

template<typename StringType>
static void test_string_printf()
{
    {
        StringType ptr;
        ptr.printf("");
        HK_TEST(ptr == "");
    }

    {
        StringType ptr;
        ptr.printf("H%cv%dk %s", 'a', 0, "test");
        HK_TEST(ptr == "Hav0k test");
    }

    {
        StringType ptr;
        hkStringBuf sample(
            "Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod"
            "tempor incididunt ut labore et dolore magna aliqua. Ut enim ad minim"
            "veniam, quis nostrud exercitation ullamco laboris nisi ut aliquip ex ea"
            "commodo consequat. Duis aute irure dolor in reprehenderit in voluptate"
            "velit esse cillum dolore eu fugiat nulla pariatur. Excepteur sint "
            "occaecat cupidatat non proident, sunt in culpa qui officia deserunt "
            "mollit anim id est laborum");
        hkStringBuf format(sample);
        format.replace("Lorem", "%s", hkStringBuf::REPLACE_ONE);
        format.replace("ullamco", "%s", hkStringBuf::REPLACE_ONE);
        format.replace("reprehenderit", "%s", hkStringBuf::REPLACE_ONE);

        ptr.printf(format.cString(), "Lorem", "ullamco", "reprehenderit");
        HK_TEST_EQ(ptr.cString(), sample.cString());
    }

}

static void string_printf()
{
    test_string_printf<hkStringBuf>();
    test_string_printf<hkStringPtr>();
    {
        char buf[4];
        buf[3] = 0x0f;
        hkString::snPrintf(buf, HK_COUNT_OF(buf), HK_COUNT_OF(buf)-1, "%x", 0xffffffff);
        HK_TEST(buf[3] == 0);
    }
}

static void string_view_test()
{
    {
        hkStringView orig = "hexlo";
        HK_TEST(orig.getSize()==5);
        HK_TEST(orig[0]=='h');

        {
            hkStringView s = orig.ltrim(3);
            HK_TEST(s.getSize()==2);
            HK_TEST(s[0]=='l');
        }
        {
            hkStringView s = orig.rtrim(3);
            HK_TEST(s.getSize()==2);
            HK_TEST(s[0]=='h');
        }
        {
            hkStringView s = orig.slice(1,3);
            HK_TEST(s.getSize()==3);
            HK_TEST(s[0]=='e');
        }
    }
    {
        HK_TEST( hkStringView("hllo") == "hllo" );
        HK_TEST( hkStringView("hllo2") != "hllo" );
        HK_TEST( hkStringView("hllo") != "hllo2" );
        HK_TEST( hkStringView("hllo") <= "hllo2" );
        HK_TEST( hkStringView("hllo") < "hllo2" );
        HK_TEST( hkStringView("hllo3") >= "hllo2" );
        HK_TEST( hkStringView("hllo3") > "hllo2" );

        HK_TEST( hkStringView("blah").compareLower("BLah") == 0);
        HK_TEST( hkStringView("_aff").compareLower("_BFF") == -1);
        HK_TEST( hkStringView("_Aff").compareLower("_BFF") == -1);
        HK_TEST( hkStringView("_aff").compareLower("_bFF") == -1);
        HK_TEST( hkStringView("_aff").compareLower("_BFF") == -1);
    }
    {
        if( hkStringView() )
        {
            HK_TEST(false);
        }

        if( !hkStringView("foo") )
        {
            HK_TEST(false);
        }
    }

    {
        hkStringView haystack = "Hello";
        {
            hkStringView match = haystack.strStr("Hello");
            HK_TEST( match.begin() == haystack.begin() );
            HK_TEST( match.end() == haystack.end() );
        }

        {
            hkStringView match = haystack.strStr("Hel");
            HK_TEST( match.begin() == haystack.begin() );
            HK_TEST( match.end() == haystack.begin() + 3 );
        }

        {
            hkStringView match = haystack.strStr("");
            HK_TEST( match.begin() == haystack.begin() );
            HK_TEST( match.end() == haystack.begin() );
        }

        {
            hkStringView match = haystack.strStr("lo");
            HK_TEST( match.begin() == haystack.begin() + 3 );
            HK_TEST( match.end() == haystack.end() );
        }

        {
            hkStringView match = haystack.strStr("ell");
            HK_TEST( match.begin() == haystack.begin() + 1 );
            HK_TEST( match.end() == haystack.begin() + 4 );
        }

        HK_TEST( !haystack.strStr("sdk") );
        HK_TEST( !haystack.strStr("sdkasdsadasdas") );
        HK_TEST( haystack.strChr('H') == haystack.begin() );
        HK_TEST( haystack.strChr('l') == haystack.begin() + 2 );
        HK_TEST( haystack.strChr('o') == haystack.begin() + 4 );
        HK_TEST( haystack.strChr(0) == HK_NULL );
        HK_TEST( haystack.strChr('z') == HK_NULL );
    }

    {
        hkStringView haystack;
        HK_TEST( !haystack.strStr( "Hello" ) );
        HK_TEST( !haystack.strStr( hkStringView() ) );
        HK_TEST( !haystack.strChr( 0 ) );
    }
}

static void result_to_string()
{
    {
        hkStringBuf s; // a known value
        HK_E_UNEXPECTED.toString(s);
        HK_TEST_EQ(s, "HK_E_UNEXPECTED");
    }
    {
        hkStringBuf s; // upper case?
        HK_RESULT_FROM_ERROR_CODE(0x8421abcd).toString(s);
        HK_TEST_EQ(s, "HK_E_8421ABCD");
    }
    {
        hkStringBuf s; // zero padding
        HK_RESULT_FROM_ERROR_CODE(42).toString(s);
        HK_TEST_EQ(s, "HK_SUCCESS_0000002A");
    }
}

static int string_main()
{
    {
        hkStringBuf s;
        HK_TEST(s.cString()[0] == 0);
        HK_TEST(s.getLength() == 0);
    }
    {
        hkStringBuf s(helloworld);
        HK_TEST(s[0]=='H');
        HK_TEST(s.getLength() == sizeof(helloworld) - 1);

        HK_TEST(s.startsWith(hello));
        hkStringBuf hello2(hello);
        HK_TEST(s.startsWith(hello2));

        HK_TEST(s.endsWith(world));
        hkStringBuf world2(world);
        HK_TEST(s.endsWith(world2));

        HK_TEST(s.indexOf('l') == 2 );
        HK_TEST(s.indexOf('!') == 11);

        {
            hkStringBuf scpy( s );
            scpy.upperCase();
            HK_TEST( scpy == HELLOWORLD );
        }
        hkStringBuf t( HELLOWORLD );
        {
            hkStringBuf tcpy( t );
            tcpy.lowerCase();
            HK_TEST( tcpy != s );
        }

        HK_TEST( s.compareToIgnoreCase(t)==0 );

        {
            hkStringBuf trunc( t );
            trunc.chompEnd( 1 );
            HK_TEST( trunc < t );
            HK_TEST( trunc < t.cString() );

            hkStringBuf app( t );
            app.append( "a" );
            HK_TEST( app > t );
            HK_TEST( app > t.cString() );

            hkStringBuf diff( t );
            diff.getArray()[diff.getLength() - 1] = '?';
            HK_TEST( diff > t );
            HK_TEST( diff > t.cString() );
        }

        {
            hkStringBuf scpy( s );
            scpy.slice(3,5);
            HK_TEST( scpy == "lo wo");
        }

        HK_TEST( hkStringBuf("apples").compareTo("oranges") < 0 );
        HK_TEST( hkStringBuf("apples").compareTo("apples") == 0 );
        HK_TEST( hkStringBuf("bananas").compareTo("apples") > 0 );

        {
            hkStringBuf tfc("the", " fat", " cat");
            HK_TEST( tfc == "the fat cat" );
        }

        hkStringBuf u("one");
        u += "two";
        u += "three";
        HK_TEST( u == "onetwothree" );
    }
    {
        
        hkStringBuf sb;
        HK_TEST((sb == HK_NULL) == false); // hkStringBuf::operator const char*() returns a pointer to the internal buffer
        HK_TEST(sb.compareToIgnoreCase(HK_NULL) == 1);
        HK_TEST(sb.compareTo(HK_NULL) == 1);

        // Test against empty
        HK_TEST(sb == "");

        hkStringPtr sp;
        HK_TEST((sp == "") == false); // empty string and nullptr not considered equal
        HK_TEST(sp == HK_NULL);
    }
    {
        hkStringBuf s1(helloworld); s1.replace('o', 'X', hkStringBuf::REPLACE_ONE);
        hkStringBuf s2(helloworld); s2.replace('o', 'X', hkStringBuf::REPLACE_ALL);
        hkStringBuf s3(helloworld); s3.replace('o', 'X');
        HK_TEST( s1 == "HellX world!\n" );
        HK_TEST( s2 == "HellX wXrld!\n" );
        HK_TEST( s3 == "HellX wXrld!\n" );

        
        {
            const char* const nullString = HK_NULL; 
            const char* const emptyString = "";
            const char* const shortString = "very short";
            const char* const longString = "this is the longest string you can think of or at least the longest I want to write";
            const char* testStrings[] = { nullString, emptyString, shortString, longString };
            const int numStrings = hkSizeOf( testStrings ) / hkSizeOf( testStrings[0] );
            hkStringBuf s0( helloworld );
            for ( int i = 0; i < numStrings; ++i )
            {
                for ( int j = 0; j < numStrings; ++j )
                {
                    HK_TEST( false == s0.replace( testStrings[i], testStrings[j] ) );
                }
            }
        }

        s1 = "this is the longest string you can think of or at least the longest I want to write.not short";
        s1 = "very short";
        s1.replace("very", "this is the longest string you can think of or at least the longest I want to write.not");
        HK_TEST( s1 == "this is the longest string you can think of or at least the longest I want to write.not short" );
    }
    {
        hkStringBuf s1("Lo! hello, hello!");
        hkStringBuf s2( s1 );
        hkStringBuf s3( s1 );

        s1.replace("lo", "[0123]", hkStringBuf::REPLACE_ONE);
        s2.replace("lo", "[0123]", hkStringBuf::REPLACE_ALL);
        s3.replace("lo", "[0123]");
        HK_TEST( s1 == "Lo! hel[0123], hello!" );
        HK_TEST( s2 == "Lo! hel[0123], hel[0123]!" );
        HK_TEST( s3 == "Lo! hel[0123], hel[0123]!" );
    }
    {
        hkStringBuf s("Lo! hello, hello!");
        hkStringBuf s1( s );
        hkStringBuf s2 = s1;
        s1.slice(0,3);
        s2.slice(4,5);
        HK_TEST( s1 == "Lo!");
        HK_TEST( s2 == "hello");
    }

    {
        // standard
        HK_TEST( hkString::atoi("123") == 123);
        HK_TEST( hkString::atoi("-123") == -123);

        // Unary + is allowed
        HK_TEST( hkString::atoi("+123") == 123);

        // Whitespace
        HK_TEST( hkString::atoi(" 123") == 123);
        HK_TEST( hkString::atoi("  123") == 123);
        HK_TEST( hkString::atoi("\t123") == 123);
        HK_TEST( hkString::atoi("\t\t123") == 123);

        // Garbage allowed at end
        HK_TEST( hkString::atoi("123BAD") == 123);

        // Other bases
        HK_TEST( hkString::atoi("0x400") == 1024);

        // Uber test
        HK_TEST( hkString::atoi("  \t  \t +123BAD") == 123);

    }

    {
        // standard
        HK_TEST( 0          == hkString::atof( "0" ) );
        HK_TEST( 5          == hkString::atof( "5" ) );
        HK_TEST( -5         == hkString::atof( "-5" ) );
        HK_TEST( 0          == hkString::atof( "0.0" ) );
        HK_TEST( 5.05f - hkString::atof( "5.05" ) < 1e-6f);
        HK_TEST( 5.0f       == hkString::atof( "5." ) );
        HK_TEST( -0.5f      == hkString::atof( "-.5" ) );
        HK_TEST( -0.5f      == hkString::atof( "-0.5" ) );
        HK_TEST( 5.5f       == hkString::atof( "5.5" ) );
        HK_TEST( -5.5f      == hkString::atof( "-5.5" ) );
        HK_TEST( 500000.0f  == hkString::atof( "5e5" ) );
        HK_TEST( 500000.0f  == hkString::atof( "5E5" ) );
        HK_TEST( 0.0f       == hkString::atof( "0.0e5" ) );
        HK_TEST( 500000.0f  == hkString::atof( "5.e5" ) );
        HK_TEST( 50000.0f   == hkString::atof( ".5e5" ) );
        HK_TEST( -550000.0f == hkString::atof( "-5.5e5" ) );
//          HK_TEST( 0.001f > hkMath::fabs( 5e-5f - hkString::atof( "5e-5" ) ) );
//          HK_TEST( 0.001f > hkMath::fabs( 5e-5f - hkString::atof( "5E-5" ) ) );
//          HK_TEST( 0.001f > hkMath::fabs( -5e-5f - hkString::atof( "-5e-5" ) ) );
//          HK_TEST( 0.001f > hkMath::fabs( -5e-5f - hkString::atof( "-5E-5" ) ) );
//          HK_TEST( 0.001f > hkMath::fabs( 5e-5f - hkString::atof( "5.0e-5" ) ) );
//          HK_TEST( 0.001f > hkMath::fabs( 5e-6f - hkString::atof( "0.5e-5" ) ) );
//          HK_TEST( 0.001f > hkMath::fabs( -5.5e-5f - hkString::atof( "-5.5e-5" ) ) );

        // error really but does return result
        HK_TEST( 0          == hkString::atof( "" ) );
//          These tests seem to be invalid.
//          HK_TEST( 0.001f > hkMath::fabs( -5.5e-5f - hkString::atof( "-5.-5e-5" ) ) );
//          HK_TEST( -550000.0f == hkString::atof( "-5.-5e5" ) );
//          HK_TEST( -5.5f      == hkString::atof( "-5.-5" ) );
    }

    {
        char string1[] = "a";
        char string2[] = "AB";
        HK_TEST( hkString::strCasecmp(string1, string2) == -1 );
        HK_TEST( hkString::strCasecmp(string2, string1) == 1 );

        char string3[] = "ab";
        char string4[] = "AB";
        HK_TEST( hkString::strCasecmp(string3, string4) == 0 );
        HK_TEST( hkString::strCasecmp(string4, string3) == 0 );
    }
    {
        char string1[] = "ab";
        char string2[] = "ABC";
        int n = 1;
        HK_TEST( hkString::strNcasecmp(string1, string2, n) == 0 );
        HK_TEST( hkString::strNcasecmp(string2, string1, n) == 0 );

        n = 2;
        HK_TEST( hkString::strNcasecmp(string1, string2, n) == 0 );
        HK_TEST( hkString::strNcasecmp(string2, string1, n) == 0 );

        n = 3;
        HK_TEST( hkString::strNcasecmp(string1, string2, n) == -1 );
        HK_TEST( hkString::strNcasecmp(string2, string1, n) == 1 );

        n = 4;
        HK_TEST( hkString::strNcasecmp(string1, string2, n) == -1 );
        HK_TEST( hkString::strNcasecmp(string2, string1, n) == 1 );
    }
    {
        HK_TEST( hkString::indexOf("foo", 'x') == -1 );
        HK_TEST( hkString::indexOf("foo", 'o', 100, 1000) == -1 );
        HK_TEST( hkString::indexOf("foo", 'o', 1) == 1 );
        HK_TEST( hkString::indexOf("foo", 'o', 2) == 2 );
        HK_TEST( hkString::indexOf("foo", 'o', 0,1) == -1 );
        HK_TEST( hkString::indexOf("foo", 'o', 1,1) == -1 );
    }
    {
        hkStringBuf roman;
        roman.printf( "Write %s if you want %i.", "IV", 4 );
        HK_TEST( roman == "Write IV if you want 4." );
    }

    {
        hkStringPtr greeting = "hello";
        hkStringPtr goodbye = "goodbye";
        hkStringPtr null;

        HK_TEST( greeting == "hello" );
        HK_TEST( greeting != HK_NULL );
        HK_TEST( greeting != goodbye );
        HK_TEST( null  == HK_NULL );
    }

    {
        // simple
        testPathNormalize("../test.txt", "../test.txt" );
        testPathNormalize("..", ".." );
        testPathNormalize("", "" );
        testPathNormalize("test.txt", "test.txt" );

        // combinations of ..
        testPathNormalize("C:\\temp\\test\\..\\..\\testing.txt", "C:/testing.txt");
        testPathNormalize("C:\\temp\\test\\..\\../testing\\..\\more/testing\\test.txt", "C:/more/testing/test.txt" );
        testPathNormalize("a/b/../..", "" );

        // leading ..
        testPathNormalize("../foo/../bar/../test.txt", "../test.txt" );
        testPathNormalize("../foo/../bar/../../test.txt", "../../test.txt" );
        testPathNormalize("../../a/b", "../../a/b" );
        testPathNormalize("../a/..", ".." );
        testPathNormalize("../a/../..", "../.." );

        // absolute paths
        testPathNormalize("/a/b/../c", "/a/c");
        testPathNormalize("/", "/");
        testPathNormalize("/a/..", "/");
        testPathNormalize("/a/b/..", "/a");

        // trailing slashes
        testPathNormalize("/a/b/../", "/a");

        // dots and leading dots
        testPathNormalize("./a/b/../", "a");
        testPathNormalize("./a/./b", "a/b");
        testPathNormalize("/a/./b", "/a/b");
        testPathNormalize("/a/./b/.", "/a/b");

        // double slashes at incorrect locations
        testPathNormalize("//patha//pathb/../", "//patha");

        // ill formed paths & the resulting path
        testPathNormalize("//../foo", "//../foo");
        testPathNormalize("//server//path//..//../../", "//..");
        testPathNormalize("/a/../..", "/..");

        // double slashes? we want to preserve unc paths
        testPathNormalize("//foo/bar/.", "//foo/bar");
        testPathNormalize("\\\\foo\\bar", "//foo/bar");
    }

    {
        // simple
        testPathChildname("foo/bar/test", "bar/test");
        testPathChildname("foo/bar", "bar");
        testPathChildname("foo\\bar\\test", "bar\\test");
        testPathChildname("foo\\bar", "bar");

        // mix '\' and '/'
        testPathChildname("foo\\bar/test", "bar/test");
        testPathChildname("foo/bar\\test", "bar\\test");

        // leading slashes
        testPathChildname("/foo/bar/test", "bar/test");
        testPathChildname("//foo/bar/test", "bar/test");

        // combinations with '..' and '.'
        testPathChildname("/foo/bar/..", "bar/..");
        testPathChildname("/../bar/test", "bar/test");
        testPathChildname("../../bar/test", "../bar/test");
        testPathChildname("//foo/bar/test/..", "bar/test/..");
        testPathChildname(".\\foo/./bar/../test", "foo/./bar/../test");
        testPathChildname(".././foo/bar/test", "./foo/bar/test");

        // double slashes in invalid place
        testPathChildname("..//bar/test", "/bar/test");
        testPathChildname("\\..//\\bar/test", "/\\bar/test");
        testPathChildname("\\\\foo/bar/test", "bar/test");
        testPathChildname("\\/foo/bar/test", "bar/test");

        // no output
        testPathChildname("foo", "");
        testPathChildname("\foo", "");
        testPathChildname("//", "");
        testPathChildname("////", "");
    }

    {
        testPathAppend( "/foo/bar", "", "/foo", "/bar");
        testPathAppend( "/foo/bar", "/foo", "/bar", "");
        testPathAppend( "foo/bar",  "", "foo", "/bar");
        testPathAppend( "foo/bar",  "", "foo", "bar");
        testPathAppend( "foo/b/ar", "", "foo", "b/ar");
        testPathAppend( "",         "", "", "");
        testPathAppend( "//netDrive/file.dat", "", "//netDrive", "file.dat");
        testPathAppend( "a",        "", "", "a");
        testPathAppend( "/a/b",     "", "/a", "b");
        testPathAppend( "/a/b",     "", "/a", "b/");
        testPathAppend( "/a/b",     "", "/a", "/b/");
        testPathAppend( "/root/a",  "/root", "", "a");
        testPathAppend( "/root/x/y","/root", "/x", "/y");
        testPathAppend( "root/x/y", "root",  "x", "y");
        testPathAppend( "root/x/y", "root",  "x/", "/y");
        testPathAppend( "root/x/y", "root",  "x/", "/y/");
        testPathAppend( "root/y",   "root",  "/", "//y//");
    }

    {
        hkStringBuf p;

        p = "a/b\\c/d\\\\e/f/g";
        p.pathRelativeTo("a\\b/c");
        HK_TEST(hkString::strCmp(p.cString(), "d/e/f/g") == 0);

        p = "a/b\\c//d\\\\e/f/g";
        p.pathRelativeTo("a\\b/c");
        HK_TEST(hkString::strCmp(p.cString(), "d/e/f/g") == 0);

        p = "a/b\\c/d\\\\e/f/g";
        p.pathRelativeTo("a\\b/c/");
        HK_TEST(hkString::strCmp(p.cString(), "d/e/f/g") == 0);

        p = "a/b\\c//d\\\\e/f/g";
        p.pathRelativeTo("a\\b/c/");
        HK_TEST(hkString::strCmp(p.cString(), "d/e/f/g") == 0);

        p = "a/b\\c//d\\\\e/f/g/";
        p.pathRelativeTo("a\\b/c/");
        HK_TEST(hkString::strCmp(p.cString(), "d/e/f/g") == 0);

        p = "a/b\\c//d\\\\e/f/g";
        p.pathRelativeTo("a\\b/c\\/d/\\e\\f/g");
        HK_TEST(hkString::strCmp(p.cString(), "") == 0);

        p = "a/b\\c//d\\\\e/f/g/";
        p.pathRelativeTo("a\\b/c\\/d//e\\f/g\\h/i");
        HK_TEST(hkString::strCmp(p.cString(), "../..") == 0);

        p = "a/b\\c//d\\\\e/f/g/";
        p.pathRelativeTo("a\\b/c\\/d//e\\f/g\\h/i\\");
        HK_TEST(hkString::strCmp(p.cString(), "../..") == 0);

        p = "a/b\\c//d\\\\e/f/g/";
        p.pathRelativeTo("a\\b/c\\/d//e\\f/g\\h/i/");
        HK_TEST(hkString::strCmp(p.cString(), "../..") == 0);

        p = "a/b\\c//d\\\\e/f/g/j/k";
        p.pathRelativeTo("a\\b/c\\/d//e\\f/g\\h/i");
        HK_TEST(hkString::strCmp(p.cString(), "../../j/k") == 0);

        p = "a/b\\c//d\\\\e/f/ge";
        p.pathRelativeTo("a\\b/c//d/\\e\\f/g\\h/i");
        HK_TEST(hkString::strCmp(p.cString(), "../../../ge") == 0);

        p = "a/b\\c//d\\\\e/f/g.txt";
        p.pathRelativeTo("a\\b/c//d//e\\f/g\\h/i");
        HK_TEST(hkString::strCmp(p.cString(), "../../../g.txt") == 0);

        p = "a/b\\c//d\\\\e/f/g";
        p.pathRelativeTo("a\\b/c//d//e\\f/g\\h/i");
        HK_TEST(hkString::strCmp(p.cString(), "../..") == 0);

        p = "D:/Havok\\Vision/Creatures_Common\\Models/Warrior/WarriorEvents.MODEL";
        p.pathRelativeTo("D:/Havok/Vision/Creatures_Common/");
        HK_TEST(hkString::strCmp(p.cString(), "Models/Warrior/WarriorEvents.MODEL") == 0);

        p = "e:/foo/a/b";
        HK_TEST(p.pathRelativeTo("a/b").isFailure());

        p = "e:/foo/a/b";
        HK_TEST(p.pathRelativeTo("a:/b").isFailure());

        p = "a:/b";
        HK_TEST(p.pathRelativeTo("a/b").isFailure());
        HK_TEST(hkString::strCmp(p.cString(), "a:/b") == 0);

        p = "a:/b/g.txt";
        HK_TEST(p.pathRelativeTo("a:/b/").isSuccess());
        HK_TEST(hkString::strCmp(p.cString(), "g.txt") == 0);

        p = "a:/b/";
        HK_TEST(p.pathRelativeTo("a:/b").isSuccess());
        HK_TEST(hkString::strCmp(p.cString(), "") == 0);

        p = "D:/Havok\\Vision/Creatures_Common\\Models/Warrior/WarriorEvents.MODEL";
        HK_TEST(p.pathRelativeTo("C:/Havok/Vision/Creatures_Common/").isFailure());
    }

    {
        const char* fooBarBaz[] = {"foo", "bar", "baz", HK_NULL };
        testSplitEqual("foo/bar/baz", fooBarBaz);

        const char* empty[] = {"", HK_NULL };
        testSplitEqual("", empty);

        const char* empty2[] = {"", "", HK_NULL };
        testSplitEqual("/", empty2);

        const char* slashFoo[] = {"", "foo", HK_NULL };
        testSplitEqual("/foo", slashFoo);

        const char* fooSlash[] = {"foo", "", HK_NULL };
        testSplitEqual("foo/", fooSlash);

        const char* empty3[] = {"", "", "", HK_NULL };
        testSplitEqual("//", empty3);
    }

    {
        // checking hkStringPtr::setPointerAligned()

        HK_ALIGN(const char testStr[],2) = "Hello Test!";
        hkStringPtr str("Old Content");
        str.setPointerAligned(testStr);
        HK_TEST(str.cString() == testStr);

        str.setPointerAligned(HK_NULL);
        HK_TEST(str.cString() == HK_NULL);

        hkStringPtr str2("Hello");
        str.setPointerAligned(str2.cString());
        HK_TEST(str.cString() == str2.cString());

        HK_DETAIL_DIAG_MSVC_PUSH()
        HK_DETAIL_DIAG_MSVC_OFF(4611) // interaction between '_setjmp' and C++ object destruction is non-portable
        HK_TEST_ASSERT(0x3d02cfc8, str.setPointerAligned(testStr+1));
        HK_DETAIL_DIAG_MSVC_POP()
    }

    {
        // checking hkString::tokenize()
        const char* str = "   abc|\td$  ef\n||ghi\t\t ";

        hkArray<hkStringView> tokens;

        hkString::tokenize(str, "|", " \n\t", tokens);
        HK_TEST(tokens.getSize() == 4);
        HK_TEST(tokens[0] == "abc");
        HK_TEST(tokens[1] == "d$  ef");
        HK_TEST(tokens[2] == "");
        HK_TEST(tokens[3] == "ghi");

        hkString::tokenize(str, "$", " \n\t", tokens);
        HK_TEST(tokens.getSize() == 2);
        HK_TEST(tokens[0] == "abc|\td");
        HK_TEST(tokens[1] == "ef\n||ghi");
    }

    string_memset16_test();
    string_int24w();
    hkarray_stringbuf_not_allowed();
    string_encoding_equal();
    string_printf();
    string_view_test();
    result_to_string();
    return 0;
}

HK_TEST_REGISTER(string_main, "Fast", "Common/Test/UnitTest/Base/", __FILE__);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
