// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/Serialize/Detail/hkReadBuffer.h>
#include <Common/Base/Serialize/Detail/hkWriteBuffer.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>
#include <Common/Base/System/Io/Reader/Buffered/hkBufferedStreamReader.h>
#include <Common/Base/System/Io/Reader/Memory/hkMemoryStreamReader.h>
#include <Common/Base/System/Io/IStream/hkIStream.h>

namespace HK_UNITY_ANONYMOUS_NAMESPACE
{
    struct BlobStreamReader : public hkStreamReader
    {
        hkArray<const char*> m_bits;
        bool m_isOk;

        BlobStreamReader(_In_opt_z_ const char* b0 = HK_NULL, _In_opt_z_ const char* b1 = HK_NULL, _In_opt_z_ const char* b2 = HK_NULL )
            : m_isOk( true )
        {
            if (b2) { m_bits.pushBack( b2 ); }
            if (b1) { m_bits.pushBack( b1 ); }
            if (b0) { m_bits.pushBack( b0 ); }
        }

        virtual hkBool isOk() const { return m_isOk; }

        virtual int read( void* buf, int nb )
        {
            if (nb == 0 || m_isOk == false)
            {
                return 0;
            }
            if (m_bits.getSize() == 0)
            {
                m_isOk = false;
                return false;
            }
            int l = hkString::strLen( m_bits.back() );
            int n = hkMath::min2( l, nb );
            hkMemUtil::memCpy( buf, m_bits.back(), n );
            if (l == n)
            {
                m_bits.popBack(); // read the whole thing
            }
            else
            {
                m_bits.back() += n; // only read some
            }
            return n;
        }
    };


    struct BufferedStreamTester : public hkIo::ReadBuffer
    {
        BufferedStreamTester( const char* a, const char* b ) : m_bsr( a, b )
        {
            attach( &m_bsr );
        }
        ~BufferedStreamTester()
        {
            detach();
        }
        BlobStreamReader m_bsr;
    };

    /// Read round and round in a circle!  There's no write buffer synchronized with this
    class RingBufferTestImpl : public hkIo::Detail::ReadBufferImpl
    {
        hkArray<int> m_data;
        int m_startOffset;
        int m_availableChunkAtBufBegin;

    public:

        bool m_readToSecondSegment;
        bool m_rewindHitImpl;

        RingBufferTestImpl()
            :  m_startOffset(1024 * 2), m_availableChunkAtBufBegin(1024), m_readToSecondSegment(false), m_rewindHitImpl(false)
        {
            /// Segment at beginning, segment at end, voided area in middle
            m_data.setSize(1024 * 3);
            for (int i = 0; i < 1024 * 3; ++i)
            {
                m_data[i] = i;
            }
        }

        virtual void _attachTo(hkIo::Detail::ReadBufferState& rs) HK_OVERRIDE
        {
            /// Start somewhere in the middle
            rs.m_offsetAtStart = 0;
            rs.m_start = rs.m_cur = &m_data[m_startOffset];
            rs.m_end = m_data.end();
        }

        virtual void _detachFrom(hkIo::Detail::ReadBufferState& rs) HK_OVERRIDE
        {
            rs.clear();
        }

        /// Get next segment
        virtual hkLong _prefetch(hkIo::Detail::ReadBufferState& rs, hkLong n) HK_OVERRIDE
        {
            /// We started 2048 bytes into the buffer
            rs.m_offsetAtStart = m_startOffset;

            rs.m_start = rs.m_cur = m_data.begin();

            /// Open up a 1024 chunk at the beginning of the buffer
            rs.m_end = &m_data[m_availableChunkAtBufBegin];

            return hkGetByteOffset(rs.m_end, rs.m_start);
        }

        virtual hkLong _prefetchAll(hkIo::Detail::ReadBufferState& rs) HK_OVERRIDE
        {
            HK_ASSERT(0x73f56740, 0, "Cannot prefetch all on a ring buffer");
            return 0;
        }

        virtual hkLong _skip(hkIo::Detail::ReadBufferState& rs, hkLong n) HK_OVERRIDE
        {
            return 0;
        }


        virtual hkLong _read(hkIo::Detail::ReadBufferState& rs, _Out_writes_bytes_(_Inexpressible_()) void* buf, hkLong n) HK_OVERRIDE
        {
            HK_ASSERT_EXPR(0x17caa618, rs.remaining(), < , n, "hRPCReadBufferImpl::_read "
                "shouldn't be called when there is still more data remaining "
                "in the ReadBufferState's buffer than we're trying to read.");

            /// If we're here, we can assume that
            const hkLong amountInCurrentBuf = rs.remaining();

            /// Copy the beginning of the data
            hkString::memCpy(buf, rs.m_cur, static_cast<int>(amountInCurrentBuf));

            /// Fetch the next segment for the read
            _prefetch(rs, amountInCurrentBuf + 1);

            /// We overshot our buffer
            if (rs.remaining() < (n - amountInCurrentBuf))
            {
                HK_ASSERT(0x33ba3a2a, 0, "Insufficient space in ring buffer to read data request of length {}", n);
                return amountInCurrentBuf;
            }

            m_readToSecondSegment = true;

            const hkLong amountToCopy = hkMath::min2(rs.remaining(), static_cast<int>(n - amountInCurrentBuf));

            /// Copy the remaining bytes
            hkString::memCpy(hkAddByteOffset(buf, amountInCurrentBuf), rs.m_cur, static_cast<int>(amountToCopy));

            rs.m_cur = hkAddByteOffset(rs.m_cur, amountToCopy);

            return n;
        }

        virtual hkResult _setReadLimit(hkIo::Detail::ReadBufferState& rs, hkLong n) HK_OVERRIDE
        {
            return HK_FAILURE;
        }

        virtual hkLong _getReadLimit(const hkIo::Detail::ReadBufferState& rs) const HK_OVERRIDE
        {
            return 0;
        }

        virtual hkResult _rewind(hkIo::Detail::ReadBufferState& rs, hkLong offset) HK_OVERRIDE
        {
            const void* ptr = hkAddByteOffset(rs.m_rewind, offset);
            const void* startPtr = &m_data[m_startOffset];
            const void* endPtr = m_data.end();
            m_rewindHitImpl = true;

            if (startPtr <= ptr && ptr <= endPtr)
            {
                rs.m_cur = ptr;
                rs.m_start = startPtr;
                rs.m_end = endPtr;
                rs.m_offsetAtStart = 0;
                return HK_SUCCESS;
            }

            return HK_FAILURE;
        }
    };

    static void buffer_read_test()
    {
        {
            BufferedStreamTester rb("hello", "world");
            rb.prefetchAll();
        }

        char buf[1024] = {};
        {
            BufferedStreamTester rb( "hello", "world" );
            HK_TEST( rb.read( buf, 100 ) == 10 );
        }

        for (int i = 0; i < 10; ++i)
        {
            BufferedStreamTester rb( "hello", "world" );
            HK_TEST2( rb.prefetch( i ) >= i, i );
            HK_TEST2( rb.read( buf, i ) == i, i );
            HK_TEST2( rb.prefetch( 100 ) >= 10 - i, i );
            HK_TEST2( rb.read( buf, 100 ) == 10 - i, i );
        }
        {
            BufferedStreamTester rb( "hello", "world" );
            HK_TEST( rb.prefetch( 4 ) >= 4 );
            HK_TEST( rb.read( buf, 4 ) == 4 );
            HK_TEST( rb.prefetch( 100 ) >= 6 );
            HK_TEST( rb.read( buf, 100 ) == 6 );
        }

        for (int i = 0; i < 100; ++i)
        {
            BufferedStreamTester rb( "hello", "world" );
            HK_TEST2( rb.skip( i ) == hkMath::min2( i, 10 ), i );
        }
        {
            BufferedStreamTester rb( "hello", "world" );
            HK_TEST( rb.skip( 100 ) == 10 );
        }
        {
            BufferedStreamTester rb( "hello", "world" );
            for (int i = 0; i < 5; ++i)
            {
                HK_TEST( rb.read( buf, 2 ) == 2 );
            }
        }
        {
            BufferedStreamTester rb( "hello", "world" );
            for (int i = 0; i < 5; ++i)
            {
                HK_TEST( rb.skip( 2 ) == 2 );
            }
        }
        {
            BufferedStreamTester rb( "hello", "world" );
            HK_TEST( rb.prefetch( 4 ) >= 4 );
            HK_TEST( rb.read( buf, 2 ) == 2 );
            HK_TEST( buf[0] == 'h' );
            HK_TEST( buf[1] == 'e' );
            HK_TEST( rb.prefetch( 15 ) >= 8 );
            HK_TEST( rb.read( buf, 2 ) == 2 );
            HK_TEST( buf[0] == 'l' );
            HK_TEST( buf[1] == 'l' );
            HK_TEST( rb.read( buf, 2 ) == 2 );
            HK_TEST( buf[0] == 'o' );
            HK_TEST( buf[1] == 'w' );
            HK_TEST( rb.prefetch( 10 ) >= 4 );
            HK_TEST( rb.skip( 100 ) == 4 );
        }
    }

    static void archive_test()
    {
        //
        // tests the hkIArchive and hkOArchive classes for binary input and output
        // In this case, file streams are created
        // first, some test data of each supported type is written to file.
        // Then the file is read back in and compared to the original.
        //
        {
            // output
            {
                /*
                hkOfArchive oa("testbindata.txt");

                // test data
                char c = 'a';
                unsigned char uc = 25;
                short s = -12345;
                unsigned short us = 12345;
                int i = -12345;
                unsigned int ui = 12345;
                long int li = -12345;
                unsigned long int uli = 12345;
                float f = -1.2345f;
                double d = -1.2345;

                // write
                oa.write8(c);

                oa.write8u(uc);

                oa.write16(s);

                oa.write16u(us);

                oa.write32(i);

                oa.write32u(ui);

                oa.write64(li);

                oa.write64u(uli);

                oa.writeFloat32(f);

                oa.writeDouble64(d);
                */
            }

            // input
            {
                /*
                hkIfArchive ia("testbindata.txt");

                char c = ia.read8();
                HK_TEST(c == 'a');

                unsigned char uc = ia.read8u();
                HK_TEST(uc == 25);

                short s = ia.read16();
                HK_TEST(s == -12345);

                unsigned short us = ia.read16u();
                HK_TEST(us == 12345);

                int i = ia.read32();
                HK_TEST(i == -12345);

                unsigned int ui = ia.read32u();
                HK_TEST(ui == 12345);

                hkInt64 li = ia.read64();
                HK_TEST(li == -12345);

                hkUint64 uli = ia.read64u();
                HK_TEST(uli == 12345);

                float f = ia.readFloat32();
                HK_TEST(f == -1.2345f);

                double d = ia.readDouble64();
                HK_TEST(d == -1.2345);
                */
            }
        }
    }

    static void buffer_view_test()
    {
        struct Data
        {
            int len;
            int cmd;
            char msg[12];
        };
        Data d_correct = { 14, 6, {'h','e','l','l','o',0} };
        Data d_truncated0 = { 7, 1, {} };
        Data d_truncated1 = { 9, 2, {'h'} };

        {
            hkIo::ReadBufferView view( &d_truncated0, d_truncated0.len );
            int len = view.readPod<int>();
            HK_TEST( view.isOk() );
            int cmd = view.readPod<int>();
            HK_TEST( view.isOk() == false );
            const char* msg = view.access<char>( cmd );
            HK_TEST( view.isOk() == false );
            msg = HK_NULL;
            len = 0;
        }
        {
            hkIo::ReadBufferView view( &d_truncated1, d_truncated1.len );
            view.readPod<int>();
            int cmd = view.readPod<int>();
            HK_TEST_ASSERT(0x0990e92b, view.access<char>( cmd ));
            cmd = 0;
        }
        {
            hkIo::ReadBufferView view(&d_correct, d_correct.len);
            view.skip(d_correct.len + 1234);
            HK_TEST(view.tell() == d_correct.len);
        }
        {
            hkIo::ReadBufferView view( &d_correct, d_correct.len );
            int len = view.readPod<int>();
            int cmd = view.readPod<int>();
            HK_TEST( view.isOk() );
            const char* msg = view.access<char>( cmd );
            HK_TEST( view.isOk() );
            HK_TEST( view.remaining() == 0 );
            msg = HK_NULL;
            len = 0;
        }
        {
            hkIo::ReadBufferView view( &d_correct, d_correct.len );
            HK_TEST_ASSERT(0x0990e92b, view.peekAt<int>( 100 ));
        }
        {
            hkUint32Be ube = 1234;
            hkIo::ReadBufferView view( &ube, sizeof( ube ) );
            hkUint32 r = view.readPod<hkUint32Be>();
            HK_TEST( r == 1234 );
            HK_TEST( view.remaining() == 0 && view.isOk() );
        }
    }


    static void inplace_test()
    {
        static const char hello[] = "hello world";
        hkIo::ReadBuffer rb( hello, sizeof( hello ) );

        if (HK_TEST( rb.prefetch( 5 ) >= 5 ))
        {
            HK_TEST( hkString::strNcmp( "hello", rb.peekAt<const char>( 0, 5 ), 5 ) == 0 );
        }
        if (HK_TEST( rb.prefetch( 25 ) >= sizeof( hello ) )) // past the end
        {
            HK_TEST( hkString::strNcmp( "hello", rb.peekAt<const char>( 0, 5 ), 5 ) == 0 );
        }

        // go to a nonzero offset
        const int SKIP = 6;
        rb.skip( SKIP );
        if (HK_TEST( rb.prefetch( 5 ) >= 5 ))
        {
            HK_TEST( hkString::strNcmp( "world", rb.peekAt<const char>( 0, 5 ), 5 ) == 0 );
        }
        if (HK_TEST( rb.prefetch( 25 ) >= sizeof( hello ) - SKIP )) // past the end
        {
            HK_TEST( hkString::strNcmp( &hello[0] + SKIP, rb.peekAt<const char>( 0, 5 ), 5 ) == 0 );
        }
    }

    static void rewind_test()
    {
        static const char hello[] = "hello world, I am a bit longer than other strings, but I am still nice";
        BufferedStreamTester rb(hello, hello + sizeof(hello));

        // go to a nonzero offset
        rb.skip(6);
        if (HK_TEST(rb.prefetch(5) >= 5))
        {
            HK_TEST(hkString::strNcmp("world", rb.peekAt<const char>(0, 5), 5) == 0);
        }
        rb.setRewindPosition();
        rb.setReadLimit(sizeof(hello));
        HK_TEST(rb.skip(60) == 60);
        if (HK_TEST(rb.prefetch(4) >= 4))
        {
            HK_TEST(hkString::strNcmp("nice", rb.peekAt<const char>(0, 4), 4) == 0);
        }
        rb.rewindTo(0);
        HK_TEST(hkString::strNcmp("world", rb.peekAt<const char>(0, 5), 5) == 0);
    }

    static void rewind_ringbuffer_test()
    {
        int buf[2048];

        RingBufferTestImpl* testImpl = new RingBufferTestImpl;
        hkIo::ReadBuffer rdBuf(testImpl);

        rdBuf.read(buf, 345 * sizeof(int));
        rdBuf.setRewindPosition();
        rdBuf.read(buf, sizeof(int));
        const int testValue = buf[0];

        /// Read into the next segment
        rdBuf.read(buf, (1096 - 345) * sizeof(int));
        HK_TEST(buf[0] != testValue);
        HK_TEST(testImpl->m_readToSecondSegment);

        HK_TEST(rdBuf.rewindTo(0).isSuccess());
        HK_TEST(testImpl->m_rewindHitImpl);
        rdBuf.clearRewindPosition();
        rdBuf.read(buf, sizeof(int));

        HK_TEST(buf[0] == testValue);
        testImpl = NULL;
    }
}

static void nonexisting_file_test()
{
    hkIo::ReadBuffer readBuffer("NonExistingFile.txt");

    char buff[32];
    hkLong numRead = readBuffer.read(buff, 32);
    HK_TEST(numRead == 0);
}

static int readbuffer_main()
{
    HK_UNITY_USING_ANONYMOUS_NAMESPACE;

    buffer_view_test();
    buffer_read_test();
    archive_test();
    inplace_test();
    rewind_test();
    nonexisting_file_test();
    rewind_ringbuffer_test();

    return 0;
}

HK_TEST_REGISTER(readbuffer_main, "Fast", "Common/Test/UnitTest/Base/", __FILE__);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
