// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>

#include <Common/Base/UnitTest/Serialize/urlObjects.h>

#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>
#include <Common/Base/Serialize/Resource/hkResourceLinker.h>
#include <Common/Serialize/Util/hkSerializeUtil.h>
#include <Common/Base/Container/Set/hkSet.h>
#include <Common/Base/Serialize/Detail/hkWriteBuffer.h>
#include <Common/Base/Serialize/hkSerialize.h>
#include <Common/Base/Fwd/hkcstdio.h> // scanf
#include <Common/Base/Types/hkEndian.h>
#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/System/Io/Reader/Memory/hkMemoryStreamReader.h>

#ifndef HK_NONSDK_BUILD
#include <Common/Base/Serialize/Format/Xml/hkXmlWriteFormat.h>
#include <Common/Base/Serialize/Format/Xml/hkXmlReadFormat.h>
#endif

#include <Common/Base/Serialize/Format/Tagfile/hkTagfileReadFormat.h>
#include <Common/Base/Serialize/Format/Tagfile/hkTagfileWriteFormat.h>
#include <Common/Base/Config/hkOptionalComponent.h>

namespace
{
    template<typename T>
    const T* bundleContentsIs( hkViewPtr<hkSerialize::Bundle> bundle )
    {
        // We're accessing the vars in the bundle *before* copying to native
        // Thus var.dynCast<> and friends won't work.
        // This is a hack assuming that the type of T matches the native type
        hkReflect::QualType want = hkReflect::getType<T>();
        hkReflect::Var have = bundle->getContents();
        if( hkString::strCmp( want->getName(), have.getType()->getName()) == 0 )
        {
            return static_cast<const T*>( have.getAddress() );
        }
        return HK_NULL;
    }

    template<typename Format>
    void test_notes()
    {
        using namespace UnitTest::UrlObject;
        hkArray<char> wbuf;
        {
            hkInt32Be i32be = 0x01020304;
            hkStringPtr name = "myname";
            hkUint32 color = 0xff112233;

            typename Format::Writer wformat;
            hkIo::WriteBuffer writeBuffer(&wbuf);
            hkSerialize::BundleBuilder bb(&wformat, &writeBuffer);
            bb.add(&i32be);
            bb.addNote(&i32be, hkReflect::Var(&name));
            bb.addNote(&i32be, hkReflect::Var(&color));
        }
        HK_TEST(wbuf.getSize());
        {
            hkRefPtr<hkObjectResource> res = hkSerialize::Load().withFormat<typename Format::Reader>().toResource(wbuf);
            if(HK_TEST(res))
            {
            }
        }
    }


    void HK_CALL addLotsOfNotes(const hkReflect::Var& var, hkSerialize::BundleBuilder& bb, void*)
    {
        hkStringPtr name = "myname";
        hkUint32 color = 0xff112233;
        int index = -2;
        float mag = 4.1f;
        bb.addNote(var, hkReflect::Var(&name));
        bb.addNote(var, hkReflect::Var(&color));
        bb.addNote(var, hkReflect::Var(&index));
        bb.addNote(var, hkReflect::Var(&mag));
    }


    template<typename Format>
    void test_notes2()
    {
        using namespace UnitTest::UrlObject;
        hkArray<char> wbuf;
        {
            hkInt32Be i32be = 0x01020304;

            typename Format::Writer wformat;
            hkIo::WriteBuffer writeBuffer(&wbuf);
            hkSerialize::BundleBuilder bb(&wformat, &writeBuffer);
            bb.setCallbacks(HK_NULL, addLotsOfNotes, HK_NULL);
            bb.add(&i32be);
            bb.recursiveAdd();
        }
        HK_TEST(wbuf.getSize());
        {
            hkRefPtr<hkObjectResource> res = hkSerialize::Load().withFormat<typename Format::Reader>().toResource(wbuf);
            if(HK_TEST(res))
            {
            }
        }
    }

    
    template<typename Format, typename IntegerType>
    void test_endian()
    {
        using namespace UnitTest::UrlObject;
        hkArray<char> wbuf;
        {
            typename Format::Writer wformat;
            wformat.enableMultiBundle();
            {
                IntegerType i32be = 0x01020304;
                hkIo::WriteBuffer writeBuffer(&wbuf);
                hkSerialize::BundleBuilder bb(&wformat, &writeBuffer);
                bb.add(&i32be);
            }
            {
                IntegerType i32be = 0x05060708;

                hkIo::WriteBuffer writeBuffer(&wbuf);
                hkSerialize::BundleBuilder bb(&wformat, &writeBuffer);
                bb.add(&i32be);
            }
        }
        HK_TEST(wbuf.getSize());

        // Read two bundles back
        if(1)
        {
            hkIo::ReadBuffer msw(wbuf.begin(), wbuf.getSize());
            typename Format::Reader rformat;
            hkViewPtr<hkSerialize::Bundle> res = rformat.read(msw);
            if (HK_TEST(res))
            {
                const IntegerType* be = bundleContentsIs<IntegerType>(res);
                if (HK_TEST(be))
                {
                    HK_TEST(int(*be) == 0x01020304);
                }
            }
            res = rformat.read(msw);
            if( HK_TEST(res))
            {
                const IntegerType* be = bundleContentsIs<IntegerType>(res);
                if( HK_TEST( be ) )
                {
                    HK_TEST( int(*be) == 0x05060708 );
                }
            }
        }

        if(0)
        {
            hkRefPtr<hkObjectResource> res0 = hkSerialize::Load().toResource(wbuf);
            if(HK_TEST(res0))
            {
                const IntegerType* be = res0->getContents<IntegerType>();
                HK_TEST( int(*be) == 0x01020304 );
            }

            // We can't load the second part because it shares types from res0 which we've discarded.
            // hkObjectResource* res1 = hkSerialize::Load();
        }
    }

    template<typename Format>
    void test_noSubstitution()
    {
        using namespace UnitTest::UrlObject;
        hkArray<char> buf;
        {
            // Make sure these die when we go out of scope
            Shape* shape;
            Body* rb;
            {
                shape = new Shape(0.01f);
                rb = new Body(shape);
                shape->removeReference();
            }

            typename Format::Writer wformat;
            hkIo::WriteBuffer writeBuffer(&buf);
            hkSerialize::BundleBuilder bb(&wformat, &writeBuffer);
            bb.add(rb);
            while( bb.nextUnresolved() )
            {
                hkReflect::Var obj = bb.currentUnresolved();
                bb.add(obj);
                bb.addExport(obj, obj.getType()->getName());
            }

            delete rb;
        }

        // A normal read
        {
            hkRefPtr<hkObjectResource> container = hkSerialize::Load()
                .withFormat<typename Format::Reader>()
                .toResource(buf);
            if (HK_TEST(container))
            {
                const Body* rb = container->getContents<Body>();
                if (HK_TEST(rb) && HK_TEST(rb->m_shape))
                {
                    HK_TEST(hkMath::equal(rb->m_shape->m_size, 0.01f, 0.0001f));
                }
            }

        }
    }

    template<typename Format>
    void test_withSubstitution()
    {
        using namespace UnitTest::UrlObject;

        hkArray<char> outputFile;
        {
            // Make sure these die when we go out of scope
            Shape* shape;
            Body* rb;
            {
                shape = new Shape(0.01f);
                rb = new Body(shape);
                shape->removeReference();
            }

            typename Format::Writer wformat;
            hkIo::WriteBuffer writeBuffer(&outputFile);
            hkSerialize::BundleBuilder bb(&wformat, &writeBuffer);
            bb.add(rb);
            bb.addExport(rb, "ROOT");


            while(bb.nextUnresolved())
            {
                hkReflect::Var obj = bb.currentUnresolved();

                if( obj.dynCast<Shape>() )
                {
                    bb.addImport(obj, "URL00000000");
                }
                else
                {
                    bb.add(obj);
                }
            }
            delete rb;
        }


        // Now read this back with a url substitution
        {
            hkResourceLinker linker;
            linker.add(hkSerialize::Load().withFormat<typename Format::Reader>().toResource(outputFile));

            hkReflect::Var shape(new Shape(0.15f));
            linker.addExport(shape, "URL00000000", true);

            const Body* rb = linker.getContainer(0)->getContents<Body>();

            HK_TEST(rb->m_shape);
            if(rb->m_shape)
            {
                HK_TEST(hkMath::equal(rb->m_shape->m_size, 0.15f, 0.0001f));
            }
        }
    }

    template<typename Format>
    void test_withEmpty()
    {
        using namespace UnitTest::UrlObject;

        hkArray<char> outputFile;
        {
            // Make sure these die when we go out of scope
            Shape* shape;
            Body* rb;
            {
                shape = new Shape(0.01f);
                rb = new Body(shape);
                shape->removeReference();
            }

            typename Format::Writer wformat;
            hkIo::WriteBuffer writeBuffer(&outputFile);
            hkSerialize::BundleBuilder bb(&wformat, &writeBuffer);
            bb.add(rb);
            bb.addExport(rb, "ROOT");


            while (bb.nextUnresolved())
            {
                hkReflect::Var obj = bb.currentUnresolved();

                if (obj.dynCast<Shape>())
                {
                    bb.addEmpty(obj);
                }
                else
                {
                    bb.add(obj);
                }
            }
            delete rb;
        }


        // Now read this back
        {
            hkRefPtr<hkResource> res = hkSerialize::Load().withFormat<typename Format::Reader>().toResource(outputFile);
            auto rb = res->getContents<Body>();
            if (HK_TEST(rb))
            {
                HK_TEST_EQ(rb->m_shape, (void*)HK_NULL);
            }
        }
    }

    template<typename Format>
    void test_objectStream()
    {
        using namespace UnitTest::UrlObject;

        // Make a more complicated object graph to test
        hkArray<char> outputFile;
        {
            TestGraph testGraph0(0); // [1, 2, 3] // these two links to [1] will be unlinked
            TestGraph testGraph1(1); // []
            TestGraph testGraph2(2); // []
            TestGraph testGraph3(3); // [4, 2]
            TestGraph testGraph4(4); // []

            testGraph0.m_childA = &testGraph1;
            testGraph0.m_childB = &testGraph2;
            testGraph0.m_childC = &testGraph3;

            //testGraph2.m_childA = &testGraph2;

            testGraph3.m_childA = &testGraph4;
            testGraph3.m_childB = &testGraph2;

            //testGraph4.m_childA = &testGraph4;

            typename Format::Writer wformat;
            hkIo::WriteBuffer writeBuffer(&outputFile);
            hkSerialize::BundleBuilder bb(&wformat, &writeBuffer);
            bb.add(&testGraph0);
            while(bb.nextUnresolved())
            {
                hkReflect::Var obj = bb.currentUnresolved();

                bool replaced = false;
                if(const TestGraph* tg = obj.dynCast<TestGraph>())
                {
                    // Swap out these objects, replace with URLs
                    if(tg->m_childA == HK_NULL)
                    {
                        bb.addImport(obj, "LEAF");
                        replaced = true;
                    }
                }

                if(!replaced)
                {
                    bb.add(obj);
                }
            }
        }

        {
            hkRefPtr<hkResource> container = hkSerialize::Load()
                .withFormat<typename Format::Reader>().toResource(outputFile);

            hkArray<hkResource::Import> containerImports;
            hkArray<hkResource::Export> containerExports;
            container->getImportsExports(containerImports, containerExports);

            // No exports
            HK_TEST(containerExports.getSize() == 0);
            HK_TEST(containerImports.getSize() == 4); // 4 pointers to imported objects

            // In this test, if we have two pointers to the same URL object they will
            // show up as two separate imports (with the same URL)

            hkRefPtr<const TestGraph> contents = container->getContents<TestGraph>(); // no steal, just view

            for(int importIndex = 0; importIndex < containerImports.getSize(); importIndex++)
            {
                hkStringPtr const& thisImportName = containerImports[importIndex].m_name;

                // Link in the new objects
                if(!thisImportName.compareTo("LEAF"))
                {
                    hkRefPtr<TestGraph> newGraph(hkRefNew<TestGraph>(new TestGraph(-1)));
                    hkReflect::Var newGraphVar(newGraph.val(), hkReflect::getType<TestGraph>());
                    containerImports[importIndex].resolve(newGraphVar);
                }
                else
                {
                    HK_TEST(0); // Unknown import
                }
            }

            //      TestGraph testGraph0; // [LEAF, LEAF, 3]
            //      TestGraph testGraph3; // [LEAF, LEAF]

            const TestGraph* testLeaf1 = contents->m_childA;
            const TestGraph* testLeaf2 = contents->m_childB;
            const TestGraph* test3 = contents->m_childC;
            const TestGraph* testLeaf3 = test3->m_childA;
            const TestGraph* testLeaf4 = test3->m_childB;

            HK_TEST(testLeaf1->m_childA == HK_NULL);
            HK_TEST(test3->m_childC == HK_NULL);
            HK_TEST(testLeaf2->m_childA == HK_NULL);
            HK_TEST(testLeaf3->m_childA == HK_NULL);
            HK_TEST(testLeaf4->m_childA == HK_NULL);

//          for(int i=0; i < ourAllocatedGraphNodes.getSize(); i++)
//          {
//              delete ourAllocatedGraphNodes[i];
//          }
        }

        // Leaking here as the container is only freeing object 0
    }

    // This usage is a bit more complicated since we don't control the
    // map anymore
    template<typename Format>
    void test_linkingStreams()
    {
        using namespace UnitTest::UrlObject;

        hkArray<char> mainOutputFile;
        hkArray< hkArray<char> > assetOutputFiles;

        {
            // Can't have loops in here, as they use ref counting...
            // Make a more complicated object graph to test
                                     //  Stream A    Stream B
            TestGraph testGraph0(0); // [1, 2, 3]
            TestGraph testGraph1(1); //              [4, 5]
            TestGraph testGraph2(2); // [6]
            TestGraph testGraph3(3); //              [7]
            TestGraph testGraph4(4); //              []
            TestGraph testGraph5(5); //              [8]
            TestGraph testGraph6(6); // []
            TestGraph testGraph7(7); //              []
            TestGraph testGraph8(8); //              []

            testGraph0.m_childA = &testGraph1;
            testGraph0.m_childB = &testGraph2;
            testGraph0.m_childC = &testGraph3;

            testGraph1.m_childA = &testGraph4;
            testGraph1.m_childB = &testGraph5;

            testGraph2.m_childA = &testGraph6;

            testGraph3.m_childA = &testGraph7;

            testGraph5.m_childA = &testGraph8;

            hkArray<const TestGraph*> toWriteTodos;
            hkSet<int> queuedIds;
            {
                {
                    typename Format::Writer wformat;
                    hkIo::WriteBuffer writeBuffer(&mainOutputFile);
                    hkSerialize::BundleBuilder mainBb(&wformat, &writeBuffer);
                    // write the root node and any even numbered ids which it references

                    mainBb.add(&testGraph0);
                    mainBb.addExport(&testGraph0, "ROOT");
                    while(mainBb.nextUnresolved())
                    {
                        hkReflect::Var obj = mainBb.currentUnresolved();
                        if(const TestGraph* tg = obj.dynCast<TestGraph>())
                        {
                            hkStringBuf objectName; objectName.appendPrintf("OBJ%d", tg->m_id);
                            if(tg->m_id & 1)
                            {
                                mainBb.addImport(obj, objectName);

                                toWriteTodos.pushBack(tg);
                                queuedIds.insert(tg->m_id);
                            }
                            else
                            {
                                queuedIds.insert(tg->m_id);
                                mainBb.add(obj);
                                mainBb.addExport(obj, objectName);
                            }
                        }
                    }
                }

                // Any unresolved objects at this point go into their own file

                for(int todoIndex=0; todoIndex < toWriteTodos.getSize(); todoIndex++)
                {
                    const TestGraph* tg = toWriteTodos[todoIndex];
                    typename Format::Writer wformat;
                    hkIo::WriteBuffer writeBuffer(&assetOutputFiles.expandOne());
                    hkSerialize::BundleBuilder assetBb(&wformat, &writeBuffer);
                    hkStringBuf objectName; objectName.appendPrintf("OBJ%d", tg->m_id);
                    assetBb.add(tg);
                    assetBb.addExport(tg, objectName);

                    while(assetBb.nextUnresolved())
                    {
                        hkReflect::Var obj = assetBb.currentUnresolved();
                        if(const TestGraph* tgn = obj.dynCast<TestGraph>())
                        {
                            if(!queuedIds.contains(tgn->m_id))
                            {
                                queuedIds.insert(tgn->m_id);
                                toWriteTodos.pushBack(tgn);
                            }

                            hkStringBuf assetTodoName; assetTodoName.appendPrintf("OBJ%d", tgn->m_id);
                            assetBb.addImport(obj, assetTodoName);
                        }
                    }
                }
            }
        }

        {
            hkArray<hkRefPtr<hkObjectResource> > objectResources;
            {
                objectResources.expandOne() = hkSerialize::Load()
                    .withFormat<typename Format::Reader>().toResource(mainOutputFile);
            }


            // Containers are now loaded with objects and URLs

            hkResourceLinker linkContainers;
            linkContainers.add(objectResources[0]);

            for(int assetIndex=0; assetIndex < assetOutputFiles.getSize(); assetIndex++)
            {
                // Only destructs top-level object
                objectResources.expandOne() = hkSerialize::Load()
                    .withFormat<typename Format::Reader>().toResource(assetOutputFiles[assetIndex].begin(), assetOutputFiles[assetIndex].getSize());
                linkContainers.add(objectResources.back());
            }

            {

//              TestGraph testGraph0; // [1, 2, 3]
//              TestGraph testGraph1; // [4, 5]
//              TestGraph testGraph2; // [6]
//              TestGraph testGraph3; // [7]
//              TestGraph testGraph4; // []
//              TestGraph testGraph5; // [8]
//              TestGraph testGraph6; // []
//              TestGraph testGraph7; // []
//              TestGraph testGraph8; // []

                const TestGraph* testGraph0 = objectResources[0]->getContents<TestGraph>(); // [1., 2., 3.]
                const TestGraph* testGraph1 = testGraph0->m_childA; // [4., 5.]
                const TestGraph* testGraph2 = testGraph0->m_childB; // [6.]
                const TestGraph* testGraph3 = testGraph0->m_childC; // [7.]
                const TestGraph* testGraph4 = testGraph1->m_childA; // []
                const TestGraph* testGraph5 = testGraph1->m_childB; // [8.]
                const TestGraph* testGraph6 = testGraph2->m_childA; // []
                const TestGraph* testGraph7 = testGraph3->m_childA; // []
                const TestGraph* testGraph8 = testGraph5->m_childA; // []

                HK_TEST(testGraph1->m_childC == HK_NULL);
                HK_TEST(testGraph2->m_childB == HK_NULL);
                HK_TEST(testGraph3->m_childB == HK_NULL);
                HK_TEST(testGraph4->m_childA == HK_NULL);
                HK_TEST(testGraph5->m_childB == HK_NULL);
                HK_TEST(testGraph6->m_childA == HK_NULL);
                HK_TEST(testGraph7->m_childA == HK_NULL);
                HK_TEST(testGraph8->m_childA == HK_NULL);
            }
        }
    }

    template<typename Format>
    void test_linkingWithNative()
    {
        using namespace UnitTest::UrlObject;

        hkArray<char> outputFile;
        {
            // Make sure these die when we go out of scope
            Shape shape1(1.0f);
            Body body1(&shape1);
            Shape shape2(2.0f);
            Body body2(&shape2);
            Shape shape3(3.0f);
            Body body3(&shape3);

            body1.m_child = &body2;
            body2.m_child = &body3;

            typename Format::Writer wformat;
            hkIo::WriteBuffer writeBuffer(&outputFile);
            hkSerialize::BundleBuilder bb(&wformat, &writeBuffer);
            bb.add(&body1);
            while(bb.nextUnresolved())
            {
                hkReflect::Var obj = bb.currentUnresolved();
                if(const Shape* shape = obj.dynCast<Shape>())
                {
                    hkStringBuf name; name.appendPrintf("SHAPE%f", shape->m_size);
                    bb.addImport(obj, name.cString());
                }
                else
                {
                    bb.add(obj);
                }
            }

            // Overwrite just in case we accidentally re-read them
            body1.m_child = HK_NULL; body1.m_shape = HK_NULL; shape1.m_size = -6666.0f;
            body2.m_child = HK_NULL; body2.m_shape = HK_NULL; shape2.m_size = -6666.0f;
            body3.m_child = HK_NULL; body3.m_shape = HK_NULL; shape3.m_size = -6666.0f;
        }


        // Now read this back with a url substitution
        {
            hkRefPtr<hkObjectResource> fileObjectsContainer = hkSerialize::Load()
                .withFormat<typename Format::Reader>().toResource(outputFile.begin(), outputFile.getSize());
            hkResourceLinker linker;
            linker.add(fileObjectsContainer);

            hkArray<hkResource::Import> objectsImports;
            hkArray<hkResource::Export> objectsExports;
            fileObjectsContainer->getImportsExports(objectsImports, objectsExports);

            for(int i=0; i<objectsImports.getSize(); i++)
            {
                hkResource::Import const& thisImport = objectsImports[i];

                {
                    float shapeSize = -1.0f;
                    // Skip "SHAPE"
                    if(sscanf(thisImport.m_name.cString() + 5, "%f", &shapeSize) == 1)
                    {
                        Shape* newShape = new Shape(shapeSize);
                        linker.addExport(hkReflect::Var(newShape, hkReflect::getType<Shape>()), thisImport.m_name);
                    }
                }
            }

            // Test and steal contents
            {
                const Body* body1 = fileObjectsContainer->stealContents<Body>();
                HK_TEST(body1 != HK_NULL);
                const Body* body2 = body1->m_child;
                HK_TEST(body2 != HK_NULL);
                const Body* body3 = body2->m_child;
                HK_TEST(body3 != HK_NULL);

                HK_TEST(body1->m_shape != HK_NULL);
                HK_TEST(body2->m_shape != HK_NULL);
                HK_TEST(body3->m_shape != HK_NULL);

                HK_TEST(hkMath::equal(body1->m_shape->m_size, 1.0f, 0.001f));
                HK_TEST(hkMath::equal(body2->m_shape->m_size, 2.0f, 0.001f));
                HK_TEST(hkMath::equal(body3->m_shape->m_size, 3.0f, 0.001f));

                delete body3;
                delete body2;
                delete body1;
            }
        }
    }

    template<typename Format>
    void test_hkVariantArray()
    {
        using namespace UnitTest::UrlObject;

        hkArray<char> fileBuffer;
        hkOstream fileStream(fileBuffer);

        {
            VariantTest testObject;

            testObject.m_array1.allocate(4, hkReflect::getType<hkReal>());
            testObject.m_array1.setElement<hkReal>(0, 0.1f);
            testObject.m_array1.setElement<hkReal>(1, -0.4f);
            testObject.m_array1.setElement<hkReal>(2, -2.44f);
            testObject.m_array1.setElement<hkReal>(3, 40.5f);

            testObject.m_a1 = 25;

            testObject.m_array2.allocate(2, hkReflect::getType<hkUint32>());
            testObject.m_array2.setElement<hkUint32>(0, 1000);
            testObject.m_array2.setElement<hkUint32>(1, 20000);

            testObject.m_a4 = -8888.88;

            hkSerialize::Save().withFormat<typename Format::Writer>().contentsPtr(&testObject, fileStream.getStreamWriter());
        }
        {
            if (VariantTest* loadedObj = hkSerialize::Load().withFormat<typename Format::Reader>().template toObject<VariantTest>(fileBuffer.begin(), fileBuffer.getSize()))
            {
//              printf("Loaded %p [%d/%d/%d/%f]\n", loadedObj, loadedObj->m_array1.m_size, loadedObj->m_a1, loadedObj->m_array2.m_size, loadedObj->m_a4);
//              printf("[%f %f %f %f]\n", loadedObj->m_array1.getElement<hkReal>(0), loadedObj->m_array1.getElement<hkReal>(1), loadedObj->m_array1.getElement<hkReal>(2), loadedObj->m_array1.getElement<hkReal>(3));
//              printf("[%d %d]\n", loadedObj->m_array2.getElement<hkUint32>(0), loadedObj->m_array2.getElement<hkUint32>(1));

                delete loadedObj;
            }
        }
    }

    struct PointerNoteCallbacks : public hkSerialize::NoteHandler
    {
        static hkReflect::Var HK_CALL pointer(const hkReflect::PointerVar& ptr, hkSerialize::BundleBuilder& bb, void* data)
        {
            using namespace UnitTest::UrlObject;
            PointerNoteCallbacks* thisPtr = static_cast<PointerNoteCallbacks*>(data);
            if (hkReflect::FieldDecl asField = hkReflect::QualType(ptr.getType()))
            {
                if (const TestGraph* pointed = ptr.getValue().dynCast<TestGraph>())
                {
                    Import* note = new Import;
                    note->m_id = pointed->m_id;

                    // Generate a strong import if the obj is pointed by "childA"
                    if (hkString::strCmp(asField.getName(), "childA") == 0)
                    {
                        note->m_strong = true;
                    }
                    else
                    {
                        note->m_strong = false;
                    }
                    thisPtr->m_imports.expandOne().reset(note);
                    bb.addNote(pointed, note);
                    bb.addEmpty(pointed);
                    return note;
                }
            }
            return hkReflect::Var();
        }

        virtual void addNotes(const hkArrayView<hkReflect::Var>& notes, hkReflect::Var& dst, hkReflect::Var& src) HK_OVERRIDE
        {
            using namespace UnitTest::UrlObject;
            // Should all be import notes
            for (int i = 0; i < notes.getSize(); ++i)
            {
                Import* cloned = hkDynCast<Import>(notes[i]);
                if (HK_TEST(cloned))
                {
                    m_imports.expandOne().reset(cloned);
                }
            }
        }

        virtual bool atPointerNote(const hkReflect::Var& note, hkReflect::PointerVar& dst, hkReflect::PointerVar& src) HK_OVERRIDE
        {
            using namespace UnitTest::UrlObject;
            Import* import = hkDynCast<Import>(note);
            if (HK_TEST(import))
            {
                int id = import->m_id;
                if (import->m_strong)
                {
                    // Strong import
                    if (m_loaded[id] == HK_NULL)
                    {
                        m_loaded[id] = hkSerialize::Load().withNoteHandler(this).toResource(m_saved[id]);
                    }
                    hkReflect::Detail::getPlain(dst).setValue(m_loaded[id]->getContents());
                }
                else
                {
                    // Weak import
                    m_fixups.pushBack(hkTupleT::make(dst, id));
                }
                return false;
            }
            return true;
        }

        hkArray<char> m_saved[4];
        hkRefPtr<hkResource> m_loaded[4];
        hkArray< hkScopedPtr<UnitTest::UrlObject::Import> > m_imports;
        hkArray< hkTuple<hkReflect::PointerVar, int> > m_fixups;
    };

    template<typename Format>
    void test_pointerNotes()
    {
        using namespace UnitTest::UrlObject;

        PointerNoteCallbacks callbacks;
        {
            hkArray<TestGraph> testGraphs;
            testGraphs.pushBack(0); // [1, 2, 3]
            testGraphs.pushBack(1); // [2, 3,]
            testGraphs.pushBack(2); // [, 3,]
            testGraphs.pushBack(3); // [,,]

            testGraphs[0].m_childA = &testGraphs[1];
            testGraphs[0].m_childB = &testGraphs[2];
            testGraphs[0].m_childC = &testGraphs[3];
            testGraphs[1].m_childA = &testGraphs[2];
            testGraphs[1].m_childB = &testGraphs[3];
            testGraphs[2].m_childB = &testGraphs[3];

            for (int i = 0; i < testGraphs.getSize(); ++i)
            {
                hkSerialize::Save().withFormat<typename Format::Writer>()
                    .withCallbacks(&callbacks, HK_NULL, &PointerNoteCallbacks::pointer).contentsPtr(&testGraphs[i], &callbacks.m_saved[i]);
            }
        }

        {
            hkRefPtr<TestGraph> loaded = hkRefNew<TestGraph>(hkSerialize::Load().withNoteHandler(&callbacks).toObject<TestGraph>(callbacks.m_saved[0]));
            for (int i = 0; i < callbacks.m_fixups.getSize(); ++i)
            {
                auto& fixup = callbacks.m_fixups[i];
                if (hkResource* imported = callbacks.m_loaded[fixup.m_1])
                {
                    fixup.m_0.setValue(imported->getContents());
                }
            }
            if (HK_TEST(loaded))
            {
                HK_TEST(loaded->m_id == 0);

                if (HK_TEST(loaded->m_childA))
                {
                    HK_TEST(loaded->m_childA->m_id == 1);
                    if (HK_TEST(loaded->m_childA->m_childA))
                    {
                        HK_TEST(loaded->m_childA->m_childA == loaded->m_childB);
                    }
                    HK_TEST(loaded->m_childA->m_childB == HK_NULL);
                    HK_TEST(loaded->m_childA->m_childC == HK_NULL);
                }
                if (HK_TEST(loaded->m_childB))
                {
                    HK_TEST(loaded->m_childB->m_id == 2);
                    HK_TEST(loaded->m_childB->m_childA == HK_NULL);
                    HK_TEST(loaded->m_childB->m_childB == HK_NULL);
                    HK_TEST(loaded->m_childB->m_childC == HK_NULL);
                }
            }
        }
    }

    template<typename Format>
    static void test()
    {
        test_notes<Format>();
        test_notes2<Format>();
        test_endian<Format, hkInt32Le>();
        test_endian<Format, hkInt32Be>();
        test_noSubstitution<Format>();
        test_withSubstitution<Format>();
        test_withEmpty<Format>();
        test_objectStream<Format>();
        test_linkingStreams<Format>();
        test_linkingWithNative<Format>();
        test_hkVariantArray<Format>();
    }
    struct Tagfile
    {
        typedef hkSerialize::TagfileReadFormat Reader;
        typedef hkSerialize::TagfileWriteFormat Writer;
    };

#ifndef HK_NONSDK_BUILD
    struct Xml
    {
        typedef hkSerialize::XmlReadFormat Reader;
        typedef hkSerialize::XmlWriteFormat Writer;
    };
#if defined(HK_BUILDING_WITH_ENGINE)
    struct Yaml
    {
        typedef UnitTest::OptionalReadFormatWrapper<&hkSerialize::Detail::fileFormatYamlfile> Reader;
        typedef UnitTest::OptionalWriteFormatWrapper<&hkSerialize::Detail::fileFormatYamlfile> Writer;
    };

    template<> void test_endian<Yaml, hkInt32Le>() {}
    template<> void test_endian<Yaml, hkInt32Be>() {}
#endif

    
    template<> void test_endian<Xml, hkInt32Le>() {}
    template<> void test_endian<Xml, hkInt32Be>() {}
#endif
}

static int urlPhysicsObjects2_main()
{
    test<Tagfile>();
    test_pointerNotes<Tagfile>();

#ifndef HK_NONSDK_BUILD
    test<Xml>();
    test_pointerNotes<Xml>();
#if defined(HK_BUILDING_WITH_ENGINE)
    if ( HK_TEST( Yaml::Reader::isAvailable() ) && HK_TEST( Yaml::Writer::isAvailable() ) )
    {
        test<Yaml>();
        test_pointerNotes<Yaml>();
    }
#endif
#endif
    return 0;
}

HK_TEST_REGISTER(urlPhysicsObjects2_main, "Fast", "Common/Test/UnitTest/Base/", __FILE__ );

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
