// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/UnitTest/Cloning/CloneTestClasses.h>
#include <Common/Base/Reflect/Builder/hkTypeCopier.h>
#include <Common/Base/Reflect/TypeReg/hkTypeReg.h>
#include <Common/Base/Container/Hash/hkHashSet.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Reflect/Detail/hkTypeHasher.h>

// helper macro which prints the name of the type for which the test fails.
// Use the convention that Copy==LHS, Orig==RHS
#define TYPE_TEST(CONDITION) { hkStringBuf b1, b2; HK_TEST2(CONDITION, " Copy = " << printType(copy, b2) << ", Orig = " << printType(orig, b1)); }
#define TYPE_TEST_EQ(A,B) if((A)!=(B)) { hkStringBuf b1, b2; HK_TEST2( (A)==(B), "A=" << (A) << ", B="<< (B) << " Copy = " << printType(copy, b2) << ", Orig = " << printType(orig, b1) ); }

namespace UnitTest
{
    static const char* printType(const hkReflect::Type* type, hkStringBuf& buf)
    {
        if(!type)
        {
            return "HK_NULL";
        }
        return type->getFullName(buf);
    }

    struct TypeEqualTester
    {
        enum Mode { EQUAL_MINIFY, EQUAL_EXACT };
        typedef hkTuple< const hkReflect::Type*, const hkReflect::Type*> TypePair;

        Mode m_mode;
        hkHashSet<TypePair> m_alreadyTested;
        const hkReflect::TypeReg* m_copyReg;

        TypeEqualTester(Mode mode, const hkReflect::TypeReg* reg) : m_mode(mode), m_copyReg(reg) {}

        void testRegisteredOrEqual(const hkReflect::Type* copy, const hkReflect::Type* orig)
        {
            if (copy != orig)
            {
                // if native copy, checks if the type is registered
                if(m_copyReg && orig)
                {
                    if(const hkReflect::Type* registered = m_copyReg->typeFromType(orig))
                    {
                        HK_TEST_EQ(copy, registered);
                        return;
                    }
                }
                else
                {
                    test(copy, orig);
                }
            }
        }

        bool alreadyTested(const hkReflect::Type* copy, const hkReflect::Type* orig)
        {
            return m_alreadyTested.insert(hkTupleT::make(copy, orig)) == false;
        }

        void test(const hkReflect::Type* copy, const hkReflect::Type* orig)
        {
            if (orig == HK_NULL || copy == HK_NULL)
            {
                HK_TEST_EQ(copy, orig);
                return;
            }

            if(alreadyTested(copy, orig))
            {
                return;
            }

            // Skip things we have excluded from the versioning check, some of these are not set up correctly
            if (orig->findAttribute<hk::ExcludeFromVersionCheck>())
            {
                return;
            }

            TYPE_TEST(copy->getKind() == orig->getKind()
                /*|| (copy->getKind() == hkReflect::KIND_ARRAY && orig->getKind() == hkReflect::KIND_CONTAINER)*/);
            HK_TEST_EQ(copy->isDecorator(), orig->isDecorator());
            //HK_TEST_EQ(copy->getFormat().get(), orig->getFormat().get());

            if (orig->asOpaque())
            {
                // Might be a tracker type, just skip.
                return;
            }

            if(m_mode == EQUAL_EXACT)
            {
                TYPE_TEST_EQ(copy->getSizeOf(), orig->getSizeOf());
                TYPE_TEST_EQ(copy->getAlignOf(), orig->getAlignOf());
            }

            if (const hkReflect::PointerType* p = orig->asPointer())
            {
                testRegisteredOrEqual(copy->asPointer()->getSubType(), p->getSubType());
            }
            else if (const hkReflect::ArrayType* a = orig->asArray())
            {
                HK_TEST_EQ(copy->asArray()->getFixedCount(), a->getFixedCount());
                testRegisteredOrEqual(copy->asArray()->getSubType(), a->getSubType());
            }
            else if (const hkReflect::RecordType* rOrig = orig->asRecord())
            {
                if ( hkReflect::TypeDetail::localHasOptional( rOrig, hkReflect::Opt::DECLS ) )
                {
                    const hkReflect::RecordType* rCopy = copy->asRecord();
                    if(m_mode == EQUAL_MINIFY)
                    {
                        TYPE_TEST(rCopy->getNumFields() <= rOrig->getNumFields());
                        for(int fieldIndex = 0; fieldIndex < rOrig->getNumFields(); ++fieldIndex)
                        {
                            hkReflect::FieldDecl origField = rOrig->getField(fieldIndex);
                            if(origField.isSerializable())
                            {
                                const hkReflect::FieldDecl copyField = rCopy->findField(origField.getName(), false);

                                HK_TEST_EQ(copyField.getName(), origField.getName());
                                HK_TEST_EQ(copyField.getType()->getKind(), origField.getType()->getKind());
                                //TYPE_TEST(emptyBaseClass || copyField->getOffset() <= origField->getOffset());
                                testRegisteredOrEqual(copyField.getType()->getParent(), origField.getType()->getParent());
                            }
                        }
                    }
                    else // EXACT
                    {
                        TYPE_TEST_EQ(rCopy->getNumDataFields(), rOrig->getNumDataFields());
                        for(int fieldIndex = 0; fieldIndex < rOrig->getNumDataFields(); ++fieldIndex)
                        {
                            hkReflect::FieldDecl origField = rOrig->getField(fieldIndex);
                            const hkReflect::FieldDecl copyField = rCopy->getField(fieldIndex);

                            HK_TEST_EQ(copyField.getName(), origField.getName());
                            HK_TEST_EQ(copyField.getType()->getKind(), origField.getType()->getKind());
                            if(HK_TEST_EQ(copyField.getOffset(), origField.getOffset()) == false)
                            {
                                hkStringBuf sb;
                                HK_TEST2(false, rCopy->getFullName(sb) << "::" << copyField.getName());
                            }
                            testRegisteredOrEqual(copyField.getType()->getParent(), origField.getType()->getParent());
                        }
                    }
                }
            }

            // value and incomplete types are remapped on the default types, so optionals can be different
            if (m_copyReg==HK_NULL|| !orig->asValue())
            {
                HK_TEST_EQ(copy->getFullName(), orig->getFullName());
                HK_TEST_EQ(copy->getVersion(), orig->getVersion());
                int ignoreFlags = hkReflect::Type::TYPE_DYNAMIC | hkReflect::Type::TYPE_OWN_VTABLE; // vtable flag can become cleared when copy & compacting
                HK_TEST_EQ((hkReflect::TypeDetail::getFlags(copy).get() | ignoreFlags), (hkReflect::TypeDetail::getFlags(orig).get() | ignoreFlags));

                if (orig->getTemplate() && copy->getTemplate())
                {
                    const hkReflect::Template* origParams = orig->getTemplate();
                    const hkReflect::Template* copyParams = copy->getTemplate();

                    HK_TEST_EQ(copyParams->getNumParams(), origParams->getNumParams());
                    if (copyParams->getNumParams() == origParams->getNumParams())
                    {
                        for (int i = 0; i < copyParams->getNumParams(); ++i)
                        {
                            HK_TEST_EQ(copyParams->getParam(i)->getKind(), origParams->getParam(i)->getKind());
                            if(copyParams->getParam(i)->getKind() == origParams->getParam(i)->getKind())
                            {
                                if(copyParams->getParam(i)->isType())
                                {
                                    testRegisteredOrEqual(copyParams->getParam(i)->getAsType(), origParams->getParam(i)->getAsType());
                                }
                                else
                                {
                                    HK_TEST_EQ(copyParams->getParam(i)->getAsValue(), origParams->getParam(i)->getAsValue());
                                }
                            }
                        }
                    }
                }
                else
                {
                    HK_TEST_EQ(copy->getTemplate(), orig->getTemplate());
                }
            }
        }
    };

    
    static bool isRetargetable(const hkReflect::Type* type)
    {
        if (!type->isRetargetable() || !type->isSerializable())
        {
            return false;
        }

        // Check flag on the parent.
        if (type->getParent() && !type->getParent()->isRetargetable())
        {
            return false;
        }

        // Recursively check fields.
        for (hkReflect::DeclIter<hkReflect::DataFieldDecl> it(type); it.advance();)
        {
            if (!isRetargetable(it.current().getType()))
            {
                return false;
            }
        }
        return true;
    }
}


int typeCopier_main()
{
    using namespace hkReflect;
    {
        // test minified (portable) types
        TypeCopier::Options options;
        TypeCopier copier(options);
        UnitTest::TypeEqualTester tester(UnitTest::TypeEqualTester::EQUAL_MINIFY, HK_NULL);
        for(TypeRegIterator it(getTypeReg()); it.advance(); )
        {
            hkArray< TypeCopier::TypePair > copiesOut;
            const Type* srcType = it.current();
            const Type* dstType = copier.copy(srcType, &copiesOut);
            for(int i = 0; i < copiesOut.getSize(); ++i)
            {
                tester.test(copiesOut[i].m_0, copiesOut[i].m_1);
            }
            HK_TEST(tester.alreadyTested(dstType, srcType));
        }
    }

    {
        // test retargeted (packfile) types
        TypeCopier::Options options;
        TypeCopier copier(options.setHostTarget());
        UnitTest::TypeEqualTester tester(UnitTest::TypeEqualTester::EQUAL_EXACT, HK_NULL);
        hkReflect::TypeHasher hasher;
        #if defined(HK_PLATFORM_LINUX) && HK_POINTER_SIZE==4
        static const hkReflect::Type* ignoredTypes[] =
        {
            // linux32-gcc claims these are size(8), align(8) but when they are used in structures
            // they behave like size(8), align(4). Let's just ignore them.
            HK_REFLECT_GET_TYPE(double),
            HK_REFLECT_GET_TYPE(unsigned long long),
            HK_REFLECT_GET_TYPE(long long),
            HK_REFLECT_GET_TYPE(hkReflect::Typedef::hkUint64_Tag),
            HK_REFLECT_GET_TYPE(hkReflect::Typedef::hkInt64_Tag),
            HK_REFLECT_GET_TYPE(long long[4]),
        };
        for(int i = 0; i < HK_COUNT_OF(ignoredTypes); ++i)
        {
            const hkReflect::Type* srcType = ignoredTypes[i];
            hkArray< TypeCopier::TypePair > copiesOut;
            const hkReflect::Type* dstType = copier.copy(srcType, &copiesOut);
            tester.alreadyTested(dstType, srcType);
        }
        #endif
        for(TypeRegIterator it(getTypeReg()); it.advance();)
        {
            hkArray< TypeCopier::TypePair > copiesOut;
            const Type* srcType = it.current();
            if(srcType->asRecord() && srcType->isAttribute()==false && UnitTest::isRetargetable(srcType) && srcType->isSerializable())
            {
                const Type* dstType = copier.copy(srcType, &copiesOut);
                if(HK_TEST(dstType))
                {
                    if(HK_TEST_EQ(hasher.calc(srcType), hasher.calc(dstType))==false)
                    {
                        hkStringBuf buf;
                        (*hkTestReportFunction)(false, srcType->getFullName(buf), __FILE__, __LINE__);
                    }
                    for(int i = 0; i < copiesOut.getSize(); ++i)
                    {
                        tester.test(copiesOut[i].m_0, copiesOut[i].m_1);
                    }
                    HK_TEST(tester.alreadyTested(dstType, srcType));
                }
            }
        }
    }

    return 0;
}

HK_TEST_REGISTER(typeCopier_main, "Fast", "Common/Test/UnitTest/Base/", __FILE__);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
