// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/UnitTest/Cloning/CloneTest.h>
#include <Common/Base/Serialize/Format/Tagfile/hkTagfileWriteFormat.h>
#include <Common/Base/Reflect/Util/hkReflectClone.h>
#include <Common/Base/System/Io/Writer/hkStreamWriter.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>
#include <Common/Base/System/Io/FileSystem/hkFileSystem.h>
#include <Common/Base/Serialize/Detail/hkSerializeDetail.h>
#include <Common/Base/Container/Hash/hkHashSet.h>
#include <Common/Base/Container/Hash/hkHashMap.h>
#include <Common/Base/Reflect/Builder/hkTypeCopier.h>
#include <Common/Base/System/Io/Util/hkLoadUtil.h>
#include <Common/Base/Config/hkOptionalComponent.h>

// See NOTES at the end

#define DEBUG_LOG_IDENTIFIER "unittest.CrossPlatformSaveLoad"
#include <Common/Base/System/Log/hkLog.hxx>

#define TEST_FOLDER "Resources/Common/Api/Serialize/CrossPlatformTests"


static int testCrossPlatformLoad_main();
static int testCrossPlatformSave_main();

namespace hkCrossPlatformSaveLoadTest
{


hkReflect::Var justClone( const hkReflect::Var& root )
{
    hkReflect::Var res = hkReflect::Cloner().cloneVarRecursive(root);

    if ( hkReferencedObject* refObj = res.dynCast<hkReferencedObject>() )
    {
        refObj->addReference();
    }
    return res;
}

struct TestNameDatabase
{
    void appendTest( const char* testName, hkStringBuf& fileName )
    {
        fileName = testName;
        fileName.replace(':', '_');
        fileName.replace(' ', '_');

        HK_ASSERT(0x40980186, !m_testNamesInUse.contains( fileName.cString() ), "More than one test shares the same name. Use the 'testName' argument to the testObject function to distinguish them." );
        m_testNamesInUse.insert( fileName.cString() );
        fileName.append(".hkt");
    }

    typedef hkHashSet<hkStringPtr> TestNamesInUse;
    TestNamesInUse m_testNamesInUse;
};

static void ensureDirExists(hkFileSystem& fs, const char* p)
{
    fs.mkdir(p, hkFileSystem::ALLOW_EXISTING);
}

static bool existsInFs(hkFileSystem& fs, const char* p)
{
    hkFileSystem::Entry entry;
    return fs.stat(p, entry).isSuccess();
}

typedef void (*Callback)(const char* filename, const hkArray<char>& contents);

struct SaveCloner : public UnitTest::Cloning::Cloner
{
    enum Type
    {
        INPLACE,
        BINARY,
        TEXT
    };

    SaveCloner(Type type, Callback cb)
        : m_type(type)
        , m_indexOfHost(-1)
        , m_callback(cb)
    {
        if( m_type == INPLACE )
        {
            hkReflect::TypeCopier::Options hostOptions;
            hostOptions.setHostTarget();
            hkArrayView<const hkReflect::TypeCopier::Target> targets = hkReflect::TypeCopier::getTargetStrings();
            for(int i = 0; i < targets.getSize(); ++i)
            {
                hkReflect::TypeCopier::Options options;
                if(HK_TEST(options.parseTarget(targets[i].m_name).isOk()))
                {
                    if(hkMemUtil::memCmp(&options, &hostOptions, sizeof(options)) == 0)
                    {
                        m_indexOfHost = i;
                        break;
                    }
                }
            }
            HK_ASSERT_NO_MSG(0x4e9256e, m_indexOfHost != -1);
        }
    }

    virtual hkReflect::Var clone(const char* testNameOrig, const hkReflect::Var& root) HK_OVERRIDE
    {
        Log_Scope(Dev,"Starting test {}", testNameOrig);
        hkStringBuf testName;
        m_testNameDatabase.appendTest(testNameOrig, testName);

        if(m_type == INPLACE)
        {
            hkArrayView<const hkReflect::TypeCopier::Target> targets = hkReflect::TypeCopier::getTargetStrings();
            for(int i = 0; i < targets.getSize(); ++i)
            {
                const char* plat = targets[i].m_name;
                hkStringBuf testFileName(TEST_FOLDER, "/", plat, "/", testName);
                hkArray<char> out;
                {
                    hkSerialize::Save().withTarget(plat).contentsVar(root, &out);
                }

                (*m_callback)(testFileName.cString(), out);

                // there are 2 ways of writing
                // 1 (above) copies the type & recalculates the layouts etc
                // 2 (below) uses the native types directly (only for 1 of the layouts).
                // This checks to see if 1 & 2 match for the host.
                if (false) 
                {
                    hkArray<char> out2;
                    hkSerialize::Save().withTarget(HK_NULL).contentsVar(root, &out2);
                    bool same = (out2.getSize() == out.getSize()) && (hkMemUtil::memCmp(out2.begin(), out.begin(), out.getSize()) == 0);
                    HK_TEST(same);
                    #if 0
                    if(!same)
                    {
                        hkOstream("fail0.bin").write(out.begin(), out.getSize());
                        hkOstream("fail1.bin").write(out2.begin(), out2.getSize());
                    }
                    #endif
                }
            }
        }
        else if(m_type == BINARY)
        {
            hkStringBuf testFileName(TEST_FOLDER, "/Portable/", testName);
            hkArray<char> out;
            hkSerialize::Save().withFormat<hkSerialize::TagfileWriteFormat>().contentsVar(root, &out);
            (*m_callback)(testFileName.cString(), out);
        }
#if defined(HK_BUILDING_WITH_ENGINE)
        else
        {
            hkStringBuf testFileName(TEST_FOLDER, "/Text/", testName);
            hkArray<char> out;

            hkRefPtr<hkSerialize::WriteFormat> writeFormat = hkSerialize::Detail::fileFormatYamlfile.createWriteFormat();
            if (HK_TEST(writeFormat))
            {
                hkSerialize::Save().withFormatState(writeFormat).contentsVar(root, &out);
                (*m_callback)(testFileName.cString(), out);
            }
        }
#endif
        // To placate the enclosing test, also return a clone.
        return justClone(root);
    }

    // We return allocated clones.
    virtual bool mustDeleteCopy() HK_OVERRIDE { return true; }

    TestNameDatabase m_testNameDatabase;
    const char* m_folderPath;
    Type m_type;
    int m_indexOfHost;
    Callback m_callback;
};


static void runTests(Callback cb)
{
    // the "portable" versions
    {
        SaveCloner ser(SaveCloner::BINARY, cb);
        TEST_MAIN_BREAK_HERE(ser, UnitTest::Cloning::DISABLE_NONE);
    }

    // the "packfile" versions
    {
        SaveCloner ser(SaveCloner::INPLACE, cb);
        TEST_MAIN_BREAK_HERE(ser, UnitTest::Cloning::DISABLE_INPLACE);
    }

#if defined(HK_BUILDING_WITH_ENGINE)
    // the "text" versions
    {
        SaveCloner ser(SaveCloner::TEXT, cb);
        TEST_MAIN_BREAK_HERE(ser, UnitTest::Cloning::DISABLE_NONE);
    }
#endif
}


static void overwriteOnFilesystem(const char* fname, const hkArray<char>& contents)
{
    // Check if the file contents are the same first
    hkArray<char> existing;
    hkLoadUtil(fname).toArray(existing);
    if (contents.getSize() != existing.getSize() ||
        hkString::memCmp(existing.begin(), contents.begin(), contents.getSize()) != 0)
    {
        // they differ
        hkOstream ofstr(fname);
        if (ofstr.isOk())
        {
            if (ofstr.write(contents.begin(), contents.getSize()) == contents.getSize())
            {
                return; // ok
            }
        }
        hkStringBuf warning;
        warning.printf("Cannot overwrite \"%s\" with the new reference data.", fname);
        HK_WARN(0xd5ea70a, warning);
    }
}


static void checkAgainstFilesystem(const char* fname, const hkArray<char>& contents)
{
    hkFileSystem& fs = hkFileSystem::getInstance();
    if(existsInFs(fs,fname))
    {
        hkIo::ReadBuffer rbuf(fname);
        hkLong eSize = rbuf.prefetchAll();
        if(eSize > 0)
        {
            bool sameAsReference = true;
            HK_TEST2(contents.getSize() == eSize, fname);
            if(contents.getSize() == eSize)
            {
                sameAsReference = hkString::memCmp(contents.begin(), rbuf.peekAt<void>(0, eSize), hkLosslessCast<int>(eSize)) == 0;
                HK_TEST2(sameAsReference, fname);
            }
            else
            {
                sameAsReference = false;
            }
            #if 1 // for debugging failures
            if(!sameAsReference)
            {
                hkOstream(hkStringBuf(fname, ".fail")).write(contents.begin(), contents.getSize());
            }
            #endif
        }
    }
    else
    {
        hkStringBuf warning;
        warning.printf("Missing test \"%s\". When the CloneTest changes, the files in " TEST_FOLDER " should be updated.", fname);
        HK_WARN(0xd5ea70a, warning);
    }
}


struct FromFileCloner : public UnitTest::Cloning::Cloner
{
    FromFileCloner(const char* folderPath, bool inplace)
        : m_folderPath(folderPath)
        , m_wantInplace(inplace)
    {
    }

    ~FromFileCloner()
    {
        clearSelf();
    }

    void clearSelf()
    {
        if(m_inplaceVar)
        {
            hkSerialize::InplaceLoad::unload(m_inplaceBuf.begin(), m_inplaceBuf.getSize());
        }
        m_inplaceVar.clear();
        m_inplaceBuf.clear();
        m_mustDelete = true;
    }

    virtual hkReflect::Var clone(const char* testNameOrig, const hkReflect::Var& root) HK_OVERRIDE
    {
        clearSelf();

        hkStringBuf testFileName;
        m_testNameDatabase.appendTest( testNameOrig, testFileName );
        hkStringBuf testFilePath(m_folderPath, "/", testFileName);
        if(hkRefPtr<hkStreamReader> streamReader = hkFileSystem::getInstance().openReader(testFilePath))
        {
            if(m_wantInplace)
            {
                m_mustDelete = false;
                hkLoadUtil(streamReader).toArray(m_inplaceBuf);
                m_inplaceVar = hkSerialize::InplaceLoad().toVar(m_inplaceBuf.begin(), m_inplaceBuf.getSize());
                return m_inplaceVar;
            }
            else
            {
                return hkSerialize::Load().toVar(streamReader.val());
            }
        }
        else
        {
            // Missing test file, return empty Var, nothing was cloned
            Log_Warning("Missing test {}. When the CloneTest changes, the files in {} should be updated.", testFilePath, TEST_FOLDER);
            return hkReflect::Var();
        }
    }

    virtual bool mustDeleteCopy() HK_OVERRIDE { return m_mustDelete; }

    TestNameDatabase m_testNameDatabase;
    const char* m_folderPath;
    hkArray<char> m_inplaceBuf;
    hkReflect::Var m_inplaceVar;
    bool m_wantInplace;
    bool m_mustDelete;
};


// Check that all files from all platforms are usable via the normal "clone" load
static void loadTestDataAndTest()
{
    hkFileSystem& fs = hkFileSystem::getInstance();

    // the "portable" versions
    {
        FromFileCloner ser(TEST_FOLDER "/Portable", false);
        TEST_MAIN_BREAK_HERE(ser, UnitTest::Cloning::DISABLE_NONE);
    }

    // the "packfile" versions
    hkArrayView<const hkReflect::TypeCopier::Target> targets = hkReflect::TypeCopier::getTargetStrings();
    for(int i = 0; i < targets.getSize(); ++i)
    {
        const char* plat = targets[i].m_name;
        hkStringBuf path(TEST_FOLDER, "/", plat);
        if(existsInFs(fs, path))
        {
            FromFileCloner ser(path, false);
            TEST_MAIN_BREAK_HERE(ser, UnitTest::Cloning::DISABLE_INPLACE);
        }
    }

#if defined(HK_BUILDING_WITH_ENGINE) // needs YAML
    // the "text" versions
    {
        FromFileCloner ser(TEST_FOLDER "/Text", false);
        TEST_MAIN_BREAK_HERE(ser, UnitTest::Cloning::DISABLE_NONE);
    }
#endif
}


// Check that the inplace load actually works
static void loadInplaceAndTest()
{
    hkStringBuf folderIn(TEST_FOLDER "/", hkReflect::TypeCopier::getHostString());
    if(existsInFs(hkFileSystem::getInstance(), folderIn))
    {
        FromFileCloner ser(folderIn, true);
        TEST_MAIN_BREAK_HERE(ser, UnitTest::Cloning::DISABLE_INPLACE);
    }
}

// NOTES
// The general idea is that we should be able to create all the checked-in gold standard data from any platform.
// Each platform then tests if the data which it *would have written* is the same as the existing data.
// This allows quick update of the gold standard when the file format or tests changes.
HK_EXPORT_COMMON int createData_CrossPlatformSaveLoad()
{
    hkFileSystem& fs = hkFileSystem::getInstance();

    // Ensure the required directories exist.
    ensureDirExists(fs, TEST_FOLDER);
    ensureDirExists(fs, TEST_FOLDER "/Portable");
    ensureDirExists(fs, TEST_FOLDER "/Text");

    hkArrayView<const hkReflect::TypeCopier::Target> targets = hkReflect::TypeCopier::getTargetStrings();
    for(int i = 0; i < targets.getSize(); ++i)
    {
        ensureDirExists(fs, hkStringBuf(TEST_FOLDER, "/", targets[i].m_name));
    }

    // Create new gold standard
    runTests(&overwriteOnFilesystem);
    return 0;
}


HK_EXPORT_COMMON int testCrossPlatformSaveLoad_full()
{
    testCrossPlatformLoad_main();
    testCrossPlatformSave_main();
    return 0;
}

}

// Load all reference files and tests if the loaded contents are correct.
// This test MUST pass after changes to the serialization formats. Any failure means a breakage of compatibility with
// files saved before the changes.
static int testCrossPlatformLoad_main()
{
#ifdef HK_REAL_IS_FLOAT
    hkCrossPlatformSaveLoadTest::loadTestDataAndTest();
    hkCrossPlatformSaveLoadTest::loadInplaceAndTest();
#endif
    return 0;
}

// Save the test data and compare it against the reference files.
// This test might fail after changes to the serialization formats. The reference files will need to be regenerated
// in that case.
static int testCrossPlatformSave_main()
{
#ifdef HK_REAL_IS_FLOAT
    hkCrossPlatformSaveLoadTest::runTests(&hkCrossPlatformSaveLoadTest::checkAgainstFilesystem);
#endif
    return 0;
}

HK_TEST_REGISTER(testCrossPlatformLoad_main, "Slow", "Common/Test/UnitTest/Base/", __FILE__);
HK_TEST_REGISTER(testCrossPlatformSave_main, "Slow", "Common/Test/UnitTest/Base/", __FILE__);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
