// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT    : COMMON
// VISIBILITY : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/Algorithm/Find/hkFind.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/UnitTest/Serialize/Tagfile/BundleLoadSave.h>
#include <Common/Base/Serialize/hkSerialize.h>
#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>
#include <Common/Base/System/Io/Reader/Memory/hkMemoryStreamReader.h>
#include <Common/Base/System/Io/Structured/hkStructuredStream.h>
#include <Common/Base/UnitTest/Cloning/CloneTestClasses.h>
#include <Common/Base/Reflect/TypeReg/hkTypeReg.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Container/StringMap/hkStringMap.h>

#include <Common/Base/Serialize/Format/Tagfile/hkTagfileWriteFormat.h>
#include <Common/Base/Serialize/Format/Tagfile/hkTagfileReadFormat.h>
#include <Common/Base/Serialize/Format/Tagfile/Detail/hkTagfileDetail.h>

namespace TestBundleLoadSave
{
    static const hkUint64 u64 = 99;
    static const char one = '1';
    static const short two = 0x0202;
    static const int three = 0x03030303;

    struct TestVarsContainer
    {
        enum { VAR_COUNT = 7 };
        TestVarsContainer()
        {
            a0.pushBack(1);

            a1.pushBack(10);
            a1.pushBack(11);
            a1.pushBack(12);
            a1.pushBack(13);

            a2.pushBack(100);
            a2.pushBack(101);
            a2.pushBack(102);
            a2.pushBack(104);
            a2.pushBack(105);
            a2.pushBack(106);
            a2.pushBack(107);
            a2.pushBack(108);
            a2.pushBack(109);

            m_vars[0] = &u64;
            m_vars[1] = &one;
            m_vars[2] = &two;
            m_vars[3] = &three;
            m_vars[4] = &a0;
            m_vars[5] = &a1;
            m_vars[6] = &a2;
        }
        hkArray<int> a0;
        hkArray<int> a1;
        hkArray<int> a2;
        hkReflect::Var m_vars[VAR_COUNT];
    };

template<typename T>
static hkUlong testLoad(hkSerialize::Load& load, const hkArray<char>& buf, hkUlong cursor, T expected)
{
    hkUlong used;
    T* ptr = load.toObject<T>(buf.begin() + cursor, buf.getSize() - cursor, &used);
    if (HK_TEST(ptr))
    {
        HK_TEST_EQ(*ptr, expected);
        hkReflect::Var(ptr).destroy();
    }

    return used;
}

namespace
{
    using namespace hkSerialize::Detail;

    hkResult findSection( TagfileTypeSection::Enum sectionType, const hkArray<HffSection>& sections, const HffSection*& typeSectionOut )
    {
        auto view = sections.view();
        auto const it = hkAlgorithm::findIf( view, [sectionType]( const HffSection& section ) -> bool { return ( section.m_ident == (hkUint32)sectionType ); } );
        if ( it != view.end() )
        {
            typeSectionOut = &( *it );
            return HK_SUCCESS;
        }
        return HK_FAILURE;
    }

    hkResult findTypeSection( const hkArray<HffSection>& sections, const HffSection*& typeSectionOut )
    {
        return findSection( TagfileTypeSection::IDENTITY, sections, typeSectionOut );
    }

    hkResult findRollingHashSection( const hkArray<HffSection>& sections, const HffSection*& rollinhHashSectionOut )
    {
        return findSection( TagfileTypeSection::ROLLING_HASH, sections, rollinhHashSectionOut );
    }

    void extractRollingHashes( const hkArray<char>& buf, const HffSection& hashSection, hkUint32& oldHash, hkUint32& newHash )
    {
        HK_TEST3( hashSection.m_ident == TagfileTypeSection::ROLLING_HASH, "Invalid rolling hash section." );
        const hkUint32Le* ptr = (const hkUint32Le*)hkAddByteOffset( buf.begin(), hashSection.m_offset + 8 );
        oldHash = ptr[0];
        newHash = ptr[1];
    }
}

static void testSequenceNumber()
{
    const int anInt = 10;
    const float aFloat = 11.1;
    const double aDouble = 3.14;

    // Single output (concatenation)
    {
        hkArray<char> buf;

        // Saving
        {
            hkIo::WriteBuffer writeBuffer( &buf );
            hkSerialize::Save save;
            save.withMultiBundle();
            HK_TEST( save.contentsPtr( &anInt, writeBuffer ).isSuccess() );
            HK_TEST( save.contentsPtr( &aFloat, writeBuffer ).isSuccess() );
            HK_TEST( save.contentsPtr( &anInt, writeBuffer ).isSuccess() );
            HK_TEST( save.contentsPtr( &aDouble, writeBuffer ).isSuccess() );
        }

        // Reloading
        {
            hkSerialize::Load load;
            hkUlong cur = testLoad( load, buf, 0, anInt );
            cur += testLoad( load, buf, cur, aFloat );
            cur += testLoad( load, buf, cur, anInt );
            testLoad( load, buf, cur, aDouble );
        }
    }

    // Multiple outputs (streaming)
    {
        using namespace hkSerialize::Detail;

        hkArray<char> buf0, buf1, buf2, buf3;
        hkSerialize::Save save;
        save.withMultiBundle();
        HK_TEST( save.contentsPtr( &anInt, &buf0 ).isSuccess() );
        HK_TEST( save.contentsPtr( &aFloat, &buf1 ).isSuccess() );
        HK_TEST( save.contentsPtr( &anInt, &buf2 ).isSuccess() );
        HK_TEST( save.contentsPtr( &aDouble, &buf3 ).isSuccess() );

        // Multi-bundle should skip already-sent types
        HK_TEST( buf0.getSize() > buf2.getSize() );

        hkArray<HffSection> sec0, sec1, sec2, sec3;
        HK_TEST( indexHff( sec0, buf0.begin(), buf0.getSize() ).isSuccess() );
        HK_TEST( indexHff( sec1, buf1.begin(), buf1.getSize() ).isSuccess() );
        HK_TEST( indexHff( sec2, buf2.begin(), buf2.getSize() ).isSuccess() );
        HK_TEST( indexHff( sec3, buf3.begin(), buf3.getSize() ).isSuccess() );

        const HffSection* typeSec0;
        const HffSection* typeSec1;
        const HffSection* typeSec2;
        const HffSection* typeSec3;
        HK_TEST( findTypeSection( sec0, typeSec0 ).isSuccess() );
        HK_TEST( findTypeSection( sec1, typeSec1 ).isSuccess() );
        HK_TEST( findTypeSection( sec2, typeSec2 ).isFailure() ); // no new type, no type section
        HK_TEST( findTypeSection( sec3, typeSec3 ).isSuccess() );

        const HffSection* hashSec0;
        const HffSection* hashSec1;
        const HffSection* hashSec2;
        const HffSection* hashSec3;
        HK_TEST( findRollingHashSection( sec0, hashSec0 ).isSuccess() );
        HK_TEST( findRollingHashSection( sec1, hashSec1 ).isSuccess() );
        HK_TEST( findRollingHashSection( sec2, hashSec2 ).isFailure() ); // no new type, no type section
        HK_TEST( findRollingHashSection( sec3, hashSec3 ).isSuccess() );

        hkUint32 oldHash0, newHash0;
        hkUint32 oldHash1, newHash1;
        hkUint32 oldHash3, newHash3;
        extractRollingHashes( buf0, *hashSec0, oldHash0, newHash0 );
        extractRollingHashes( buf1, *hashSec1, oldHash1, newHash1 );
        extractRollingHashes( buf3, *hashSec3, oldHash3, newHash3 );
        HK_TEST3( oldHash0 == 0, "Invalid initial type rolling hash (expected zero, got {:x8}).", oldHash0 );
        HK_TEST3( oldHash1 == newHash0, "Invalid #2 type rolling hash (expected {:x8}, got {:x8}).", newHash0, oldHash1 );
        HK_TEST3( oldHash3 == newHash1, "Invalid #3 type rolling hash (expected {:x8}, got {:x8}).", newHash1, oldHash3 );
    }

    // Multiple outputs (streaming)
    {
        using namespace hkSerialize::Detail;

        hkArray<char> buf0, buf1, buf2, buf3;
        hkSerialize::Save save;
        save.withMultiBundle();
        HK_TEST( save.contentsPtr( &anInt, &buf0 ).isSuccess() );
        HK_TEST( save.contentsPtr( &aFloat, &buf1 ).isSuccess() );
        HK_TEST( save.contentsPtr( &anInt, &buf2 ).isSuccess() );
        HK_TEST( save.contentsPtr( &aDouble, &buf3 ).isSuccess() );

        // Multi-bundle should skip already-sent types
        HK_TEST( buf0.getSize() > buf2.getSize() );

        hkArray<HffSection> sec0, sec1, sec2, sec3;
        HK_TEST( indexHff( sec0, buf0.begin(), buf0.getSize() ).isSuccess() );
        HK_TEST( indexHff( sec1, buf1.begin(), buf1.getSize() ).isSuccess() );
        HK_TEST( indexHff( sec2, buf2.begin(), buf2.getSize() ).isSuccess() );
        HK_TEST( indexHff( sec3, buf3.begin(), buf3.getSize() ).isSuccess() );

        const HffSection* typeSec0;
        const HffSection* typeSec1;
        const HffSection* typeSec2;
        const HffSection* typeSec3;
        HK_TEST( findTypeSection( sec0, typeSec0 ).isSuccess() );
        HK_TEST( findTypeSection( sec1, typeSec1 ).isSuccess() );
        HK_TEST( findTypeSection( sec2, typeSec2 ).isFailure() ); // no new type, no type section
        HK_TEST( findTypeSection( sec3, typeSec3 ).isSuccess() );

        const HffSection* hashSec0;
        const HffSection* hashSec1;
        const HffSection* hashSec2;
        const HffSection* hashSec3;
        HK_TEST( findRollingHashSection( sec0, hashSec0 ).isSuccess() );
        HK_TEST( findRollingHashSection( sec1, hashSec1 ).isSuccess() );
        HK_TEST( findRollingHashSection( sec2, hashSec2 ).isFailure() ); // no new type, no type section
        HK_TEST( findRollingHashSection( sec3, hashSec3 ).isSuccess() );

        hkUint32 oldHash0, newHash0;
        hkUint32 oldHash1, newHash1;
        hkUint32 oldHash3, newHash3;
        extractRollingHashes( buf0, *hashSec0, oldHash0, newHash0 );
        extractRollingHashes( buf1, *hashSec1, oldHash1, newHash1 );
        extractRollingHashes( buf3, *hashSec3, oldHash3, newHash3 );
        HK_TEST3( oldHash0 == 0, "Invalid initial type rolling hash (expected zero, got {:x8}).", oldHash0 );
        HK_TEST3( oldHash1 == newHash0, "Invalid #2 type rolling hash (expected {:x8}, got {:x8}).", newHash0, oldHash1 );
        HK_TEST3( oldHash3 == newHash1, "Invalid #3 type rolling hash (expected {:x8}, got {:x8}).", newHash1, oldHash3 );
    }
}


static void testSaveToOneStream(hkSerialize::WriteFormat& wformat, hkSerialize::ReadFormat& rformat)
{
    // All items in stream are written consecutively and read back out of the same buffer

    // There is trickiness with alignment.
    // In order to use the inplace() method, the caller needs to ensure the buffer is aligned

    hkArray<char> buf;
    hkArray<hkUlong> endOffsets;
    {
        hkArrayStreamWriter asw(&buf, hkArrayStreamWriter::ARRAY_BORROW);
        TestVarsContainer container;

        for( int i=0; i < TestVarsContainer::VAR_COUNT; ++i )
        {
            hkSerialize::Save().withFormatState(&wformat).contentsVar(container.m_vars[i], &asw);
            endOffsets.pushBack(buf.getSize());
            asw.seek( HK_NEXT_MULTIPLE_OF(16, asw.tell()), hkStreamWriter::STREAM_SET );
        }
    }
    {
        hkUlong offset = 0;
        for( int i=0; i < TestVarsContainer::VAR_COUNT; ++i )
        {
            hkUlong used = 0;
            if (hkViewPtr<hkSerialize::Bundle> bun = rformat.view(buf.begin() + offset, buf.getSize() - offset, &used))
            {
                hkReflect::Var v = bun->getContents();
                v.getType();

                offset += used;
                HK_TEST( offset == endOffsets[i] );
                offset = HK_NEXT_MULTIPLE_OF(16, offset);
            }
            else
            {
                HK_TEST(0);
                break;
            }
        }
    }
}

static void testSaveToLogicalStream(hkSerialize::WriteFormat& wformat, hkSerialize::ReadFormat& rformat)
{
    // emulate network case - stream is written in independent frames
    hkArray<char> streams[TestVarsContainer::VAR_COUNT];

    {
        TestVarsContainer container;
        for( int i=0; i < TestVarsContainer::VAR_COUNT; ++i )
        {
            hkArrayStreamWriter asw(&streams[i], hkArrayStreamWriter::ARRAY_BORROW);
            hkSerialize::Save().withFormatState(&wformat).contentsVar(container.m_vars[i], &asw);
        }
    }
    {
        for( int i=0; i < TestVarsContainer::VAR_COUNT; ++i )
        {
            hkUlong used = 0;
            rformat.view(streams[i].begin(), streams[i].getSize(), &used);
            HK_TEST( hkUlong(streams[i].getSize()) == used);
        }
    }
}

static void testImportsSave(hkSerialize::WriteFormat& wformat, hkSerialize::ReadFormat& rformat)
{
    using namespace hkSerialize;

    hkArray<char> buf;
    {
        UnitTest::Ref0 obj0;
        UnitTest::Ref1 obj1;
        hkArray<hkReferencedObject*> arr;
        arr.pushBack(&obj0);
        arr.pushBack(&obj1);

        {
            hkIo::WriteBuffer writeBuffer(&buf);
            BundleBuilder bb(&wformat, &writeBuffer);
            bb.add(&arr);
            bb.addImport(&obj1, "obj1");
            bb.recursiveAdd();
        }
    }
    {
        hkUlong used = 0;
        if (hkViewPtr<hkSerialize::Bundle> bundle = rformat.view(buf.begin(), buf.getSize(), &used))
        {
            hkReflect::Var v = bundle->getContents();
            HK_TEST(v.getAddress());
            HK_TEST( hkUlong(buf.getSize()) == used);

        }
    }
    {
        hkReflect::Var var = hkSerialize::Load().toVar(buf.begin(), buf.getSize());
        HK_TEST(var.getAddress());
        hkArray<hkReferencedObject*>* arr = var.dynCast< hkArray<hkReferencedObject*> >();
        if(HK_TEST(arr))
        {
            for(int i = 0; i < arr->getSize(); ++i)
            {
                if(hkReferencedObject* o = (*arr)[i])
                {
                    o->removeReference();
                }
            }
            delete arr;
        }
    }
}

static void testMismatchedReadWrite(hkSerialize::WriteFormat& wformat, hkSerialize::ReadFormat& rformatA, hkSerialize::ReadFormat& rformatB)
{
    using namespace hkSerialize;
    hkArray<char> buf0;
    {
        int anInt = 42;
        hkIo::WriteBuffer writeBuffer(&buf0);
        BundleBuilder bb(&wformat, &writeBuffer);
        bb.add(&anInt);
        bb.recursiveAdd();
    }

    hkArray<char> buf1;
    {
        float aFloat = 101; // save a different type to previous to increment the sequence number
        hkIo::WriteBuffer writeBuffer(&buf1);
        BundleBuilder bb(&wformat, &writeBuffer);
        bb.add(&aFloat);
        bb.recursiveAdd();
    }

    // send the two parts separately
    {
        hkUlong used0 = 0;
        hkViewPtr<hkSerialize::Bundle> bundle0 = rformatA.view(buf0.begin(), buf0.getSize(), &used0);
        HK_TEST(bundle0);
        HK_TEST(used0 == hkUlong(buf0.getSize()));

        hkUlong used1 = 0;
        hkViewPtr<hkSerialize::Bundle> bundle1 = rformatA.view(buf1.begin(), buf1.getSize(), &used1);
        HK_TEST(bundle1);
        HK_TEST(used1 == hkUlong(buf1.getSize()));
    }
    // missing a part should fail
    {
        UnitTest::Raii::CaptureThreadLogOutput capture;
        hkUlong used1 = 0;
        hkViewPtr<hkSerialize::Bundle> bundle1 = rformatB.view(buf1.begin(), buf1.getSize(), &used1);
        HK_TEST(bundle1 == HK_NULL);
    }
}

static void testMultiBundle(hkSerialize::WriteFormat& wformat, hkSerialize::ReadFormat& rformat)
{
    using namespace hkSerialize;

    TestVarsContainer container;

    hkArray<char> buf;
    {
        hkArrayStreamWriter asw(&buf, hkArrayStreamWriter::ARRAY_BORROW);
        Save save; save.withFormatState(&wformat);

        for( int i = 0; i < TestVarsContainer::VAR_COUNT; ++i )
        {
            save.contentsVar(container.m_vars[i], &asw);
        }
    }

    {
        hkUlong cur = 0;
        Load load;
        load.withFormatState(&rformat);
        for( int i=0; i < TestVarsContainer::VAR_COUNT; ++i )
        {
            hkUlong used;
            hkReflect::Var var = load.toVar(buf.begin() + cur, buf.getSize() - cur, HK_NULL, &used);
            cur += used;
            HK_TEST(var.equals(container.m_vars[i]));
            var.destroy();
        }
    }
}

static void testAfterReflectNewHandler(hkSerialize::WriteFormat& wformat, hkSerialize::ReadFormat& rformat)
{
    hkArray<char> buf;
    {
        AfterReflectNewTestObj testObj;
        testObj.m_afterReflectNewCalled = false;

        hkArrayStreamWriter asw(&buf, hkArrayStreamWriter::ARRAY_BORROW);
        hkSerialize::Save().withFormatState(&wformat).contentsVar(&testObj, &asw);
    }

    {
        hkReflect::Cloner::AfterReflectNewHandler handler;

        hkReflect::Var var = hkSerialize::Load().withFormatState(&rformat).withAfterReflectNewHandler(&handler).toVar(buf.begin(), buf.getSize());

        if ( HK_TEST( var ) )
        {
            HK_TEST( var.dynCast<AfterReflectNewTestObj>()->m_afterReflectNewCalled == false );
            handler.callAfterReflectNews();
            HK_TEST( var.dynCast<AfterReflectNewTestObj>()->m_afterReflectNewCalled == true );

            var.destroy();
        }
    }
}

}

static int testBundleLoadSave()
{
    using namespace TestBundleLoadSave;

    // New writer versions
    {
        {
            testSequenceNumber();
        }
        {
            hkSerialize::TagfileWriteFormat writeFormat;
            hkSerialize::TagfileReadFormat readFormat;
            writeFormat.enableMultiBundle();
            testSaveToOneStream(writeFormat, readFormat);
        }
        {
            hkSerialize::TagfileWriteFormat writeFormat;
            hkSerialize::TagfileReadFormat readFormat;
            writeFormat.enableMultiBundle();
            testSaveToLogicalStream(writeFormat, readFormat);
        }
        {
            hkSerialize::TagfileWriteFormat writeFormat;
            hkSerialize::TagfileReadFormat readFormatA;
            hkSerialize::TagfileReadFormat readFormatB;
            writeFormat.enableMultiBundle();
            testMismatchedReadWrite(writeFormat, readFormatA, readFormatB);
        }
        {
            hkSerialize::TagfileWriteFormat writeFormat;
            hkSerialize::TagfileReadFormat readFormat;
            testImportsSave(writeFormat, readFormat);
        }

        {
            hkSerialize::TagfileWriteFormat writeFormat;
            hkSerialize::TagfileReadFormat readFormat;
            writeFormat.enableMultiBundle();
            testMultiBundle(writeFormat, readFormat);
        }

        {
            hkSerialize::TagfileWriteFormat writeFormat;
            hkSerialize::TagfileReadFormat readFormat;
            testAfterReflectNewHandler(writeFormat, readFormat);
        }
    }
    return 0;
}

HK_TEST_REGISTER(testBundleLoadSave, "Fast", "Common/Test/UnitTest/Base/", __FILE__);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
