// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/Reflect/TypeVm/hkTypeVmCompiler.h>
#include <Common/Base/UnitTest/Reflection/TypeVm.h>
#include <Common/Base/Reflect/TypeVm/hkTypeVmCompilerPasses.h>
#include <Common/Base/Reflect/TypeVm/hkTypeVmFastCopyInterpreter.h>

namespace
{
    hkTypeVm::Program::InstructionType mapKind(const hkReflect::Kind kind)
    {
        switch(kind)
        {
        case hkReflect::KIND_INT: return hkTypeVm::Program::INST_INT;
        case hkReflect::KIND_FLOAT: return hkTypeVm::Program::INST_FLOAT;
        case hkReflect::KIND_STRING: return hkTypeVm::Program::INST_STRING;
        case hkReflect::KIND_RECORD: return hkTypeVm::Program::INST_RECORD;
        case hkReflect::KIND_ARRAY: return hkTypeVm::Program::INST_ARRAY;
        case hkReflect::KIND_POINTER: return hkTypeVm::Program::INST_POINTER;
        default:
            return hkTypeVm::Program::INST_NONE;
        }
    }
}

int typeVm_main()
{
    // Simple type.
    {
        hkTypeVm::Compiler compiler;
        hkViewPtr<hkTypeVm::Program> program = compiler.compile(hkReflect::getType<int>());
        if (HK_TEST(program) && HK_TEST_EQ(program->end()-program->begin(), 1))
        {
            HK_TEST_EQ(program->begin()->getInstructionType(), hkTypeVm::Program::INST_INT);
            HK_TEST_EQ(program->begin()->getSrcOffset(), 0);
            HK_TEST_EQ(program->begin()->getDstOffset(), 0);
        }
    }

    // Record
    {
        const hkReflect::Type* type = hkReflect::getType<UnitTest::TypeVm::Record>();
        hkTypeVm::Compiler compiler;
        hkViewPtr<hkTypeVm::Program> program = compiler.compile(type);
        if (HK_TEST(program))
        {
            hkTypeVm::Program::Iterator progIt = program->begin();
            for (hkReflect::DeclIter<hkReflect::FieldDecl> it(type); it.advance(); )
            {
                if (HK_TEST(progIt != program->end()))
                {
                    const hkTypeVm::Program::InstructionBase& fieldInst = *progIt++;
                    hkReflect::FieldDecl field = it.current();

                    HK_TEST_EQ(fieldInst.getInstructionType(), mapKind(field.getType()->getKind()));
                    HK_TEST_EQ(fieldInst.getSrcOffset(), field.getOffset());
                    HK_TEST_EQ(fieldInst.getDstOffset(), field.getOffset());

                    if (fieldInst.getInstructionType() == hkTypeVm::Program::INST_RECORD)
                    {
                        const hkTypeVm::Program* embProg = static_cast<const hkTypeVm::Program::ProgramInstruction&>(fieldInst).getProgram();
                        HK_TEST_EQ(embProg, compiler.compile(type->findDecl("embedded").getType()).val());
                    }
                }
            }
            HK_TEST(progIt == program->end());
        }
    }

    // Inline passes
    {
        const hkReflect::Type* type = hkReflect::getType<UnitTest::TypeVm::Record>();
        hkTypeVm::Compiler compiler;
        compiler.addPass(new hkTypeVm::InlineFixedArrayPass);
        compiler.addPass(new hkTypeVm::InlineRecordPass);
        hkViewPtr<hkTypeVm::Program> program = compiler.compile(type);
        if (HK_TEST(program))
        {
            hkTypeVm::Program::Iterator progIt = program->begin();
            HK_TEST_EQ(progIt->getInstructionType(), hkTypeVm::Program::INST_ARRAY); // m_dynamicProperties
            ++progIt;
            HK_TEST_EQ(progIt->getInstructionType(), hkTypeVm::Program::INST_INT); // m_memSizeAndFlags
            ++progIt;
            HK_TEST_EQ(progIt->getInstructionType(), hkTypeVm::Program::INST_INT); // m_refCount
            ++progIt;

#define TEST_OFFSET(FIELD) \
        HK_TEST_EQ(progIt->getSrcOffset(), HK_OFFSET_OF(UnitTest::TypeVm::Record, FIELD)); \
        HK_TEST_EQ(progIt->getDstOffset(), HK_OFFSET_OF(UnitTest::TypeVm::Record, FIELD))

            HK_TEST_EQ(progIt->getInstructionType(), hkTypeVm::Program::INST_INT);
            TEST_OFFSET(m_int);
            ++progIt;
            HK_TEST_EQ(progIt->getInstructionType(), hkTypeVm::Program::INST_FLOAT);
            TEST_OFFSET(m_floats[0]);
            ++progIt;
            HK_TEST_EQ(progIt->getInstructionType(), hkTypeVm::Program::INST_FLOAT);
            TEST_OFFSET(m_floats[1]);
            ++progIt;
            HK_TEST_EQ(progIt->getInstructionType(), hkTypeVm::Program::INST_INT);
            TEST_OFFSET(m_embedded.m_int);
            ++progIt;
            HK_TEST_EQ(progIt->getInstructionType(), hkTypeVm::Program::INST_FLOAT);
            TEST_OFFSET(m_embedded.m_floats[0]);
            ++progIt;
            HK_TEST_EQ(progIt->getInstructionType(), hkTypeVm::Program::INST_FLOAT);
            TEST_OFFSET(m_embedded.m_floats[1]);
            ++progIt;
            HK_TEST_EQ(progIt->getInstructionType(), hkTypeVm::Program::INST_POINTER);
            TEST_OFFSET(m_embedded.m_embedded.m_ptr);
            ++progIt;
            HK_TEST_EQ(progIt->getInstructionType(), hkTypeVm::Program::INST_ARRAY);
            TEST_OFFSET(m_array);
            ++progIt;
            HK_TEST_EQ(progIt->getInstructionType(), hkTypeVm::Program::INST_STRING);
            TEST_OFFSET(m_string);
            ++progIt;
#undef TEST_OFFSET
            HK_TEST(progIt == program->end());
        }
    }

    return 0;
}

int typeVm_regression()
{
    // MSFT:10583932 - Handle type mismatch
    {
        typedef hkArray<float> SrcType;
        typedef hkArray<hkStringPtr> DstType;

        // Without CheckTypeKindsPass, this produces a program which eventually asserts on execution
        {
            hkTypeVm::Compiler compiler;
            hkTypeVm::addDefaultPasses( compiler );
            compiler.addPass<hkTypeVm::IntAndFloatConversionPass>();

            hkViewPtr<hkTypeVm::Program> program = compiler.compile( hkReflect::getType<SrcType>(), hkReflect::getType<DstType>() );
            if ( HK_TEST( program ) )
            {
                SrcType srcArray;
                srcArray.pushBack( 1.0f );
                srcArray.pushBack( 2.0f );
                srcArray.pushBack( 3.0f );

                DstType dstArray;
                dstArray.setSize( 3 );

                hkReflect::ArrayVar src( &srcArray );
                hkReflect::ArrayVar dst( &dstArray );

                hkReflect::ArrayValue valSrc;
                if ( HK_TEST( src.getValue( &valSrc ).isSuccess() ) == false )
                {
                    return 0;
                }

                hkReflect::ArrayValue valDst;
                if ( HK_TEST( dst.getValue( &valDst ).isSuccess() ) == false )
                {
                    return 0;
                }

                const int count = valDst.getCount();
                if ( HK_TEST( count > 0 ) )
                {
                    if ( HK_TEST( src.getCount() == count ) == false )
                    {
                        return 0;
                    }

                    HK_ON_TEST_ASSERT_ENABLED( hkTypeVm::FastCopyInterpreter machine );

#if defined(HK_DEBUG_SLOW)
#define ASSERT_EXPECTED 0x5dcef9a9
#else
#define ASSERT_EXPECTED 0x3d485f34
#endif

                    HK_TEST_ASSERT2(ASSERT_EXPECTED, hkTypeVm::FastCopyInterpreter::execN( machine, program,
                        valDst.getAddress(), count*valDst.getStride(),
                        valSrc.getAddress(), count*valSrc.getStride(),
                        valDst.getStride(), valSrc.getStride(), valSrc.getCount() ),
                        "FastCopyInterpreter::execN() was expected to assert (0x5dcef9a9 or 0x3d485f34) on array cloning with mismatching subtype (float -> hkStringPtr)." );

#undef ASSERT_EXPECTED
                }
            }
        }

        // With CheckTypeKindsPass, the compilation fails as expected
        {
            hkTypeVm::Compiler compiler;
            hkTypeVm::addDefaultPasses( compiler );
            compiler.addPass<hkTypeVm::IntAndFloatConversionPass>();
            compiler.addPass<hkTypeVm::CheckTypeKindsPass>();

            UnitTest::Raii::CaptureThreadLogOutput capture; // Capture warning emitted by CheckTypeKindsPass (expected)

            hkViewPtr<hkTypeVm::Program> program = compiler.compile( hkReflect::getType<SrcType>(), hkReflect::getType<DstType>() );
            HK_TEST( !program );
        }
    }

    return 0;
}

HK_TEST_REGISTER( typeVm_main, "Fast", "Common/Test/UnitTest/Base/", __FILE__ );
HK_TEST_REGISTER( typeVm_regression, "Fast", "Common/Test/UnitTest/Base/", __FILE__ );

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
