// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT    : COMMON
// VISIBILITY : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>

#include <Common/Base/Reflect/TypeReg/hkTypeReg.h>
#include <Common/Base/Reflect/Util/hkReflectionCheckUtil.h>
#include <Common/Base/Types/Geometry/Aabb/hkAabb.h>
#include <Common/Base/Reflect/Builder/hkTypeBuilder.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>

#define DEBUG_LOG_IDENTIFIER "test.reflect.types"
#include <Common/Base/System/Log/hkLog.hxx>

int reflectedTypesTest_main()
{
    // Incremental type checker
    {
        hkReflect::Detail::CheckUtil::Config config;
        config.m_earlyOutOnFirstFailure = false;
        config.m_origin = &DEBUG_LOG_OBJECT;
        config.m_level = hkLog::Level::Error;
        config.m_isTypeReachableCallback = nullptr; // skip reachability tests
        hkReflect::Detail::CheckUtil::IncrementalChecker checker( config );

        const hkReflect::Type* const voidType = hkReflect::getType<void>();
        HK_TEST3( false == voidType->isSerializable(), "Expected void type to be non-serializable; fix test if that changed." );
        HK_TEST3( false == hkString::isNullOrEmpty( voidType->getName() ), "Expected void type to have a non-empty name (Opt::NAME); fix test if that changed." );

        // Verify that checker catches invalid flags
        {
            checker.reset();

            config.m_earlyOutOnFirstFailure = true;
            config.m_allowedFlags = ~hkUint32( hkReflect::Type::TYPE_NOT_SERIALIZABLE ); // cannot be non-serializable
            config.m_allowedOptMask = hkUlong( -1 ); // no restriction
            checker.setConfig( config );

            // Don't use void because there's a workaround for it. Use any other type which is not serializable.
            const hkReflect::Type* const dummyType = hkReflect::getType<hkPtrAndInt<void*,int,0xF>>();
            HK_TEST3( false == dummyType->isSerializable(), "Expected hkPtrAndInt type to be non-serializable; fix test if that changed." );

            hkResult result;
            {
                // The following test will fail reflection checks (expected); capture its output to remove warnings from the logs.
                UnitTest::Raii::CaptureThreadLogOutput capture;
                result = checker.addAndCheckTypes( hkArrayView<const hkReflect::Type* const>( &dummyType, 1 ) );
            }
            HK_TEST3( result.isFailure(), "void type has forbidden flag yet passed type checks." );
        }

        // Verify that checker catches invalid Opt
        {
            checker.reset();

            config.m_earlyOutOnFirstFailure = true;
            config.m_allowedFlags = hkUint32( -1 ); // no restriction
            config.m_allowedOptMask = ~hkUlong( hkReflect::Opt::NAME );
            checker.setConfig( config );

            hkResult result;
            {
                // The following test will fail reflection checks (expected); capture its output to remove warnings from the logs.
                UnitTest::Raii::CaptureThreadLogOutput capture;
                result = checker.addAndCheckTypes( hkArrayView<const hkReflect::Type* const>( &voidType, 1 ) );
            }
            HK_TEST3( result.isFailure(), "void type has forbidden Opt::NAME yet passed type checks." );
        }

        // Run checker on all registered built-in types
        {
            checker.reset();

            config.m_earlyOutOnFirstFailure = false;
            config.m_allowedFlags = hkUint32( -1 );
            config.m_allowedOptMask = hkUlong( -1 );
            checker.setConfig( config );

            for ( hkReflect::TypeRegIterator it( hkReflect::getTypeReg() ); it.advance(); )
            {
                auto* const type = it.current();
                checker.addAndCheckTypes( hkArrayView<const hkReflect::Type* const>( &type, 1 ) );
            }
            HK_TEST3( checker.getStatus().isSuccess(), "Reflected types checks failed, for details see errors in " DEBUG_LOG_IDENTIFIER " logs." );
        }
    }

    // Fields
    {
        using namespace hkReflect;
        {
            const RecordType* t = getType<hkAabb>()->asRecord();
            if ( HK_TEST( t ) )
            {
                const Type* ft = t->getDataFields()[1].getType();
                DataFieldDecl rf = ft->isField().asDataField();
                if ( HK_TEST( rf ) )
                {
                    HK_TEST( rf.getOffset() == t->getDataFields()[1].getOffset() );
                }
            }
        }
        {
            QualType t = getType<int>();
            FieldDecl rf = t->isField();
            HK_TEST(!rf);
        }
    }

    // Detection of loops in types
    {
        hkReflect::Detail::CheckUtil::Config config;

        config.m_earlyOutOnFirstFailure = false;
        config.m_origin = &DEBUG_LOG_OBJECT;
        config.m_level = hkLog::Level::Error;

        hkReflect::Detail::CheckUtil::IncrementalChecker checker(config);

        hkReflect::Type* dummyTypeA;
        hkReflect::Type* dummyTypeB;

        {
            hkReflect::TypeBuilder builder;
            builder.beginRecord("", nullptr);
            builder.addDeleteInfo();
            dummyTypeA = builder.allocate(hkMemHeapAllocator())->asRecord();
        }

        {
            hkReflect::TypeBuilder builder;
            builder.beginRecord("", nullptr);
            builder.addDeleteInfo();
            dummyTypeB = builder.allocate(hkMemHeapAllocator())->asRecord();
        }

        hkReflect::TypeDetail::setParent(dummyTypeA, dummyTypeB);
        hkReflect::TypeDetail::setParent(dummyTypeB, dummyTypeA);

        hkResult result;
        {
            // The following test will fail reflection checks (expected); capture its output to remove warnings from the logs.
            UnitTest::Raii::CaptureThreadLogOutput capture;
            result = checker.addAndCheckTypes(hkArrayView<const hkReflect::Type* const>(&dummyTypeA, 1));
        }
        HK_TEST3(result.isFailure(), "Reflected type contains a loop in its parent linked list yet passed type checks.");

        HK_TEST(hk::DeleteTypeInfo::deleteType(dummyTypeA));
        HK_TEST(hk::DeleteTypeInfo::deleteType(dummyTypeB));
    }

    return 0;
}

HK_TEST_REGISTER(reflectedTypesTest_main, "Fast", "Common/Test/UnitTest/Base/", __FILE__);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
