// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/Reflect/Attributes/hkAttributeParser.h>
#include <Common/Base/UnitTest/hkUnitTest.h>

namespace
{
    bool testContents(hkAttrData& attrData, hkArrayView<const char*> testValues)
    {
        int dataIdx = 0;
        for (int fieldIdx = 0; fieldIdx < attrData.getNumFields(); ++fieldIdx)
        {
            hkAttrData::Field field = attrData.getField(fieldIdx);

            if (field.isSet() != (testValues[dataIdx] != HK_NULL))
            {
                return false;
            }

            if (testValues[dataIdx] == HK_NULL)
            {
                ++dataIdx;
                continue;
            }

            if (field.getDecl().isSimple())
            {
                if (field.getValue() != testValues[dataIdx])
                {
                    return false;
                }
                ++dataIdx;
            }
            else
            {
                hkArrayView<const hkStringView> elements = field.getValuesArray();
                for (int elemIdx = 0; elemIdx < elements.getSize(); ++elemIdx)
                {
                    if (elements[elemIdx] != testValues[dataIdx])
                    {
                        return false;
                    }
                    ++dataIdx;
                }
            }
        }
        return true;
    }

    bool parseAttrs(const hkArrayView<const hkAttrData>& fields, const char* str, const hkArrayView< hkArrayView<const char*> > testValues)
    {
        hkAttributeParser parser(str);
        hkStringBuf errorBuf;

        for (int i = 0; i < fields.getSize(); ++i)
        {
            if (!parser.advance())
            {
                return false;
            }

            if (!parser.currentAttrName(errorBuf).isSuccess())
            {
                return false;
            }

            hkAttrData attrData = fields[i];
            if (!parser.currentAttrValue(attrData, errorBuf).isSuccess())
            {
                return false;
            }

            if (testValues[i].getSize() != 0 && !testContents(attrData, testValues[i]))
            {
                return false;
            }
        }
        return true;
    }

    bool parseAttr(const hkAttrData& fields, const char* str, hkArrayView<const char*> testValues = hkArrayView<const char*>())
    {
        return parseAttrs(hkArrayView<const hkAttrData>(&fields, 1), str, hkArrayView< hkArrayView<const char*> >(&testValues, 1));
    }

}

int attributeParserTests_main()
{
    // Empty attribute
    {
        hkAttrData fields;
        HK_TEST(parseAttr(fields, "Attr "));
        HK_TEST(parseAttr(fields, "hk::Attr"));
        HK_TEST(parseAttr(fields, "hk ::  Attr"));
        HK_TEST(parseAttr(fields, "hk ::  hk\t ::\nAttr"));
        HK_TEST(parseAttr(fields, "Attr()"));
        HK_TEST(!parseAttr(fields, "Attr(,)"));
        HK_TEST(!parseAttr(fields, "Attr("));
        HK_TEST(!parseAttr(fields, "Attr(123)"));
        HK_TEST(!parseAttr(fields, "1Attr"));
        HK_TEST(!parseAttr(fields, "Attr::"));
        HK_TEST(!parseAttr(fields, "A ttr"));
        HK_TEST(!parseAttr(fields, "hk::Reflect true"));
    }

    // Simple values
    {
        hkAttrData fields;
        fields.addDecl(hkAttrData::FieldDecl::simple("int"));
        fields.addDecl(hkAttrData::FieldDecl::simple("string1"));
        fields.addDecl(hkAttrData::FieldDecl::simple("string2"));

        HK_TEST(!parseAttr(fields, "Attr()"));
        HK_TEST(!parseAttr(fields, "Attr(something)"));
        HK_TEST(!parseAttr(fields, "Attr({with_braces}, something else)"));
        HK_TEST(!parseAttr(fields, "Attr(123, my string1, string1 = a, string 2 = b)"));
        HK_TEST(!parseAttr(fields, "Attr(123, my string1, string1 = a, string1 = b)"));
        HK_TEST(!parseAttr(fields, "Attr(123, my string1, string1 = a, string2 = b,)"));

        const char* v[] = {"123", "my string1", "\"my string2\""};
        hkArrayView<const char*> values = hkArrayViewT::make(v);
        HK_TEST(parseAttr(fields, "Attr(123, my string1, \"my string2\") ", values));
        HK_TEST(parseAttr(fields, "Attr(string1 = my string1, string2 = \"my string2\", 123)", values));
    }

    // Fixed array
    {
        hkAttrData fields;
        fields.addDecl(hkAttrData::FieldDecl::fixedArray("arr", 4));

        {
            const char* v[] = { "0", "1" };
            hkArrayView<const char*> values = hkArrayViewT::make( v );
            HK_TEST( parseAttr( fields, "Attr(0, 1)", values ) );
            HK_TEST( parseAttr( fields, "Attr({0, 1})", values ) );
        }

        HK_TEST(!parseAttr(fields, "Attr({0, 1, 2, 3)"));

        {
            const char* v[] = { "0", "1", "2", "3" };
            hkArrayView<const char*> values = hkArrayViewT::make( v );
            HK_TEST( parseAttr( fields, "Attr({0, 1, 2, 3})", values ) );
            HK_TEST( parseAttr( fields, "Attr(0, 1, 2, 3)", values ) );
        }
    }

    // Variable array
    {
        hkAttrData fields;
        fields.addDecl(hkAttrData::FieldDecl::variableArray("arr"));
        HK_TEST(!parseAttr(fields, "Attr({0, 1, 2, 3)"));
        HK_TEST(!parseAttr(fields, "Attr(0, 1, named = 2, 3)"));

        {
            hkArrayView<const char*> values;
            HK_TEST(parseAttr(fields, "Attr"));
            HK_TEST(parseAttr(fields, "Attr()"));
            HK_TEST(parseAttr(fields, "Attr({})"));
        }

        {
            const char* v[] = { "0", "1" };
            hkArrayView<const char*> values = hkArrayViewT::make( v );
            HK_TEST(parseAttr(fields, "Attr(0, 1)", values));
            HK_TEST(parseAttr(fields, "Attr({0, 1})", values));
        }

        {
            const char* v[] = {"0", "1", "2", "3"};
            hkArrayView<const char*> values = hkArrayViewT::make(v);
            HK_TEST(parseAttr(fields, "Attr({0, 1, 2, 3})", values));
            HK_TEST(parseAttr(fields, "Attr(0, 1, 2, 3)", values));
        }
        {
            const char* v[] = { "0", "1", "{2}", "3" };
            hkArrayView<const char*> values = hkArrayViewT::make(v);
            HK_TEST(parseAttr(fields, "Attr({0, 1, {2}, 3})", values));
            HK_TEST(parseAttr(fields, "Attr(0, 1, {2}, 3)", values));
        }
    }

    // complex record
    {
        hkAttrData fields;
        fields.addDecl(hkAttrData::FieldDecl::simple("int"));
        fields.addDecl(hkAttrData::FieldDecl::fixedArray("fixedArr", 2));
        fields.addDecl(hkAttrData::FieldDecl::simple("string1"));
        fields.addDecl(hkAttrData::FieldDecl::variableArray("arr"));

        HK_TEST(!parseAttr(fields, "Attr(1, 2, 3, 4)"));

        const char* v[] = {"1", "2", "2", "3", "4", "4", "4"};
        hkArrayView<const char*> values = hkArrayViewT::make(v);
        HK_TEST(parseAttr(fields, "Attr(1, {2, 2}, 3, {4, 4, 4})", values));
        HK_TEST(parseAttr(fields, "Attr(int=1, fixedArr={2, 2}, string1=3, arr={4, 4, 4})", values));
        HK_TEST(parseAttr(fields, "Attr(string1=3, arr={4, 4, 4}, 1, {2, 2})", values));
    }

    // optional fields
    {
        hkAttrData fields;
        fields.addDecl(hkAttrData::FieldDecl::simple("int"));
        fields.addDecl(hkAttrData::FieldDecl::fixedArray("fixedArr", 2, hkAttrData::FieldDecl::FLAG_OPTIONAL | hkAttrData::FieldDecl::FLAG_EXPLICIT));
        fields.addDecl(hkAttrData::FieldDecl::simple("string1", hkAttrData::FieldDecl::FLAG_OPTIONAL | hkAttrData::FieldDecl::FLAG_EXPLICIT));
        fields.addDecl(hkAttrData::FieldDecl::variableArray("arr"));

        HK_TEST(!parseAttr(fields, "Attr(1, {2, 2}, 3, {4, 4, 4})"));

        {
            const char* v[] = { "1", "2", "2", "3", nullptr, nullptr, nullptr };
            hkArrayView<const char*> values = hkArrayViewT::make( v );
            HK_TEST(parseAttr(fields, "Attr(int=1, fixedArr={2, 2}, string1=3)", values));
            HK_TEST(parseAttr(fields, "Attr(fixedArr={2, 2}, string1=3, 1)", values));
        }

        {
            const char* v[] = {"1", "2", "2", "3", "4", "4", "4"};
            hkArrayView<const char*> values = hkArrayViewT::make(v);
            HK_TEST(parseAttr(fields, "Attr(int=1, fixedArr={2, 2}, string1=3, arr={4, 4, 4})", values));
            HK_TEST(parseAttr(fields, "Attr(string1=3, 1, fixedArr={2, 2}, {4, 4, 4})", values));
            HK_TEST(parseAttr(fields, "Attr(fixedArr={2, 2}, 1, string1=3, {4, 4, 4})", values));
        }
        {
            const char* v[] = {"1", HK_NULL, "3", "4", "4", "4"};
            hkArrayView<const char*> values = hkArrayViewT::make(v);
            HK_TEST(parseAttr(fields, "Attr(int=1, string1=3, arr={4, 4, 4})", values));
            HK_TEST(parseAttr(fields, "Attr(string1=3, 1, {4, 4, 4})", values));
        }
        {
            const char* v[] = {"1", "2", "2", HK_NULL, "4", "4", "4"};
            hkArrayView<const char*> values = hkArrayViewT::make(v);
            HK_TEST(parseAttr(fields, "Attr(fixedArr={2, 2}, int=1, arr={4, 4, 4})", values));
            HK_TEST(parseAttr(fields, "Attr(fixedArr={2, 2}, 1, {4, 4, 4})", values));
        }
    }

    // multiple attributes
    {
        hkAttrData fields[2];

        fields[0].addDecl(hkAttrData::FieldDecl::simple("int"));
        fields[0].addDecl(hkAttrData::FieldDecl::fixedArray("fixedArr", 2, hkAttrData::FieldDecl::FLAG_OPTIONAL | hkAttrData::FieldDecl::FLAG_EXPLICIT));
        fields[0].addDecl(hkAttrData::FieldDecl::simple("string1", hkAttrData::FieldDecl::FLAG_OPTIONAL | hkAttrData::FieldDecl::FLAG_EXPLICIT));
        fields[0].addDecl(hkAttrData::FieldDecl::variableArray("arr"));

        fields[1].addDecl(hkAttrData::FieldDecl::simple("field1"));
        fields[1].addDecl(hkAttrData::FieldDecl::simple("field2"));

        {
            const char* v1[] = {"1", "2", "2", HK_NULL, "4", "4", "4"};
            const char* v2[] =  {"1", "2"};
            hkArrayView<const char*> v[] = { hkArrayViewT::make(v1), hkArrayViewT::make(v2) };

            const char* attrString = "Attr1(fixedArr={2, 2}, 1, {4, 4, 4}), Attr2(1, field2 = 2)";
            HK_TEST(parseAttrs(hkArrayViewT::make(fields), attrString, hkArrayViewT::make(v)));
        }
    }

    return 0;
}

HK_TEST_REGISTER(attributeParserTests_main, "Fast", "Common/Test/UnitTest/Base/", __FILE__);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
