// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/UnitTest/Memory/tracker_tests.h>
#include <Common/Base/Memory/Tracker/hkMemoryTracker.h>
#include <Common/Base/Memory/Tracker/hkMemoryTrackerSnapshot.h>
#include <Common/Base/Memory/Allocator/Malloc/hkMallocAllocator.h>
#include <Common/Base/Container/StringMap/hkStorageStringMap.h>
#include <Common/Base/Container/PointerMap/hkMap.hxx>
#include <Common/Base/Container/PointerMultiMap/hkMultiMap.hxx>
#include <Common/Base/Types/hkScopedPtr.h>
#include <Common/Base/Container/Set/hkSet.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>

namespace UnitTest
{
    void TrackerTestC::trackerHandler(const hkReflect::Var& var, hkMemoryTrackerSnapshot& snapshot)
    {
        TrackerTestC* obj = static_cast<TrackerTestC*>(var.getAddress());
        if (obj->m_ptrA)
        {
            snapshot.addLink(obj->m_ptrA);
        }
    }

// TODO this test fails when built in non-DLL config, because HK_EXCLUDE_FEATURE_MemoryTracker has to be defined (otherwise it won't link)
// it is only a problem for the one of our XXXXXCppTests project, so disabling it completely for Razzle is a hack
#if defined(HK_MEMORY_TRACKER_ENABLE) && !defined(HK_COMPILER_RAZZLE)
    static void testTrackerA()
    {
        // allocate some objects
        TrackerTestA* a = new TrackerTestA();
        a->m_ptrA = a;
        a->m_ptrB = new TrackerTestB();

        // take a tracker snapshot
        hkMallocAllocator alloc;
        hkMemoryTrackerSnapshot snapshot(&alloc);
        hkResult res = hkMemoryTracker::getInstance().getSnapshot(snapshot);
        if (!HK_TEST(res.isSuccess()))
        {
            return;
        }

        hkMemoryTrackerSnapshot::BlockIterator aBlockIt = snapshot.getBlockFor(a);

        if (HK_TEST(aBlockIt >= 0))
        {
            const hkMemoryTrackerSnapshot::Block& aBlock = snapshot.getBlock(aBlockIt);

            HK_TEST_EQ(aBlock.m_ptr, a);
            HK_TEST_EQ(aBlock.m_size, (int)sizeof(TrackerTestA));
            HK_TEST_EQ(aBlock.m_name, "TrackerTestA");
            HK_TEST_EQ(aBlock.m_num, -1);

            for (hkMemoryTrackerSnapshot::LinkIterator it = snapshot.getLinksFor(aBlockIt); it.advance();)
            {
                hkMemoryTrackerSnapshot::BlockIterator pointedBlockIt = it.current();
                const hkMemoryTrackerSnapshot::Block& pointedBlock = snapshot.getBlock(pointedBlockIt);
                if (HK_TEST(pointedBlock.m_type))
                {
                    HK_TEST(&pointedBlock == &aBlock || (hkStringView(pointedBlock.m_name) == "TrackerTestB" && pointedBlock.m_ptr == a->m_ptrB));
                    if (pointedBlock.m_ptr == a->m_ptrB)
                    {
                        HK_TEST(!snapshot.containsLinks(pointedBlockIt));
                    }
                }
            }
        }

        delete a->m_ptrB;
        delete a;
    }

    static void testTrackerB()
    {
        hkDisableError allowGlobalNew(0xf0c34ded); // new Data()

        // allocate some objects
        TrackerTestB* b = new TrackerTestB();
        b->m_data = new Data();
        b->m_array.pushBack(new TrackerTestC(new TrackerTestA()));
        b->m_array.pushBack(HK_NULL);
        TrackerTestC*& c = b->m_array[0];
        c->m_ptrA->m_ptrB = b;
        b->m_manual.m_ptrA = c->m_ptrA;

        // take a tracker snapshot
        hkMallocAllocator alloc;
        hkMemoryTrackerSnapshot snapshot(&alloc);
        hkResult res = hkMemoryTracker::getInstance().getSnapshot(snapshot);
        if (!HK_TEST(res.isSuccess()))
        {
            return;
        }

        hkMemoryTrackerSnapshot::BlockIterator bBlockIt = snapshot.getBlockFor(b);
        hkMemoryTrackerSnapshot::BlockIterator cBlockIt = snapshot.getBlockFor(c);
        hkMemoryTrackerSnapshot::BlockIterator aBlockIt = snapshot.getBlockFor(c->m_ptrA);
        hkMemoryTrackerSnapshot::BlockIterator arrayBlockIt = snapshot.getBlockFor(&b->m_array[0]);


        if (HK_TEST(bBlockIt>= 0 && cBlockIt>= 0 && aBlockIt>= 0))
        {
            const hkMemoryTrackerSnapshot::Block& bBlock = snapshot.getBlock(bBlockIt);
            HK_TEST_EQ(bBlock.m_ptr, b);
            HK_TEST_EQ(bBlock.m_size, (int)sizeof(TrackerTestB));
            HK_TEST_EQ(bBlock.m_name, "TrackerTestB");
            HK_TEST_EQ(bBlock.m_num, -1);

            // links from B
            bool arrayMatched = false, aMatched = false;
            for (hkMemoryTrackerSnapshot::LinkIterator it = snapshot.getLinksFor(bBlockIt); it.advance();)
            {
                hkMemoryTrackerSnapshot::BlockIterator pointedBlockIt = it.current();
                const hkMemoryTrackerSnapshot::Block& pointedBlock = snapshot.getBlock(pointedBlockIt);
                if (HK_TEST(pointedBlock.m_type))
                {
                    HK_TEST(pointedBlockIt == arrayBlockIt || pointedBlockIt == aBlockIt);
                    if (pointedBlockIt == arrayBlockIt) // array buffer
                    {
                        const hkMemoryTrackerSnapshot::Block& arrayBlock = snapshot.getBlock(arrayBlockIt);

                        HK_TEST_EQ(arrayBlock.m_ptr, &b->m_array[0]);
                        HK_TEST_EQ(arrayBlock.m_size, (int)sizeof(TrackerTestC*)*b->m_array.getSize());
                        
                        HK_TEST_EQ(arrayBlock.m_name, "buffer_hkArray");
                        HK_TEST_EQ(arrayBlock.m_num, b->m_array.getSize());

                        int counter = 0;
                        for (hkMemoryTrackerSnapshot::LinkIterator arrayIt = snapshot.getLinksFor(arrayBlockIt); arrayIt.advance();)
                        {
                            hkMemoryTrackerSnapshot::BlockIterator elementIt = arrayIt.current();
                            HK_TEST_EQ(elementIt, cBlockIt);
                            counter++;
                        }
                        HK_TEST(counter == 1); // all the block pointers were found
                        arrayMatched = true;
                    }
                    else if (pointedBlockIt == aBlockIt)
                    {
                        aMatched = true;
                    }
                }
            }
            HK_TEST(aMatched);
            HK_TEST(arrayMatched);

            // C should contain just one tracked link
            const hkMemoryTrackerSnapshot::Block& element = snapshot.getBlock(cBlockIt);
            HK_TEST_EQ(element.m_ptr, b->m_array[0]);
            HK_TEST_EQ(element.m_num, -1);
            hkMemoryTrackerSnapshot::LinkIterator cLinks = snapshot.getLinksFor(cBlockIt);
            if (HK_TEST(cLinks.advance()))
            {
                HK_TEST_EQ(cLinks.current(), aBlockIt);
                HK_TEST(!cLinks.advance());
            }


            // A should contain just one link
            hkMemoryTrackerSnapshot::LinkIterator aLinks = snapshot.getLinksFor(aBlockIt);
            if (HK_TEST(aLinks.advance()))
            {
                HK_TEST_EQ(aLinks.current(), bBlockIt);
                HK_TEST(!aLinks.advance());
            }
        }

        delete c->m_ptrA;
        delete c;
        delete b->m_data;
        delete b;
    }

    static hkResult getAllLinks(hkMemoryTrackerSnapshot& snapshot, const void* map, hkSet<const char*>& links, int& numLinks)
    {
        hkResult res = hkMemoryTracker::getInstance().getSnapshot(snapshot);
        if (!HK_TEST(res.isSuccess())) { return HK_FAILURE; }

        hkMemoryTrackerSnapshot::BlockIterator mapIt = snapshot.getBlockFor(map);
        if (!HK_TEST(mapIt != snapshot.end())) { return HK_FAILURE; }

        hkMemoryTrackerSnapshot::LinkIterator linkIt = snapshot.getLinksFor(mapIt);
        numLinks = 0;
        while (linkIt.advance())
        {
            const hkMemoryTrackerSnapshot::Block& block = snapshot.getBlock(linkIt.current());
            // Add only links to strings
            if (HK_TEST(block.m_name) &&
                (hkString::strCmp("buffer_hkStringPtr", block.m_name) == 0 ||
                hkString::strCmp("buffer_hkStorageStringMap_elem", block.m_name) == 0))
            {
                links.insert((const char*)snapshot.getBlock(linkIt.current()).m_ptr);
                ++numLinks;
            }
        }
        return HK_SUCCESS;
    }

    static bool cmpLess(const char* a, const char* b)
    {
        return hkString::strCmp(a, b) < 0;
    }

    static void testTrackerMaps()
    {
        hkArray<hkStringPtr> testStrings;
        HK_MEMORY_TRACKER_ADD_EXPLICIT(hkArray<hkStringPtr>, &testStrings, sizeof(hkArray<hkStringPtr>), false);

        testStrings.reserve(255);
        for (int i = 1; i < 32; ++i)
        {
            hkStringBuf string;
            string.appendRepeat(20, (char)i);
            testStrings.pushBack(string.cString());
        }

        {
            hkScopedPtr< hkMap<const char*, hkUlong> > map(new hkMap<const char*, hkUlong>(testStrings.getSize()));
            for (int i = 0; i < testStrings.getSize(); ++i)
            {
                map->insert(testStrings[i].cString(), i);
            }
            hkMallocAllocator alloc;
            hkMemoryTrackerSnapshot snapshot(&alloc);
            hkSet<const char*> links;
            int numLinks;
            if (HK_TEST(getAllLinks(snapshot, map.get(), links, numLinks).isSuccess()))
            {
                // Each string should be pointed once.
                HK_TEST_EQ(numLinks, testStrings.getSize());
                HK_TEST_EQ(links.getSize(), testStrings.getSize());
                for (int i = 0; i < testStrings.getSize(); ++i)
                {
                    links.contains(testStrings[i].cString());
                }
            }
        }

        {
            hkScopedPtr< hkMap<const char*, const char*> > map(new hkMap<const char*, const char*>(testStrings.getSize()));
            for (int i = 0; i < testStrings.getSize(); ++i)
            {
                map->insert(testStrings[i].cString(), testStrings[i].cString());
            }
            hkMallocAllocator alloc;
            hkMemoryTrackerSnapshot snapshot(&alloc);
            hkSet<const char*> links;
            int numLinks;
            if (HK_TEST(getAllLinks(snapshot, map.get(), links, numLinks).isSuccess()))
            {
                // Each string should be pointed twice.
                HK_TEST_EQ(numLinks, 2 * testStrings.getSize());
                HK_TEST_EQ(links.getSize(), testStrings.getSize());
                for (int i = 0; i < testStrings.getSize(); ++i)
                {
                    HK_TEST(links.contains(testStrings[i].cString()));
                }
            }
        }

        {
            hkScopedPtr< hkMultiMap<const char*, const char*> > map(new hkMultiMap<const char*, const char*>(testStrings.getSize()));
            for (int i = 0; i < testStrings.getSize(); ++i)
            {
                for (int j = 0; j < 5; ++j)
                {
                    map->insert(testStrings[i].cString(), testStrings[(i + j) % testStrings.getSize()].cString());
                }
            }
            hkMallocAllocator alloc;
            hkMemoryTrackerSnapshot snapshot(&alloc);
            hkSet<const char*> links;
            int numLinks;
            if (HK_TEST(getAllLinks(snapshot, map.get(), links, numLinks).isSuccess()))
            {
                // Each string should be pointed 10 times.
                HK_TEST_EQ(numLinks, 10 * testStrings.getSize());
                HK_TEST_EQ(links.getSize(), testStrings.getSize());
                for (int i = 0; i < testStrings.getSize(); ++i)
                {
                    HK_TEST(links.contains(testStrings[i].cString()));
                }
            }
        }

        {
            hkScopedPtr< hkStringMap<const char*> > map(new hkStringMap<const char*>());
            for (int i = 0; i < testStrings.getSize(); ++i)
            {
                map->insert(testStrings[i].cString(), testStrings[i].cString());
            }
            hkMallocAllocator alloc;
            hkMemoryTrackerSnapshot snapshot(&alloc);
            hkSet<const char*> links;
            int numLinks;
            if (HK_TEST(getAllLinks(snapshot, map.get(), links, numLinks).isSuccess()))
            {
                // Each string should be pointed twice.
                HK_TEST_EQ(numLinks, 2 * testStrings.getSize());
                HK_TEST_EQ(links.getSize(), testStrings.getSize());
                for (int i = 0; i < testStrings.getSize(); ++i)
                {
                    HK_TEST(links.contains(testStrings[i].cString()));
                }
            }
        }

        {
            hkScopedPtr< hkStorageStringMap<const char*> > map(new hkStorageStringMap<const char*>());
            for (int i = 0; i < testStrings.getSize(); ++i)
            {
                map->insert(testStrings[i].cString(), testStrings[i].cString());
            }
            hkMallocAllocator alloc;
            hkMemoryTrackerSnapshot snapshot(&alloc);
            hkSet<const char*> links;
            int numLinks;
            if (HK_TEST(getAllLinks(snapshot, map.get(), links, numLinks).isSuccess()))
            {
                // Each string should be pointed twice.
                HK_TEST_EQ(numLinks, 2 * testStrings.getSize());
                HK_TEST_EQ(links.getSize(), 2 * testStrings.getSize());
                for (hkSet<const char*>::Iterator it = links.getIterator(); links.isValid(it); it = links.getNext(it))
                {
                    HK_TEST(hkAlgorithm::binarySearch(links.getElement(it), testStrings.begin(), testStrings.getSize(), cmpLess) != -1);
                }
            }
        }

        HK_MEMORY_TRACKER_REMOVE(&testStrings);
    }

    static void testTrackerTemplate()
    {
        hkScopedPtr< TrackerTestTemplate<int> > obj1(new TrackerTestTemplate<int>);
        int* intPtr = hkMemHeapCreate<int>();
        obj1->m_ptr = intPtr;

        hkScopedPtr< TrackerTestTemplate<void*> > obj2(new TrackerTestTemplate<void*>);
        obj2->m_value = intPtr;
        void** ptrToPtr = hkMemHeapCreate<void*>();
        obj2->m_ptr = ptrToPtr;

        hkMallocAllocator alloc;
        hkMemoryTrackerSnapshot snapshot(&alloc);
        hkResult res = hkMemoryTracker::getInstance().getSnapshot(snapshot);
        HK_TEST(res.isSuccess());

        // TrackerTestTemplate<int>
        {
            hkMemoryTrackerSnapshot::BlockIterator objIt = snapshot.getBlockFor(obj1.get());
            const hkMemoryTrackerSnapshot::Block& block = snapshot.getBlock(objIt); block;
            hkMemoryTrackerSnapshot::LinkIterator linkIt = snapshot.getLinksFor(objIt);
            HK_TEST(linkIt.advance());

            auto ptdBlock = snapshot.getBlock(linkIt.current());
            HK_TEST_EQ(ptdBlock.m_ptr, intPtr);

            HK_TEST(!linkIt.advance());
        }

        // TrackerTestTemplate<void*>
        {
            hkMemoryTrackerSnapshot::BlockIterator objIt = snapshot.getBlockFor(obj2.get());
            const hkMemoryTrackerSnapshot::Block& block = snapshot.getBlock(objIt); block;
            hkMemoryTrackerSnapshot::LinkIterator linkIt = snapshot.getLinksFor(objIt);

            int numLinks = 0;
            while (linkIt.advance())
            {
                ++numLinks;

                auto ptdBlock = snapshot.getBlock(linkIt.current());
                HK_TEST(ptdBlock.m_ptr == intPtr || ptdBlock.m_ptr == ptrToPtr);
            }
            HK_TEST_EQ(numLinks, 2);
        }
        hkMemHeapDestroy(intPtr);
        hkMemHeapDestroy(ptrToPtr);
    }

    static void testTrackerManual()
    {
        hkScopedPtr<hkReferencedObject> ptd(new hkReferencedObject);
        hkScopedPtr< UnitTest::ManualTrack<hkReferencedObject, 2> > obj(new UnitTest::ManualTrack<hkReferencedObject, 2>(ptd.get()));
        hkMallocAllocator alloc;
        hkMemoryTrackerSnapshot snapshot(&alloc);
        hkResult res = hkMemoryTracker::getInstance().getSnapshot(snapshot);
        HK_TEST(res.isSuccess());
        int numLinks = 0;
        hkMemoryTrackerSnapshot::BlockIterator objIt = snapshot.getBlockFor(obj.get());
        const hkMemoryTrackerSnapshot::Block& block = snapshot.getBlock(objIt); block;
        for (hkMemoryTrackerSnapshot::LinkIterator linkIt = snapshot.getLinksFor(objIt); linkIt.advance();)
        {
            const void* linked = snapshot.getBlock(linkIt.current()).m_ptr;
            HK_TEST(linked == ptd.get() || linked == obj->m_arr.begin());
            ++numLinks;
        }
        HK_TEST(numLinks == 3);
    }
#endif
}

static int trackerTestMain()
{
// TODO re-enable on razzle when fixed.
#if defined(HK_MEMORY_TRACKER_ENABLE) && !defined(HK_COMPILER_RAZZLE)
    if (hkMemoryTracker::getInstancePtr())
    {
        UnitTest::testTrackerA();
        UnitTest::testTrackerB();
        UnitTest::testTrackerMaps();
        UnitTest::testTrackerTemplate();
        UnitTest::testTrackerManual();
    }
#endif
    return 0;
}

HK_TEST_REGISTER(trackerTestMain, "Slow", "Common/Test/UnitTest/Base/", __FILE__);

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
