// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/Math/Vector/hkVectorSort.h>
#include <Common/Base/System/Hardware/hkHardwareInfo.h>
#include <cstdlib>
#include <cstdio>

bool areVectorsReversed(const hkVector4& a, const hkVector4& b)
{
    return a.getComponent<0>() == b.getComponent<3>() &&
           a.getComponent<1>() == b.getComponent<2>() &&
           a.getComponent<2>() == b.getComponent<1>() &&
           a.getComponent<3>() == b.getComponent<0>();
}

void compactAndCountTest(const hkIntVector& values, int (*compactFunc)(const hkVector4Comparison&, const hkIntVector&, hkIntVector* ) )
{
    // test compactAndCount
    {
        static const hkVector4ComparisonMask::Mask masks[] = {  hkVector4ComparisonMask::MASK_X,
            hkVector4ComparisonMask::MASK_Y,
            hkVector4ComparisonMask::MASK_Z,
            hkVector4ComparisonMask::MASK_W
        };


        for(int type=0; type<16; type++)
        {
            const hkVector4ComparisonMask::Mask mask =  hkVector4ComparisonMask::Mask(  ((type & 1) ? hkVector4ComparisonMask::MASK_X : 0) |
                ((type & 2) ? hkVector4ComparisonMask::MASK_Y : 0) |
                ((type & 4) ? hkVector4ComparisonMask::MASK_Z : 0) |
                ((type & 8) ? hkVector4ComparisonMask::MASK_W : 0)
                );

            hkIntVector compactedValues;

            hkVector4Comparison comp; comp.set(mask);
            const int               numBits = compactFunc(comp, values, &compactedValues);

            int numBitsCheck = 0;
            for(int temp=mask; temp; temp >>= 1) if(temp & 1) numBitsCheck++;
            // Invalid bit count
            HK_TEST(numBits == numBitsCheck);

            int j = 0;
            for(int i=0; i<4; ++i)
            {
                if(mask & masks[i])
                {
                    // Invalid element?
                    const bool validElementHasSameValue = values.getU32( i ) == compactedValues.getU32( j );
                    HK_TEST( validElementHasSameValue );
                    j++;
                    if ( !validElementHasSameValue ) return;
                }
            }

            // Unmatched elements and bit count?
            HK_TEST( j == numBits );
        }
    }
}

int vectorSort_main()
{
    hkUnitTest::Prng random(1577200837);
    const int k_numTests = 100;

    for (int testNo = 0; testNo < k_numTests; ++testNo)
    {

        // test sorting
        {
            // floating point

            // ascending
            hkVector4 vA; random.nextVector(vA);
            hkVector4 vB; random.nextVector(vB);
            hkVector4Comparison comp = hkVectorSort::vectorCompareT<HK_SORT_ASCENDING>( vA, vB  );
            hkVector4ComparisonMask::Mask mask = comp.getMask();

            hkVector4ComparisonMask::Mask verificationMask;
            verificationMask =  (hkVector4ComparisonMask::Mask) (   (vA.getComponent<0>() < vB.getComponent<0>() ? hkVector4ComparisonMask::MASK_X : hkVector4ComparisonMask::MASK_NONE) |
                                                                    (vA.getComponent<1>() < vB.getComponent<1>() ? hkVector4ComparisonMask::MASK_Y : hkVector4ComparisonMask::MASK_NONE) |
                                                                    (vA.getComponent<2>() < vB.getComponent<2>() ? hkVector4ComparisonMask::MASK_Z : hkVector4ComparisonMask::MASK_NONE) |
                                                                    (vA.getComponent<3>() < vB.getComponent<3>() ? hkVector4ComparisonMask::MASK_W : hkVector4ComparisonMask::MASK_NONE) );
            HK_TEST(mask == verificationMask);

            // descending
            comp = hkVectorSort::vectorCompareT<HK_SORT_DESCENDING>( vA, vB  );
            mask = comp.getMask();

            verificationMask =  (hkVector4ComparisonMask::Mask) (   (vA.getComponent<0>() > vB.getComponent<0>() ? hkVector4ComparisonMask::MASK_X : hkVector4ComparisonMask::MASK_NONE) |
                                                                    (vA.getComponent<1>() > vB.getComponent<1>() ? hkVector4ComparisonMask::MASK_Y : hkVector4ComparisonMask::MASK_NONE) |
                                                                    (vA.getComponent<2>() > vB.getComponent<2>() ? hkVector4ComparisonMask::MASK_Z : hkVector4ComparisonMask::MASK_NONE) |
                                                                    (vA.getComponent<3>() > vB.getComponent<3>() ? hkVector4ComparisonMask::MASK_W : hkVector4ComparisonMask::MASK_NONE));
            HK_TEST(mask == verificationMask);

            // integer

            hkIntVector viA; viA.loadAsFloat32BitRepresentation(vA);
            hkIntVector viB; viB.loadAsFloat32BitRepresentation(vB);

            // ascending
            comp = hkVectorSort::vectorCompareT<HK_SORT_ASCENDING>( viA, viB );
            mask = comp.getMask();
            verificationMask =  (hkVector4ComparisonMask::Mask) (   (viA.getComponent<0>() < viB.getComponent<0>() ? hkVector4ComparisonMask::MASK_X : hkVector4ComparisonMask::MASK_NONE) |
                                                                    (viA.getComponent<1>() < viB.getComponent<1>() ? hkVector4ComparisonMask::MASK_Y : hkVector4ComparisonMask::MASK_NONE) |
                                                                    (viA.getComponent<2>() < viB.getComponent<2>() ? hkVector4ComparisonMask::MASK_Z : hkVector4ComparisonMask::MASK_NONE) |
                                                                    (viA.getComponent<3>() < viB.getComponent<3>() ? hkVector4ComparisonMask::MASK_W : hkVector4ComparisonMask::MASK_NONE) );
            HK_TEST(mask == verificationMask);

            // descending
            comp = hkVectorSort::vectorCompareT<HK_SORT_DESCENDING>( viA, viB );
            mask = comp.getMask();
            verificationMask =  (hkVector4ComparisonMask::Mask) (   (viA.getComponent<0>() > viB.getComponent<0>() ? hkVector4ComparisonMask::MASK_X : hkVector4ComparisonMask::MASK_NONE) |
                                                                    (viA.getComponent<1>() > viB.getComponent<1>() ? hkVector4ComparisonMask::MASK_Y : hkVector4ComparisonMask::MASK_NONE) |
                                                                    (viA.getComponent<2>() > viB.getComponent<2>() ? hkVector4ComparisonMask::MASK_Z : hkVector4ComparisonMask::MASK_NONE) |
                                                                    (viA.getComponent<3>() > viB.getComponent<3>() ? hkVector4ComparisonMask::MASK_W : hkVector4ComparisonMask::MASK_NONE) );
            HK_TEST(mask == verificationMask);
        }

        // test sorting
        {
            hkVector4 vIn0 , vIn1;
            random.nextVector(vIn0); random.nextVector(vIn1);
            hkVector4 vOut0, vOut1;

            // ascending
            hkVector4 keys = hkVector4::getConstant<HK_QUADREAL_8421>();
            hkVector4 sortedKeys;
            hkVectorSort::sort<HK_SORT_ASCENDING>(keys, &sortedKeys);
            hkVector4 verificationKeys = hkVector4::getConstant<HK_QUADREAL_1248>();
            hkVector4Comparison comp;

            // keys only
            comp = sortedKeys.equal(verificationKeys);
            HK_TEST( comp.getMask() == hkVector4ComparisonMask::MASK_XYZW );

            // one vector
            hkVectorSort::sort<HK_SORT_ASCENDING>(keys, vIn0, &sortedKeys, &vOut0);
            HK_TEST( areVectorsReversed(vIn0, vOut0) );
            vOut0 = vIn0;

            // two vectors
            hkVectorSort::sort<HK_SORT_ASCENDING>(keys, vIn0, vIn1, &sortedKeys, &vOut0, &vOut1);
            HK_TEST( areVectorsReversed(vIn0, vOut0) );
            vOut0 = vIn0;
            HK_TEST( areVectorsReversed(vIn1, vOut1) );
            vOut1 = vIn1;

            // descending
            keys = hkVector4::getConstant<HK_QUADREAL_1248>();
            hkVectorSort::sort<HK_SORT_DESCENDING>(keys, &sortedKeys);
            verificationKeys = hkVector4::getConstant<HK_QUADREAL_8421>();

            //keys only
            comp = sortedKeys.equal(verificationKeys);
            HK_TEST( comp.getMask() == hkVector4ComparisonMask::MASK_XYZW );

            // one vector
            hkVectorSort::sort<HK_SORT_DESCENDING>(keys, vIn0, &sortedKeys, &vOut0);
            HK_TEST( areVectorsReversed(vIn0, vOut0) );
            vOut0 = vIn0;

            // two vectors
            hkVectorSort::sort<HK_SORT_DESCENDING>(keys, vIn0, vIn1, &sortedKeys, &vOut0, &vOut1);
            HK_TEST( areVectorsReversed(vIn0, vOut0) );
            vOut0 = vIn0;
            HK_TEST( areVectorsReversed(vIn1, vOut1) );
            vOut1 = vIn1;
        }
    }

    // test compactAndCount
    {
        hkIntVector values; values.set( 419, 421, 431, 433 );

        // generic impl.
        compactAndCountTest( values, &( hkVectorSort::GenericSimd::compactAndCount ) );

        // sse42 impl. (temporarily disabled to help build go green)
        //#if (defined(HK_PLATFORM_WIN32) || defined(HK_PLATFORM_PS4) || defined(HK_PLATFORM_DURANGO)) && defined(HK_SSE_VERSION) && (HK_CONFIG_SIMD == HK_CONFIG_SIMD_ENABLED) &&  defined(HK_ARCH_INTEL )
        //if ( hkHardwareInfo::hasFeatureSet( hkHardwareInfo::SSSE3, hkHardwareInfo::POPCNT ) )
        //{
        //  compactAndCountTest( values, &( hkVectorSort::SSE42::compactAndCount ) );
        //}
        //#endif

        #ifdef HK_INT_VECTOR_NATIVE_PERMUTE8
        compactAndCountTest( values, &( hkVectorSort::Shuffle8Bit::compactAndCount ) );
        #endif
    }

    return 0;
}

HK_TEST_REGISTER(vectorSort_main, "Fast", "Common/Test/UnitTest/Base/", "UnitTest/Math/Linear/vectorSort.cpp"     );

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
