// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>

#include <Common/Base/Math/LargeInt/hkLargeIntTypes.h>

// C4611 interaction between '_setjmp' and C++ object destruction is non-portable yes we know.
HK_DETAIL_DIAG_MSVC_PUSH()
HK_DETAIL_DIAG_MSVC_OFF(4611)

//  Test the set operations

static void testSetOps()
{
    // Component set
    {
        hkInt64Vector4 v;
        v.setZero();
        v.setComponent<0>(1);
        v.setComponent<1>(2);
        v.setComponent<2>(3);
        v.setComponent<3>(4);
#if !(defined(HK_PLATFORM_ANDROID) && defined(HK_COMPILER_GCC) && defined(HK_COMPILER_HAS_INTRINSICS_NEON))
        
        HK_TEST(v.getComponent<0>() == 1);
        HK_TEST(v.getComponent<1>() == 2);
        HK_TEST(v.getComponent<2>() == 3);
        HK_TEST(v.getComponent<3>() == 4);
#endif
        v.set(4, 3, 2, 1);
        HK_TEST(v.getComponent<0>() == 4);
        HK_TEST(v.getComponent<1>() == 3);
        HK_TEST(v.getComponent<2>() == 2);
        HK_TEST(v.getComponent<3>() == 1);

        hkInt128 iv;
        iv.setFromInt32(100);
        HK_TEST(iv.getDoubleWord(0) == 100);
        HK_TEST(iv.getDoubleWord(1) == 0);

        iv.setFromInt64(100);
        HK_TEST(iv.getDoubleWord(0) == 100);
        HK_TEST(iv.getDoubleWord(1) == 0);

        iv.setFromInt32(-100);
        HK_TEST(iv.getDoubleWord(0) == -100);
        HK_TEST(iv.getDoubleWord(1) == -1);

        iv.setFromInt64(-100);
        HK_TEST(iv.getDoubleWord(0) == -100);
        HK_TEST(iv.getDoubleWord(1) == -1);
    }
}

//
//  Test the arithmetic operations

static void testArithmeticOps()
{
    // Addition
    {
        hkInt64Vector4 a;   a.set(1, 2, 3, 4);
        hkInt64Vector4 b;   b.set(-5, -6, -7, -8);

        hkInt64Vector4 c;   c.setSub(a, b);
        HK_TEST((c.getComponent<0>() == 6) &&
            (c.getComponent<1>() == 8) &&
            (c.getComponent<2>() == 10) &&
            (c.getComponent<3>() == 12));

        hkInt128 ia;
        hkInt128 ib;
#if defined(HK_PLATFORM_ANDROID)
        ia.setFromInt64(0xFFFFFFFFFFFFLL);
        ib.setFromInt64(0xFFFFFFFFFFFFLL);
#else
        ia.setFromInt64(0xFFFFFFFFFFFFL);
        ib.setFromInt64(0xFFFFFFFFFFFFL);
#endif

        hkInt128 ic;    ic.setAdd(ia, ib);
        HK_TEST((ic.getDoubleWord(0) == 0x1FFFFFFFFFFFELL) &&
            (ic.getDoubleWord(1) == 0));

        ia.setFromInt32(-100);
        ib.setFromInt32(100);
        ic.setAdd(ia, ib);
        HK_TEST((ic.getDoubleWord(0) == 0) &&
            (ic.getDoubleWord(1) == 0));

        ia.setFromInt32(-100);
        ib.setFromInt32(2);
        ic.setAdd(ia, ib);
        HK_TEST((ic.getDoubleWord(0) == -98) &&
            (ic.getDoubleWord(1) == -1));

        ia.setFromInt32(-100);
        ib.setFromInt32(102);
        ic.setAdd(ia, ib);
        HK_TEST((ic.getDoubleWord(0) == 2) &&
            (ic.getDoubleWord(1) == 0));

#if defined(HK_PLATFORM_ANDROID)
        ia.setFromInt64(0xF904DA07B303383FLL);  // -0503037531755694017 = -0x06FB25F84CFCC7C1
        ib.setFromInt64(0x13678E394909DE00LL);  // +1398242586011491840 = +0x13678E394909DE00
#else
        ia.setFromInt64(0xF904DA07B303383FL);   // -0503037531755694017 = -0x06FB25F84CFCC7C1
        ib.setFromInt64(0x13678E394909DE00L);   // +1398242586011491840 = +0x13678E394909DE00
#endif

        ic.setAdd(ia, ib);                      // +0895205054255797823 = +0x0C6C6840FC0D163F
        HK_TEST((ic.getDoubleWord(0) == 0x0C6C6840FC0D163FLL) &&
            (ic.getDoubleWord(1) == 0));
    }

    // Subtraction
    {
        hkInt64Vector4 a;   a.set(1, 2, 3, 4);
        hkInt64Vector4 b;   b.set(5, 6, 7, 8);

        hkInt64Vector4 c;   c.setSub(b, a);
        HK_TEST((c.getComponent<0>() == 4) &&
            (c.getComponent<1>() == 4) &&
            (c.getComponent<2>() == 4) &&
            (c.getComponent<3>() == 4));
    }

    // Multiplication
    {
        hkIntVector a;  a.set(1, 2, 3, 4);
        hkIntVector b;  b.set(5, 6, 7, 8);

        hkInt64Vector4 c;   c.setMul(a, b);
        HK_TEST((c.getComponent<0>() == 5) &&
            (c.getComponent<1>() == 12) &&
            (c.getComponent<2>() == 21) &&
            (c.getComponent<3>() == 32));
    }

    // 128-bit add
    {
        hkInt128 a; a.setFromUint64((hkUint64)-1);
        hkInt128 b; b.setFromUint64((hkUint64)-1);
        hkInt128 ab;    ab.setAdd(a, b);
        HK_TEST(ab.getDoubleWord(1) == 1);
        HK_TEST(ab.getDoubleWord(0) == 0xFFFFFFFFFFFFFFFELL);
    }

    // Division
    {
        struct TestCase
        {
            hkUint64 a[4];
            hkUint64 b[4];
            hkUint64 q[4];
            hkUint64 r[4];
        } testCases[] = {
            {
                { 0x0000000000000000ULL, 0x00000005c8bb1678ULL, 0xf861dc550f15dee6ULL, 0x307ab9ecfc97e0bcULL },
                { 0x0000000000000000ULL, 0x0000000000002c42ULL, 0x830d3c6317170ac7ULL, 0xc07f207c5e198875ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000217496ULL },
                { 0x0000000000000000ULL, 0x0000000000001f69ULL, 0x667d99d6b2f33a3eULL, 0x9b3d351fb0b7e82eULL },
            },
            {
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000007ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000010003ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000007ULL },
            },
            {
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000001ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL },
            },
            {
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000003ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000002ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000001ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000001ULL },
            },
            {
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000003ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000003ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000001ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL },
            },
            {
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000003ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000004ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000003ULL },
            },
            {
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0xFFFFFFFFFFFFFFFFULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL },
            },
            {
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0xFFFFFFFFFFFFFFFFULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000001ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0xFFFFFFFFFFFFFFFFULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL },
            },
            {
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0xFFFFFFFFFFFFFFFFULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0xFFFFFFFFFFFFFFFFULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000001ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL },
            },
            {
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0xFFFFFFFFFFFFFFFFULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000003ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x5555555555555555ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL },
            },
            {
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0xFFFFFFFFFFFFFFFFULL, 0xFFFFFFFFFFFFFFFFULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000001ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0xFFFFFFFFFFFFFFFFULL, 0xFFFFFFFFFFFFFFFFULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL },
            },
            {
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0xFFFFFFFFFFFFFFFFULL, 0xFFFFFFFFFFFFFFFFULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0xFFFFFFFFFFFFFFFFULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000001ULL, 0x0000000000000001ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL },
            },
            {
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0xFFFFFFFFFFFFFFFEULL, 0xFFFFFFFFFFFFFFFFULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0xFFFFFFFFFFFFFFFFULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0xFFFFFFFFFFFFFFFFULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0xFFFFFFFFFFFFFFFEULL },
            },
            {
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000012345678ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000009abcULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000001e1eULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000002c70ULL },
            },
            {
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000080000001ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000040007000ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000001ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x000000003FFF9001ULL },
            },
            {
                { 0x7FFFFFFFFFFFFFFFULL, 0xFFFFFFFFFFFFFFFFULL, 0xFFFFFFFFFFFFFFFFULL, 0xFFFFFFFFFFFFFFFFULL },
                { 0x7FFFFFFFFFFFFFFFULL, 0xFFFFFFFFFFFFFFFFULL, 0xFFFFFFFFFFFFFFFFULL, 0xFFFFFFFFFFFFFFFFULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000001ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL },
            },
            {
                { 0x7FFFFFFFFFFFFFFFULL, 0xFFFFFFFFFFFFFFFFULL, 0xFFFFFFFFFFFFFFFFULL, 0xFFFFFFFFFFFFFFFFULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000001ULL },
                { 0x7FFFFFFFFFFFFFFFULL, 0xFFFFFFFFFFFFFFFFULL, 0xFFFFFFFFFFFFFFFFULL, 0xFFFFFFFFFFFFFFFFULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL },
            },
            {
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000080000000ULL, 0xFFFFFFFE00000000ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x80000000FFFFFFFFULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x00000000FFFFFFFFULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x7FFFFFFFFFFFFFFFULL },
            },
            {
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000080000000ULL, 0x0000000000000003ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000020000000ULL, 0x0000000000000001ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000003ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000020000000ULL, 0x0000000000000000ULL },
            },
            {
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x7FFFFFFF80000000ULL, 0x0000000000000000ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000080000000ULL, 0x0000000000000001ULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x0000000000000000ULL, 0x00000000FFFFFFFEULL },
                { 0x0000000000000000ULL, 0x0000000000000000ULL, 0x000000007FFFFFFFULL, 0xFFFFFFFF00000002ULL }
            }
        };

        hkInt256 int32Max;
        int32Max.setZero();
        int32Max.setDoubleWord(0, HK_INT32_MAX);

        hkInt256 zero;
        zero.setZero();

        // Division by zero must assert
        HK_TEST_ASSERT(0x72638521, hkInt256::computeDiv32(int32Max, zero));
        HK_TEST_ASSERT(0x72638521, hkInt256::computeDiv32(zero, zero));

        for (int i = 0; i < sizeof(testCases) / sizeof(testCases[0]); ++i)
        {
            hkInt256 a, b, q, r, qRef, rRef;

            for (int j = 0; j < 4; ++j)
            {
                a.setDoubleWord(j, testCases[i].a[3 - j]);
                b.setDoubleWord(j, testCases[i].b[3 - j]);
                qRef.setDoubleWord(j, testCases[i].q[3 - j]);
                rRef.setDoubleWord(j, testCases[i].r[3 - j]);
            }

            hkInt256::computeUnsignedDivMod(a, b, q, r);

            HK_TEST(q.equal(qRef));
            HK_TEST(r.equal(rRef));

            // Check if q * b + r == a
            hkInt256 test;
            test.setMul(q, b);
            test.setAdd(test, r);
            HK_TEST(test.equal(a));

            hkInt256 qMinusInt32Max;
            qMinusInt32Max.setSub(q, int32Max);
            if (qMinusInt32Max.lessZero() || qMinusInt32Max.equalZero())
            {
                int qInt = hkInt256::computeDiv32(a, b);
                HK_TEST(qInt == q.getDoubleWord(0));

                // Try permutations of different signs
                a.setNeg(a);
                qInt = hkInt256::computeDiv32(a, b);
                HK_TEST(-qInt == q.getDoubleWord(0));

                b.setNeg(b);
                qInt = hkInt256::computeDiv32(a, b);
                HK_TEST(qInt == q.getDoubleWord(0));
            }
            else
            {
                // Must assert if result can't fit in 32bit
#if defined HK_DEBUG
                HK_TEST_ASSERT(0x72638520, hkInt256::computeDiv32(a, b));
#endif
            }
        }
    }

    // CLZ
    {
        int clz;
        clz = hkMath::countLeadingZeros<hkUint32>((hkUint32)1);     HK_TEST(clz == 31);
        clz = hkMath::countLeadingZeros<hkUint32>((hkUint32)-1);    HK_TEST(clz == 0);
        clz = hkMath::countLeadingZeros<hkUint32>((hkUint32)0);     HK_TEST(clz == 32);

        clz = hkMath::countLeadingZeros<hkUint64>((hkUint64)1);     HK_TEST(clz == 63);
        clz = hkMath::countLeadingZeros<hkUint64>((hkUint64)-1);    HK_TEST(clz == 0);
        clz = hkMath::countLeadingZeros<hkUint64>((hkUint64)0);     HK_TEST(clz == 64);

        hkInt128 i;
        i.setFromInt32(1);  clz = i.countLeadingZeros();    HK_TEST(clz == 127);
        i.setFromInt32(-1); clz = i.countLeadingZeros();    HK_TEST(clz == 0);
        i.setZero();        clz = i.countLeadingZeros();    HK_TEST(clz == 128);
    }

    // count bits set
    {
        for (int i =0; i < 1024; i++ )
        {
            int numBits = 0;
            for (int q = i; q; q = q >>1 ){ numBits += q&1; }
            HK_TEST( hkMath::countBitsSet( i ) == numBits );
            if ( i < 256 )
            {
                HK_TEST( hkMath::countBitsSet8( i ) == numBits );
            }
            if ( i < 16 )
            {
                HK_TEST( hkMath::countBitsSet4( i ) == numBits );
            }
        }
    }
}

// https://connect.microsoft.com/VisualStudio/feedback/details/1716320/addcarry-u32-causes-an-internal-compiler-error-in-a-certain-case
// Seems to be the same issue here - the compiler doesn't crash (unless you reduce the code), but optimizes constant expressions incorrectly, e.g. 0 - -100 becomes -100.

HK_DETAIL_OPTIMIZATIONS_OFF()
HK_DETAIL_DIAG_MSVC_OFF(4748)

//
//  Tests comparisons
static int testComparisons()
{
    hkInt64Vector4 v1;  v1.set(1, 4, 8, 7);
    hkInt64Vector4 v2;  v2.set(2, 4, 3, 6);

    hkVector4fComparison eq = v1.equal(v2);
    HK_TEST(eq.getMask() == hkVector4ComparisonMask::MASK_Y);

    v2.setComponent<0>(1);
    eq = v1.equal(v2);
    HK_TEST(eq.getMask() == hkVector4ComparisonMask::MASK_XY);

    v2.setComponent<2>(8);
    eq = v1.equal(v2);
    HK_TEST(eq.getMask() == hkVector4ComparisonMask::MASK_XYZ);

    v2.setComponent<3>(7);
    eq = v1.equal(v2);
    HK_TEST(eq.getMask() == hkVector4ComparisonMask::MASK_XYZW);

    {
        hkInt128 ia;
        ia.setFromInt32(-100);
        HK_TEST(ia.lessZero());
        ia.setFromInt32(-1);
        HK_TEST(ia.lessZero());
        ia.setFromInt32(0);
        HK_TEST(!ia.lessZero());
        ia.setFromInt32(1);
        HK_TEST(!ia.lessZero());
        ia.setFromInt32(100);
        HK_TEST(!ia.lessZero());
    }

    {
        hkInt128 ia, ib;
        ia.setFromInt32(-100); ib.setFromInt32(-2);     ia.setSub(ib, ia);
        HK_TEST(!ia.lessZero());
        ia.setFromInt32(-100); ib.setFromInt32(-200);   ia.setSub(ib, ia);
        HK_TEST(ia.lessZero());
        ia.setFromInt32(-100); ib.setFromInt32(-100);   ia.setSub(ib, ia);
        HK_TEST(!ia.lessZero());
        ia.setFromInt32(-100); ib.setFromInt32(100);    ia.setSub(ib, ia);
        HK_TEST(!ia.lessZero());
        ia.setFromInt32(-100); ib.setFromInt32(0);      ia.setSub(ib, ia);
        HK_TEST(!ia.lessZero());
        ia.setFromInt32(100); ib.setFromInt32(-2);      ia.setSub(ib, ia);
        HK_TEST(ia.lessZero());
        ia.setFromInt32(100); ib.setFromInt32(-200);    ia.setSub(ib, ia);
        HK_TEST(ia.lessZero());
        ia.setFromInt32(100); ib.setFromInt32(-100);    ia.setSub(ib, ia);
        HK_TEST(ia.lessZero());
        ia.setFromInt32(100); ib.setFromInt32(100);     ia.setSub(ib, ia);
        HK_TEST(!ia.lessZero());
        ia.setFromInt32(100); ib.setFromInt32(0);       ia.setSub(ib, ia);
        HK_TEST(ia.lessZero());
        ia.setFromInt32(100); ib.setFromInt32(2);       ia.setSub(ib, ia);
        HK_TEST(ia.lessZero());
        ia.setFromInt32(100); ib.setFromInt32(200);     ia.setSub(ib, ia);
        HK_TEST(!ia.lessZero());
    }

    // Success!
    return 0;
}

HK_DETAIL_OPTIMIZATIONS_ON()

//////////////////////////////////////////////////////////////////////////
//  Tests dot & cross

static int testDotCross()
{
    hkInt64Vector4 vA;  vA.set(1, 2, 3, 4);
    hkInt64Vector4 vB;  vB.set(8, 6, 1, 2);

    hkInt128 lenA, lenB;

    lenA = vA.dot<3>(vA);
    lenB = vB.dot<3>(vB);
    HK_TEST((lenA.getDoubleWord(0) == 14) &&
        (lenA.getDoubleWord(1) == 0));
    HK_TEST((lenB.getDoubleWord(0) == 101) &&
        (lenB.getDoubleWord(1) == 0));

    hkIntVector ivA;    ivA.set(1, 2, 3, 4);
    hkIntVector ivB;    ivB.set(8, 6, 1, 2);
    hkInt64Vector4 vC;  vC.setCross(ivA, ivB);
    HK_TEST((vC.getComponent<0>() == -16) &&
        (vC.getComponent<1>() == 23) &&
        (vC.getComponent<2>() == -10) &&
        (vC.getComponent<3>() == 0));

    // Success!
    return 0;
}

int LargeInt_main()
{
    {
        testSetOps();
        testArithmeticOps();
        testComparisons();
        testDotCross();
    }
    return 0;
}

HK_TEST_REGISTER(LargeInt_main, "Fast", "Common/Test/UnitTest/Base/", __FILE__);

HK_DETAIL_DIAG_MSVC_POP()

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
