// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : ALL
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/Math/ExtendedMath/hkFpMath.h>
#include <Common/Base/Math/LargeInt/hkLargeIntTypes.h>
#include <Common/Base/Algorithm/PseudoRandom/hkMT19937RandomGenerator.h>

namespace
{
    enum { REFERENCE_INT_BITS = 256 };

    typedef hkInt256 ReferenceInt;
    typedef hkInt512 DoubleReferenceInt;

    const int NUM_RANDOM_TESTS = 100;

    template<int BITS>
    hkFpMath::FpUint<BITS> makeRandomUint(hkMT19937RandomGenerator & rand)
    {
        hkFpMath::FpUint<BITS> res;
        res.setZero();

        for(int iw = 0; iw < BITS / hkFpMath::Detail::MULTI_WORD_UINT_BITS; iw++)
        {
#ifdef HKFPMATH_64_BIT_WORDS
            hkFpMath::Detail::MultiWordUintType word = (hkUint64(rand.getRand()) << 32) | rand.getRand();
#else
            hkFpMath::Detail::MultiWordUintType word = rand.getRand();
#endif

            hkFpMath::Detail::inPlaceSetWordUint(res.m_storage, iw, word);
        }

        int extraBits = BITS % hkFpMath::Detail::MULTI_WORD_UINT_BITS;
        if(extraBits != 0)
        {
#ifdef HKFPMATH_64_BIT_WORDS
            hkFpMath::Detail::MultiWordUintType word = (hkUint64(rand.getRand()) << 32) | rand.getRand();
#else
            hkFpMath::Detail::MultiWordUintType word = rand.getRand();
#endif
            int topWordIdx = BITS / hkFpMath::Detail::MULTI_WORD_UINT_BITS;
            hkFpMath::Detail::MultiWordUintType mask = (hkFpMath::Detail::MultiWordUintType(1) << extraBits) - 1;
            hkFpMath::Detail::inPlaceSetWordUint(res.m_storage, topWordIdx, mask & word);
        }

        return res;
    }

    template<int BITS>
    hkFpMath::FpInt<BITS> makeRandomInt(hkMT19937RandomGenerator & rand)
    {
        hkFpMath::FpUint<BITS-1> mag = makeRandomUint<BITS-1>(rand);

        hkFpMath::FpInt<BITS> res = mag.toSigned();
        if(rand.getRand() & 0x1)
        {
            return -res;
        }
        else
        {
            return res;
        }
    }

    template<typename ReferenceIntType, int BITS>
    void setReferenceInt(ReferenceIntType & dst, hkFpMath::FpUint<BITS> const& src)
    {
        dst.setZero();
        for(int iw = 0; iw < REFERENCE_INT_BITS / hkFpMath::Detail::MULTI_WORD_UINT_BITS; iw++)
        {
            hkFpMath::Detail::MultiWordUintType word = hkFpMath::Detail::getWordUint(src.m_storage, iw);
            ReferenceIntType refWord;
            refWord.setFromUint64(word);
            ReferenceIntType refWordShifted;
            refWordShifted.setShiftLeft(refWord, iw*hkFpMath::Detail::MULTI_WORD_UINT_BITS);
            dst.setOr(dst, refWordShifted);
        }
    }

    template<typename ReferenceIntType, int BITS>
    void setReferenceInt(ReferenceIntType & dst, hkFpMath::FpInt<BITS> const& src)
    {
        hkFpMath::FpInt<BITS> absSrc = src.abs();
        dst.setZero();
        for(int iw = 0; iw < REFERENCE_INT_BITS / hkFpMath::Detail::MULTI_WORD_UINT_BITS; iw++)
        {
            hkFpMath::Detail::MultiWordUintType word = hkFpMath::Detail::getWordUint(absSrc.m_storage, iw);
            ReferenceIntType refWord;
            refWord.setFromUint64(word);
            ReferenceIntType refWordShifted;
            refWordShifted.setShiftLeft(refWord, iw*hkFpMath::Detail::MULTI_WORD_UINT_BITS);
            dst.setOr(dst, refWordShifted);
        }

        if(src.lessZero())
        {
            dst.setNeg(dst);
        }
    }

    template<int BITS>
    bool referenceIntEqual(ReferenceInt const& a, hkFpMath::FpUint<BITS> const& b)
    {
        ReferenceInt cmp;
        setReferenceInt(cmp, b);

        return hkMemUtil::memCmp(&a, &cmp, sizeof(ReferenceInt)) == 0;
    }

    template<int BITS>
    bool referenceIntEqual(ReferenceInt const& a, hkFpMath::FpInt<BITS> const& b)
    {
        ReferenceInt cmp;
        setReferenceInt(cmp, b);

        return hkMemUtil::memCmp(&a, &cmp, sizeof(ReferenceInt)) == 0;
    }

    template<int BITS>
    bool referenceIntEqual(DoubleReferenceInt const& a, hkFpMath::FpUint<BITS> const& b)
    {
        DoubleReferenceInt cmp;
        setReferenceInt(cmp, b);

        return hkMemUtil::memCmp(&a, &cmp, sizeof(ReferenceInt)) == 0;
    }

    template<int BITS>
    bool referenceIntEqual(DoubleReferenceInt const& a, hkFpMath::FpInt<BITS> const& b)
    {
        DoubleReferenceInt cmp;
        setReferenceInt(cmp, b);

        return hkMemUtil::memCmp(&a, &cmp, sizeof(ReferenceInt)) == 0;
    }

    template<int ABITS>
    int testSanity()
    {
        hkMT19937RandomGenerator rand; rand.initialize(0x927351);

        for(int it = 0; it < NUM_RANDOM_TESTS; it++)
        {
            hkFpMath::FpUint<ABITS> a = makeRandomUint<ABITS>(rand);
            ReferenceInt aRef; setReferenceInt(aRef, a);

            HK_TEST(referenceIntEqual(a, aRef));
        }

        return 0;
    }

    template<int ABITS>
    int uintToFloat32_fuzz()
    {
        hkMT19937RandomGenerator rand; rand.initialize(0x927351);

        for(int it = 0; it < NUM_RANDOM_TESTS; it++)
        {
            hkFpMath::FpUint<ABITS> a = makeRandomUint<ABITS>(rand);

            hkFloat32 aFloat = a.toFloat32();

            while(!a.template canNarrow<32>())
            {
                a = a.template shr<1>();
                aFloat *= 0.5f;
            }

            hkUint32 aWord = a.template narrow<32>().toUint32();

            hkFloat32 aWordFloat = hkFloat32(aWord);
            hkFloat32 error = hkMath::abs(aFloat - aWordFloat);
            HK_TEST(error <= aFloat * HK_FLOAT_EPSILON);
        }
        return 0;
    }

    template<int ABITS>
    int intToFloat32_fuzz()
    {
        hkMT19937RandomGenerator rand; rand.initialize(0x927351);

        for(int it = 0; it < NUM_RANDOM_TESTS; it++)
        {
            hkFpMath::FpInt<ABITS> a = makeRandomInt<ABITS>(rand);

            hkFloat32 aFloat = a.toFloat32();

            HK_TEST(a.lessZero() == (aFloat < 0));

            hkFpMath::FpUint<ABITS-1> aMag = a.abs();

            while(!aMag.template canNarrow<32>())
            {
                aMag = aMag.template shr<1>();
                aFloat *= 0.5f;
            }

            hkUint32 aWord = aMag.template narrow<32>().toUint32();

            hkFloat32 aWordFloat = hkFloat32(aWord);
            if(a.lessZero()) aWordFloat = -aWordFloat;

            hkFloat32 error = hkMath::abs(aFloat - aWordFloat);
            HK_TEST(error <= hkMath::abs(aFloat) * HK_FLOAT_EPSILON);
        }
        return 0;
    }
    /*
    template<int ABITS, int BBITS>
    int uintRatioToFloat32_fuzz()
    {
        hkMT19937RandomGenerator rand; rand.initialize(0x927351);

        for(int it = 0; it < NUM_RANDOM_TESTS; it++)
        {
            hkFpMath::FpUint<ABITS> n = makeRandomUint<ABITS>(rand);
            hkFpMath::FpUint<ABITS> d = makeRandomUint<BBITS>(rand);

            hkFloat32 aFloat = a.toFloat32();

            hkUint32 aWord = a.template narrow<32>().toUint32();

            hkFloat32 aWordFloat = hkFloat32(aWord);
            hkFloat32 error = hkMath::abs(aFloat - aWordFloat);
            HK_TEST(error <= aFloat * HK_FLOAT_EPSILON);
        }
        return 0;
    }
    */
    template<int ABITS>
    int float32ToInt_fuzz()
    {
        hkMT19937RandomGenerator rand; rand.initialize(0x927351);

        const hkFloat32 max = hkMath::pow(2.0f, hkFloat32(ABITS-1)) - 1;

        for(int it = 0; it < NUM_RANDOM_TESTS; it++)
        {
            hkFloat32 aFloat = rand.getRandFloat32(max*2)-max;

            hkFpMath::FpInt<ABITS> a; a.setFromFloat32(aFloat);

            hkFloat32 aRes = a.toFloat32();

            hkFloat32 error = hkMath::abs(aFloat-aRes);

            HK_TEST(error < 1.0f);
        }

        return 0;
    }

    template<int ABITS, int BIASBITS>
    int biasedFloat32ToInt_negBias_fuzz()
    {
        hkMT19937RandomGenerator rand; rand.initialize(0x927351);

        const hkFloat32 max = hkMath::pow(2.0f, hkFloat32(ABITS-1)) - 1;

        const hkFloat32 bias = hkMath::pow(2.0f, -BIASBITS);

        for(int it = 0; it < NUM_RANDOM_TESTS; it++)
        {
            hkFloat32 aFloat = rand.getRandFloat32(max*2)-max;

            hkFpMath::FpInt<ABITS> a; a.template setFromBiasedFloat32<-BIASBITS>(aFloat * bias);

            hkFloat32 aRes = a.toFloat32();

            hkFloat32 error = hkMath::abs(aFloat-aRes);

            HK_TEST(error < 1.0f);
        }

        return 0;
    }

    template<int ABITS, int BIASBITS>
    int biasedFloat32ToInt_posBias_fuzz()
    {
        hkMT19937RandomGenerator rand; rand.initialize(0x927351);

        const hkFloat32 max = hkMath::pow(2.0f, hkFloat32(ABITS-1)) - 1;

        const hkFloat32 bias = hkMath::pow(2.0f, BIASBITS);

        for(int it = 0; it < NUM_RANDOM_TESTS; it++)
        {
            hkFloat32 aFloat = rand.getRandFloat32(max*2)-max;

            hkFpMath::FpInt<ABITS> a; a.template setFromBiasedFloat32<BIASBITS>(aFloat * bias);

            hkFloat32 aRes = a.toFloat32();

            hkFloat32 error = hkMath::abs(aFloat-aRes);

            HK_TEST(error < 1.0f);
        }

        return 0;
    }


    template<int ABITS, int BBITS>
    int addUintUint_fuzz()
    {
        hkMT19937RandomGenerator rand; rand.initialize(0x927351);

        for(int it = 0; it < NUM_RANDOM_TESTS; it++)
        {
            hkFpMath::FpUint<ABITS> a = makeRandomUint<ABITS>(rand);
            hkFpMath::FpUint<BBITS> b = makeRandomUint<BBITS>(rand);
            hkFpMath::FpUint<(ABITS>BBITS ? ABITS : BBITS) + 1 > res = a + b;

            ReferenceInt aRef; setReferenceInt(aRef, a);
            ReferenceInt bRef; setReferenceInt(bRef, b);
            ReferenceInt resRef; resRef.setAdd(aRef, bRef);

            HK_TEST(referenceIntEqual(resRef, res));
        }

        return 0;
    }

    template<int ABITS, int BBITS>
    int mulUintUint_fuzz()
    {
        hkMT19937RandomGenerator rand; rand.initialize(0x927351);

        for(int it = 0; it < NUM_RANDOM_TESTS; it++)
        {
            hkFpMath::FpUint<ABITS> a = makeRandomUint<ABITS>(rand);
            hkFpMath::FpUint<BBITS> b = makeRandomUint<BBITS>(rand);
            hkFpMath::FpUint<ABITS+BBITS> res = a * b;

            ReferenceInt aRef; setReferenceInt(aRef, a);
            ReferenceInt bRef; setReferenceInt(bRef, b);
            DoubleReferenceInt resRef; resRef.setMul(aRef, bRef);

            bool eq = referenceIntEqual(resRef, res);
            HK_TEST(eq);
        }

        return 0;
    }

    template<int ABITS, int BBITS>
    int divModUintUint_fuzz()
    {
        hkMT19937RandomGenerator rand; rand.initialize(0x927351);

        for(int it = 0; it < NUM_RANDOM_TESTS; it++)
        {
            // For good coverage of the test space (with expected u >> v), work backwards from result
            hkFpMath::FpUint<ABITS> qRef = makeRandomUint<ABITS>(rand);

            hkFpMath::FpUint<BBITS> rRef, v;
            do
            {
                rRef = makeRandomUint<BBITS>(rand);
                v = makeRandomUint<BBITS>(rand);
            } while (rRef >= v);

            hkFpMath::FpUint<ABITS+BBITS+1> u = qRef*v + rRef;

            hkFpMath::FpUint<ABITS+BBITS+1> q;
            hkFpMath::FpUint<BBITS> r;

            divMod(u, v, q, r);

            HK_TEST(q == qRef);
            HK_TEST(r == rRef);
        }

        return 0;
    }

    int divModUintUint_addback()
    {
        hkFpMath::FpUint<hkFpMath::Detail::MULTI_WORD_UINT_BITS * 4> u;
        hkFpMath::Detail::inPlaceSetWordUint(u.m_storage, 0, 0);
        hkFpMath::Detail::inPlaceSetWordUint(u.m_storage, 1, 0);
        hkFpMath::Detail::inPlaceSetWordUint(u.m_storage, 2, hkFpMath::Detail::WORD_SIGN_MASK);
        hkFpMath::Detail::inPlaceSetWordUint(u.m_storage, 3, hkFpMath::Detail::WORD_SIGN_MASK-1);

        hkFpMath::FpUint<hkFpMath::Detail::MULTI_WORD_UINT_BITS * 3> v;
        hkFpMath::Detail::inPlaceSetWordUint(v.m_storage, 0, 1);
        hkFpMath::Detail::inPlaceSetWordUint(v.m_storage, 1, 0);
        hkFpMath::Detail::inPlaceSetWordUint(v.m_storage, 2, hkFpMath::Detail::WORD_SIGN_MASK);

        hkFpMath::FpUint<hkFpMath::Detail::MULTI_WORD_UINT_BITS * 4> q;
        hkFpMath::FpUint<hkFpMath::Detail::MULTI_WORD_UINT_BITS * 3> r;

        divMod(u, v, q, r);

        HK_TEST(r < v);
        HK_TEST(q*v+r == u);

        return 0;
    }

    template<int ABITS, int BBITS>
    int addIntInt_fuzz()
    {
        hkMT19937RandomGenerator rand; rand.initialize(0x927351);

        for(int it = 0; it < NUM_RANDOM_TESTS; it++)
        {
            hkFpMath::FpInt<ABITS> a = makeRandomInt<ABITS>(rand);
            hkFpMath::FpInt<BBITS> b = makeRandomInt<BBITS>(rand);
            hkFpMath::FpInt<(ABITS>BBITS ? ABITS : BBITS) + 1 > res = a + b;

            ReferenceInt aRef; setReferenceInt(aRef, a);
            ReferenceInt bRef; setReferenceInt(bRef, b);
            ReferenceInt resRef; resRef.setAdd(aRef, bRef);

            HK_TEST(referenceIntEqual(resRef, res));
        }

        return 0;
    }

    template<int ABITS, int BBITS>
    int subIntInt_fuzz()
    {
        hkMT19937RandomGenerator rand; rand.initialize(0x927351);

        for(int it = 0; it < NUM_RANDOM_TESTS; it++)
        {
            hkFpMath::FpInt<ABITS> a = makeRandomInt<ABITS>(rand);
            hkFpMath::FpInt<BBITS> b = makeRandomInt<BBITS>(rand);
            hkFpMath::FpInt<(ABITS>BBITS ? ABITS : BBITS) + 1 > res = a - b;

            ReferenceInt aRef; setReferenceInt(aRef, a);
            ReferenceInt bRef; setReferenceInt(bRef, b);
            ReferenceInt resRef; resRef.setSub(aRef, bRef);

            HK_TEST(referenceIntEqual(resRef, res));
        }

        return 0;
    }

    template<int ABITS, int BBITS>
    int mulIntInt_fuzz()
    {
        hkMT19937RandomGenerator rand; rand.initialize(0x927351);

        for(int it = 0; it < NUM_RANDOM_TESTS; it++)
        {
            hkFpMath::FpInt<ABITS> a = makeRandomInt<ABITS>(rand);
            hkFpMath::FpInt<BBITS> b = makeRandomInt<BBITS>(rand);
            hkFpMath::FpInt<ABITS + BBITS> res = a * b;

            ReferenceInt aRef; setReferenceInt(aRef, a);
            ReferenceInt bRef; setReferenceInt(bRef, b);
            DoubleReferenceInt resRef; resRef.setMul(aRef, bRef);

            bool eq = referenceIntEqual(resRef, res);
            HK_TEST(eq);
        }

        return 0;
    }

    template<int ABITS, int BBITS>
    int divModIntInt_fuzz()
    {
        hkMT19937RandomGenerator rand; rand.initialize(0x927351);

        for(int it = 0; it < NUM_RANDOM_TESTS; it++)
        {
            // For good coverage of the test space (with expected u >> v), work backwards from result
            hkFpMath::FpInt<ABITS> qRef = makeRandomInt<ABITS>(rand);

            hkFpMath::FpInt<BBITS> rRef, v;
            hkFpMath::FpInt<ABITS+BBITS+1> u;
            do
            {
                do
                {
                    rRef = makeRandomInt<BBITS>(rand);
                    v = makeRandomInt<BBITS>(rand);
                } while(rRef.abs() >= v.abs()); // Force legal remainder (and no div0)

                u = qRef*v + rRef;
            } while(!(rRef.equalZero() || u.lessZero() == rRef.lessZero())); // Force truncating division

            hkFpMath::FpInt<ABITS+BBITS+1> q;
            hkFpMath::FpInt<BBITS> r;

            divMod(u, v, q, r);

            HK_TEST(q == qRef);
            HK_TEST(r == rRef);
        }

        return 0;
    }

    void gcdRefImpl(ReferenceInt const& aIn, ReferenceInt const& bIn, ReferenceInt & gcdOut)
    {
        ReferenceInt a = aIn;
        ReferenceInt b = bIn;

        while(!b.equalZero())
        {
            ReferenceInt t = b;
            ReferenceInt qIgnored;
            ReferenceInt::computeUnsignedDivMod(a, b, qIgnored, b);
            a = t;
        }
        gcdOut = a;
    }

    template<int ABITS, int BBITS>
    int gcdUintUint_fuzz()
    {
        hkMT19937RandomGenerator rand; rand.initialize(0x927351);

        for(int it = 0; it < NUM_RANDOM_TESTS; it++)
        {
            hkFpMath::FpUint<ABITS> a;
            do {
                a = makeRandomUint<ABITS>(rand);
            } while(a.equalZero());

            hkFpMath::FpUint<BBITS> b;
            do {
                b = makeRandomUint<BBITS>(rand);
            } while(b.equalZero());

            // In the first round, just check against reference GCD impl
            // This probably won't flex the algorithm for large GCDs

            hkFpMath::FpUint<(ABITS<BBITS)?ABITS:BBITS> abGcd = gcd(a, b);

            ReferenceInt aRef; setReferenceInt(aRef, a);
            ReferenceInt bRef; setReferenceInt(bRef, b);
            ReferenceInt abGcdRef;
            gcdRefImpl(aRef, bRef, abGcdRef);

            HK_TEST(referenceIntEqual(abGcdRef, abGcd));

            // Now reduce by the GCD.
            {
                hkFpMath::FpUint<(ABITS<BBITS) ? ABITS : BBITS> rIgnored;
                divMod(a, abGcd, a, rIgnored);
                divMod(b, abGcd, b, rIgnored);
            }

            // Now we expect a and b to be coprime. Multiply by a common factor,
            // and expect that factor as the GCD.

            hkFpMath::FpUint<BBITS> commonFactor;
            do {
                commonFactor = makeRandomUint<BBITS>(rand);
            } while(commonFactor.equalZero());

            hkFpMath::FpUint<ABITS+BBITS> c = a*commonFactor;
            hkFpMath::FpUint<BBITS+BBITS> d = b*commonFactor;

            hkFpMath::FpUint<((ABITS<BBITS)?ABITS:BBITS)+BBITS> cdGcd = gcd(c, d);

            HK_TEST(cdGcd == commonFactor);
        }

        return 0;
    }

    template<int ABITS>
    int shiftUint_fuzz()
    {
        hkMT19937RandomGenerator rand; rand.initialize(0x927351);

        for(int it = 0; it < NUM_RANDOM_TESTS; it++)
        {
            hkFpMath::FpUint<ABITS> a = makeRandomUint<ABITS>(rand);
            ReferenceInt aRef; setReferenceInt(aRef, a);

            int bits = rand.getRand() % ABITS;

            a >>= bits;
            aRef.setShiftRight(aRef, bits);

            HK_TEST(referenceIntEqual(aRef, a));

            a <<= bits;
            aRef.setShiftLeft(aRef, bits);

            HK_TEST(referenceIntEqual(aRef, a));
        }

        return 0;
    }

    template<int ABITS, int BBITS>
    int testCmp()
    {
        int vals[] = { -5, -3, 0, 3, 5 };

        for(int i = 0; i < HK_COUNT_OF(vals); i++)
        {
            hkFpMath::FpInt<ABITS> a(vals[i]);
            for(int j = 0; j < HK_COUNT_OF(vals); j++)
            {
                hkFpMath::FpInt<BBITS> b(vals[j]);

                int res = cmp(a, b);

                if(i < j)
                {
                    HK_TEST(res < 0);
                }
                else if(i > j)
                {
                    HK_TEST(res > 0);
                }
                else
                {
                    HK_TEST(res == 0);
                }
            }
        }

        return 0;
    }

#define DEFINE_TEST_FUNC_UNARY(NAME, A) int NAME ## _ ## A () { return NAME<A>(); }
#define DEFINE_TEST_FUNC_BINARY(NAME, A, B) int NAME ## _ ## A ## _ ## B () { return NAME<A, B>(); }

    DEFINE_TEST_FUNC_BINARY(addUintUint_fuzz, 3, 5)
    DEFINE_TEST_FUNC_BINARY(addUintUint_fuzz, 3, 14)
    DEFINE_TEST_FUNC_BINARY(addUintUint_fuzz, 14, 14)
    DEFINE_TEST_FUNC_BINARY(addUintUint_fuzz, 14, 19)
    DEFINE_TEST_FUNC_BINARY(addUintUint_fuzz, 35, 19)
    DEFINE_TEST_FUNC_BINARY(addUintUint_fuzz, 7, 140)
    DEFINE_TEST_FUNC_BINARY(addUintUint_fuzz, 35, 140)
    DEFINE_TEST_FUNC_BINARY(addUintUint_fuzz, 77, 140)

    DEFINE_TEST_FUNC_BINARY(mulUintUint_fuzz, 3, 5)
    DEFINE_TEST_FUNC_BINARY(mulUintUint_fuzz, 3, 14)
    DEFINE_TEST_FUNC_BINARY(mulUintUint_fuzz, 14, 14)
    DEFINE_TEST_FUNC_BINARY(mulUintUint_fuzz, 14, 19)
    DEFINE_TEST_FUNC_BINARY(mulUintUint_fuzz, 35, 19)
    DEFINE_TEST_FUNC_BINARY(mulUintUint_fuzz, 7, 140)
    DEFINE_TEST_FUNC_BINARY(mulUintUint_fuzz, 35, 140)
    DEFINE_TEST_FUNC_BINARY(mulUintUint_fuzz, 77, 140)

    DEFINE_TEST_FUNC_BINARY(divModUintUint_fuzz, 3, 5)
    DEFINE_TEST_FUNC_BINARY(divModUintUint_fuzz, 3, 14)
    DEFINE_TEST_FUNC_BINARY(divModUintUint_fuzz, 14, 14)
    DEFINE_TEST_FUNC_BINARY(divModUintUint_fuzz, 14, 19)
    DEFINE_TEST_FUNC_BINARY(divModUintUint_fuzz, 35, 19)
    DEFINE_TEST_FUNC_BINARY(divModUintUint_fuzz, 35, 140)
    DEFINE_TEST_FUNC_BINARY(divModUintUint_fuzz, 140, 35)

    DEFINE_TEST_FUNC_BINARY(addIntInt_fuzz, 3, 5)
    DEFINE_TEST_FUNC_BINARY(addIntInt_fuzz, 3, 14)
    DEFINE_TEST_FUNC_BINARY(addIntInt_fuzz, 14, 14)
    DEFINE_TEST_FUNC_BINARY(addIntInt_fuzz, 14, 19)
    DEFINE_TEST_FUNC_BINARY(addIntInt_fuzz, 35, 19)
    DEFINE_TEST_FUNC_BINARY(addIntInt_fuzz, 7, 140)
    DEFINE_TEST_FUNC_BINARY(addIntInt_fuzz, 35, 140)
    DEFINE_TEST_FUNC_BINARY(addIntInt_fuzz, 77, 140)

    DEFINE_TEST_FUNC_BINARY(subIntInt_fuzz, 3, 5)
    DEFINE_TEST_FUNC_BINARY(subIntInt_fuzz, 3, 14)
    DEFINE_TEST_FUNC_BINARY(subIntInt_fuzz, 14, 14)
    DEFINE_TEST_FUNC_BINARY(subIntInt_fuzz, 14, 19)
    DEFINE_TEST_FUNC_BINARY(subIntInt_fuzz, 35, 19)
    DEFINE_TEST_FUNC_BINARY(subIntInt_fuzz, 7, 140)
    DEFINE_TEST_FUNC_BINARY(subIntInt_fuzz, 35, 140)
    DEFINE_TEST_FUNC_BINARY(subIntInt_fuzz, 77, 140)

    DEFINE_TEST_FUNC_BINARY(mulIntInt_fuzz, 3, 5)
    DEFINE_TEST_FUNC_BINARY(mulIntInt_fuzz, 3, 14)
    DEFINE_TEST_FUNC_BINARY(mulIntInt_fuzz, 14, 14)
    DEFINE_TEST_FUNC_BINARY(mulIntInt_fuzz, 14, 19)
    DEFINE_TEST_FUNC_BINARY(mulIntInt_fuzz, 35, 19)
    DEFINE_TEST_FUNC_BINARY(mulIntInt_fuzz, 7, 140)
    DEFINE_TEST_FUNC_BINARY(mulIntInt_fuzz, 35, 140)
    DEFINE_TEST_FUNC_BINARY(mulIntInt_fuzz, 77, 140)

    DEFINE_TEST_FUNC_BINARY(divModIntInt_fuzz, 3, 5)
    DEFINE_TEST_FUNC_BINARY(divModIntInt_fuzz, 3, 14)
    DEFINE_TEST_FUNC_BINARY(divModIntInt_fuzz, 14, 14)
    DEFINE_TEST_FUNC_BINARY(divModIntInt_fuzz, 14, 19)
    DEFINE_TEST_FUNC_BINARY(divModIntInt_fuzz, 35, 19)
    DEFINE_TEST_FUNC_BINARY(divModIntInt_fuzz, 7, 140)
    DEFINE_TEST_FUNC_BINARY(divModIntInt_fuzz, 35, 140)
    DEFINE_TEST_FUNC_BINARY(divModIntInt_fuzz, 77, 140)

    DEFINE_TEST_FUNC_BINARY(gcdUintUint_fuzz, 3, 5)
    DEFINE_TEST_FUNC_BINARY(gcdUintUint_fuzz, 3, 14)
    DEFINE_TEST_FUNC_BINARY(gcdUintUint_fuzz, 14, 14)
    DEFINE_TEST_FUNC_BINARY(gcdUintUint_fuzz, 14, 19)
    DEFINE_TEST_FUNC_BINARY(gcdUintUint_fuzz, 35, 19)
    DEFINE_TEST_FUNC_BINARY(gcdUintUint_fuzz, 7, 140)
    DEFINE_TEST_FUNC_BINARY(gcdUintUint_fuzz, 35, 140)
    DEFINE_TEST_FUNC_BINARY(gcdUintUint_fuzz, 77, 140)

    DEFINE_TEST_FUNC_UNARY(uintToFloat32_fuzz, 45);
    DEFINE_TEST_FUNC_UNARY(uintToFloat32_fuzz, 98);
    DEFINE_TEST_FUNC_UNARY(uintToFloat32_fuzz, 125);

    DEFINE_TEST_FUNC_UNARY(intToFloat32_fuzz, 45);
    DEFINE_TEST_FUNC_UNARY(intToFloat32_fuzz, 98);
    DEFINE_TEST_FUNC_UNARY(intToFloat32_fuzz, 125);

    DEFINE_TEST_FUNC_UNARY(float32ToInt_fuzz, 6);
    DEFINE_TEST_FUNC_UNARY(float32ToInt_fuzz, 45);
    DEFINE_TEST_FUNC_UNARY(float32ToInt_fuzz, 98);
    DEFINE_TEST_FUNC_UNARY(float32ToInt_fuzz, 125);

    DEFINE_TEST_FUNC_BINARY(biasedFloat32ToInt_posBias_fuzz, 6, 8);
    DEFINE_TEST_FUNC_BINARY(biasedFloat32ToInt_posBias_fuzz, 98, 8);
    DEFINE_TEST_FUNC_BINARY(biasedFloat32ToInt_posBias_fuzz, 6, 17);
    DEFINE_TEST_FUNC_BINARY(biasedFloat32ToInt_posBias_fuzz, 98, 17);
    DEFINE_TEST_FUNC_BINARY(biasedFloat32ToInt_posBias_fuzz, 6, 64);
    DEFINE_TEST_FUNC_BINARY(biasedFloat32ToInt_negBias_fuzz, 6, 8);
    DEFINE_TEST_FUNC_BINARY(biasedFloat32ToInt_negBias_fuzz, 98, 8);
    DEFINE_TEST_FUNC_BINARY(biasedFloat32ToInt_negBias_fuzz, 6, 17);
    DEFINE_TEST_FUNC_BINARY(biasedFloat32ToInt_negBias_fuzz, 98, 17);
    DEFINE_TEST_FUNC_BINARY(biasedFloat32ToInt_negBias_fuzz, 6, 64);
    DEFINE_TEST_FUNC_BINARY(biasedFloat32ToInt_negBias_fuzz, 98, 64);

    DEFINE_TEST_FUNC_UNARY(shiftUint_fuzz, 6);
    DEFINE_TEST_FUNC_UNARY(shiftUint_fuzz, 45);
    DEFINE_TEST_FUNC_UNARY(shiftUint_fuzz, 98);
    DEFINE_TEST_FUNC_UNARY(shiftUint_fuzz, 125);
}

HK_TEST_REGISTER(addUintUint_fuzz_3_5, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(addUintUint_fuzz_3_14, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(addUintUint_fuzz_14_14, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(addUintUint_fuzz_14_19, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(addUintUint_fuzz_35_19, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(addUintUint_fuzz_7_140, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(addUintUint_fuzz_35_140, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(addUintUint_fuzz_77_140, "Fast", "Common/Test/UnitTest/Base/", "FpMath");

HK_TEST_REGISTER(mulUintUint_fuzz_3_5, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(mulUintUint_fuzz_3_14, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(mulUintUint_fuzz_14_14, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(mulUintUint_fuzz_14_19, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(mulUintUint_fuzz_35_19, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(mulUintUint_fuzz_7_140, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(mulUintUint_fuzz_35_140, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(mulUintUint_fuzz_77_140, "Fast", "Common/Test/UnitTest/Base/", "FpMath");

HK_TEST_REGISTER(divModUintUint_fuzz_3_5, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(divModUintUint_fuzz_3_14, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(divModUintUint_fuzz_14_14, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(divModUintUint_fuzz_14_19, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(divModUintUint_fuzz_35_19, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(divModUintUint_fuzz_35_140, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(divModUintUint_fuzz_140_35, "Fast", "Common/Test/UnitTest/Base/", "FpMath");

HK_TEST_REGISTER(addIntInt_fuzz_3_5, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(addIntInt_fuzz_3_14, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(addIntInt_fuzz_14_14, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(addIntInt_fuzz_14_19, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(addIntInt_fuzz_35_19, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(addIntInt_fuzz_7_140, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(addIntInt_fuzz_35_140, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(addIntInt_fuzz_77_140, "Fast", "Common/Test/UnitTest/Base/", "FpMath");

HK_TEST_REGISTER(subIntInt_fuzz_3_5, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(subIntInt_fuzz_3_14, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(subIntInt_fuzz_14_14, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(subIntInt_fuzz_14_19, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(subIntInt_fuzz_35_19, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(subIntInt_fuzz_7_140, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(subIntInt_fuzz_35_140, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(subIntInt_fuzz_77_140, "Fast", "Common/Test/UnitTest/Base/", "FpMath");

HK_TEST_REGISTER(mulIntInt_fuzz_3_5, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(mulIntInt_fuzz_3_14, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(mulIntInt_fuzz_14_14, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(mulIntInt_fuzz_14_19, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(mulIntInt_fuzz_35_19, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(mulIntInt_fuzz_7_140, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(mulIntInt_fuzz_35_140, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(mulIntInt_fuzz_77_140, "Fast", "Common/Test/UnitTest/Base/", "FpMath");


HK_TEST_REGISTER(divModIntInt_fuzz_3_5, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(divModIntInt_fuzz_3_14, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(divModIntInt_fuzz_14_14, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(divModIntInt_fuzz_14_19, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(divModIntInt_fuzz_35_19, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(divModIntInt_fuzz_7_140, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(divModIntInt_fuzz_35_140, "Fast", "Common/Test/UnitTest/Base/", "FpMath");


HK_TEST_REGISTER(gcdUintUint_fuzz_3_5, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(gcdUintUint_fuzz_3_14, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(gcdUintUint_fuzz_14_14, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(gcdUintUint_fuzz_14_19, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(gcdUintUint_fuzz_35_19, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(gcdUintUint_fuzz_7_140, "Slow", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(gcdUintUint_fuzz_35_140, "Slow", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(gcdUintUint_fuzz_77_140, "Slow", "Common/Test/UnitTest/Base/", "FpMath");


HK_TEST_REGISTER(uintToFloat32_fuzz_45, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(uintToFloat32_fuzz_98, "Slow", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(uintToFloat32_fuzz_125, "Slow", "Common/Test/UnitTest/Base/", "FpMath");

HK_TEST_REGISTER(intToFloat32_fuzz_45, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(intToFloat32_fuzz_98, "Slow", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(intToFloat32_fuzz_125, "Slow", "Common/Test/UnitTest/Base/", "FpMath");

HK_TEST_REGISTER(float32ToInt_fuzz_6, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(float32ToInt_fuzz_45, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(float32ToInt_fuzz_98, "Slow", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(float32ToInt_fuzz_125, "Slow", "Common/Test/UnitTest/Base/", "FpMath");

HK_TEST_REGISTER(biasedFloat32ToInt_posBias_fuzz_6_8, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(biasedFloat32ToInt_posBias_fuzz_98_8, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(biasedFloat32ToInt_posBias_fuzz_6_17, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(biasedFloat32ToInt_posBias_fuzz_98_17, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(biasedFloat32ToInt_posBias_fuzz_6_64, "Fast", "Common/Test/UnitTest/Base/", "FpMath");

HK_TEST_REGISTER(biasedFloat32ToInt_negBias_fuzz_6_8, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(biasedFloat32ToInt_negBias_fuzz_98_8, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(biasedFloat32ToInt_negBias_fuzz_6_17, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(biasedFloat32ToInt_negBias_fuzz_98_17, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(biasedFloat32ToInt_negBias_fuzz_6_64, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(biasedFloat32ToInt_negBias_fuzz_98_64, "Fast", "Common/Test/UnitTest/Base/", "FpMath");

HK_TEST_REGISTER(divModUintUint_addback, "Fast", "Common/Test/UnitTest/Base/", "FpMath");

HK_TEST_REGISTER(shiftUint_fuzz_6, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(shiftUint_fuzz_45, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(shiftUint_fuzz_98, "Fast", "Common/Test/UnitTest/Base/", "FpMath");
HK_TEST_REGISTER(shiftUint_fuzz_125, "Fast", "Common/Test/UnitTest/Base/", "FpMath");

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
