// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM   : ALL
// PRODUCT   : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0
#include <Common/Base/hkBase.h>
#include <Common/Base/UnitTest/hkUnitTest.h>
#include <Common/Base/Math/AxialTransform/hkAxialRotation.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>

static void testGetRotation()
{
    hkVector4 v;
    v.set(1, -2, 5);

    for(int i = 0; i < HK_NUM_AXIAL_ROTATIONS; i++)
    {
        hkAxialRotation axial = hkAxialRotation::getConstant((hkAxialRotationTag)i);
        hkVector4 axialRotated = v;
        axial.rotateVector(axialRotated);

        hkRotation rot;
        axial.getRotation(rot);
        hkVector4 rotRotated;
        rotRotated.setRotatedDir(rot, v);

        HK_TEST(axialRotated.allExactlyEqual<3>(rotRotated));
    }
}

static void testSetComposition()
{
    for(int i = 0; i < HK_NUM_AXIAL_ROTATIONS; i++)
    {
        for(int j = 0; j < HK_NUM_AXIAL_ROTATIONS; j++)
        {
            hkAxialRotation a = hkAxialRotation::getConstant((hkAxialRotationTag)i);
            hkAxialRotation b = hkAxialRotation::getConstant((hkAxialRotationTag)j);

            hkAxialRotation comp;
            comp.setComposition(a, b);

            // See if the result applied to a vector, gives the same result as the original rotations applied to that vector.
            {
                hkVector4 v;
                v.set(1, -2, 5);

                hkVector4 rotated = v;
                comp.rotateVector(rotated);

                hkVector4 refRotated = v;
                b.rotateVector(refRotated);
                a.rotateVector(refRotated);

                HK_TEST(rotated.allExactlyEqual<3>(refRotated));
            }

            // See if it matches hkRotation's composition
            {
                hkRotation aRot, bRot;
                a.getRotation(aRot);
                b.getRotation(bRot);

                hkRotation refRot;
                refRot.setMul(aRot, bRot);

                hkRotation asRot;
                comp.getRotation(asRot);

                HK_TEST(refRot.isApproximatelyEqual(asRot, .001f));
            }
        }
    }
}

static void testRotate90Deg()
{
    {
        hkAxialRotation axialRot;
        axialRot.setIdentity();
        axialRot.rotateX90Deg();

        HK_TEST(axialRot == hkAxialRotation::getConstant(HK_AXIAL_ROT_X90));

        hkRotation asRot;
        axialRot.getRotation(asRot);

        hkRotation refRot;
        refRot.setAxisAngle(hkVector4::getConstant<HK_QUADREAL_1000>(), HK_REAL_PI / 2);
        HK_TEST(asRot.isApproximatelyEqual(refRot, .001f));
    }

    {
        hkAxialRotation axialRot;
        axialRot.setIdentity();
        axialRot.rotateY90Deg();

        HK_TEST(axialRot == hkAxialRotation::getConstant(HK_AXIAL_ROT_Y90));

        hkRotation asRot;
        axialRot.getRotation(asRot);

        hkRotation refRot;
        refRot.setAxisAngle(hkVector4::getConstant<HK_QUADREAL_0100>(), HK_REAL_PI / 2);
        HK_TEST(asRot.isApproximatelyEqual(refRot, .001f));
    }

    {
        hkAxialRotation axialRot;
        axialRot.setIdentity();
        axialRot.rotateZ90Deg();

        HK_TEST(axialRot == hkAxialRotation::getConstant(HK_AXIAL_ROT_Z90));

        hkRotation asRot;
        axialRot.getRotation(asRot);

        hkRotation refRot;
        refRot.setAxisAngle(hkVector4::getConstant<HK_QUADREAL_0010>(), HK_REAL_PI / 2);
        HK_TEST(asRot.isApproximatelyEqual(refRot, .001f));
    }
}

static void testRotate90DegRepeatedly()
{
    hkAxialRotation xRot, yRot, zRot;

    xRot.setIdentity();
    xRot.rotateX90Deg();

    yRot.setIdentity();
    yRot.rotateY90Deg();

    zRot.setIdentity();
    zRot.rotateZ90Deg();

    {
        hkAxialRotation rot;
        rot.setIdentity();
        rot.rotateX90Deg();
        rot.rotateY90Deg();

        hkAxialRotation ref;
        ref.setComposition(yRot, xRot);
        HK_TEST(rot == ref);
    }
}

static void testSetInverse()
{
    hkAxialRotation rot;
    rot.setIdentity();
    rot.rotateX90Deg();
    rot.rotateZ90Deg();
    rot.rotateZ90Deg();
    rot.rotateZ90Deg();

    hkAxialRotation inv;
    inv.setInverse(rot);

    hkAxialRotation ident;
    ident.setComposition(rot, inv);

    HK_TEST(ident == hkAxialRotation::getConstant(HK_AXIAL_ROT_IDENTITY));
}

static void testRotateAabb()
{
    hkAabb aabb;
    aabb.m_min.set(-3, -2, 5);
    aabb.m_max.set(4, -1, 7);

    for(int i = 0; i < HK_NUM_AXIAL_ROTATIONS; i++)
    {
        hkAxialRotation rot = hkAxialRotation::getConstant((hkAxialRotationTag)i);

        hkAabb rotatedAabb = aabb;
        rot.rotateAabb(rotatedAabb);

        hkRotation refRot;
        rot.getRotation(refRot);

        hkAabb refRotatedAabb;
        refRotatedAabb.setEmpty();

        for(int j = 0; j < 8; j++)
        {
            hkVector4 corner;
            aabb.getVertex(j, corner);

            corner.setRotatedDir(refRot, corner);
            refRotatedAabb.includePoint(corner);
        }

        HK_TEST(rotatedAabb.equals(refRotatedAabb));
    }
}

static void testSetRotation()
{
    for(int i = 0; i < HK_NUM_AXIAL_ROTATIONS; i++)
    {
        hkAxialRotation refAxialRot = hkAxialRotation::getConstant((hkAxialRotationTag)i);

        hkRotation rot;
        refAxialRot.getRotation(rot);

        hkAxialRotation axialRot;
        axialRot.setRotation(rot);

        HK_TEST(axialRot == refAxialRot);
    }
}

static void testSetRotationPerturbed()
{
    hkPseudoRandomGenerator prng(0);

    for(int i = 0; i < HK_NUM_AXIAL_ROTATIONS; i++)
    {
        hkAxialRotation refAxialRot = hkAxialRotation::getConstant((hkAxialRotationTag)i);

        hkRotation rot;
        refAxialRot.getRotation(rot);

        hkVector4 perturbAxis;
        prng.getRandomVector11(perturbAxis);
        perturbAxis.normalize<3>();

        hkRotation perturbRot;
        perturbRot.setAxisAngle(perturbAxis, prng.getRandReal11() * HK_REAL_PI / 6);

        hkRotation perturbedRot;
        perturbedRot.setMul(perturbRot, rot);

        hkAxialRotation axialRot;
        axialRot.setRotation(perturbedRot);

        HK_TEST(axialRot == refAxialRot);
    }
}

int axialRotationMain()
{
    testGetRotation();
    testSetComposition();
    testRotate90Deg();
    testRotate90DegRepeatedly();
    testSetInverse();
    testRotateAabb();
    testSetRotation();
    testSetRotationPerturbed();
    return 0;
}

HK_TEST_REGISTER(axialRotationMain, "Fast", "Common/Test/UnitTest/Base/", "UnitTest/Math/AxialRotation.cpp"     );

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
